/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree.
 */
#include <fizz/crypto/test/SignatureTestData.h>

namespace fizz::test {

// test data created using openssl cli
const std::vector<SignatureTestData>
    kSignatureTestVectors =
        {
            // 0 - valid rsa_pss_pss_sha256 test case
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::rsa_pss_sha256,
                .sig =
                    "8b197d42fe05e89476f6d584241f8a16de80e450f20a6e8922a1fc6cd140b72a16cb59512a4606840ee049c24cd04885ad9bf67b7058eb55b497ddaa83e025b5559bee96dda40da60e75fbe0039c5eb6e9dbd790c39d42af1db606d9fa4fb7afe79d6d61b449229468f84904fd9ddc5dec45b60543a840730ed162f2ca13e7db0bd5a5a7714fa2f2ffa3170b79b8d0cdeaa5f799b220f5e279c1a46bbfb80233ce3d8836aa110d5fc0375dc739ed82bcdaa5229931f172a5d214ba4f4aaaedfdf57f2f56f5ba463f4f0458bff9b69f0a7c2edc1cc2ed072bd5a3bc622d2ecda8142cbe9675c23d20a1abe5b790703409d7c53bc65c0879e0fdd3cc940677dea1",
                .validSig = true,
                .certDer =
                    "3082038930820271021412fe60cf43404b614e6c1e233efa620d3121c9bd300d06092a864886f70d01010b0500308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e74657374301e170d3234313032333233323935335a170d3235313032333233323935335a308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e7465737430820122300d06092a864886f70d01010105000382010f003082010a0282010100c47082ee102c6abc6b59c8c364a71539af5e52e03673eab65096c38418bc38560613a02d0953f3124b47f2533ebf1834ca53fde451a1a801fa3f1067ed3059c5c350f41cd681ab1443d0aeca48a6c88605e75f261180b7d8e9bf8a52a93a4ddc24b623bb6677f9ee14f6b6667916b09d512777214a283a76fb518edebb132680f47a24b7b3efc7b56746c4ae96251bff855a1b3267a8580f2a4e64fec2ed0e733e7eb937dfe7aac0b615890823c71f917b0c6efb0d0de3267d6510a94068417173df3cf5ee68155c5a226822063a2f1c78a75de0467519e72bb715ec729271a9253486caacd0a62e48b4258c4c4198ed559c6b81567e24c99928427b9938eb9b0203010001300d06092a864886f70d01010b050003820101003ddfadc14a5e317576bd5f0495080ce6f3d89dc1060fa8dec887c3abb8b8780ceac4d2d1c8016379abbed386dabc418e1fd6173c5d49cd39fe50e2495ac325af4f7f0b9f9a6197a56b0c42b2f306b7d7ed4e9d89aeb537f2d3b1057b1cbd5f0c81e687515ec94994fa73d9e10ecee99fcd72c2022ac75f9be89c0f79a90f226bfc2f20feedc0dee913d00b77e707e92ee075e14841c5854133f938418d3f230226fa30ea6f04f3da98abdcd80737a7519b8c90e9df9dab5c3739039c8f9c67997a7537339dd00630fd8c22c4fc9ce304e19d11a71a1a17b91512b062ed9bee48bea693f9c117948473728bd2d308a05c279c443e190c7c8090ef9400fc63bcd9",
                .validCert = true,
                .msg = "48656c6c6f20576f726c6421",
            },

            // 1 - valid ecdsa_secp256r1_sha256 test case
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::ecdsa_secp256r1_sha256,
                .sig =
                    "3046022100ea79c4e612c78fe9059898c20487fb0dce74cdf43cdbb9a3290862562825c5a8022100d2cf34085d91b2a57b5c862057996c63528e142d4f260c1a0c7722080abc14e4",
                .validSig = true,
                .certDer =
                    "308201fc308201a3021438989b4fd10d76c412b9f2ab4fd0a15219c5d0f0300a06082a8648ce3d040302308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e74657374301e170d3234313032333233353232395a170d3235313032333233353232395a308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e746573743059301306072a8648ce3d020106082a8648ce3d03010703420004c1498bc4669f751e83f1635e49e2649efb9a1ddf1943eb88bcea1774ece59ff73b0e23330c704c9b20d3960efa530013655dfd711037f862c1bed92e40964283300a06082a8648ce3d0403020347003044022037db03f282bcd55841650be7f0f6320c487e25773a20f4103ab9d9734c7c38a902205ee2bc22bd6d1cc7da841c6435ef0afe7f96ac82c461066a9cfc4c333e0c51da",
                .validCert = true,
                .msg =
                    "596f752061726520617765736f6d6521204b65657020676f696e672e",
            },

            // 2 - valid ecdsa_secp384r1_sha384 test case
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::ecdsa_secp384r1_sha384,
                .sig =
                    "306502306c9f88a073d6581e0a370c1f416f356a448ee390cc198aaf14e9418feaf5dff328915aae3f48e7d6d8a4f7ef6a96e514023100b2392a7130fde1daf2cd691564e854dc1481e60a10c392832ef8adb86878877ccbf3520fb468e79661235687a18acc95",
                .validSig = true,
                .certDer =
                    "30820239308201c002146ab260f5a8655743e1b4781358999f038fd0c279300a06082a8648ce3d040302308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e74657374301e170d3234313032333233353533365a170d3235313032333233353533365a308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e746573743076301006072a8648ce3d020106052b810400220362000486077eb65e38d6031d5724ad606508a1e94c65c638d8e423b659309b053068d0ae9b56dff26e0918a0be0635d0b83b9cb64d9d0f56dc5a6b3ec972fe9d7843c0caa48c0d657f0a8615f326fa07786578a0d935480889e7a9009c21b8ec2c241f300a06082a8648ce3d040302036700306402307f7691403239e8959b76daaf5bc1eb2389bad1a4ca2e57bb72a6b7ef7e04625c43595f6da70597e2df0a4974605dd8ad0230421e2c9a9aa11fa8cd349ca4b9472cce9dc61defecb29b72c7bcaab74737601aba37a53d01658ab98b2e6f55b9e03562",
                .validCert = true,
                .msg =
                    "596f752061726520617765736f6d6521204b65657020676f696e672e",
            },

            // 3 - invalid cert - rsassa-pss test case
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::rsa_pss_sha256,
                .sig =
                    "3bffb6dcfef59e3fabbda732a29f3c7364421534f5c145791ce5c118017717945501ad48facb09f1bffe061d10b10a216c040e015287ab2e85a9dd487af4845ad0f1e87c81a613dee44fb6b2ab7781d6a9de55c0b6620233dcd9e569e64d9e0fa469661021dfe520aff5e20f8d969b2dac8ac0c0339b1f3ae4e08510c7298e194154644a01b5ac61f3b86b3a4510c129dc1344e05a1151c08d2678717d4aea6211889a61d39977aa85503a18b320e8feeca8ce0dae974c3ce8d212861084806f2a04a6f916de382bc77e64b0b11bd847449ba19a863bae0873b70267f274324ee39a63fecb2b6fcbc1a197a4c17fb43b82aef8f4b50ba3b35af35db78681b5de",
                .validSig = true,
                .certDer =
                    "308203893082027102147804939721a8d913035bc9aa7d601fcc742bad300d06092a864886f70d01010b0500308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e74657374301e170d3234313032333233353734325a170d3235313032333233353734325a308180310b090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e7465737430820122300d06092a864886f70d01010105000382010f003082010a0282010100a99f6b9e289cc213c51ac23661b0b30fb961b55347a93a1d38427ef5f0c3bd980f8510b109b6d33dfb92e4cd77b72537199887c599f5d8f43525c21a94e46a24fe2a1e68d36a0f7812ba8eaec600d8a6d10079b8c825492bd01ef83bba72ec3ca9b54e697a8eb223e20e1497344d5b1fc09476ad2a4bc0a0380760362622e23fcc0c665a6ea19d93af674abd3bac10873db56afc8963b10ef3b0146ec20f9549bc5b292acbd32e52c6f62a943b7134b94ee445ddfbd37963be903d21add5bf22f923610f52f29bd25801325df156b2ad326bf404e2dbf068aa819ff00a742182670130e36eb6b7020590d0345434f2595447ff80def85e0e4897dcdb7248f5130203010001300d06092a864886f70d01010b050003820101002dbe9ad3adace7569c6d409910f67bcd4159ff1d6d03eefe32ffc35d08dfca3a5d284d448261bb43560e0d6491637a45e69367e225eff787a7576159e3a79df3cdf65a924966f0995260b19d672f65888dad883374c5c16ec6c04289e4236bfcbfd85361c698acfa0bd5319441c1ed051402339efeaccc891705b0088348d2a5171ad70b12019ae2819784692e7156088d061d95d2c65272e850d6072352d87bf47bf4aa931c8a7c06f311025539e767209a8d696c2d0c6dcebe825940e1f586d2622ed52b66919323ca4c380cabdfa1a21810c99d4b172552a7d5a7f5a7f48aaf2be4102e4e33fe5da01da70984f5921229ddb5a7efdf4a13d28e840f5b4035",
                .validCert = false,
                .msg =
                    "546865206d617374657220686173206661696c6564206d6f72652074696d6573207468616e2074686520626567696e6e657220686173206576656e2074726965642e",
            },

            // 4 - invalid signature - rsassa-pss test case
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::rsa_pss_sha256,
                .sig =
                    "af1e7d4c3c7cd94c7037dcc3b1ec522c043fb9a05f3f3f8002714f74b6fb32f3a6d1a7fc7b5a6510454df7544111a254783a376ef15543030316c8f9c3bdf119b4ffd3f9996aeb694604252d6bd67e6d4fe4ef2158896261b6696bda263cdadae43d0344a6c626ea1ea26c3571a46a9146d7985f795e3a4114687a4298bc05e0b69c9e3000ad833a135179c36f2fece7a1a6e6a74c352c843f65ad63de1641d3e310e59dd9028891c74a38f0a8891162d7b287e52bd1bd086d8dc18f477bf98dc2a036857e988d7e0f83bcca84937943bbd5debd7261700936a4f10b780356f6f90657fa34b85314eef7df2430d77f46688d214c82eb5edc6a5614cae1b4c9",
                .validSig = false,
                .certDer =
                    "30820389308202710214220b6bfcf57efea7a07fc5c1f7603ebfdd49a86f300d06092a864886f70d01010b0500308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e74657374301e170d3234313032333233353934315a170d3235313032333233353934315a308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e7465737430820122300d06092a864886f70d01010105000382010f003082010a0282010100b11f666217fa6a3376db26c357feac4dec5dcb6580fee06454c309b64932f686e69472e4796aa047f4e34c15384dd926170a57fce6ff567dfed2b5661ecd276813b3ca0022901327f9ea5bb25389d913f8aca17799cab1f8015da7e945a08f8f44a3ad95714f3a013bdb7513b5e41db77a6dbf827efa202c9cc6b8ff92a2e3cd3ccedd6f1c11c1de77fb6d17ed7e309e868211a0359a7f38d4514c6f041081265a583f5f9207c759694dacd052c30e103934e36891d684616a76c6cbc58c22b35190c2edc1b46df401867a908286dc3fcd222172db4e117e4fa0b85eb3897ec3de6cb3a20677c0fe6c45df242b6c460cc0a4ffbec1f6e7cf62f18784cbedc2390203010001300d06092a864886f70d01010b050003820101008119368d78cc19a73665d674f4be694c561a99c4a3a39fe7186844a5daf2bf100d8bedcea806b5844394f0b613bcacc5aaf211855ab369de86c4bf570fc1f07937ccbb3a47170f381cfc2a0efeb58faecbd47f60eaeabe7daea14c9e813b6254ac2c3784a5ce41a97f9023210ff9789f7545672b6a305cfe44829d8a6883f1172988488b81a659ebfd31de507034066bbf04aa7ccab7bab86fcd69e91865f1eb10fd4008ae24ca8d62fe987d95b2e1496f505d5969d30011709333aa4a87a0c4ec7881f31bf0a8e0f86da5b44cd9aa11a164d53e0c030d740277cbfc61dcea6d53745f9edfbb9c54b202a82bc8a42629bbcb6d21c589679d91fff226644b22b1",
                .validCert = true,
                .msg =
                    "546865206d617374657220686173206661696c6564206d6f72652074696d6573207468616e2074686520626567696e6e657220686173206576656e2074726965642e",
            },

            // 5 - invalid cert - ecdsa_secp256r1_sha256 test case
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::ecdsa_secp256r1_sha256,
                .sig =
                    "304502200a54cb5bd65fd24127bd590b566d99ef057e28ce15cedf4d633258c87a170f0e0221009513c0b4469ac0321bc3ee288bb7256d607c47d083d44d1d17b24fdf8c464a99",
                .validSig = true,
                .certDer =
                    "308201fc3082011468eb79dd88a7661433aa29bae26546349e868c53300a06082a8648ce3d040302308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e74657374301e170d3234313032343030303033305a170d3235313032343030303033305a308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c10746573696e672e666f6f2e746573743059301306072a8648ce3d020106082a8648ce3d0301070342000414be41dd94daf3dbe46fd503bc449a78dd12bd36957828149e5be0faf788f764d192ab3b3f878a2cefdc9e12291c458a52a328000d649369d3513b0707dcc29d300a06082a8648ce3d0403020347003044022046aa91b5d81edb04d88c6d6ccaf0e9e502976ee27625953eae56b266ce0290f802203825d9b52cb9490f7519a522d98467f7679dcc3dee0dadcca7d2f86a3308d0ee",
                .validCert = false,
                .msg =
                    "546865206d617374657220686173206661696c6564206d6f72652074696d6573207468616e2074686520626567696e6e657220686173206576656e2074726965642e",
            },

            // 6 -  invalid signature - ecdsa_secp256r1_sha256 test case
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::ecdsa_secp256r1_sha256,
                .sig =
                    "3045022100d7f9d2cb15fa01f349ecadd85ecd904f6c35f50540a837386526c8f91b02205c66f6977dae47522a3f202ed9a5cfda85adb07d92546b128e79d73f9422e497",
                .validSig = false,
                .certDer =
                    "308201fc308201a3021475b18128545490f9eb65ea148d9cbd1b77a52bcb300a06082a8648ce3d040302308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e74657374301e170d3234313032343030303230365a170d3235313032343030303230365a308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e746573743059301306072a8648ce3d020106082a8648ce3d03010703420004853483d4efd8a3529a92fcb7e2b1a1ee8b3281b2145bf5be92b37d8e503e9c17ca49e2c82bf4ed46b10e83ee5c266be6dd7e16b3c0cdae3a622c4563b8ca42ac300a06082a8648ce3d040302034700304402207cce6cb8bd489c2333cfc7dc16a1ca9c062746efd2d799793f89f6848cec496102205bceae49e1fc52c5ebd727a68ec064fc7e1ee914681351ab84be9e3fdab143e8",
                .validCert = true,
                .msg =
                    "546865206d617374657220686173206661696c6564206d6f72652074696d6573207468616e2074686520626567696e6e657220686173206576656e2074726965642e",
            },

            // 7 - invalid cert - ecdsa_secp384r1_sha384 test case
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::ecdsa_secp384r1_sha384,
                .sig =
                    "30650230695e901b8917a8de466ff3307a890a2e567f3d82fefc2c16e9eb0f20497c44c86c3f66d6b3748f9a94eeab8d38c48a3c023100f16d3f7e9d233676cf1f603f883ab196b112e875b524646fc831f4d25db64b863b8fa46eb0a747fb49f50a56169a5be2",
                .validSig = true,
                .certDer =
                    "3082023a3082c002146b26dae110621da123f8538118a72b9f3c9f27d3300a06082a8648ce3d040302308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e74657374301e170d3234313032343030303935345a170d3235313032343030303935345a308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c10747374696e672e666f6f2e746573743076301006072a8648ce3d020106052b8104002203620004b7fa1c4978959769fe6684c9c437260c33cdb203d380958f94b2d3bd88deefc6f463d01f21e995d9bd3043554933f195747e64b6c877157df7028f54aac436ed7fb4bc06ab0b06eccf3dad7cd7ecf247927b3e2ec651f2c14db8bef3b5ab01d4300a06082a8648ce3d0403020368003065023100b72ac1dd1fdd9edd2cd7683192ec35ef21320241d5573b54fc2b6170ba121c8d6f53e31787d9b4cae0c679a654bce429023003e678716c16080a7db22a55677360f6f22a6acd453befe77329e988d91d43cf27839cef60bf7c731b0ea73e3b814f73",
                .validCert = false,
                .msg =
                    "546865206d617374657220686173206661696c6564206d6f72652074696d6573207468616e2074686520626567696e6e657220686173206576656e2074726965642e",
            },

            // 8 - invalid signature - ecdsa_secp384r1_sha384 test case
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::ecdsa_secp384r1_sha384,
                .sig =
                    "30640230627ed1954eb5b6bee18559d3b63ec9ef6be0b6b739b2d17afc458daacace71f7780670504fa16b357b9f15b67f27c0023055931a2a1334e08ebf2231a2ac0b13b43360c6230a052e89d181660a25515f18a28b2e73ace6666e267f69b5676a0651",
                .validSig = false,
                .certDer =
                    "3082023a308201c0021423bbd56718a9c006feb21eaab16bf0c423727da6300a06082a8648ce3d040302308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e74657374301e170d3234313032343030313130375a170d3235313032343030313130375a308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e746573743076301006072a8648ce3d020106052b8104002203620004c7bccc95bd4b1ff192011aac1d189414ab9cb08904d7f613cc0161f5ea4ee5ba04ace2ab300744ed7a3439ed9ce3b85df9b6d889beefbc9d0a196e386d6951e1f1d02952349908e4141c177bf51e2011a1bb33fdae436868ba7fca73017b2f35300a06082a8648ce3d04030203680030650231008b270435399bf1aa2c215c5ff72d1ced89022158094a4a448ff9fee0e1ece4fc6c45a572feec95f4383f32df304c119a02307d1917c82770e2267aa34d9ac7f28e931297b5675820173037a166f95d09516ed57fe0b4ced566095a1aa44be85d1c52",
                .validCert = true,
                .msg =
                    "546865206d617374657220686173206661696c6564206d6f72652074696d6573207468616e2074686520626567696e6e657220686173206576656e2074726965642e",
            },

            // 9 - rsa_pss_pss_sha256 with short signature
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::rsa_pss_sha256,
                .sig = "01",
                .validSig = false,
                .certDer =
                    "308203533082023b0214643747549d8066efac36572cf70d19e1d8306ab9300d06092a864886f70d01010b05003066310b30090603550406130255533113301106035504080c0a43616c69666f726e69613115301306035504070c0c44656661756c742043697479311c301a060355040a0c1344656661756c7420436f6d70616e79204c7464310d300b06035504030c0474657374301e170d3234303932363233353533375a170d3235303932363233353533375a3066310b30090603550406130255533113301106035504080c0a43616c69666f726e69613115301306035504070c0c44656661756c742043697479311c301a060355040a0c1344656661756c7420436f6d70616e79204c7464310d300b06035504030c047465737430820122300d06092a864886f70d01010105000382010f003082010a02820101008a6c4d2b1d7a4843df11bd545ad7658e8a1d6b961fbafbf94b1f8118d0885d83a320be6875d9f2c231a154fc9aa80849d6773f914b2fe279a67503db221b7215c8983e92bfeb16e6077ed5d1f47eaca1bd83e545276a79021ec43590b2b17283256fec5c8a8fbbe2642c7413236d0e11228406ddf2e7f36562f2c51d79892c1751b22b5588985f51eb21636e83aae01809bf847b501c79353879829ab0f2bc307894b1297be51ff5c15f3109413258071fd6e80216ab3856e8133a18ed05e142da8d90b7fb4c3bd08fd7e68f399e22fdc499287ae20ae9c88c059b41af9c60b570b73d041233ff4c938bc06d94a906c823333b971c900f50d5e731f8dd9640fb0203010001300d06092a864886f70d01010b05000382010100225d83bca5171d13b7034e34ec23116c6213547acf385177b25236cf3a8e6a49c1723dc1e11fa545379975ef72dcdda2b3442251d764c20c8d0ddc033aa8aa65591a9529bb67445839c608ba1c3c462f69548d72b317c5a6bbd6e63bfb804aded926ba9bb5db1db0421d8f658b0046622d0d9369d1eea7d84dff27656d2789b658deab9c0703af5381123e8714d2f8eeff7c0f8fda6ec1190e25e545c6c0acf415d9cc053a96880fa30de62b0d2486c40eb3fe575ff342d0540853001eee1b4a5d04e26aa248fe3544ea9eba9af2ef47ad079a95293b89b596021b744d65578d020d980ec5ecdc7ad496e17e3b99a3883defa3ab6a4e5a78cd5e4d1343e85a02",
                .validCert = true,
                .msg =
                    "426520796f757273656c66203a292054686520776f726c6420697320612062657474657220706c616365207769746820796f7520696e2069742e",
            },

            // 10 - ecdsa_secp256r1_sha256 with short signature
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::ecdsa_secp256r1_sha256,
                .sig = "01",
                .validSig = false,
                .certDer =
                    "308201f33082019902146319d6fd924a04fb0811d41f851256f44da86bfe300a06082a8648ce3d040302307c310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e310d300b060355040b0c04546573743118301606035504030c0f746573742e646f6d61696e2e666f6f301e170d3234303933303231303233375a170d3235303933303231303233375a307c310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e310d300b060355040b0c04546573743118301606035504030c0f746573742e646f6d61696e2e666f6f3059301306072a8648ce3d020106082a8648ce3d0301070342000462a9238aaf2d52630b64325ddf600009592d9144848ddb9c349b83750363e6177c4ca51ab865b66401e6fa720148183a2dde815dad0bf6fcf43c49f9b0985594300a06082a8648ce3d040302034800304502210097eddaa3ddf5c6a3f7e993413252bd7c9c2b04f734d0824f9548aa3724b6a4ff02200be7b4a218285a96c83a18ec22b80cad27d99781b08cdf3fc1c8dd96e10435e9",
                .validCert = true,
                .msg =
                    "426520796f757273656c66203a292054686520776f726c6420697320612062657474657220706c616365207769746820796f7520696e2069742e",
            },

            // 11 - ecdsa_secp384r1_sha384 with short signature
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::ecdsa_secp384r1_sha384,
                .sig = "01",
                .validSig = false,
                .certDer =
                    "30820230308201b602144789a43d59094e62d0a185f3f389fe23dce66996300a06082a8648ce3d040302307c310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e310d300b060355040b0c04546573743118301606035504030c0f746573742e646f6d61696e2e666f6f301e170d3234303933303231303834305a170d3235303933303231303834305a307c310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e310d300b060355040b0c04546573743118301606035504030c0f746573742e646f6d61696e2e666f6f3076301006072a8648ce3d020106052b81040022036200049aed07504902c74140e63dd62e1d66fdca4cecc89769d4f99b16cf5d5976d07f6dd1f7ea5e29fa1f485bacff61ab8d2b6efab6fcca5c2f4866c553f94f5777c3413459bae774005c3e49f933726c448bbb3c0a632e1a7fd8eacb9887ac31ce0e300a06082a8648ce3d04030203680030650231009624cb719e4938c0d3820abbace751601716e57592e6b6f6df8bb7ca2f8eed9843f3599f379d1c621f82f9e371ffee190230634387dc6c396373382f940a9f65f67261294d7e4208fd6a2da96fb54ac11588ab72760f877e32007be0f5f74ee46c50",
                .validCert = true,
                .msg =
                    "426520796f757273656c66203a292054686520776f726c6420697320612062657474657220706c616365207769746820796f7520696e2069742e",
            },

            // 12 - mismatched key: ecdsa_secp384r1_sha384 with a p256 key
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::ecdsa_secp384r1_sha384,
                .sig =
                    "3065023041418c5758af36b904ab320f12f807414ea389e3a028e89807b28b9e7903cfad32fd23297f3427fdd437885ac64cc6c7023100e539230235adf03e81e112fba9cef44c2f1aec77006446bfd031b3482fcb513eede661135471f2e70bc9db8d7aaf846a",
                .validSig = false,
                .certDer =
                    "308201fe308201a30214467b6fc2a8e7c64929a64fa8a0f8d4445b73065f300a06082a8648ce3d040302308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e74657374301e170d3234313032343030313931345a170d3235313032343030313931345a308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e746573743059301306072a8648ce3d020106082a8648ce3d030107034200040b548a0b07004db5804f8fde9ec3863a27cccf56cd296004bc880c083e602e7459947b2cb7ff7372b7216768ffe61e87a0f2fd6fa59ae1d2fae5b893b827a1ab300a06082a8648ce3d0403020349003046022100ada731e4ff6bb4d7013d5083d5ba6fc605fbcc6732d7ca161873dcafa52ce93d0221008dda565e71f4a9623580934aece778a95910370bfc6359bedb53742ab091ab4e",
                .validCert = true,
                .msg = "466f6c6c6f7720796f757220647265616d7321",
            },

            // 13 - mismatched key: rsa_pss_sha256 with an ecdsa key
            SignatureTestData{
                .sigScheme = fizz::SignatureScheme::rsa_pss_sha256,
                .sig =
                    "d7e50433f47be0efc28e233dce086f6b4ef5cd057be41989c0dd9f104a478cef612c39ea45f035f9343a2e56340a6b0e4263097a0e8bb0997d056bcae4a2dbac7eab5ce6c08a5176c54a6d2ed0979f7d9ae8d16802578fa1991c88e9b8fc1697018e5a80910ac4705dd128bba6da0db1b6639b95d92632b15950b8ea3b030f580088a53f5a4048fb92e2ba0eacabe16ac3300586fb70c92e795ff7dedb55aee1f4e7c387a3108fef231fe749e6341ec35af9b0ff3afcb49e3f24edaad5fa1d0a154f8b6c6be394d9ec688d8986c39b2a1b4d2131e774fc6609d000bf068d4c0c8e79c195539bed87d09640e7c06747f13e4abb3af1bf0d3bfe6257030f765d0f",
                .validSig = false,
                .certDer =
                    "308201fd308201a30214014edbf8b3e105d3641b90d47ad243ebc80e1c60300a06082a8648ce3d040302308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e74657374301e170d3234313032343030323233305a170d3235313032343030323233305a308180310b30090603550406130255533113301106035504080c0a43616c69666f726e69613113301106035504070c0a4d656e6c6f205061726b311a3018060355040a0c11596f7572204f7267616e697a6174696f6e3110300e060355040b0c07546573744f72673119301706035504030c1074657374696e672e666f6f2e746573743059301306072a8648ce3d020106082a8648ce3d03010703420004828c1d65790a589ff09009663ed9c38b9247ca7da76f3b7401552c5bdd67814442611bf039e7de30b4a459a852578d8e613db27beed481c2edde3c625cf35aec300a06082a8648ce3d040302034800304502202c5b56f81905a43e0cb2df3e297c758dcde9d1e469fecd5f02308c63d3e1a55f022100a549d6bd6075c114738c5cd7023057d12070a82df42937be45ca65b148159c1d",
                .validCert = true,
                .msg = "466f6c6c6f7720796f757220647265616d7321",
            },
};
} // namespace fizz::test
