// Copyright 2020 The Gitea Authors. All rights reserved.
// SPDX-License-Identifier: MIT

package interact

import (
	"fmt"
	"os"

	"code.gitea.io/sdk/gitea"
	"code.gitea.io/tea/cmd/flags"
	"code.gitea.io/tea/modules/context"
	"code.gitea.io/tea/modules/print"
	"code.gitea.io/tea/modules/theme"

	"github.com/charmbracelet/huh"
	"golang.org/x/term"
)

// ShowCommentsMaybeInteractive fetches & prints comments, depending on the --comments flag.
// If that flag is unset, and output is not piped, prompts the user first.
func ShowCommentsMaybeInteractive(ctx *context.TeaContext, idx int64, totalComments int) error {
	if ctx.Bool("comments") {
		opts := gitea.ListIssueCommentOptions{ListOptions: flags.GetListOptions()}
		c := ctx.Login.Client()
		comments, _, err := c.ListIssueComments(ctx.Owner, ctx.Repo, idx, opts)
		if err != nil {
			return err
		}
		print.Comments(comments)
	} else if print.IsInteractive() && !ctx.IsSet("comments") {
		// if we're interactive, but --comments hasn't been explicitly set to false
		if err := ShowCommentsPaginated(ctx, idx, totalComments); err != nil {
			fmt.Printf("error while loading comments: %v\n", err)
		}
	}
	return nil
}

// ShowCommentsPaginated prompts if issue/pr comments should be shown and continues to do so.
func ShowCommentsPaginated(ctx *context.TeaContext, idx int64, totalComments int) error {
	c := ctx.Login.Client()
	opts := gitea.ListIssueCommentOptions{ListOptions: flags.GetListOptions()}
	prompt := "show comments?"
	commentsLoaded := 0

	// paginated fetch
	// NOTE: as of gitea 1.13, pagination is not provided by this endpoint, but handles
	// this function gracefully anyways.
	for {
		loadComments := true
		if err := huh.NewConfirm().
			Title(prompt).
			Value(&loadComments).
			WithTheme(theme.GetTheme()).
			Run(); err != nil {
			return err
		} else if !loadComments {
			break
		} else {
			if comments, _, err := c.ListIssueComments(ctx.Owner, ctx.Repo, idx, opts); err != nil {
				return err
			} else if len(comments) != 0 {
				print.Comments(comments)
				commentsLoaded += len(comments)
			}
			if commentsLoaded >= totalComments {
				break
			}
			opts.ListOptions.Page++
			prompt = "load more?"
		}
	}
	return nil
}

// IsStdinPiped checks if stdin is piped
func IsStdinPiped() bool {
	return !term.IsTerminal(int(os.Stdin.Fd()))
}
