#include <errno.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>
#ifdef	IPV6
#include <netinet/ip6.h>
#endif // IPV6
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <time.h>

#include "f_ping.h"

int	datalen = 56;		/* data that goes with ICMP echo request */


extern unsigned short packets;

int f_ping(char* hostname)
{

    struct proto	proto_v4 = { proc_v4, send_v4, NULL, NULL, 0, IPPROTO_ICMP, 0, 0};
#ifdef	IPV6
    struct proto	proto_v6 = { proc_v6, send_v6, NULL, NULL, 0, IPPROTO_ICMPV6, 0, 0};
#endif

    struct addrinfo	*ai;
    int  n;
    struct addrinfo hints;

    nsent = 0;

    pid = getpid();

    bzero(&hints, sizeof(struct addrinfo));
    hints.ai_flags = AI_CANONNAME;	/* always return canonical name */
    hints.ai_family = 0;		/* 0, AF_INET, AF_INET6, etc. */
    hints.ai_socktype = 0;	/* 0, SOCK_STREAM, SOCK_DGRAM, etc. */

    if ( (n = getaddrinfo(hostname, NULL, &hints, &ai)) != 0)
    {
    //    err_quit("host_serv error for %s, %s: %s",
    //    	 (host == NULL) ? "(no hostname)" : host,
    //    	 (serv == NULL) ? "(no service name)" : serv,
	//    	 gai_strerror(n));

	return 0;
    }
#ifdef DBG
    printf("PING %s (%s): %d data bytes\n", ai->ai_canonname,
	   sock_ntop_host(ai->ai_addr, ai->ai_addrlen), datalen);
#endif
    /* 4initialize according to protocol */
    if (ai->ai_family == AF_INET) {
	pr = &proto_v4;
#ifdef	IPV6
    } else if (ai->ai_family == AF_INET6) {
	pr = &proto_v6;
	if (IN6_IS_ADDR_V4MAPPED(&(((struct sockaddr_in6 *)
				    ai->ai_addr)->sin6_addr)))
	{
	    //   err_quit("cannot ping IPv4-mapped IPv6 address");
            return 0;
	}
#endif
    }
    else
    {
	//   err_quit("unknown address family %d", ai->ai_family);
        return 0;
    }

    pr->sasend = ai->ai_addr;
    pr->sarecv = (struct sockaddr*)calloc(1, ai->ai_addrlen);
    pr->salen = ai->ai_addrlen;

    stopPing = 0;

    signal(SIGALRM, sig_alrm);

    return readloop();
}

void sig_alrm(int signo)
{
    (*pr->fsend)();
    if(pr->n_send < packets)
    {
	alarm(1);
    }
    else
    {
/*	struct timespec req={3, 0};
	struct timespec rem;
	int ret = 1;
	while(ret)
	{
	    ret = nanosleep(&req, &rem);
	    if(ret == -1)
	    {
		if( errno == EINTR)
		{
		    nanosleep(&rem, &rem);
		}
		else
		{
		    break;
		}
	    }
        }*/
	stopPing = 1;
    }
    return;		/* probably interrupts recvfrom() */
}

int readloop(void)
{
    int				size;
    char			recvbuf[BUFSIZE];
    socklen_t		len;
    ssize_t			n;
    struct timeval	tval;
    fd_set rset;
    struct timeval timeout;
    sockfd = socket(pr->sasend->sa_family, SOCK_RAW, pr->icmpproto);

    if(sockfd == -1)
    {
	//error
        return 0;
    }
    //    setuid(getuid());		/* don't need special permissions any more */

    size = 60 * 1024;		/* OK if setsockopt fails */
    setsockopt(sockfd, SOL_SOCKET, SO_RCVBUF, &size, sizeof(size));

    //sig_alrm(SIGALRM);		/* send first packet */

    (*pr->fsend)();

    FD_ZERO(&rset);
    FD_SET(sockfd, &rset);

    while(!stopPing || n)
    {
	timeout.tv_sec = 3;
	timeout.tv_usec = 0;
	n = select(sockfd + 1, &rset, NULL, NULL, &timeout);
	if(n)
	{
	    len = pr->salen;
	    if (FD_ISSET(sockfd, &rset))
	    {
		n = recvfrom(sockfd, recvbuf, sizeof(recvbuf), 0, pr->sarecv, &len);
		if (n < 0)
		{
		    if (errno == EINTR)
		    {
			continue;
		    }
		    else
		    {
			//   err_sys("recvfrom error");
			return 0;
		    }
		}
		gettimeofday(&tval, NULL);
		(*pr->fproc)(recvbuf, n, &tval);
	    }
	}

	if(pr->n_send < packets)
	{
	    (*pr->fsend)();
	}
	else
	{
	    stopPing = 1;
	}
    }

    close(sockfd);
#ifdef DBG
    printf("s = %lu;   r = %lu\n", pr->n_send, pr->n_recv);
#endif
    return 100*pr->n_recv/pr->n_send;
}

void proc_v4(char *ptr, ssize_t len, struct timeval *tvrecv)
{
    int				hlen1, icmplen;
    double			rtt;
    struct ip		*ip;
    struct icmp		*icmp;
    struct timeval	*tvsend;

    ip = (struct ip *) ptr;		/* start of IP header */
    hlen1 = ip->ip_hl << 2;		/* length of IP header */

    icmp = (struct icmp *) (ptr + hlen1);	/* start of ICMP header */
    if ( (icmplen = len - hlen1) < 8)
    {
	//err_quit("icmplen (%d) < 8", icmplen);
        return ;

    }

    if (icmp->icmp_type == ICMP_ECHOREPLY) {
	if (icmp->icmp_id != pid)
	    return;			/* not a response to our ECHO_REQUEST */
	if (icmplen < 16)
	{
	    //err_quit("icmplen (%d) < 16", icmplen);
            return ;
	}

	pr->n_recv++;

	tvsend = (struct timeval *) icmp->icmp_data;
	tv_sub(tvrecv, tvsend);
	rtt = tvrecv->tv_sec * 1000.0 + tvrecv->tv_usec / 1000.0;
#ifdef DBG
	printf("%d bytes from %s: seq=%u, ttl=%d, rtt=%.3f ms\n",
	       icmplen, sock_ntop_host(pr->sarecv, pr->salen),
	       icmp->icmp_seq, ip->ip_ttl, rtt);
#endif
    }
    else if (verbose)
    {
#ifdef DBG
	printf("  %d bytes from %s: type = %d, code = %d\n",
	       icmplen, sock_ntop_host(pr->sarecv, pr->salen),
	       icmp->icmp_type, icmp->icmp_code);
#endif
    }
}

void proc_v6(char *ptr, ssize_t len, struct timeval* tvrecv)
{
#ifdef	IPV6
	int					hlen1, icmp6len;
	double				rtt;
	struct ip6_hdr		*ip6;
	struct icmp6_hdr	*icmp6;
	struct timeval		*tvsend;

	ip6 = (struct ip6_hdr *) ptr;		/* start of IPv6 header */
	hlen1 = sizeof(struct ip6_hdr);
	if (ip6->ip6_nxt != IPPROTO_ICMPV6)
	{
	    //err_quit("next header not IPPROTO_ICMPV6");
            return ;
	}

	pr->n_recv++;

	icmp6 = (struct icmp6_hdr *) (ptr + hlen1);
	if ( (icmp6len = len - hlen1) < 8)
	{
	    //   err_quit("icmp6len (%d) < 8", icmp6len);
            return ;
	}

	if (icmp6->icmp6_type == ICMP6_ECHO_REPLY) {
		if (icmp6->icmp6_id != pid)
			return;			/* not a response to our ECHO_REQUEST */
		if (icmp6len < 16)
		{
		    //   err_quit("icmp6len (%d) < 16", icmp6len);
                    return;
		}

		tvsend = (struct timeval *) (icmp6 + 1);
		tv_sub(tvrecv, tvsend);
		rtt = tvrecv->tv_sec * 1000.0 + tvrecv->tv_usec / 1000.0;
#ifdef DBG
		printf("%d bytes from %s: seq=%u, hlim=%d, rtt=%.3f ms\n",
				icmp6len, sock_ntop_host(pr->sarecv, pr->salen),
				icmp6->icmp6_seq, ip6->ip6_hlim, rtt);
#endif
	} else if (verbose)
	{
#ifdef DBG
	    printf("  %d bytes from %s: type = %d, code = %d\n",
				icmp6len, sock_ntop_host(pr->sarecv, pr->salen),
		   icmp6->icmp6_type, icmp6->icmp6_code);
#endif
	}
#endif	/* IPV6 */
}

char *sock_ntop_host(const struct sockaddr *sa, socklen_t salen)
{
    static char str[128];		/* Unix domain is largest */

	switch (sa->sa_family) {
	case AF_INET: {
		struct sockaddr_in	*sin = (struct sockaddr_in *) sa;

		if (!inet_ntop(AF_INET, &sin->sin_addr, str, sizeof(str)))
			return(NULL);
		return(str);
	}

#ifdef	IPV6
	case AF_INET6: {
		struct sockaddr_in6	*sin6 = (struct sockaddr_in6 *) sa;

		if (inet_ntop(AF_INET6, &sin6->sin6_addr, str, sizeof(str)) == NULL)
			return(NULL);
		return(str);
	}
#endif

	default:
		snprintf(str, sizeof(str), "sock_ntop_host: unknown AF_xxx: %d, len %d",
				 sa->sa_family, salen);
		return(str);
	}
    return (NULL);
}
void tv_sub(struct timeval *out, struct timeval *in)
{
	if ( (out->tv_usec -= in->tv_usec) < 0) {	/* out -= in */
		--out->tv_sec;
		out->tv_usec += 1000000;
	}
	out->tv_sec -= in->tv_sec;
}
void send_v4(void)
{
	int			len;
	struct icmp	*icmp;

	icmp = (struct icmp *) sendbuf;
	icmp->icmp_type = ICMP_ECHO;
	icmp->icmp_code = 0;
	icmp->icmp_id = pid;
	icmp->icmp_seq = nsent++;
	gettimeofday((struct timeval *) icmp->icmp_data, NULL);

	len = 8 + datalen;		/* checksum ICMP header and data */
	icmp->icmp_cksum = 0;
	icmp->icmp_cksum = in_cksum((u_short *) icmp, len);

	sendto(sockfd, sendbuf, len, 0, pr->sasend, pr->salen);
        pr->n_send++;
}

void send_v6()
{
#ifdef	IPV6
	int					len;
	struct icmp6_hdr	*icmp6;

	icmp6 = (struct icmp6_hdr *) sendbuf;
	icmp6->icmp6_type = ICMP6_ECHO_REQUEST;
	icmp6->icmp6_code = 0;
	icmp6->icmp6_id = pid;
	icmp6->icmp6_seq = nsent++;
	gettimeofday((struct timeval *) (icmp6 + 1), NULL);

	len = 8 + datalen;		/* 8-byte ICMPv6 header */

	sendto(sockfd, sendbuf, len, 0, pr->sasend, pr->salen);
	pr->n_send++;
	/* 4kernel calculates and stores checksum for us */
#endif	/* IPV6 */
}

unsigned short in_cksum(unsigned short *addr, int len)
{
	int				nleft = len;
	int				sum = 0;
	unsigned short	*w = addr;
	unsigned short	answer = 0;

	/*
	 * Our algorithm is simple, using a 32 bit accumulator (sum), we add
	 * sequential 16 bit words to it, and at the end, fold back all the
	 * carry bits from the top 16 bits into the lower 16 bits.
	 */
	while (nleft > 1)  {
		sum += *w++;
		nleft -= 2;
	}

		/* 4mop up an odd byte, if necessary */
	if (nleft == 1) {
		*(unsigned char *)(&answer) = *(unsigned char *)w ;
		sum += answer;
	}

		/* 4add back carry outs from top 16 bits to low 16 bits */
	sum = (sum >> 16) + (sum & 0xffff);	/* add hi 16 to low 16 */
	sum += (sum >> 16);			/* add carry */
	answer = ~sum;				/* truncate to 16 bits */
	return(answer);
}
