// Copyright (c) 2019, Google Inc.
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
// SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
// OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
// CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

#include <openssl/blowfish.h>
#include <openssl/cipher.h>

#include <gtest/gtest.h>

#include "../../internal.h"
#include "../../test/test_util.h"

struct BlowfishTestCase {
  uint8_t key[16];
  uint8_t plaintext[16];
  uint8_t iv[8];
  uint8_t ecb_ciphertext[16];
  uint8_t cbc_ciphertext[24];
  uint8_t cfb_ciphertext[16];
  uint8_t ofb_ciphertext[16];
};

static const BlowfishTestCase kTests[] = {
    // Randomly generated test cases. Checked against vanilla OpenSSL.
    {
        {0xbb, 0x56, 0xb1, 0x27, 0x7c, 0x4c, 0xdd, 0x5a, 0x99, 0x90, 0x1e, 0x6f,
         0xeb, 0x36, 0x6c, 0xf3},
        {0xa6, 0x5b, 0xe0, 0x99, 0xad, 0x5d, 0x91, 0x98, 0x37, 0xc1, 0xa4, 0x7f,
         0x01, 0x24, 0x9a, 0x6b},
        {0xd5, 0x8a, 0x5c, 0x29, 0xeb, 0xee, 0xed, 0x76},
        {0xda, 0x6e, 0x18, 0x9c, 0x03, 0x59, 0x12, 0x61, 0xfa, 0x20, 0xd9, 0xce,
         0xee, 0x43, 0x9d, 0x05},
        {0x4f, 0x8b, 0x3e, 0x17, 0xa5, 0x35, 0x9b, 0xcb,
         0xdf, 0x3c, 0x52, 0xfb, 0xe5, 0x20, 0xdd, 0x53,
         0xd5, 0xf8, 0x1a, 0x6c, 0xf0, 0x99, 0x34, 0x94},
        {0xfd, 0x65, 0xc5, 0xa6, 0x07, 0x07, 0xb5, 0xf3, 0x2e, 0xfb, 0x12, 0xc3,
         0x7f, 0x45, 0x37, 0x54},
        {0xfd, 0x65, 0xc5, 0xa6, 0x07, 0x07, 0xb5, 0xf3, 0x3a, 0x27, 0x62, 0xbe,
         0xfe, 0xb8, 0x14, 0x91},
    },
    {
        {0x5d, 0x98, 0xa9, 0xd2, 0x27, 0x5d, 0xc8, 0x8c, 0x8c, 0xee, 0x23, 0x7f,
         0x8e, 0x2b, 0xd4, 0x8d},
        {0x60, 0xec, 0x31, 0xda, 0x25, 0x07, 0x02, 0x14, 0x84, 0x44, 0x96, 0xa6,
         0x04, 0x81, 0xca, 0x4e},
        {0x96, 0x4c, 0xa4, 0x07, 0xee, 0x1c, 0xd1, 0xfb},
        {0x83, 0x8a, 0xef, 0x18, 0x53, 0x96, 0xec, 0xf3, 0xf4, 0xd9, 0xe8, 0x4b,
         0x2c, 0x3f, 0xe7, 0x27},
        {0xad, 0x78, 0x70, 0x06, 0x2e, 0x5e, 0xa5, 0x21,
         0xdd, 0xe8, 0xa0, 0xb9, 0xdb, 0x0c, 0x81, 0x1d,
         0x0a, 0xd3, 0xa9, 0x63, 0x78, 0xac, 0x82, 0x64},
        {0x43, 0x2f, 0xf3, 0x23, 0xf4, 0x5c, 0xbf, 0x05, 0x53, 0x3c, 0x9e, 0xd6,
         0xd3, 0xd2, 0xc0, 0xf0},
        {0x43, 0x2f, 0xf3, 0x23, 0xf4, 0x5c, 0xbf, 0x05, 0xeb, 0x7e, 0xde, 0xc5,
         0xd7, 0xe4, 0xf4, 0x3e},
    },
};

TEST(Blowfish, ECB) {
  unsigned test_num = 0;
  for (const auto &test : kTests) {
    test_num++;
    SCOPED_TRACE(test_num);

    uint8_t out[sizeof(test.ecb_ciphertext)];
    int out_bytes, final_bytes;

    bssl::ScopedEVP_CIPHER_CTX ctx;
    ASSERT_TRUE(EVP_EncryptInit_ex(ctx.get(), EVP_bf_ecb(), nullptr, test.key,
                                   nullptr));
    ASSERT_TRUE(EVP_CIPHER_CTX_set_padding(ctx.get(), 0 /* no padding */));
    ASSERT_TRUE(EVP_EncryptUpdate(ctx.get(), out, &out_bytes, test.plaintext,
                                  sizeof(test.plaintext)));
    ASSERT_TRUE(EVP_EncryptFinal_ex(ctx.get(), out + out_bytes, &final_bytes));
    EXPECT_EQ(static_cast<size_t>(out_bytes + final_bytes),
              sizeof(test.plaintext));
    EXPECT_EQ(Bytes(test.ecb_ciphertext), Bytes(out));

    bssl::ScopedEVP_CIPHER_CTX decrypt_ctx;
    ASSERT_TRUE(EVP_DecryptInit_ex(decrypt_ctx.get(), EVP_bf_ecb(), nullptr,
                                   test.key, nullptr));
    ASSERT_TRUE(
        EVP_CIPHER_CTX_set_padding(decrypt_ctx.get(), 0 /* no padding */));
    ASSERT_TRUE(EVP_DecryptUpdate(decrypt_ctx.get(), out, &out_bytes,
                                  test.ecb_ciphertext,
                                  sizeof(test.ecb_ciphertext)));
    ASSERT_TRUE(
        EVP_DecryptFinal_ex(decrypt_ctx.get(), out + out_bytes, &final_bytes));
    EXPECT_EQ(static_cast<size_t>(out_bytes + final_bytes),
              sizeof(test.plaintext));
    EXPECT_EQ(Bytes(test.plaintext), Bytes(out));
  }
}

TEST(Blowfish, CBC) {
  unsigned test_num = 0;
  for (const auto &test : kTests) {
    test_num++;
    SCOPED_TRACE(test_num);

    uint8_t out[sizeof(test.cbc_ciphertext)];
    int out_bytes, final_bytes;

    bssl::ScopedEVP_CIPHER_CTX ctx;
    ASSERT_TRUE(EVP_EncryptInit_ex(ctx.get(), EVP_bf_cbc(), nullptr, test.key,
                                   test.iv));
    ASSERT_TRUE(EVP_EncryptUpdate(ctx.get(), out, &out_bytes, test.plaintext,
                                  sizeof(test.plaintext)));
    EXPECT_TRUE(EVP_EncryptFinal_ex(ctx.get(), out + out_bytes, &final_bytes));
    EXPECT_EQ(static_cast<size_t>(out_bytes + final_bytes),
              sizeof(test.cbc_ciphertext));
    EXPECT_EQ(Bytes(test.cbc_ciphertext), Bytes(out));

    bssl::ScopedEVP_CIPHER_CTX decrypt_ctx;
    ASSERT_TRUE(EVP_DecryptInit_ex(decrypt_ctx.get(), EVP_bf_cbc(), nullptr,
                                   test.key, test.iv));
    ASSERT_TRUE(EVP_DecryptUpdate(decrypt_ctx.get(), out, &out_bytes,
                                  test.cbc_ciphertext,
                                  sizeof(test.cbc_ciphertext)));
    EXPECT_TRUE(
        EVP_DecryptFinal_ex(decrypt_ctx.get(), out + out_bytes, &final_bytes));
    EXPECT_EQ(static_cast<size_t>(out_bytes + final_bytes),
              sizeof(test.plaintext));
    EXPECT_EQ(Bytes(test.plaintext), Bytes(out, out_bytes + final_bytes));
  }
}

TEST(Blowfish, CFB) {
  unsigned test_num = 0;
  for (const auto &test : kTests) {
    test_num++;
    SCOPED_TRACE(test_num);

    uint8_t out[sizeof(test.cfb_ciphertext)];
    int out_bytes, final_bytes;

    bssl::ScopedEVP_CIPHER_CTX ctx;
    ASSERT_TRUE(EVP_EncryptInit_ex(ctx.get(), EVP_bf_cfb(), nullptr, test.key,
                                   test.iv));
    ASSERT_TRUE(EVP_EncryptUpdate(ctx.get(), out, &out_bytes, test.plaintext,
                                  sizeof(test.plaintext)));
    ASSERT_TRUE(EVP_EncryptFinal_ex(ctx.get(), out + out_bytes, &final_bytes));
    EXPECT_EQ(static_cast<size_t>(out_bytes + final_bytes),
              sizeof(test.plaintext));
    EXPECT_EQ(Bytes(test.cfb_ciphertext), Bytes(out));

    bssl::ScopedEVP_CIPHER_CTX decrypt_ctx;
    ASSERT_TRUE(EVP_DecryptInit_ex(decrypt_ctx.get(), EVP_bf_cfb(), nullptr,
                                   test.key, test.iv));
    ASSERT_TRUE(EVP_DecryptUpdate(decrypt_ctx.get(), out, &out_bytes,
                                  test.cfb_ciphertext,
                                  sizeof(test.cfb_ciphertext)));
    ASSERT_TRUE(
        EVP_DecryptFinal_ex(decrypt_ctx.get(), out + out_bytes, &final_bytes));
    EXPECT_EQ(static_cast<size_t>(out_bytes + final_bytes),
              sizeof(test.plaintext));
    EXPECT_EQ(Bytes(test.plaintext), Bytes(out));
  }
}

TEST(Blowfish, CFB64) {
  unsigned test_num = 0;
  for (const auto &test : kTests) {
    test_num++;
    SCOPED_TRACE(test_num);

    uint8_t out[sizeof(test.cfb_ciphertext)];
    int out_bytes, final_bytes;

    bssl::ScopedEVP_CIPHER_CTX ctx;
    ASSERT_TRUE(EVP_EncryptInit_ex(ctx.get(), EVP_bf_cfb64(), nullptr, test.key,
                                   test.iv));
    ASSERT_TRUE(EVP_EncryptUpdate(ctx.get(), out, &out_bytes, test.plaintext,
                                  sizeof(test.plaintext)));
    ASSERT_TRUE(EVP_EncryptFinal_ex(ctx.get(), out + out_bytes, &final_bytes));
    EXPECT_EQ(static_cast<size_t>(out_bytes + final_bytes),
              sizeof(test.plaintext));
    EXPECT_EQ(Bytes(test.cfb_ciphertext), Bytes(out));

    bssl::ScopedEVP_CIPHER_CTX decrypt_ctx;
    ASSERT_TRUE(EVP_DecryptInit_ex(decrypt_ctx.get(), EVP_bf_cfb64(), nullptr,
                                   test.key, test.iv));
    ASSERT_TRUE(EVP_DecryptUpdate(decrypt_ctx.get(), out, &out_bytes,
                                  test.cfb_ciphertext,
                                  sizeof(test.cfb_ciphertext)));
    ASSERT_TRUE(
        EVP_DecryptFinal_ex(decrypt_ctx.get(), out + out_bytes, &final_bytes));
    EXPECT_EQ(static_cast<size_t>(out_bytes + final_bytes),
              sizeof(test.plaintext));
    EXPECT_EQ(Bytes(test.plaintext), Bytes(out));
  }
}

TEST(Blowfish, OFB) {
  unsigned test_num = 0;
  for (const auto &test : kTests) {
    test_num++;
    SCOPED_TRACE(test_num);

    uint8_t out[sizeof(test.ofb_ciphertext)];
    int out_bytes = 0, final_bytes = 0;

    bssl::ScopedEVP_CIPHER_CTX ctx;
    ASSERT_TRUE(EVP_EncryptInit_ex(ctx.get(), EVP_bf_ofb(), nullptr, test.key,
                                   test.iv));
    ASSERT_TRUE(EVP_EncryptUpdate(ctx.get(), out, &out_bytes, test.plaintext,
                                  sizeof(test.plaintext)));
    ASSERT_TRUE(EVP_EncryptFinal_ex(ctx.get(), out + out_bytes, &final_bytes));
    EXPECT_EQ(static_cast<size_t>(out_bytes + final_bytes),
              sizeof(test.plaintext));
    EXPECT_EQ(Bytes(test.ofb_ciphertext), Bytes(out));

    bssl::ScopedEVP_CIPHER_CTX decrypt_ctx;
    ASSERT_TRUE(EVP_DecryptInit_ex(decrypt_ctx.get(), EVP_bf_ofb(), nullptr,
                                   test.key, test.iv));
    ASSERT_TRUE(EVP_DecryptUpdate(decrypt_ctx.get(), out, &out_bytes,
                                  test.ofb_ciphertext,
                                  sizeof(test.ofb_ciphertext)));
    ASSERT_TRUE(
        EVP_DecryptFinal_ex(decrypt_ctx.get(), out + out_bytes, &final_bytes));
    EXPECT_EQ(static_cast<size_t>(out_bytes + final_bytes),
              sizeof(test.plaintext));
    EXPECT_EQ(Bytes(test.plaintext), Bytes(out));
  }
}

// OpenSSL 1.1.1 Blowfish Test Data Below

struct ossl_bf_key {
  uint8_t key[30];
  size_t len;
};

static ossl_bf_key bf_key[2] = {
    {
        {0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69,
         0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72,
         0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a},
        26,
    },
    {{0x57, 0x68, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x4a, 0x6f, 0x68, 0x6e, 0x20,
      0x47, 0x61, 0x6c, 0x74, 0x3f},
     17},
};

/* big endian */
static uint32_t bf_plain[2][2] = {{0x424c4f57L, 0x46495348L},
                                  {0xfedcba98L, 0x76543210L}};

static uint32_t bf_cipher[2][2] = {{0x324ed0feL, 0xf413a203L},
                                   {0xcc91732bL, 0x8022f684L}};

#define NUM_TESTS 34
static uint8_t ecb_data[NUM_TESTS][8] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
    {0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    {0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11},
    {0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF},
    {0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    {0xFE, 0xDC, 0xBA, 0x98, 0x76, 0x54, 0x32, 0x10},
    {0x7C, 0xA1, 0x10, 0x45, 0x4A, 0x1A, 0x6E, 0x57},
    {0x01, 0x31, 0xD9, 0x61, 0x9D, 0xC1, 0x37, 0x6E},
    {0x07, 0xA1, 0x13, 0x3E, 0x4A, 0x0B, 0x26, 0x86},
    {0x38, 0x49, 0x67, 0x4C, 0x26, 0x02, 0x31, 0x9E},
    {0x04, 0xB9, 0x15, 0xBA, 0x43, 0xFE, 0xB5, 0xB6},
    {0x01, 0x13, 0xB9, 0x70, 0xFD, 0x34, 0xF2, 0xCE},
    {0x01, 0x70, 0xF1, 0x75, 0x46, 0x8F, 0xB5, 0xE6},
    {0x43, 0x29, 0x7F, 0xAD, 0x38, 0xE3, 0x73, 0xFE},
    {0x07, 0xA7, 0x13, 0x70, 0x45, 0xDA, 0x2A, 0x16},
    {0x04, 0x68, 0x91, 0x04, 0xC2, 0xFD, 0x3B, 0x2F},
    {0x37, 0xD0, 0x6B, 0xB5, 0x16, 0xCB, 0x75, 0x46},
    {0x1F, 0x08, 0x26, 0x0D, 0x1A, 0xC2, 0x46, 0x5E},
    {0x58, 0x40, 0x23, 0x64, 0x1A, 0xBA, 0x61, 0x76},
    {0x02, 0x58, 0x16, 0x16, 0x46, 0x29, 0xB0, 0x07},
    {0x49, 0x79, 0x3E, 0xBC, 0x79, 0xB3, 0x25, 0x8F},
    {0x4F, 0xB0, 0x5E, 0x15, 0x15, 0xAB, 0x73, 0xA7},
    {0x49, 0xE9, 0x5D, 0x6D, 0x4C, 0xA2, 0x29, 0xBF},
    {0x01, 0x83, 0x10, 0xDC, 0x40, 0x9B, 0x26, 0xD6},
    {0x1C, 0x58, 0x7F, 0x1C, 0x13, 0x92, 0x4F, 0xEF},
    {0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},
    {0x1F, 0x1F, 0x1F, 0x1F, 0x0E, 0x0E, 0x0E, 0x0E},
    {0xE0, 0xFE, 0xE0, 0xFE, 0xF1, 0xFE, 0xF1, 0xFE},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
    {0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF},
    {0xFE, 0xDC, 0xBA, 0x98, 0x76, 0x54, 0x32, 0x10}};

static uint8_t plain_data[NUM_TESTS][8] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
    {0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01},
    {0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11},
    {0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11},
    {0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    {0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF},
    {0x01, 0xA1, 0xD6, 0xD0, 0x39, 0x77, 0x67, 0x42},
    {0x5C, 0xD5, 0x4C, 0xA8, 0x3D, 0xEF, 0x57, 0xDA},
    {0x02, 0x48, 0xD4, 0x38, 0x06, 0xF6, 0x71, 0x72},
    {0x51, 0x45, 0x4B, 0x58, 0x2D, 0xDF, 0x44, 0x0A},
    {0x42, 0xFD, 0x44, 0x30, 0x59, 0x57, 0x7F, 0xA2},
    {0x05, 0x9B, 0x5E, 0x08, 0x51, 0xCF, 0x14, 0x3A},
    {0x07, 0x56, 0xD8, 0xE0, 0x77, 0x47, 0x61, 0xD2},
    {0x76, 0x25, 0x14, 0xB8, 0x29, 0xBF, 0x48, 0x6A},
    {0x3B, 0xDD, 0x11, 0x90, 0x49, 0x37, 0x28, 0x02},
    {0x26, 0x95, 0x5F, 0x68, 0x35, 0xAF, 0x60, 0x9A},
    {0x16, 0x4D, 0x5E, 0x40, 0x4F, 0x27, 0x52, 0x32},
    {0x6B, 0x05, 0x6E, 0x18, 0x75, 0x9F, 0x5C, 0xCA},
    {0x00, 0x4B, 0xD6, 0xEF, 0x09, 0x17, 0x60, 0x62},
    {0x48, 0x0D, 0x39, 0x00, 0x6E, 0xE7, 0x62, 0xF2},
    {0x43, 0x75, 0x40, 0xC8, 0x69, 0x8F, 0x3C, 0xFA},
    {0x07, 0x2D, 0x43, 0xA0, 0x77, 0x07, 0x52, 0x92},
    {0x02, 0xFE, 0x55, 0x77, 0x81, 0x17, 0xF1, 0x2A},
    {0x1D, 0x9D, 0x5C, 0x50, 0x18, 0xF7, 0x28, 0xC2},
    {0x30, 0x55, 0x32, 0x28, 0x6D, 0x6F, 0x29, 0x5A},
    {0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF},
    {0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF},
    {0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF},
    {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}};

static uint8_t cipher_data[NUM_TESTS][8] = {
    {0x4E, 0xF9, 0x97, 0x45, 0x61, 0x98, 0xDD, 0x78},
    {0x51, 0x86, 0x6F, 0xD5, 0xB8, 0x5E, 0xCB, 0x8A},
    {0x7D, 0x85, 0x6F, 0x9A, 0x61, 0x30, 0x63, 0xF2},
    {0x24, 0x66, 0xDD, 0x87, 0x8B, 0x96, 0x3C, 0x9D},
    {0x61, 0xF9, 0xC3, 0x80, 0x22, 0x81, 0xB0, 0x96},
    {0x7D, 0x0C, 0xC6, 0x30, 0xAF, 0xDA, 0x1E, 0xC7},
    {0x4E, 0xF9, 0x97, 0x45, 0x61, 0x98, 0xDD, 0x78},
    {0x0A, 0xCE, 0xAB, 0x0F, 0xC6, 0xA0, 0xA2, 0x8D},
    {0x59, 0xC6, 0x82, 0x45, 0xEB, 0x05, 0x28, 0x2B},
    {0xB1, 0xB8, 0xCC, 0x0B, 0x25, 0x0F, 0x09, 0xA0},
    {0x17, 0x30, 0xE5, 0x77, 0x8B, 0xEA, 0x1D, 0xA4},
    {0xA2, 0x5E, 0x78, 0x56, 0xCF, 0x26, 0x51, 0xEB},
    {0x35, 0x38, 0x82, 0xB1, 0x09, 0xCE, 0x8F, 0x1A},
    {0x48, 0xF4, 0xD0, 0x88, 0x4C, 0x37, 0x99, 0x18},
    {0x43, 0x21, 0x93, 0xB7, 0x89, 0x51, 0xFC, 0x98},
    {0x13, 0xF0, 0x41, 0x54, 0xD6, 0x9D, 0x1A, 0xE5},
    {0x2E, 0xED, 0xDA, 0x93, 0xFF, 0xD3, 0x9C, 0x79},
    {0xD8, 0x87, 0xE0, 0x39, 0x3C, 0x2D, 0xA6, 0xE3},
    {0x5F, 0x99, 0xD0, 0x4F, 0x5B, 0x16, 0x39, 0x69},
    {0x4A, 0x05, 0x7A, 0x3B, 0x24, 0xD3, 0x97, 0x7B},
    {0x45, 0x20, 0x31, 0xC1, 0xE4, 0xFA, 0xDA, 0x8E},
    {0x75, 0x55, 0xAE, 0x39, 0xF5, 0x9B, 0x87, 0xBD},
    {0x53, 0xC5, 0x5F, 0x9C, 0xB4, 0x9F, 0xC0, 0x19},
    {0x7A, 0x8E, 0x7B, 0xFA, 0x93, 0x7E, 0x89, 0xA3},
    {0xCF, 0x9C, 0x5D, 0x7A, 0x49, 0x86, 0xAD, 0xB5},
    {0xD1, 0xAB, 0xB2, 0x90, 0x65, 0x8B, 0xC7, 0x78},
    {0x55, 0xCB, 0x37, 0x74, 0xD1, 0x3E, 0xF2, 0x01},
    {0xFA, 0x34, 0xEC, 0x48, 0x47, 0xB2, 0x68, 0xB2},
    {0xA7, 0x90, 0x79, 0x51, 0x08, 0xEA, 0x3C, 0xAE},
    {0xC3, 0x9E, 0x07, 0x2D, 0x9F, 0xAC, 0x63, 0x1D},
    {0x01, 0x49, 0x33, 0xE0, 0xCD, 0xAF, 0xF6, 0xE4},
    {0xF2, 0x1E, 0x9A, 0x77, 0xB7, 0x1C, 0x49, 0xBC},
    {0x24, 0x59, 0x46, 0x88, 0x57, 0x54, 0x36, 0x9A},
    {0x6B, 0x5C, 0x5A, 0x9C, 0x5D, 0x9E, 0x0A, 0x5A},
};

static uint8_t cbc_key[16] = {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
                              0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87};
static uint8_t cbc_iv[8] = {0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10};

static uint8_t cbc_data[29] = {0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x20,
                               0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74,
                               0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20,
                               0x66, 0x6f, 0x72, 0x20, 0x00};

static uint8_t cbc_ok[32] = {0x6B, 0x77, 0xB4, 0xD6, 0x30, 0x06, 0xDE, 0xE6,
                             0x05, 0xB1, 0x56, 0xE2, 0x74, 0x03, 0x97, 0x93,
                             0x58, 0xDE, 0xB9, 0xE7, 0x15, 0x46, 0x16, 0xD9,
                             0x59, 0xF1, 0x65, 0x2B, 0xD5, 0xFF, 0x92, 0xCC};

static uint8_t cfb64_ok[] = {0xE7, 0x32, 0x14, 0xA2, 0x82, 0x21, 0x39, 0xCA,
                             0xF2, 0x6E, 0xCF, 0x6D, 0x2E, 0xB9, 0xE7, 0x6E,
                             0x3D, 0xA3, 0xDE, 0x04, 0xD1, 0x51, 0x72, 0x00,
                             0x51, 0x9D, 0x57, 0xA6, 0xC3};

static uint8_t ofb64_ok[] = {0xE7, 0x32, 0x14, 0xA2, 0x82, 0x21, 0x39, 0xCA,
                             0x62, 0xB3, 0x43, 0xCC, 0x5B, 0x65, 0x58, 0x73,
                             0x10, 0xDD, 0x90, 0x8D, 0x0C, 0x24, 0x1B, 0x22,
                             0x63, 0xC2, 0xCF, 0x80, 0xDA};

#define KEY_TEST_NUM 25
static uint8_t key_test[KEY_TEST_NUM] = {
    0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78,
    0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f, 0x00, 0x11,
    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88};

static uint8_t key_data[8] = {0xFE, 0xDC, 0xBA, 0x98, 0x76, 0x54, 0x32, 0x10};

static uint8_t key_out[KEY_TEST_NUM][8] = {
    {0xF9, 0xAD, 0x59, 0x7C, 0x49, 0xDB, 0x00, 0x5E},
    {0xE9, 0x1D, 0x21, 0xC1, 0xD9, 0x61, 0xA6, 0xD6},
    {0xE9, 0xC2, 0xB7, 0x0A, 0x1B, 0xC6, 0x5C, 0xF3},
    {0xBE, 0x1E, 0x63, 0x94, 0x08, 0x64, 0x0F, 0x05},
    {0xB3, 0x9E, 0x44, 0x48, 0x1B, 0xDB, 0x1E, 0x6E},
    {0x94, 0x57, 0xAA, 0x83, 0xB1, 0x92, 0x8C, 0x0D},
    {0x8B, 0xB7, 0x70, 0x32, 0xF9, 0x60, 0x62, 0x9D},
    {0xE8, 0x7A, 0x24, 0x4E, 0x2C, 0xC8, 0x5E, 0x82},
    {0x15, 0x75, 0x0E, 0x7A, 0x4F, 0x4E, 0xC5, 0x77},
    {0x12, 0x2B, 0xA7, 0x0B, 0x3A, 0xB6, 0x4A, 0xE0},
    {0x3A, 0x83, 0x3C, 0x9A, 0xFF, 0xC5, 0x37, 0xF6},
    {0x94, 0x09, 0xDA, 0x87, 0xA9, 0x0F, 0x6B, 0xF2},
    {0x88, 0x4F, 0x80, 0x62, 0x50, 0x60, 0xB8, 0xB4},
    {0x1F, 0x85, 0x03, 0x1C, 0x19, 0xE1, 0x19, 0x68},
    {0x79, 0xD9, 0x37, 0x3A, 0x71, 0x4C, 0xA3, 0x4F},
    {0x93, 0x14, 0x28, 0x87, 0xEE, 0x3B, 0xE1, 0x5C},
    {0x03, 0x42, 0x9E, 0x83, 0x8C, 0xE2, 0xD1, 0x4B},
    {0xA4, 0x29, 0x9E, 0x27, 0x46, 0x9F, 0xF6, 0x7B},
    {0xAF, 0xD5, 0xAE, 0xD1, 0xC1, 0xBC, 0x96, 0xA8},
    {0x10, 0x85, 0x1C, 0x0E, 0x38, 0x58, 0xDA, 0x9F},
    {0xE6, 0xF5, 0x1E, 0xD7, 0x9B, 0x9D, 0xB2, 0x1F},
    {0x64, 0xA6, 0xE1, 0x4A, 0xFD, 0x36, 0xB4, 0x6F},
    {0x80, 0xC7, 0xD7, 0xD4, 0x5A, 0x54, 0x79, 0xAD},
    {0x05, 0x04, 0x4B, 0x62, 0xFA, 0x52, 0xD0, 0x80},
};

TEST(Blowfish, ossl_bf_ecb_raw) {
  for (int n = 0; n < 2; n++) {
    BF_KEY key;
    uint32_t data[2];

    BF_set_key(&key, bf_key[n].len, bf_key[n].key);

    data[0] = bf_plain[n][0];
    data[1] = bf_plain[n][1];
    BF_encrypt(data, &key);
    ASSERT_EQ(0, memcmp(&(bf_cipher[n][0]), &data[0], BF_BLOCK));

    BF_decrypt(&(data[0]), &key);
    ASSERT_EQ(0, memcmp(&(bf_plain[n][0]), &data[0], BF_BLOCK));
  }
}

TEST(Blowfish, ossl_bf_ecb) {
  for (int n = 0; n < NUM_TESTS; n++) {
    BF_KEY key;
    uint8_t out[8];

    BF_set_key(&key, 8, ecb_data[n]);

    BF_ecb_encrypt(&(plain_data[n][0]), out, &key, BF_ENCRYPT);
    ASSERT_EQ(Bytes(&(cipher_data[n][0]), BF_BLOCK), Bytes(out, BF_BLOCK));

    BF_ecb_encrypt(out, out, &key, BF_DECRYPT);
    ASSERT_EQ(Bytes(&(plain_data[n][0]), BF_BLOCK), Bytes(out, BF_BLOCK));
  }
}

TEST(Blowfish, ossl_bf_set_key) {
  for (int n = 0; n < KEY_TEST_NUM - 1; n++) {
    BF_KEY key;
    uint8_t out[8];

    BF_set_key(&key, n + 1, key_test);
    BF_ecb_encrypt(key_data, out, &key, BF_ENCRYPT);
    ASSERT_EQ(Bytes(out, 8), Bytes(&(key_out[n][0]), 8));
  }
}

TEST(Blowfish, ossl_bf_cbc) {
  uint8_t cbc_in[32], cbc_out[32], iv[8];
  BF_KEY key;

  BF_set_key(&key, 16, cbc_key);
  memset(cbc_in, 0, sizeof(cbc_in));
  memset(cbc_out, 0, sizeof(cbc_out));
  memcpy(iv, cbc_iv, sizeof(iv));

  BF_cbc_encrypt(cbc_data, cbc_out, sizeof(cbc_data), &key, iv, BF_ENCRYPT);
  ASSERT_EQ(Bytes(cbc_out, sizeof(cbc_out)), Bytes(cbc_ok, sizeof(cbc_ok)));

  memcpy(iv, cbc_iv, 8);
  BF_cbc_encrypt(cbc_out, cbc_in, sizeof(cbc_out), &key, iv, BF_DECRYPT);
  ASSERT_EQ(Bytes(cbc_in, sizeof(cbc_data)), Bytes(cbc_data, sizeof(cbc_data)));
}

TEST(Blowfish, ossl_bf_cfb64) {
  uint8_t cbc_in[sizeof(cbc_data)], cbc_out[sizeof(cbc_data)], iv[8];
  int n = 0;
  BF_KEY key;

  // CFB is a streaming cipher mode so in/out len is the same.
  const size_t len = sizeof(cbc_data);

  BF_set_key(&key, sizeof(cbc_key), cbc_key);
  memset(cbc_in, 0, sizeof(cbc_in));
  memset(cbc_out, 0, sizeof(cbc_out));
  memcpy(iv, cbc_iv, sizeof(iv));

  BF_cfb64_encrypt(cbc_data, cbc_out, 13, &key, iv, &n, BF_ENCRYPT);
  BF_cfb64_encrypt(&(cbc_data[13]), &(cbc_out[13]), len - 13, &key, iv, &n,
                   BF_ENCRYPT);
  ASSERT_EQ(Bytes(cbc_out, len), Bytes(cfb64_ok, len));

  n = 0;
  memcpy(iv, cbc_iv, sizeof(cbc_iv));
  BF_cfb64_encrypt(cbc_out, cbc_in, 17, &key, iv, &n, BF_DECRYPT);
  BF_cfb64_encrypt(&(cbc_out[17]), &(cbc_in[17]), len - 17, &key, iv, &n,
                   BF_DECRYPT);
  ASSERT_EQ(Bytes(cbc_in, len), Bytes(cbc_data, len));
}

TEST(Blowfish, ossl_bf_ofb64) {
  uint8_t cbc_in[sizeof(cbc_data)], cbc_out[sizeof(cbc_data)], iv[8];
  int n = 0;
  BF_KEY key;

  // OFB is a streaming cipher mode so in/out len is the same.
  const size_t len = sizeof(cbc_data);

  BF_set_key(&key, 16, cbc_key);
  memset(cbc_in, 0, sizeof(cbc_in));
  memset(cbc_out, 0, sizeof(cbc_out));
  memcpy(iv, cbc_iv, sizeof(iv));

  BF_ofb64_encrypt(cbc_data, cbc_out, (long)13, &key, iv, &n);
  BF_ofb64_encrypt(&(cbc_data[13]), &(cbc_out[13]), len - 13, &key, iv, &n);
  ASSERT_EQ(Bytes(cbc_out, len), Bytes(ofb64_ok, len));

  n = 0;
  memcpy(iv, cbc_iv, sizeof(iv));
  BF_ofb64_encrypt(cbc_out, cbc_in, 17, &key, iv, &n);
  BF_ofb64_encrypt(&(cbc_out[17]), &(cbc_in[17]), len - 17, &key, iv, &n);
  ASSERT_EQ(Bytes(cbc_in, len), Bytes(cbc_data, len));
}
