/**
 *    Copyright (C) 2020-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#include "mongo/db/repl/tenant_migration_access_blocker_executor.h"

#include "mongo/executor/network_interface_factory.h"
#include "mongo/executor/thread_pool_task_executor.h"
#include "mongo/util/concurrency/thread_pool.h"

namespace mongo {

// Thread count to schedule ops blocked my migration blocker.
// It's dangerous to increase this pool size as the simultaneously unblocked
// operations may flood the server.
static constexpr int kBlockedOpsPoolSize = 4;

const ServiceContext::Decoration<TenantMigrationAccessBlockerExecutor>
    TenantMigrationAccessBlockerExecutor::get =
        ServiceContext::declareDecoration<TenantMigrationAccessBlockerExecutor>();

std::shared_ptr<executor::TaskExecutor>
TenantMigrationAccessBlockerExecutor::getOrCreateBlockedOperationsExecutor() {
    stdx::lock_guard<Latch> lg(_mutex);
    auto locked = _blockedOperationsExecutor.lock();
    if (locked) {
        return locked;
    }
    ThreadPool::Options threadPoolOptions;
    threadPoolOptions.maxThreads = kBlockedOpsPoolSize;
    threadPoolOptions.threadNamePrefix = "TenantMigrationBlockerAsync-";
    threadPoolOptions.poolName = "TenantMigrationBlockerAsyncThreadPool";
    threadPoolOptions.onCreateThread = [](const std::string& threadName) {
        Client::initThread(threadName.c_str());
    };
    auto executor = std::make_shared<executor::ThreadPoolTaskExecutor>(
        std::make_unique<ThreadPool>(threadPoolOptions),
        executor::makeNetworkInterface("TenantMigrationBlockerNet"));
    _blockedOperationsExecutor = executor;
    executor->startup();
    return executor;
}

}  // namespace mongo
