/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.core;

import org.apiguardian.api.API;
import org.opentest4j.reporting.events.api.ChildElement;
import org.opentest4j.reporting.events.api.Context;
import org.opentest4j.reporting.schema.Namespace;
import org.opentest4j.reporting.schema.QualifiedName;

import java.time.LocalDateTime;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * The {@code data} element of the core namespace.
 *
 * @since 0.1.0
 */
@API(status = EXPERIMENTAL, since = "0.1.0")
public class Data extends ChildElement<Attachments, Data> {

	private static final QualifiedName ELEMENT = QualifiedName.of(Namespace.REPORTING_CORE, "data");
	private static final QualifiedName TIME = QualifiedName.of(Namespace.REPORTING_CORE, "time");

	Data(Context context) {
		super(context, ELEMENT);
	}

	/**
	 * Set the {@code time} attribute of this element.
	 *
	 * @param timestamp the timestamp to set
	 * @return this element
	 */
	public Data withTime(LocalDateTime timestamp) {
		withAttribute(TIME, timestamp.toString());
		return this;
	}

	/**
	 * Add an {@code entry} child element to this element.
	 *
	 * @param key   entry key
	 * @param value entry value
	 */
	public void addEntry(String key, String value) {
		append(context -> new Entry(context, key, value));
	}

	private static class Entry extends ChildElement<Data, Entry> {

		public static final QualifiedName ELEMENT = QualifiedName.of(Namespace.REPORTING_CORE, "entry");
		public static final QualifiedName KEY = QualifiedName.of(Namespace.REPORTING_CORE, "key");

		Entry(Context context, String key, String value) {
			super(context, ELEMENT);
			withAttribute(KEY, key);
			withContent(value);
		}
	}
}
