// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief e+ e- > sigma+- sigmabar -+
  class BESIII_2020_I1814783 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2020_I1814783);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");
      for (size_t ix=0; ix<2; ++ix) {
        book(_n_plus [ix], 1+ix, 1, 1);
        book(_n_minus[ix], 1+ix, 1, 2);
        for (const string& en : _n_plus[ix].binning().edges<0>()) {
          const double eval = stod(en);
          if (isCompatibleWithSqrtS(eval)) {
            _sqs[ix] = en; break;
          }
        }
      }
      raiseBeamErrorIf(_sqs[0].empty() && _sqs[1].empty());
      book(_h_cTheta_A, 3, 1, 1);
      book(_h_cTheta_B, 3, 1, 2);
    }

    void findChildren(const Particle& p, map<long,int>& nRes, int& ncount) const {
      for (const Particle &child : p.children()) {
        if (child.children().empty()) {
          nRes[child.pid()]-=1;
          --ncount;
        }
        else {
          findChildren(child,nRes,ncount);
        }
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");
      // total hadronic and muonic cross sections
      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      // find the Sigmas
      const FinalState& ufs = apply<UnstableParticles>(event, "UFS");
      for (size_t ix=0; ix<ufs.particles().size(); ++ix) {
        const Particle& p1 = ufs.particles()[ix];
        if (p1.abspid()!=3112 && p1.abspid()!=3222) continue;
        bool matched = false;
        // check fs
        bool fs = true;
        for (const Particle& child : p1.children()) {
          if (child.pid()==p1.pid()) {
            fs = false;
            break;
          }
        }
        if (!fs) continue;
        // find the children
        map<long,int> nRes = nCount;
        int ncount = ntotal;
        findChildren(p1,nRes,ncount);
        for (size_t iy=ix+1; iy<ufs.particles().size(); ++iy) {
          const Particle& p2 = ufs.particles()[iy];
          if (p2.pid() != -p1.pid()) continue;
          // check fs
          bool fs = true;
          for (const Particle& child : p2.children()) {
            if (child.pid()==p2.pid()) {
              fs = false;
              break;
            }
          }
          if(!fs) continue;
          map<long,int> nRes2 = nRes;
          int ncount2 = ncount;
          findChildren(p2,nRes2,ncount2);
          if (ncount2!=0) continue;
          matched=true;
          for (const auto& val : nRes2) {
            if (val.second!=0) {
              matched = false;
              break;
            }
          }
          if (matched) {
            if (p1.abspid()==3222) {
              for (size_t ix=0; ix<2; ++ix) {
                if (!_sqs[ix].empty()) _n_plus[ix]->fill(_sqs[ix]);
              }
              if (_sqs[0] == "2.3960"s) {
                double cTheta = p1.pid()>0 ? cos(p1.mom().polarAngle()) : cos(p2.mom().polarAngle());
                _h_cTheta_A->fill(cTheta);
                _h_cTheta_B->fill(cTheta);
              }
            }
            else if (p1.abspid()==3112) {
              for (size_t ix=0; ix<2; ++ix) {
                if(!_sqs[ix].empty()) _n_minus[ix]->fill(_sqs[ix]);
              }
            }
            break;
          }
        }
        if (matched) break;
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      normalize(_h_cTheta_A);
      normalize(_h_cTheta_B);
      const double fact = crossSection()/ sumOfWeights() /picobarn;
      scale(_n_plus, fact);
      scale(_n_minus, fact);
    }
    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _n_plus[2], _n_minus[2];
    Histo1DPtr _h_cTheta_A, _h_cTheta_B;
    string _sqs[2];
    /// @}

  };


  RIVET_DECLARE_PLUGIN(BESIII_2020_I1814783);

}
