// Copyright 2018, VIXL authors
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   * Redistributions of source code must retain the above copyright notice,
//     this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above copyright notice,
//     this list of conditions and the following disclaimer in the documentation
//     and/or other materials provided with the distribution.
//   * Neither the name of ARM Limited nor the names of its contributors may be
//     used to endorse or promote products derived from this software without
//     specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#include <cstring>
#include <iostream>
#include <string>

#include "test-runner.h"
#include "test-utils.h"

#include "aarch64/macro-assembler-aarch64.h"
#include "aarch64/test-utils-aarch64.h"

#define __ masm->
#define TEST(name) TEST_(AARCH64_CPUFeatures_##name)


namespace vixl {
namespace aarch64 {

class CPUFeaturesTest {
 public:
  CPUFeaturesTest(const CPUFeatures& features, const char* description)
      : description_(description),
        features_(features),
        auditor_(&decoder_, features) {}

  void Run() {
    auditor_.ResetSeenFeatures();

    // Positive test: the instruction must assemble with the specified features.
    RunWithFeatures(features_);
    // Positive test: extra features are ignored.
    RunWithFeatures(CPUFeatures::All());
#ifdef VIXL_DEBUG
#ifdef VIXL_NEGATIVE_TESTING
    // Negative tests: any one missing feature causes the Assembler to fail in
    // debug mode.
    std::vector<CPUFeatures> errors;
    for (CPUFeatures::const_iterator it = features_.begin();
         it != features_.end();
         ++it) {
      try {
        CPUFeatures reduced = features_.Without(*it);
        RunWithFeatures(reduced);
        errors.push_back(reduced);
      } catch (const std::runtime_error&) {
      }
    }
    if (!errors.empty()) {
      std::cerr << "Negative CPUFeatures test failed for:\n";
      std::cerr << "  " << description_ << ";";
      std::cerr << "  // Requires: {" << features_ << "}\n";
      std::cerr << "with feature sets:\n";
      for (std::vector<CPUFeatures>::iterator it = errors.begin();
           it != errors.end();
           ++it) {
        std::cerr << "  {" << *it << "}\n";
      }
      abort();
    }
#endif  // VIXL_NEGATIVE_TESTING
#else   // VIXL_DEBUG
    // In release mode, the {Macro}Assembler doesn't check CPUFeatures.
    RunWithFeatures(CPUFeatures::None());
#endif  // VIXL_DEBUG

    // Check that the CPUFeaturesAuditor detected the correct features.
    VIXL_CHECK(auditor_.GetSeenFeatures() == features_);
  }

  virtual void GenerateTestInstruction(MacroAssembler* masm) const = 0;

 protected:
  const char* description_;

 private:
  CPUFeatures features_;

  Decoder decoder_;
  CPUFeaturesAuditor auditor_;

  // Use a separate context (and MacroAssembler) for each test, because the
  // MacroAssembler does not guarantee any particular exception safety in
  // negative testing mode.
  void RunWithFeatures(const CPUFeatures& features) {
    // Use PositionDependentCode to allow the likes of adrp.
    MacroAssembler masm(PositionDependentCode);
    masm.SetCPUFeatures(features);
    {
      SingleEmissionCheckScope guard(&masm);
      GenerateTestInstruction(&masm);
    }
    masm.FinalizeCode();

    // Pass the generated code through the CPUFeaturesAuditor.
    VIXL_ASSERT(masm.GetBuffer()->GetSizeInBytes() == kInstructionSize);
    decoder_.Decode(masm.GetInstructionAt(0));

    // Check that the CPUFeaturesAuditor detected the correct features for this
    // instruction. A simple assertion would do, but printing the missing or
    // superfluous features is useful for debugging.
    if (auditor_.GetInstructionFeatures() != features_) {
      CPUFeatures missing =
          features_.Without(auditor_.GetInstructionFeatures());
      if (missing.Count() > 0) {
        std::cout << "Error: the auditor should have detected CPUFeatures { "
                  << missing << " }\n";
      }
      CPUFeatures extra = auditor_.GetInstructionFeatures().Without(features_);
      if (extra.Count() > 0) {
        std::cout << "Error: the auditor detected superfluous CPUFeatures { "
                  << extra << " }\n";
      }
      VIXL_ABORT();
    }
  }
};

#define STRINGIFY(x) #x

#define TEST_TEMPLATE(FEATURES, NAME, ASM)                                     \
  TEST(NAME) {                                                                 \
    class TestCase : public CPUFeaturesTest {                                  \
     public:                                                                   \
      explicit TestCase(const CPUFeatures& features)                           \
          : CPUFeaturesTest(features, STRINGIFY(ASM)) {}                       \
                                                                               \
      void GenerateTestInstruction(MacroAssembler* masm) const VIXL_OVERRIDE { \
        /* Some tests need a label. */                                         \
        Label label;                                                           \
        __ bind(&label);                                                       \
        __ ASM;                                                                \
      }                                                                        \
    };                                                                         \
    TestCase test((FEATURES));                                                 \
    test.Run();                                                                \
  }


#define TEST_NONE(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures::None(), NONE_##NAME, ASM)
TEST_NONE(adc_0, adc(w0, w1, w2))
TEST_NONE(adc_1, adc(x0, x1, x2))
TEST_NONE(adcs_0, adcs(w0, w1, w2))
TEST_NONE(adcs_1, adcs(x0, x1, x2))
TEST_NONE(adds_0, adds(w0, w1, Operand(w2, UXTH, 0)))
TEST_NONE(adds_1, adds(x0, x1, Operand(x2, SXTX, 4)))
TEST_NONE(adds_2, adds(x0, x1, Operand(w2, SXTB, 1)))
TEST_NONE(adds_3, adds(w0, w1, 0x905 << 12))
TEST_NONE(adds_4, adds(x0, x1, 0x35a))
TEST_NONE(adds_5, adds(w0, w1, Operand(w2, LSL, 10)))
TEST_NONE(adds_6, adds(x0, x1, Operand(x2, LSR, 11)))
TEST_NONE(add_0, add(w0, w1, Operand(w2, UXTW, 3)))
TEST_NONE(add_1, add(x0, x1, Operand(x2, UXTX, 4)))
TEST_NONE(add_2, add(x0, x1, Operand(w2, UXTH, 4)))
TEST_NONE(add_3, add(w0, w1, 0xb7e << 12))
TEST_NONE(add_4, add(x0, x1, 0x6d2))
TEST_NONE(add_5, add(w0, w1, Operand(w2, LSL, 6)))
TEST_NONE(add_6, add(x0, x1, Operand(x2, LSR, 25)))
TEST_NONE(adr_0, adr(x0, &label))
TEST_NONE(adrp_0, adrp(x0, &label))
TEST_NONE(ands_0, ands(w0, w1, 0x3c0000))
TEST_NONE(ands_1, ands(x0, x1, 0x3c0000))
TEST_NONE(ands_2, ands(w0, w1, Operand(w2, ASR, 6)))
TEST_NONE(ands_3, ands(x0, x1, Operand(x2, ASR, 33)))
TEST_NONE(and_0, and_(w0, w1, 0x3c0))
TEST_NONE(and_1, and_(x0, x1, 0xf))
TEST_NONE(and_2, and_(w0, w1, Operand(w2, LSL, 1)))
TEST_NONE(and_3, and_(x0, x1, Operand(x2, LSL, 58)))
TEST_NONE(asrv_0, asrv(w0, w1, w2))
TEST_NONE(asrv_1, asrv(x0, x1, x2))
TEST_NONE(bfi_0, bfi(w0, w1, 5, 11))
TEST_NONE(bfi_1, bfi(x0, x1, 25, 36))
TEST_NONE(bfm_0, bfm(w0, w1, 27, 31))
TEST_NONE(bfm_1, bfm(x0, x1, 54, 57))
TEST_NONE(bfxil_0, bfxil(w0, w1, 14, 13))
TEST_NONE(bfxil_1, bfxil(x0, x1, 22, 6))
TEST_NONE(bics_0, bics(w0, w1, Operand(w2, LSR, 10)))
TEST_NONE(bics_1, bics(x0, x1, Operand(x2, LSL, 42)))
TEST_NONE(bic_0, bic(w0, w1, Operand(w2, LSL, 17)))
TEST_NONE(bic_1, bic(x0, x1, Operand(x2, LSR, 16)))
TEST_NONE(bl_0, bl(&label))
TEST_NONE(blr_0, blr(x0))
TEST_NONE(br_0, br(x0))
TEST_NONE(brk_0, brk(0x97b0))
TEST_NONE(b_0, b(&label, pl))
TEST_NONE(b_1, b(&label))
TEST_NONE(cbnz_0, cbnz(w0, &label))
TEST_NONE(cbnz_1, cbnz(x0, &label))
TEST_NONE(cbz_0, cbz(w0, &label))
TEST_NONE(cbz_1, cbz(x0, &label))
TEST_NONE(ccmn_0, ccmn(w0, 28, NZVFlag, lt))
TEST_NONE(ccmn_1, ccmn(x0, 12, ZFlag, ge))
TEST_NONE(ccmn_2, ccmn(w0, w1, NZCFlag, gt))
TEST_NONE(ccmn_3, ccmn(x0, x1, ZCFlag, cs))
TEST_NONE(ccmp_0, ccmp(w0, 8, NVFlag, mi))
TEST_NONE(ccmp_1, ccmp(x0, 19, ZVFlag, cc))
TEST_NONE(ccmp_2, ccmp(w0, w1, NVFlag, ne))
TEST_NONE(ccmp_3, ccmp(x0, x1, NZVFlag, cs))
TEST_NONE(cinc_0, cinc(w0, w1, eq))
TEST_NONE(cinc_1, cinc(x0, x1, le))
TEST_NONE(cinv_0, cinv(w0, w1, cs))
TEST_NONE(cinv_1, cinv(x0, x1, cc))
TEST_NONE(clrex_0, clrex(10))
TEST_NONE(cls_0, cls(w0, w1))
TEST_NONE(cls_1, cls(x0, x1))
TEST_NONE(clz_0, clz(w0, w1))
TEST_NONE(clz_1, clz(x0, x1))
TEST_NONE(cmn_0, cmn(w0, Operand(w1, UXTH, 2)))
TEST_NONE(cmn_1, cmn(x0, Operand(x1, SXTX, 1)))
TEST_NONE(cmn_2, cmn(x0, Operand(w1, SXTW, 3)))
TEST_NONE(cmn_3, cmn(w0, 0x2e5 << 12))
TEST_NONE(cmn_4, cmn(x0, 0xb48))
TEST_NONE(cmn_5, cmn(w0, Operand(w1, LSR, 9)))
TEST_NONE(cmn_6, cmn(x0, Operand(x1, LSL, 50)))
TEST_NONE(cmp_0, cmp(w0, Operand(w1, UXTB, 3)))
TEST_NONE(cmp_1, cmp(x0, Operand(x1, SXTX, 0)))
TEST_NONE(cmp_2, cmp(x0, Operand(w1, SXTB, 0)))
TEST_NONE(cmp_3, cmp(w0, 0xb44 << 12))
TEST_NONE(cmp_4, cmp(x0, 0x48 << 12))
TEST_NONE(cmp_5, cmp(w0, Operand(w1, LSL, 1)))
TEST_NONE(cmp_6, cmp(x0, Operand(x1, ASR, 4)))
TEST_NONE(cneg_0, cneg(w0, w1, mi))
TEST_NONE(cneg_1, cneg(x0, x1, cs))
TEST_NONE(csdb_0, csdb())
TEST_NONE(csel_0, csel(w0, w1, w2, cs))
TEST_NONE(csel_1, csel(x0, x1, x2, cc))
TEST_NONE(csetm_0, csetm(w0, vc))
TEST_NONE(csetm_1, csetm(x0, pl))
TEST_NONE(cset_0, cset(w0, vc))
TEST_NONE(cset_1, cset(x0, lt))
TEST_NONE(csinc_0, csinc(w0, w1, w2, lt))
TEST_NONE(csinc_1, csinc(x0, x1, x2, cs))
TEST_NONE(csinv_0, csinv(w0, w1, w2, cc))
TEST_NONE(csinv_1, csinv(x0, x1, x2, ne))
TEST_NONE(csneg_0, csneg(w0, w1, w2, ge))
TEST_NONE(csneg_1, csneg(x0, x1, x2, cc))
TEST_NONE(dc_0, dc(CIVAC, x0))
TEST_NONE(eon_0, eon(w0, w1, Operand(w2, ASR, 3)))
TEST_NONE(eon_1, eon(x0, x1, Operand(x2, LSL, 28)))
TEST_NONE(eor_0, eor(w0, w1, 0xf0))
TEST_NONE(eor_1, eor(x0, x1, 0x1e000))
TEST_NONE(eor_2, eor(w0, w1, Operand(w2, LSL, 24)))
TEST_NONE(eor_3, eor(x0, x1, Operand(x2, ASR, 16)))
TEST_NONE(extr_0, extr(w0, w1, w2, 17))
TEST_NONE(extr_1, extr(x0, x1, x2, 8))
TEST_NONE(extr_2, extr(x0, x1, x2, 50))
TEST_NONE(hlt_0, hlt(0x7fb7))
TEST_NONE(ic_0, ic(IVAU, x0))
TEST_NONE(ldar_0, ldar(w0, MemOperand(x1, 0)))
TEST_NONE(ldar_1, ldar(x0, MemOperand(x1, 0)))
TEST_NONE(ldarb_0, ldarb(w0, MemOperand(x1, 0)))
TEST_NONE(ldarh_0, ldarh(w0, MemOperand(x1, 0)))
TEST_NONE(ldaxp_0, ldaxp(w0, w1, MemOperand(x2, 0)))
TEST_NONE(ldaxp_1, ldaxp(x0, x1, MemOperand(x2, 0)))
TEST_NONE(ldaxr_0, ldaxr(w0, MemOperand(x1, 0)))
TEST_NONE(ldaxr_1, ldaxr(x0, MemOperand(x1, 0)))
TEST_NONE(ldaxrb_0, ldaxrb(w0, MemOperand(x1, 0)))
TEST_NONE(ldaxrh_0, ldaxrh(w0, MemOperand(x1, 0)))
TEST_NONE(ldnp_0, ldnp(w0, w1, MemOperand(x2, 12)))
TEST_NONE(ldnp_1, ldnp(x0, x1, MemOperand(x2, 304)))
TEST_NONE(ldpsw_0, ldpsw(x0, x1, MemOperand(x2, -212)))
TEST_NONE(ldpsw_1, ldpsw(x0, x1, MemOperand(x2, -36, PostIndex)))
TEST_NONE(ldpsw_2, ldpsw(x0, x1, MemOperand(x2, 104, PreIndex)))
TEST_NONE(ldp_0, ldp(w0, w1, MemOperand(x2, -212)))
TEST_NONE(ldp_1, ldp(w0, w1, MemOperand(x2, -212, PostIndex)))
TEST_NONE(ldp_2, ldp(w0, w1, MemOperand(x2, -252, PreIndex)))
TEST_NONE(ldp_3, ldp(x0, x1, MemOperand(x2, 336)))
TEST_NONE(ldp_4, ldp(x0, x1, MemOperand(x2, 8, PostIndex)))
TEST_NONE(ldp_5, ldp(x0, x1, MemOperand(x2, 360, PreIndex)))
TEST_NONE(ldrb_0, ldrb(w0, MemOperand(x1, -219, PostIndex)))
TEST_NONE(ldrb_1, ldrb(w0, MemOperand(x1, -137, PreIndex)))
TEST_NONE(ldrb_2, ldrb(w0, MemOperand(x1, 4022)))
TEST_NONE(ldrb_3, ldrb(w0, MemOperand(x1, x2, LSL, 0)))
TEST_NONE(ldrb_4, ldrb(w0, MemOperand(x1, w2, SXTW, 0)))
TEST_NONE(ldrb_5, ldrb(w0, MemOperand(x1, x2, SXTX, 0)))
TEST_NONE(ldrh_0, ldrh(w0, MemOperand(x1, -135, PostIndex)))
TEST_NONE(ldrh_1, ldrh(w0, MemOperand(x1, 52, PreIndex)))
TEST_NONE(ldrh_2, ldrh(w0, MemOperand(x1, 4410)))
TEST_NONE(ldrh_3, ldrh(w0, MemOperand(x1, w2, UXTW, 1)))
TEST_NONE(ldrh_4, ldrh(w0, MemOperand(x1, x2, SXTX, 1)))
TEST_NONE(ldrsb_0, ldrsb(w0, MemOperand(x1, 160, PostIndex)))
TEST_NONE(ldrsb_1, ldrsb(w0, MemOperand(x1, -253, PreIndex)))
TEST_NONE(ldrsb_2, ldrsb(w0, MemOperand(x1, 147)))
TEST_NONE(ldrsb_3, ldrsb(x0, MemOperand(x1, 33, PostIndex)))
TEST_NONE(ldrsb_4, ldrsb(x0, MemOperand(x1, 11, PreIndex)))
TEST_NONE(ldrsb_5, ldrsb(x0, MemOperand(x1, 2742)))
TEST_NONE(ldrsb_6, ldrsb(w0, MemOperand(x1, x2, LSL, 0)))
TEST_NONE(ldrsb_7, ldrsb(w0, MemOperand(x1, w2, UXTW, 0)))
TEST_NONE(ldrsb_8, ldrsb(w0, MemOperand(x1, x2, SXTX, 0)))
TEST_NONE(ldrsb_9, ldrsb(x0, MemOperand(x1, x2, LSL, 0)))
TEST_NONE(ldrsb_10, ldrsb(x0, MemOperand(x1, w2, SXTW, 0)))
TEST_NONE(ldrsb_11, ldrsb(x0, MemOperand(x1, x2, SXTX, 0)))
TEST_NONE(ldrsh_0, ldrsh(w0, MemOperand(x1, 11, PostIndex)))
TEST_NONE(ldrsh_1, ldrsh(w0, MemOperand(x1, -34, PreIndex)))
TEST_NONE(ldrsh_2, ldrsh(w0, MemOperand(x1, 6212)))
TEST_NONE(ldrsh_3, ldrsh(x0, MemOperand(x1, -78, PostIndex)))
TEST_NONE(ldrsh_4, ldrsh(x0, MemOperand(x1, 72, PreIndex)))
TEST_NONE(ldrsh_5, ldrsh(x0, MemOperand(x1, 6232)))
TEST_NONE(ldrsh_6, ldrsh(w0, MemOperand(x1, w2, UXTW, 0)))
TEST_NONE(ldrsh_7, ldrsh(w0, MemOperand(x1, x2, LSL, 0)))
TEST_NONE(ldrsh_8, ldrsh(x0, MemOperand(x1, w2, SXTW, 0)))
TEST_NONE(ldrsh_9, ldrsh(x0, MemOperand(x1, x2, SXTX, 1)))
TEST_NONE(ldrsw_0, ldrsw(x0, MemOperand(x1, 73, PostIndex)))
TEST_NONE(ldrsw_1, ldrsw(x0, MemOperand(x1, 13, PreIndex)))
TEST_NONE(ldrsw_2, ldrsw(x0, MemOperand(x1, 2192)))
TEST_NONE(ldrsw_3, ldrsw(x0, 0x16279))
TEST_NONE(ldrsw_4, ldrsw(x0, MemOperand(x1, w2, SXTW, 0)))
TEST_NONE(ldrsw_5, ldrsw(x0, MemOperand(x1, x2, LSL, 2)))
TEST_NONE(ldr_0, ldr(w0, MemOperand(x1, -169, PostIndex)))
TEST_NONE(ldr_1, ldr(w0, MemOperand(x1, 16, PreIndex)))
TEST_NONE(ldr_2, ldr(w0, MemOperand(x1, 13080)))
TEST_NONE(ldr_3, ldr(x0, MemOperand(x1, 3, PostIndex)))
TEST_NONE(ldr_4, ldr(x0, MemOperand(x1, -27, PreIndex)))
TEST_NONE(ldr_5, ldr(x0, MemOperand(x1, 14016)))
TEST_NONE(ldr_6, ldr(w0, 0x21ada))
TEST_NONE(ldr_7, ldr(x0, 0x2a87d))
TEST_NONE(ldr_8, ldr(w0, MemOperand(x1, w2, SXTW, 0)))
TEST_NONE(ldr_9, ldr(w0, MemOperand(x1, x2, LSL, 0)))
TEST_NONE(ldr_10, ldr(x0, MemOperand(x1, w2, UXTW, 3)))
TEST_NONE(ldr_11, ldr(x0, MemOperand(x1, x2, LSL, 3)))
TEST_NONE(ldurb_0, ldurb(w0, MemOperand(x1, 58)))
TEST_NONE(ldurh_0, ldurh(w0, MemOperand(x1, -222)))
TEST_NONE(ldursb_0, ldursb(w0, MemOperand(x1, -216)))
TEST_NONE(ldursb_1, ldursb(x0, MemOperand(x1, 196)))
TEST_NONE(ldursh_0, ldursh(w0, MemOperand(x1, -61)))
TEST_NONE(ldursh_1, ldursh(x0, MemOperand(x1, -255)))
TEST_NONE(ldursw_0, ldursw(x0, MemOperand(x1, -206)))
TEST_NONE(ldur_0, ldur(w0, MemOperand(x1, 71)))
TEST_NONE(ldur_1, ldur(x0, MemOperand(x1, 70)))
TEST_NONE(ldxp_0, ldxp(w0, w1, MemOperand(x2, 0)))
TEST_NONE(ldxp_1, ldxp(x0, x1, MemOperand(x2, 0)))
TEST_NONE(ldxr_0, ldxr(w0, MemOperand(x1, 0)))
TEST_NONE(ldxr_1, ldxr(x0, MemOperand(x1, 0)))
TEST_NONE(ldxrb_0, ldxrb(w0, MemOperand(x1, 0)))
TEST_NONE(ldxrh_0, ldxrh(w0, MemOperand(x1, 0)))
TEST_NONE(lslv_0, lslv(w0, w1, w2))
TEST_NONE(lslv_1, lslv(x0, x1, x2))
TEST_NONE(lsrv_0, lsrv(w0, w1, w2))
TEST_NONE(lsrv_1, lsrv(x0, x1, x2))
TEST_NONE(madd_0, madd(w0, w1, w2, w3))
TEST_NONE(madd_1, madd(x0, x1, x2, x3))
TEST_NONE(mneg_0, mneg(w0, w1, w2))
TEST_NONE(mneg_1, mneg(x0, x1, x2))
TEST_NONE(movk_0, movk(w0, UINT32_C(0xee40)))
TEST_NONE(movk_1, movk(x0, UINT64_C(0xfff2) << 16))
TEST_NONE(movn_0, movn(w0, UINT32_C(0xbd2f)))
TEST_NONE(movn_1, movn(x0, UINT64_C(0x560c) << 16))
TEST_NONE(movz_0, movz(w0, UINT32_C(0x9028) << 16))
TEST_NONE(movz_1, movz(x0, UINT64_C(0xf525)))
TEST_NONE(mov_0, mov(w0, w1))
TEST_NONE(mov_1, mov(x0, x1))
TEST_NONE(mov_2, mov(w0, w1))
TEST_NONE(mov_3, mov(x0, x1))
TEST_NONE(mrs_0, mrs(x0, FPCR))
TEST_NONE(msr_0, msr(FPCR, x0))
TEST_NONE(msub_0, msub(w0, w1, w2, w3))
TEST_NONE(msub_1, msub(x0, x1, x2, x3))
TEST_NONE(mul_0, mul(w0, w1, w2))
TEST_NONE(mul_1, mul(x0, x1, x2))
TEST_NONE(mvn_0, mvn(w0, Operand(w1, LSL, 7)))
TEST_NONE(mvn_1, mvn(x0, Operand(x1, LSL, 40)))
TEST_NONE(negs_0, negs(w0, Operand(w1, LSL, 21)))
TEST_NONE(negs_1, negs(x0, Operand(x1, LSL, 57)))
TEST_NONE(neg_0, neg(w0, Operand(w1, LSR, 18)))
TEST_NONE(neg_1, neg(x0, Operand(x1, ASR, 33)))
TEST_NONE(ngcs_0, ngcs(w0, w1))
TEST_NONE(ngcs_1, ngcs(x0, x1))
TEST_NONE(ngc_0, ngc(w0, w1))
TEST_NONE(ngc_1, ngc(x0, x1))
TEST_NONE(nop_0, nop())
TEST_NONE(orn_0, orn(w0, w1, Operand(w2, LSL, 6)))
TEST_NONE(orn_1, orn(x0, x1, Operand(x2, LSL, 63)))
TEST_NONE(orr_0, orr(w0, w1, 0x780000))
TEST_NONE(orr_1, orr(x0, x1, 0x1e0000))
TEST_NONE(orr_2, orr(w0, w1, Operand(w2, ASR, 24)))
TEST_NONE(orr_3, orr(x0, x1, Operand(x2, LSR, 2)))
TEST_NONE(prfm_0, prfm(PLDL2STRM, MemOperand(x0, 832)))
TEST_NONE(prfm_1, prfm(PSTL1STRM, 0x2c2fa))
TEST_NONE(prfm_2, prfm(PLDL1KEEP, MemOperand(x0, w1, SXTW, 0)))
TEST_NONE(prfm_3, prfm(PSTL3STRM, MemOperand(x0, x1, SXTX, 0)))
TEST_NONE(prfum_0, prfum(PSTL3KEEP, MemOperand(x0, 47)))
TEST_NONE(rbit_0, rbit(w0, w1))
TEST_NONE(rbit_1, rbit(x0, x1))
TEST_NONE(ret_0, ret(x0))
TEST_NONE(rev_0, rev(w0, w1))
TEST_NONE(rev_1, rev(x0, x1))
TEST_NONE(rev16_0, rev16(w0, w1))
TEST_NONE(rev16_1, rev16(x0, x1))
TEST_NONE(rev32_0, rev32(x0, x1))
TEST_NONE(rorv_0, rorv(w0, w1, w2))
TEST_NONE(rorv_1, rorv(x0, x1, x2))
TEST_NONE(sbc_0, sbc(w0, w1, w2))
TEST_NONE(sbc_1, sbc(x0, x1, x2))
TEST_NONE(sbcs_0, sbcs(w0, w1, w2))
TEST_NONE(sbcs_1, sbcs(x0, x1, x2))
TEST_NONE(sbfiz_0, sbfiz(w0, w1, 28, 2))
TEST_NONE(sbfiz_1, sbfiz(x0, x1, 46, 5))
TEST_NONE(sbfm_0, sbfm(w0, w1, 9, 11))
TEST_NONE(sbfm_1, sbfm(x0, x1, 22, 22))
TEST_NONE(sbfx_0, sbfx(w0, w1, 15, 9))
TEST_NONE(sbfx_1, sbfx(x0, x1, 21, 16))
TEST_NONE(sdiv_0, sdiv(w0, w1, w2))
TEST_NONE(sdiv_1, sdiv(x0, x1, x2))
TEST_NONE(smaddl_0, smaddl(x0, w1, w2, x3))
TEST_NONE(smsubl_0, smsubl(x0, w1, w2, x3))
TEST_NONE(smulh_0, smulh(x0, x1, x2))
TEST_NONE(smull_0, smull(x0, w1, w2))
TEST_NONE(stlr_0, stlr(w0, MemOperand(x1, 0)))
TEST_NONE(stlr_1, stlr(x0, MemOperand(x1, 0)))
TEST_NONE(stlrb_0, stlrb(w0, MemOperand(x1, 0)))
TEST_NONE(stlrh_0, stlrh(w0, MemOperand(x1, 0)))
TEST_NONE(stlxp_0, stlxp(w0, w1, w2, MemOperand(x3, 0)))
TEST_NONE(stlxp_1, stlxp(w0, x1, x2, MemOperand(x3, 0)))
TEST_NONE(stlxr_0, stlxr(w0, w1, MemOperand(x2, 0)))
TEST_NONE(stlxr_1, stlxr(w0, x1, MemOperand(x2, 0)))
TEST_NONE(stlxrb_0, stlxrb(w0, w1, MemOperand(x2, 0)))
TEST_NONE(stlxrh_0, stlxrh(w0, w1, MemOperand(x2, 0)))
TEST_NONE(stnp_0, stnp(w0, w1, MemOperand(x2, -136)))
TEST_NONE(stnp_1, stnp(x0, x1, MemOperand(x2, 480)))
TEST_NONE(stp_0, stp(w0, w1, MemOperand(x2, 184)))
TEST_NONE(stp_1, stp(w0, w1, MemOperand(x2, -16, PostIndex)))
TEST_NONE(stp_2, stp(w0, w1, MemOperand(x2, -44, PreIndex)))
TEST_NONE(stp_3, stp(x0, x1, MemOperand(x2, 472)))
TEST_NONE(stp_4, stp(x0, x1, MemOperand(x2, 320, PostIndex)))
TEST_NONE(stp_5, stp(x0, x1, MemOperand(x2, -392, PreIndex)))
TEST_NONE(strb_0, strb(w0, MemOperand(x1, 146, PostIndex)))
TEST_NONE(strb_1, strb(w0, MemOperand(x1, -120, PreIndex)))
TEST_NONE(strb_2, strb(w0, MemOperand(x1, 1477)))
TEST_NONE(strb_3, strb(w0, MemOperand(x1, x2, LSL, 0)))
TEST_NONE(strb_4, strb(w0, MemOperand(x1, w2, SXTW, 0)))
TEST_NONE(strb_5, strb(w0, MemOperand(x1, x2, SXTX, 0)))
TEST_NONE(strh_0, strh(w0, MemOperand(x1, -228, PostIndex)))
TEST_NONE(strh_1, strh(w0, MemOperand(x1, 240, PreIndex)))
TEST_NONE(strh_2, strh(w0, MemOperand(x1, 742)))
TEST_NONE(strh_3, strh(w0, MemOperand(x1, w2, UXTW, 0)))
TEST_NONE(strh_4, strh(w0, MemOperand(x1, x2, LSL, 1)))
TEST_NONE(str_0, str(w0, MemOperand(x1, 161, PostIndex)))
TEST_NONE(str_1, str(w0, MemOperand(x1, -86, PreIndex)))
TEST_NONE(str_2, str(w0, MemOperand(x1, 9200)))
TEST_NONE(str_3, str(x0, MemOperand(x1, 121, PostIndex)))
TEST_NONE(str_4, str(x0, MemOperand(x1, -6, PreIndex)))
TEST_NONE(str_5, str(x0, MemOperand(x1, 28952)))
TEST_NONE(str_6, str(w0, MemOperand(x1, w2, SXTW, 2)))
TEST_NONE(str_7, str(w0, MemOperand(x1, x2, LSL, 2)))
TEST_NONE(str_8, str(x0, MemOperand(x1, w2, UXTW, 0)))
TEST_NONE(str_9, str(x0, MemOperand(x1, x2, SXTX, 0)))
TEST_NONE(sturb_0, sturb(w0, MemOperand(x1, 67)))
TEST_NONE(sturh_0, sturh(w0, MemOperand(x1, 173)))
TEST_NONE(stur_0, stur(w0, MemOperand(x1, 151)))
TEST_NONE(stur_1, stur(x0, MemOperand(x1, 134)))
TEST_NONE(stxp_0, stxp(w0, w1, w2, MemOperand(x3, 0)))
TEST_NONE(stxp_1, stxp(w0, x1, x2, MemOperand(x3, 0)))
TEST_NONE(stxr_0, stxr(w0, w1, MemOperand(x2, 0)))
TEST_NONE(stxr_1, stxr(w0, x1, MemOperand(x2, 0)))
TEST_NONE(stxrb_0, stxrb(w0, w1, MemOperand(x2, 0)))
TEST_NONE(stxrh_0, stxrh(w0, w1, MemOperand(x2, 0)))
TEST_NONE(subs_0, subs(w0, w1, Operand(w2, SXTH, 0)))
TEST_NONE(subs_1, subs(x0, x1, Operand(x2, UXTX, 3)))
TEST_NONE(subs_2, subs(x0, x1, Operand(w2, UXTB, 3)))
TEST_NONE(subs_3, subs(w0, w1, 0x812))
TEST_NONE(subs_4, subs(x0, x1, 0x915))
TEST_NONE(subs_5, subs(w0, w1, Operand(w2, ASR, 7)))
TEST_NONE(subs_6, subs(x0, x1, Operand(x2, LSR, 45)))
TEST_NONE(sub_0, sub(w0, w1, Operand(w2, SXTW, 0)))
TEST_NONE(sub_1, sub(x0, x1, Operand(x2, UXTX, 3)))
TEST_NONE(sub_2, sub(x0, x1, Operand(w2, UXTB, 2)))
TEST_NONE(sub_3, sub(w0, w1, 0x492 << 12))
TEST_NONE(sub_4, sub(x0, x1, 0xa56 << 12))
TEST_NONE(sub_5, sub(w0, w1, Operand(w2, ASR, 2)))
TEST_NONE(sub_6, sub(x0, x1, Operand(x2, ASR, 45)))
TEST_NONE(svc_0, svc(0x1220))
TEST_NONE(sxtb_0, sxtb(w0, w1))
TEST_NONE(sxtb_1, sxtb(x0, w1))
TEST_NONE(sxth_0, sxth(w0, w1))
TEST_NONE(sxth_1, sxth(x0, w1))
TEST_NONE(sxtw_0, sxtw(x0, w1))
TEST_NONE(sys_0, sys(6, 10, 8, 3, x0))
TEST_NONE(sys_1, sys(0x2ae9, x0))
TEST_NONE(tbnz_0, tbnz(x0, 0, &label))
TEST_NONE(tbnz_1, tbnz(x0, 37, &label))
TEST_NONE(tbnz_2, tbnz(w0, 20, &label))
TEST_NONE(tbz_0, tbz(x0, 24, &label))
TEST_NONE(tbz_1, tbz(x0, 60, &label))
TEST_NONE(tbz_2, tbz(w0, 13, &label))
TEST_NONE(tst_0, tst(w0, 0x3c0))
TEST_NONE(tst_1, tst(x0, 0x3c00))
TEST_NONE(tst_2, tst(w0, Operand(w1, LSL, 12)))
TEST_NONE(tst_3, tst(x0, Operand(x1, ASR, 16)))
TEST_NONE(ubfiz_0, ubfiz(w0, w1, 10, 9))
TEST_NONE(ubfiz_1, ubfiz(x0, x1, 13, 49))
TEST_NONE(ubfm_0, ubfm(w0, w1, 18, 3))
TEST_NONE(ubfm_1, ubfm(x0, x1, 54, 30))
TEST_NONE(ubfx_0, ubfx(w0, w1, 24, 7))
TEST_NONE(ubfx_1, ubfx(x0, x1, 11, 21))
TEST_NONE(udiv_0, udiv(w0, w1, w2))
TEST_NONE(udiv_1, udiv(x0, x1, x2))
TEST_NONE(umaddl_0, umaddl(x0, w1, w2, x3))
TEST_NONE(umsubl_0, umsubl(x0, w1, w2, x3))
TEST_NONE(umulh_0, umulh(x0, x1, x2))
TEST_NONE(umull_0, umull(x0, w1, w2))
TEST_NONE(uxtb_0, uxtb(w0, w1))
TEST_NONE(uxth_0, uxth(w0, w1))

#define TEST_FP(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kFP), FP_##NAME, ASM)
TEST_FP(fabs_0, fabs(d0, d1))
TEST_FP(fabs_1, fabs(s0, s1))
TEST_FP(fadd_0, fadd(d0, d1, d2))
TEST_FP(fadd_1, fadd(s0, s1, s2))
TEST_FP(fccmpe_0, fccmpe(d0, d1, NCFlag, hi))
TEST_FP(fccmpe_1, fccmpe(s0, s1, NZCFlag, ne))
TEST_FP(fccmp_0, fccmp(d0, d1, NZCVFlag, pl))
TEST_FP(fccmp_1, fccmp(s0, s1, ZCVFlag, eq))
TEST_FP(fcmpe_0, fcmpe(d0, 0.0))
TEST_FP(fcmpe_1, fcmpe(d0, d1))
TEST_FP(fcmpe_2, fcmpe(s0, 0.0))
TEST_FP(fcmpe_3, fcmpe(s0, s1))
TEST_FP(fcmp_0, fcmp(d0, 0.0))
TEST_FP(fcmp_1, fcmp(d0, d1))
TEST_FP(fcmp_2, fcmp(s0, 0.0))
TEST_FP(fcmp_3, fcmp(s0, s1))
TEST_FP(fcsel_0, fcsel(d0, d1, d2, cs))
TEST_FP(fcsel_1, fcsel(s0, s1, s2, ne))
TEST_FP(fcvtas_0, fcvtas(w0, d1))
TEST_FP(fcvtas_1, fcvtas(w0, s1))
TEST_FP(fcvtas_2, fcvtas(x0, d1))
TEST_FP(fcvtas_3, fcvtas(x0, s1))
TEST_FP(fcvtau_0, fcvtau(w0, d1))
TEST_FP(fcvtau_1, fcvtau(w0, s1))
TEST_FP(fcvtau_2, fcvtau(x0, d1))
TEST_FP(fcvtau_3, fcvtau(x0, s1))
TEST_FP(fcvtms_0, fcvtms(w0, d1))
TEST_FP(fcvtms_1, fcvtms(w0, s1))
TEST_FP(fcvtms_2, fcvtms(x0, d1))
TEST_FP(fcvtms_3, fcvtms(x0, s1))
TEST_FP(fcvtmu_0, fcvtmu(w0, d1))
TEST_FP(fcvtmu_1, fcvtmu(w0, s1))
TEST_FP(fcvtmu_2, fcvtmu(x0, d1))
TEST_FP(fcvtmu_3, fcvtmu(x0, s1))
TEST_FP(fcvtns_0, fcvtns(w0, d1))
TEST_FP(fcvtns_1, fcvtns(w0, s1))
TEST_FP(fcvtns_2, fcvtns(x0, d1))
TEST_FP(fcvtns_3, fcvtns(x0, s1))
TEST_FP(fcvtnu_0, fcvtnu(w0, d1))
TEST_FP(fcvtnu_1, fcvtnu(w0, s1))
TEST_FP(fcvtnu_2, fcvtnu(x0, d1))
TEST_FP(fcvtnu_3, fcvtnu(x0, s1))
TEST_FP(fcvtps_0, fcvtps(w0, d1))
TEST_FP(fcvtps_1, fcvtps(w0, s1))
TEST_FP(fcvtps_2, fcvtps(x0, d1))
TEST_FP(fcvtps_3, fcvtps(x0, s1))
TEST_FP(fcvtpu_0, fcvtpu(w0, d1))
TEST_FP(fcvtpu_1, fcvtpu(w0, s1))
TEST_FP(fcvtpu_2, fcvtpu(x0, d1))
TEST_FP(fcvtpu_3, fcvtpu(x0, s1))
TEST_FP(fcvtzs_0, fcvtzs(w0, d1, 5))
TEST_FP(fcvtzs_1, fcvtzs(w0, s1, 5))
TEST_FP(fcvtzs_2, fcvtzs(x0, d1, 5))
TEST_FP(fcvtzs_3, fcvtzs(x0, s1, 5))
TEST_FP(fcvtzs_4, fcvtzs(w0, d1))
TEST_FP(fcvtzs_5, fcvtzs(w0, s1))
TEST_FP(fcvtzs_6, fcvtzs(x0, d1))
TEST_FP(fcvtzs_7, fcvtzs(x0, s1))
TEST_FP(fcvtzu_0, fcvtzu(w0, d1, 5))
TEST_FP(fcvtzu_1, fcvtzu(w0, s1, 5))
TEST_FP(fcvtzu_2, fcvtzu(x0, d1, 5))
TEST_FP(fcvtzu_3, fcvtzu(x0, s1, 5))
TEST_FP(fcvtzu_4, fcvtzu(w0, d1))
TEST_FP(fcvtzu_5, fcvtzu(w0, s1))
TEST_FP(fcvtzu_6, fcvtzu(x0, d1))
TEST_FP(fcvtzu_7, fcvtzu(x0, s1))
TEST_FP(fcvt_0, fcvt(d0, h1))
TEST_FP(fcvt_1, fcvt(d0, s1))
TEST_FP(fcvt_2, fcvt(h0, d1))
TEST_FP(fcvt_3, fcvt(h0, s1))
TEST_FP(fcvt_4, fcvt(s0, d1))
TEST_FP(fcvt_5, fcvt(s0, h1))
TEST_FP(fdiv_0, fdiv(d0, d1, d2))
TEST_FP(fdiv_1, fdiv(s0, s1, s2))
TEST_FP(fmadd_0, fmadd(d0, d1, d2, d3))
TEST_FP(fmadd_1, fmadd(s0, s1, s2, s3))
TEST_FP(fmaxnm_0, fmaxnm(d0, d1, d2))
TEST_FP(fmaxnm_1, fmaxnm(s0, s1, s2))
TEST_FP(fmax_0, fmax(d0, d1, d2))
TEST_FP(fmax_1, fmax(s0, s1, s2))
TEST_FP(fminnm_0, fminnm(d0, d1, d2))
TEST_FP(fminnm_1, fminnm(s0, s1, s2))
TEST_FP(fmin_0, fmin(d0, d1, d2))
TEST_FP(fmin_1, fmin(s0, s1, s2))
TEST_FP(fmov_0, fmov(d0, d1))
TEST_FP(fmov_1, fmov(s0, s1))
TEST_FP(fmov_2, fmov(w0, s1))
TEST_FP(fmov_3, fmov(x0, d1))
TEST_FP(fmov_4, fmov(d0, x1))
TEST_FP(fmov_5, fmov(s0, w1))
TEST_FP(fmov_6, fmov(d0, -0.15625))
TEST_FP(fmov_7, fmov(s0, -2.875f))
TEST_FP(fmsub_0, fmsub(d0, d1, d2, d3))
TEST_FP(fmsub_1, fmsub(s0, s1, s2, s3))
TEST_FP(fmul_0, fmul(d0, d1, d2))
TEST_FP(fmul_1, fmul(s0, s1, s2))
TEST_FP(fneg_0, fneg(d0, d1))
TEST_FP(fneg_1, fneg(s0, s1))
TEST_FP(fnmadd_0, fnmadd(d0, d1, d2, d3))
TEST_FP(fnmadd_1, fnmadd(s0, s1, s2, s3))
TEST_FP(fnmsub_0, fnmsub(d0, d1, d2, d3))
TEST_FP(fnmsub_1, fnmsub(s0, s1, s2, s3))
TEST_FP(fnmul_0, fnmul(d0, d1, d2))
TEST_FP(fnmul_1, fnmul(s0, s1, s2))
TEST_FP(frinta_0, frinta(d0, d1))
TEST_FP(frinta_1, frinta(s0, s1))
TEST_FP(frinti_0, frinti(d0, d1))
TEST_FP(frinti_1, frinti(s0, s1))
TEST_FP(frintm_0, frintm(d0, d1))
TEST_FP(frintm_1, frintm(s0, s1))
TEST_FP(frintn_0, frintn(d0, d1))
TEST_FP(frintn_1, frintn(s0, s1))
TEST_FP(frintp_0, frintp(d0, d1))
TEST_FP(frintp_1, frintp(s0, s1))
TEST_FP(frintx_0, frintx(d0, d1))
TEST_FP(frintx_1, frintx(s0, s1))
TEST_FP(frintz_0, frintz(d0, d1))
TEST_FP(frintz_1, frintz(s0, s1))
TEST_FP(fsqrt_0, fsqrt(d0, d1))
TEST_FP(fsqrt_1, fsqrt(s0, s1))
TEST_FP(fsub_0, fsub(d0, d1, d2))
TEST_FP(fsub_1, fsub(s0, s1, s2))
TEST_FP(ldnp_0, ldnp(d0, d1, MemOperand(x2, -32)))
TEST_FP(ldnp_1, ldnp(s0, s1, MemOperand(x2, -180)))
TEST_FP(ldp_0, ldp(d0, d1, MemOperand(x2, 112)))
TEST_FP(ldp_1, ldp(d0, d1, MemOperand(x2, 24, PostIndex)))
TEST_FP(ldp_2, ldp(d0, d1, MemOperand(x2, 192, PreIndex)))
TEST_FP(ldp_3, ldp(s0, s1, MemOperand(x2, -72)))
TEST_FP(ldp_4, ldp(s0, s1, MemOperand(x2, 0, PostIndex)))
TEST_FP(ldp_5, ldp(s0, s1, MemOperand(x2, 144, PreIndex)))
TEST_FP(ldr_0, ldr(d0, MemOperand(x1, -240, PostIndex)))
TEST_FP(ldr_1, ldr(d0, MemOperand(x1, 110, PreIndex)))
TEST_FP(ldr_2, ldr(d0, MemOperand(x1, 4048)))
TEST_FP(ldr_3, ldr(h0, MemOperand(x1, -76, PostIndex)))
TEST_FP(ldr_4, ldr(h0, MemOperand(x1, -124, PreIndex)))
TEST_FP(ldr_5, ldr(h0, MemOperand(x1, 6230)))
TEST_FP(ldr_6, ldr(s0, MemOperand(x1, 148, PostIndex)))
TEST_FP(ldr_7, ldr(s0, MemOperand(x1, 68, PreIndex)))
TEST_FP(ldr_8, ldr(s0, MemOperand(x1, 7424)))
TEST_FP(ldr_9, ldr(d0, 0x35a20))
TEST_FP(ldr_10, ldr(s0, 0x1053a))
TEST_FP(ldr_11, ldr(d0, MemOperand(x1, w2, SXTW, 3)))
TEST_FP(ldr_12, ldr(d0, MemOperand(x1, x2, SXTX, 3)))
TEST_FP(ldr_13, ldr(h0, MemOperand(x1, w2, UXTW, 1)))
TEST_FP(ldr_14, ldr(h0, MemOperand(x1, x2, SXTX, 0)))
TEST_FP(ldr_15, ldr(s0, MemOperand(x1, w2, SXTW, 2)))
TEST_FP(ldr_16, ldr(s0, MemOperand(x1, x2, SXTX, 2)))
TEST_FP(ldur_0, ldur(d0, MemOperand(x1, -146)))
TEST_FP(ldur_1, ldur(h0, MemOperand(x1, -117)))
TEST_FP(ldur_2, ldur(s0, MemOperand(x1, 209)))
TEST_FP(scvtf_0, scvtf(d0, w1, 5))
TEST_FP(scvtf_1, scvtf(d0, x1, 5))
TEST_FP(scvtf_2, scvtf(s0, w1, 5))
TEST_FP(scvtf_3, scvtf(s0, x1, 5))
TEST_FP(scvtf_4, scvtf(d0, w1))
TEST_FP(scvtf_5, scvtf(d0, x1))
TEST_FP(scvtf_6, scvtf(s0, w1))
TEST_FP(scvtf_7, scvtf(s0, x1))
TEST_FP(stnp_0, stnp(d0, d1, MemOperand(x2, 304)))
TEST_FP(stnp_1, stnp(s0, s1, MemOperand(x2, -12)))
TEST_FP(stp_0, stp(d0, d1, MemOperand(x2, 168)))
TEST_FP(stp_1, stp(d0, d1, MemOperand(x2, -376, PostIndex)))
TEST_FP(stp_2, stp(d0, d1, MemOperand(x2, 296, PreIndex)))
TEST_FP(stp_3, stp(s0, s1, MemOperand(x2, -256)))
TEST_FP(stp_4, stp(s0, s1, MemOperand(x2, 208, PostIndex)))
TEST_FP(stp_5, stp(s0, s1, MemOperand(x2, -4, PreIndex)))
TEST_FP(str_0, str(d0, MemOperand(x1, -181, PostIndex)))
TEST_FP(str_1, str(d0, MemOperand(x1, 91, PreIndex)))
TEST_FP(str_2, str(d0, MemOperand(x1, 32672)))
TEST_FP(str_3, str(h0, MemOperand(x1, -5, PostIndex)))
TEST_FP(str_4, str(h0, MemOperand(x1, 213, PreIndex)))
TEST_FP(str_5, str(h0, MemOperand(x1, 6406)))
TEST_FP(str_6, str(s0, MemOperand(x1, -81, PostIndex)))
TEST_FP(str_7, str(s0, MemOperand(x1, -126, PreIndex)))
TEST_FP(str_8, str(s0, MemOperand(x1, 15692)))
TEST_FP(str_9, str(d0, MemOperand(x1, w2, SXTW, 0)))
TEST_FP(str_10, str(d0, MemOperand(x1, x2, LSL, 0)))
TEST_FP(str_11, str(h0, MemOperand(x1, w2, UXTW, 1)))
TEST_FP(str_12, str(h0, MemOperand(x1, x2, SXTX, 1)))
TEST_FP(str_13, str(s0, MemOperand(x1, w2, UXTW, 0)))
TEST_FP(str_14, str(s0, MemOperand(x1, x2, SXTX, 2)))
TEST_FP(stur_0, stur(d0, MemOperand(x1, 22)))
TEST_FP(stur_1, stur(h0, MemOperand(x1, -236)))
TEST_FP(stur_2, stur(s0, MemOperand(x1, 23)))
TEST_FP(ucvtf_0, ucvtf(d0, w1, 5))
TEST_FP(ucvtf_1, ucvtf(d0, x1, 5))
TEST_FP(ucvtf_2, ucvtf(s0, w1, 5))
TEST_FP(ucvtf_3, ucvtf(s0, x1, 5))
TEST_FP(ucvtf_4, ucvtf(d0, w1))
TEST_FP(ucvtf_5, ucvtf(d0, x1))
TEST_FP(ucvtf_6, ucvtf(s0, w1))
TEST_FP(ucvtf_7, ucvtf(s0, x1))

#define TEST_FP_FRINT(NAME, ASM)                                 \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kFP,                    \
                            CPUFeatures::kFrintToFixedSizedInt), \
                FP_##NAME,                                       \
                ASM)
TEST_FP_FRINT(frint32x_0, frint32x(d0, d1))
TEST_FP_FRINT(frint32x_1, frint32x(s0, s1))
TEST_FP_FRINT(frint32z_0, frint32z(d0, d1))
TEST_FP_FRINT(frint32z_1, frint32z(s0, s1))
TEST_FP_FRINT(frint64x_0, frint64x(d0, d1))
TEST_FP_FRINT(frint64x_1, frint64x(s0, s1))
TEST_FP_FRINT(frint64z_0, frint64z(d0, d1))
TEST_FP_FRINT(frint64z_1, frint64z(s0, s1))

#define TEST_BTI(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kBTI), BTI_##NAME, ASM)
TEST_BTI(bti_0, bti(EmitBTI))
TEST_BTI(bti_1, bti(EmitBTI_c))
TEST_BTI(bti_2, bti(EmitBTI_j))
TEST_BTI(bti_3, bti(EmitBTI_jc))

#define TEST_RAS(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kRAS), RAS_##NAME, ASM)
TEST_RAS(esb_0, esb())

#define TEST_RNG(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kRNG), RNG_##NAME, ASM)
TEST_RNG(mrs_0, mrs(x0, RNDR))
TEST_RNG(mrs_1, mrs(x0, RNDRRS))

#define TEST_NEON(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kNEON), NEON_##NAME, ASM)
TEST_NEON(abs_0, abs(v0.V8B(), v1.V8B()))
TEST_NEON(abs_1, abs(v0.V16B(), v1.V16B()))
TEST_NEON(abs_2, abs(v0.V4H(), v1.V4H()))
TEST_NEON(abs_3, abs(v0.V8H(), v1.V8H()))
TEST_NEON(abs_4, abs(v0.V2S(), v1.V2S()))
TEST_NEON(abs_5, abs(v0.V4S(), v1.V4S()))
TEST_NEON(abs_6, abs(v0.V2D(), v1.V2D()))
TEST_NEON(abs_7, abs(d0, d1))
TEST_NEON(addhn_0, addhn(v0.V8B(), v1.V8H(), v2.V8H()))
TEST_NEON(addhn_1, addhn(v0.V4H(), v1.V4S(), v2.V4S()))
TEST_NEON(addhn_2, addhn(v0.V2S(), v1.V2D(), v2.V2D()))
TEST_NEON(addhn2_0, addhn2(v0.V16B(), v1.V8H(), v2.V8H()))
TEST_NEON(addhn2_1, addhn2(v0.V8H(), v1.V4S(), v2.V4S()))
TEST_NEON(addhn2_2, addhn2(v0.V4S(), v1.V2D(), v2.V2D()))
TEST_NEON(addp_0, addp(d0, v1.V2D()))
TEST_NEON(addp_1, addp(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(addp_2, addp(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(addp_3, addp(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(addp_4, addp(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(addp_5, addp(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(addp_6, addp(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(addp_7, addp(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(addv_0, addv(b0, v1.V8B()))
TEST_NEON(addv_1, addv(b0, v1.V16B()))
TEST_NEON(addv_2, addv(h0, v1.V4H()))
TEST_NEON(addv_3, addv(h0, v1.V8H()))
TEST_NEON(addv_4, addv(s0, v1.V4S()))
TEST_NEON(add_0, add(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(add_1, add(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(add_2, add(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(add_3, add(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(add_4, add(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(add_5, add(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(add_6, add(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(add_7, add(d0, d1, d2))
TEST_NEON(and_0, and_(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(and_1, and_(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(bic_0, bic(v0.V4H(), 0x55, 0))
TEST_NEON(bic_1, bic(v0.V8H(), 0x3a, 0))
TEST_NEON(bic_2, bic(v0.V2S(), 0xd8, 0))
TEST_NEON(bic_3, bic(v0.V4S(), 0x15, 0))
TEST_NEON(bic_4, bic(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(bic_5, bic(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(bif_0, bif(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(bif_1, bif(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(bit_0, bit(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(bit_1, bit(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(bsl_0, bsl(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(bsl_1, bsl(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(cls_0, cls(v0.V8B(), v1.V8B()))
TEST_NEON(cls_1, cls(v0.V16B(), v1.V16B()))
TEST_NEON(cls_2, cls(v0.V4H(), v1.V4H()))
TEST_NEON(cls_3, cls(v0.V8H(), v1.V8H()))
TEST_NEON(cls_4, cls(v0.V2S(), v1.V2S()))
TEST_NEON(cls_5, cls(v0.V4S(), v1.V4S()))
TEST_NEON(clz_0, clz(v0.V8B(), v1.V8B()))
TEST_NEON(clz_1, clz(v0.V16B(), v1.V16B()))
TEST_NEON(clz_2, clz(v0.V4H(), v1.V4H()))
TEST_NEON(clz_3, clz(v0.V8H(), v1.V8H()))
TEST_NEON(clz_4, clz(v0.V2S(), v1.V2S()))
TEST_NEON(clz_5, clz(v0.V4S(), v1.V4S()))
TEST_NEON(cmeq_0, cmeq(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(cmeq_1, cmeq(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(cmeq_2, cmeq(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(cmeq_3, cmeq(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(cmeq_4, cmeq(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(cmeq_5, cmeq(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(cmeq_6, cmeq(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(cmeq_7, cmeq(d0, d1, d2))
TEST_NEON(cmeq_8, cmeq(v0.V8B(), v1.V8B(), 0))
TEST_NEON(cmeq_9, cmeq(v0.V16B(), v1.V16B(), 0))
TEST_NEON(cmeq_10, cmeq(v0.V4H(), v1.V4H(), 0))
TEST_NEON(cmeq_11, cmeq(v0.V8H(), v1.V8H(), 0))
TEST_NEON(cmeq_12, cmeq(v0.V2S(), v1.V2S(), 0))
TEST_NEON(cmeq_13, cmeq(v0.V4S(), v1.V4S(), 0))
TEST_NEON(cmeq_14, cmeq(v0.V2D(), v1.V2D(), 0))
TEST_NEON(cmeq_15, cmeq(d0, d1, 0))
TEST_NEON(cmge_0, cmge(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(cmge_1, cmge(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(cmge_2, cmge(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(cmge_3, cmge(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(cmge_4, cmge(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(cmge_5, cmge(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(cmge_6, cmge(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(cmge_7, cmge(d0, d1, d2))
TEST_NEON(cmge_8, cmge(v0.V8B(), v1.V8B(), 0))
TEST_NEON(cmge_9, cmge(v0.V16B(), v1.V16B(), 0))
TEST_NEON(cmge_10, cmge(v0.V4H(), v1.V4H(), 0))
TEST_NEON(cmge_11, cmge(v0.V8H(), v1.V8H(), 0))
TEST_NEON(cmge_12, cmge(v0.V2S(), v1.V2S(), 0))
TEST_NEON(cmge_13, cmge(v0.V4S(), v1.V4S(), 0))
TEST_NEON(cmge_14, cmge(v0.V2D(), v1.V2D(), 0))
TEST_NEON(cmge_15, cmge(d0, d1, 0))
TEST_NEON(cmgt_0, cmgt(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(cmgt_1, cmgt(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(cmgt_2, cmgt(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(cmgt_3, cmgt(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(cmgt_4, cmgt(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(cmgt_5, cmgt(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(cmgt_6, cmgt(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(cmgt_7, cmgt(d0, d1, d2))
TEST_NEON(cmgt_8, cmgt(v0.V8B(), v1.V8B(), 0))
TEST_NEON(cmgt_9, cmgt(v0.V16B(), v1.V16B(), 0))
TEST_NEON(cmgt_10, cmgt(v0.V4H(), v1.V4H(), 0))
TEST_NEON(cmgt_11, cmgt(v0.V8H(), v1.V8H(), 0))
TEST_NEON(cmgt_12, cmgt(v0.V2S(), v1.V2S(), 0))
TEST_NEON(cmgt_13, cmgt(v0.V4S(), v1.V4S(), 0))
TEST_NEON(cmgt_14, cmgt(v0.V2D(), v1.V2D(), 0))
TEST_NEON(cmgt_15, cmgt(d0, d1, 0))
TEST_NEON(cmhi_0, cmhi(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(cmhi_1, cmhi(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(cmhi_2, cmhi(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(cmhi_3, cmhi(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(cmhi_4, cmhi(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(cmhi_5, cmhi(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(cmhi_6, cmhi(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(cmhi_7, cmhi(d0, d1, d2))
TEST_NEON(cmhs_0, cmhs(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(cmhs_1, cmhs(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(cmhs_2, cmhs(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(cmhs_3, cmhs(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(cmhs_4, cmhs(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(cmhs_5, cmhs(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(cmhs_6, cmhs(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(cmhs_7, cmhs(d0, d1, d2))
TEST_NEON(cmle_0, cmle(v0.V8B(), v1.V8B(), 0))
TEST_NEON(cmle_1, cmle(v0.V16B(), v1.V16B(), 0))
TEST_NEON(cmle_2, cmle(v0.V4H(), v1.V4H(), 0))
TEST_NEON(cmle_3, cmle(v0.V8H(), v1.V8H(), 0))
TEST_NEON(cmle_4, cmle(v0.V2S(), v1.V2S(), 0))
TEST_NEON(cmle_5, cmle(v0.V4S(), v1.V4S(), 0))
TEST_NEON(cmle_6, cmle(v0.V2D(), v1.V2D(), 0))
TEST_NEON(cmle_7, cmle(d0, d1, 0))
TEST_NEON(cmlt_0, cmlt(v0.V8B(), v1.V8B(), 0))
TEST_NEON(cmlt_1, cmlt(v0.V16B(), v1.V16B(), 0))
TEST_NEON(cmlt_2, cmlt(v0.V4H(), v1.V4H(), 0))
TEST_NEON(cmlt_3, cmlt(v0.V8H(), v1.V8H(), 0))
TEST_NEON(cmlt_4, cmlt(v0.V2S(), v1.V2S(), 0))
TEST_NEON(cmlt_5, cmlt(v0.V4S(), v1.V4S(), 0))
TEST_NEON(cmlt_6, cmlt(v0.V2D(), v1.V2D(), 0))
TEST_NEON(cmlt_7, cmlt(d0, d1, 0))
TEST_NEON(cmtst_0, cmtst(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(cmtst_1, cmtst(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(cmtst_2, cmtst(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(cmtst_3, cmtst(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(cmtst_4, cmtst(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(cmtst_5, cmtst(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(cmtst_6, cmtst(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(cmtst_7, cmtst(d0, d1, d2))
TEST_NEON(cnt_0, cnt(v0.V8B(), v1.V8B()))
TEST_NEON(cnt_1, cnt(v0.V16B(), v1.V16B()))
TEST_NEON(dup_0, dup(v0.V8B(), v1.B(), 11))
TEST_NEON(dup_1, dup(v0.V16B(), v1.B(), 11))
TEST_NEON(dup_2, dup(v0.V4H(), v1.H(), 2))
TEST_NEON(dup_3, dup(v0.V8H(), v1.H(), 5))
TEST_NEON(dup_4, dup(v0.V2S(), v1.S(), 3))
TEST_NEON(dup_5, dup(v0.V4S(), v1.S(), 0))
TEST_NEON(dup_6, dup(v0.V2D(), v1.D(), 0))
TEST_NEON(dup_7, dup(b0, v1.B(), 14))
TEST_NEON(dup_8, dup(h0, v1.H(), 7))
TEST_NEON(dup_9, dup(s0, v1.S(), 2))
TEST_NEON(dup_10, dup(d0, v1.D(), 0))
TEST_NEON(dup_11, dup(v0.V8B(), w1))
TEST_NEON(dup_12, dup(v0.V16B(), w1))
TEST_NEON(dup_13, dup(v0.V4H(), w1))
TEST_NEON(dup_14, dup(v0.V8H(), w1))
TEST_NEON(dup_15, dup(v0.V2S(), w1))
TEST_NEON(dup_16, dup(v0.V4S(), w1))
TEST_NEON(dup_17, dup(v0.V2D(), x1))
TEST_NEON(eor_0, eor(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(eor_1, eor(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(ext_0, ext(v0.V8B(), v1.V8B(), v2.V8B(), 3))
TEST_NEON(ext_1, ext(v0.V16B(), v1.V16B(), v2.V16B(), 5))
TEST_NEON(ins_0, ins(v0.B(), 6, v1.B(), 13))
TEST_NEON(ins_1, ins(v0.H(), 1, v1.H(), 5))
TEST_NEON(ins_2, ins(v0.S(), 2, v1.S(), 1))
TEST_NEON(ins_3, ins(v0.D(), 1, v1.D(), 1))
TEST_NEON(ins_4, ins(v0.B(), 15, w1))
TEST_NEON(ins_5, ins(v0.H(), 4, w1))
TEST_NEON(ins_6, ins(v0.S(), 2, w1))
TEST_NEON(ins_7, ins(v0.D(), 1, x1))
TEST_NEON(ld1r_0, ld1r(v0.V8B(), MemOperand(x1)))
TEST_NEON(ld1r_1, ld1r(v0.V16B(), MemOperand(x1)))
TEST_NEON(ld1r_2, ld1r(v0.V4H(), MemOperand(x1)))
TEST_NEON(ld1r_3, ld1r(v0.V8H(), MemOperand(x1)))
TEST_NEON(ld1r_4, ld1r(v0.V2S(), MemOperand(x1)))
TEST_NEON(ld1r_5, ld1r(v0.V4S(), MemOperand(x1)))
TEST_NEON(ld1r_6, ld1r(v0.V1D(), MemOperand(x1)))
TEST_NEON(ld1r_7, ld1r(v0.V2D(), MemOperand(x1)))
TEST_NEON(ld1r_8, ld1r(v0.V8B(), MemOperand(x1, 1, PostIndex)))
TEST_NEON(ld1r_9, ld1r(v0.V16B(), MemOperand(x1, 1, PostIndex)))
TEST_NEON(ld1r_10, ld1r(v0.V4H(), MemOperand(x1, 2, PostIndex)))
TEST_NEON(ld1r_11, ld1r(v0.V8H(), MemOperand(x1, 2, PostIndex)))
TEST_NEON(ld1r_12, ld1r(v0.V2S(), MemOperand(x1, 4, PostIndex)))
TEST_NEON(ld1r_13, ld1r(v0.V4S(), MemOperand(x1, 4, PostIndex)))
TEST_NEON(ld1r_14, ld1r(v0.V1D(), MemOperand(x1, 8, PostIndex)))
TEST_NEON(ld1r_15, ld1r(v0.V2D(), MemOperand(x1, 8, PostIndex)))
TEST_NEON(ld1r_16, ld1r(v0.V8B(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1r_17, ld1r(v0.V16B(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1r_18, ld1r(v0.V4H(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1r_19, ld1r(v0.V8H(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1r_20, ld1r(v0.V2S(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1r_21, ld1r(v0.V4S(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1r_22, ld1r(v0.V1D(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1r_23, ld1r(v0.V2D(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1_0, ld1(v0.V8B(), MemOperand(x1)))
TEST_NEON(ld1_1, ld1(v0.V16B(), MemOperand(x1)))
TEST_NEON(ld1_2, ld1(v0.V4H(), MemOperand(x1)))
TEST_NEON(ld1_3, ld1(v0.V8H(), MemOperand(x1)))
TEST_NEON(ld1_4, ld1(v0.V2S(), MemOperand(x1)))
TEST_NEON(ld1_5, ld1(v0.V4S(), MemOperand(x1)))
TEST_NEON(ld1_6, ld1(v0.V1D(), MemOperand(x1)))
TEST_NEON(ld1_7, ld1(v0.V2D(), MemOperand(x1)))
TEST_NEON(ld1_8, ld1(v0.V8B(), v1.V8B(), MemOperand(x2)))
TEST_NEON(ld1_9, ld1(v0.V16B(), v1.V16B(), MemOperand(x2)))
TEST_NEON(ld1_10, ld1(v0.V4H(), v1.V4H(), MemOperand(x2)))
TEST_NEON(ld1_11, ld1(v0.V8H(), v1.V8H(), MemOperand(x2)))
TEST_NEON(ld1_12, ld1(v0.V2S(), v1.V2S(), MemOperand(x2)))
TEST_NEON(ld1_13, ld1(v0.V4S(), v1.V4S(), MemOperand(x2)))
TEST_NEON(ld1_14, ld1(v0.V1D(), v1.V1D(), MemOperand(x2)))
TEST_NEON(ld1_15, ld1(v0.V2D(), v1.V2D(), MemOperand(x2)))
TEST_NEON(ld1_16, ld1(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3)))
TEST_NEON(ld1_17, ld1(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3)))
TEST_NEON(ld1_18, ld1(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3)))
TEST_NEON(ld1_19, ld1(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3)))
TEST_NEON(ld1_20, ld1(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3)))
TEST_NEON(ld1_21, ld1(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3)))
TEST_NEON(ld1_22, ld1(v0.V1D(), v1.V1D(), v2.V1D(), MemOperand(x3)))
TEST_NEON(ld1_23, ld1(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3)))
TEST_NEON(ld1_24, ld1(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4)))
TEST_NEON(ld1_25,
          ld1(v0.V16B(), v1.V16B(), v2.V16B(), v3.V16B(), MemOperand(x4)))
TEST_NEON(ld1_26, ld1(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4)))
TEST_NEON(ld1_27, ld1(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4)))
TEST_NEON(ld1_28, ld1(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4)))
TEST_NEON(ld1_29, ld1(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4)))
TEST_NEON(ld1_30, ld1(v0.V1D(), v1.V1D(), v2.V1D(), v3.V1D(), MemOperand(x4)))
TEST_NEON(ld1_31, ld1(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4)))
TEST_NEON(ld1_32, ld1(v0.V8B(), MemOperand(x1, 8, PostIndex)))
TEST_NEON(ld1_33, ld1(v0.V16B(), MemOperand(x1, 16, PostIndex)))
TEST_NEON(ld1_34, ld1(v0.V4H(), MemOperand(x1, 8, PostIndex)))
TEST_NEON(ld1_35, ld1(v0.V8H(), MemOperand(x1, 16, PostIndex)))
TEST_NEON(ld1_36, ld1(v0.V2S(), MemOperand(x1, 8, PostIndex)))
TEST_NEON(ld1_37, ld1(v0.V4S(), MemOperand(x1, 16, PostIndex)))
TEST_NEON(ld1_38, ld1(v0.V1D(), MemOperand(x1, 8, PostIndex)))
TEST_NEON(ld1_39, ld1(v0.V2D(), MemOperand(x1, 16, PostIndex)))
TEST_NEON(ld1_40, ld1(v0.V8B(), v1.V8B(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(ld1_41, ld1(v0.V16B(), v1.V16B(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(ld1_42, ld1(v0.V4H(), v1.V4H(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(ld1_43, ld1(v0.V8H(), v1.V8H(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(ld1_44, ld1(v0.V2S(), v1.V2S(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(ld1_45, ld1(v0.V4S(), v1.V4S(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(ld1_46, ld1(v0.V1D(), v1.V1D(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(ld1_47, ld1(v0.V2D(), v1.V2D(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(ld1_48,
          ld1(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(ld1_49,
          ld1(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(ld1_50,
          ld1(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(ld1_51,
          ld1(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(ld1_52,
          ld1(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(ld1_53,
          ld1(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(ld1_54,
          ld1(v0.V1D(), v1.V1D(), v2.V1D(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(ld1_55,
          ld1(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(
    ld1_56,
    ld1(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(ld1_57,
          ld1(v0.V16B(),
              v1.V16B(),
              v2.V16B(),
              v3.V16B(),
              MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    ld1_58,
    ld1(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(
    ld1_59,
    ld1(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    ld1_60,
    ld1(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(
    ld1_61,
    ld1(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    ld1_62,
    ld1(v0.V1D(), v1.V1D(), v2.V1D(), v3.V1D(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(
    ld1_63,
    ld1(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4, 64, PostIndex)))
TEST_NEON(ld1_64, ld1(v0.V8B(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1_65, ld1(v0.V16B(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1_66, ld1(v0.V4H(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1_67, ld1(v0.V8H(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1_68, ld1(v0.V2S(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1_69, ld1(v0.V4S(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1_70, ld1(v0.V1D(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1_71, ld1(v0.V2D(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1_72, ld1(v0.V8B(), v1.V8B(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld1_73, ld1(v0.V16B(), v1.V16B(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld1_74, ld1(v0.V4H(), v1.V4H(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld1_75, ld1(v0.V8H(), v1.V8H(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld1_76, ld1(v0.V2S(), v1.V2S(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld1_77, ld1(v0.V4S(), v1.V4S(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld1_78, ld1(v0.V1D(), v1.V1D(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld1_79, ld1(v0.V2D(), v1.V2D(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld1_80,
          ld1(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld1_81,
          ld1(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld1_82,
          ld1(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld1_83,
          ld1(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld1_84,
          ld1(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld1_85,
          ld1(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld1_86,
          ld1(v0.V1D(), v1.V1D(), v2.V1D(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld1_87,
          ld1(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(
    ld1_88,
    ld1(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(ld1_89,
          ld1(v0.V16B(),
              v1.V16B(),
              v2.V16B(),
              v3.V16B(),
              MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld1_90,
    ld1(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld1_91,
    ld1(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld1_92,
    ld1(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld1_93,
    ld1(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld1_94,
    ld1(v0.V1D(), v1.V1D(), v2.V1D(), v3.V1D(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld1_95,
    ld1(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(ld1_96, ld1(v0.B(), 13, MemOperand(x1)))
TEST_NEON(ld1_97, ld1(v0.D(), 0, MemOperand(x1)))
TEST_NEON(ld1_98, ld1(v0.H(), 5, MemOperand(x1)))
TEST_NEON(ld1_99, ld1(v0.S(), 2, MemOperand(x1)))
TEST_NEON(ld1_100, ld1(v0.B(), 3, MemOperand(x1, 1, PostIndex)))
TEST_NEON(ld1_101, ld1(v0.B(), 14, MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1_102, ld1(v0.D(), 0, MemOperand(x1, 8, PostIndex)))
TEST_NEON(ld1_103, ld1(v0.D(), 1, MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1_104, ld1(v0.H(), 3, MemOperand(x1, 2, PostIndex)))
TEST_NEON(ld1_105, ld1(v0.H(), 6, MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld1_106, ld1(v0.S(), 0, MemOperand(x1, 4, PostIndex)))
TEST_NEON(ld1_107, ld1(v0.S(), 3, MemOperand(x1, x2, PostIndex)))
TEST_NEON(ld2r_0, ld2r(v0.V8B(), v1.V8B(), MemOperand(x2)))
TEST_NEON(ld2r_1, ld2r(v0.V16B(), v1.V16B(), MemOperand(x2)))
TEST_NEON(ld2r_2, ld2r(v0.V4H(), v1.V4H(), MemOperand(x2)))
TEST_NEON(ld2r_3, ld2r(v0.V8H(), v1.V8H(), MemOperand(x2)))
TEST_NEON(ld2r_4, ld2r(v0.V2S(), v1.V2S(), MemOperand(x2)))
TEST_NEON(ld2r_5, ld2r(v0.V4S(), v1.V4S(), MemOperand(x2)))
TEST_NEON(ld2r_6, ld2r(v0.V1D(), v1.V1D(), MemOperand(x2)))
TEST_NEON(ld2r_7, ld2r(v0.V2D(), v1.V2D(), MemOperand(x2)))
TEST_NEON(ld2r_8, ld2r(v0.V8B(), v1.V8B(), MemOperand(x2, 2, PostIndex)))
TEST_NEON(ld2r_9, ld2r(v0.V16B(), v1.V16B(), MemOperand(x2, 2, PostIndex)))
TEST_NEON(ld2r_10, ld2r(v0.V4H(), v1.V4H(), MemOperand(x2, 4, PostIndex)))
TEST_NEON(ld2r_11, ld2r(v0.V8H(), v1.V8H(), MemOperand(x2, 4, PostIndex)))
TEST_NEON(ld2r_12, ld2r(v0.V2S(), v1.V2S(), MemOperand(x2, 8, PostIndex)))
TEST_NEON(ld2r_13, ld2r(v0.V4S(), v1.V4S(), MemOperand(x2, 8, PostIndex)))
TEST_NEON(ld2r_14, ld2r(v0.V1D(), v1.V1D(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(ld2r_15, ld2r(v0.V2D(), v1.V2D(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(ld2r_16, ld2r(v0.V8B(), v1.V8B(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2r_17, ld2r(v0.V16B(), v1.V16B(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2r_18, ld2r(v0.V4H(), v1.V4H(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2r_19, ld2r(v0.V8H(), v1.V8H(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2r_20, ld2r(v0.V2S(), v1.V2S(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2r_21, ld2r(v0.V4S(), v1.V4S(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2r_22, ld2r(v0.V1D(), v1.V1D(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2r_23, ld2r(v0.V2D(), v1.V2D(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2_0, ld2(v0.V8B(), v1.V8B(), MemOperand(x2)))
TEST_NEON(ld2_1, ld2(v0.V16B(), v1.V16B(), MemOperand(x2)))
TEST_NEON(ld2_2, ld2(v0.V4H(), v1.V4H(), MemOperand(x2)))
TEST_NEON(ld2_3, ld2(v0.V8H(), v1.V8H(), MemOperand(x2)))
TEST_NEON(ld2_4, ld2(v0.V2S(), v1.V2S(), MemOperand(x2)))
TEST_NEON(ld2_5, ld2(v0.V4S(), v1.V4S(), MemOperand(x2)))
TEST_NEON(ld2_6, ld2(v0.V2D(), v1.V2D(), MemOperand(x2)))
TEST_NEON(ld2_7, ld2(v0.V8B(), v1.V8B(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(ld2_8, ld2(v0.V16B(), v1.V16B(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(ld2_9, ld2(v0.V4H(), v1.V4H(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(ld2_10, ld2(v0.V8H(), v1.V8H(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(ld2_11, ld2(v0.V2S(), v1.V2S(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(ld2_12, ld2(v0.V4S(), v1.V4S(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(ld2_13, ld2(v0.V2D(), v1.V2D(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(ld2_14, ld2(v0.V8B(), v1.V8B(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2_15, ld2(v0.V16B(), v1.V16B(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2_16, ld2(v0.V4H(), v1.V4H(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2_17, ld2(v0.V8H(), v1.V8H(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2_18, ld2(v0.V2S(), v1.V2S(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2_19, ld2(v0.V4S(), v1.V4S(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2_20, ld2(v0.V2D(), v1.V2D(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2_21, ld2(v0.B(), v1.B(), 10, MemOperand(x2)))
TEST_NEON(ld2_22, ld2(v0.D(), v1.D(), 0, MemOperand(x2)))
TEST_NEON(ld2_23, ld2(v0.H(), v1.H(), 3, MemOperand(x2)))
TEST_NEON(ld2_24, ld2(v0.S(), v1.S(), 1, MemOperand(x2)))
TEST_NEON(ld2_25, ld2(v0.B(), v1.B(), 7, MemOperand(x2, 2, PostIndex)))
TEST_NEON(ld2_26, ld2(v0.B(), v1.B(), 4, MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2_27, ld2(v0.D(), v1.D(), 1, MemOperand(x2, 16, PostIndex)))
TEST_NEON(ld2_28, ld2(v0.D(), v1.D(), 1, MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2_29, ld2(v0.H(), v1.H(), 0, MemOperand(x2, 4, PostIndex)))
TEST_NEON(ld2_30, ld2(v0.H(), v1.H(), 6, MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld2_31, ld2(v0.S(), v1.S(), 3, MemOperand(x2, 8, PostIndex)))
TEST_NEON(ld2_32, ld2(v0.S(), v1.S(), 3, MemOperand(x2, x3, PostIndex)))
TEST_NEON(ld3r_0, ld3r(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3)))
TEST_NEON(ld3r_1, ld3r(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3)))
TEST_NEON(ld3r_2, ld3r(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3)))
TEST_NEON(ld3r_3, ld3r(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3)))
TEST_NEON(ld3r_4, ld3r(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3)))
TEST_NEON(ld3r_5, ld3r(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3)))
TEST_NEON(ld3r_6, ld3r(v0.V1D(), v1.V1D(), v2.V1D(), MemOperand(x3)))
TEST_NEON(ld3r_7, ld3r(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3)))
TEST_NEON(ld3r_8,
          ld3r(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3, 3, PostIndex)))
TEST_NEON(ld3r_9,
          ld3r(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3, 3, PostIndex)))
TEST_NEON(ld3r_10,
          ld3r(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3, 6, PostIndex)))
TEST_NEON(ld3r_11,
          ld3r(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3, 6, PostIndex)))
TEST_NEON(ld3r_12,
          ld3r(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3, 12, PostIndex)))
TEST_NEON(ld3r_13,
          ld3r(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3, 12, PostIndex)))
TEST_NEON(ld3r_14,
          ld3r(v0.V1D(), v1.V1D(), v2.V1D(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(ld3r_15,
          ld3r(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(ld3r_16,
          ld3r(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3r_17,
          ld3r(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3r_18,
          ld3r(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3r_19,
          ld3r(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3r_20,
          ld3r(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3r_21,
          ld3r(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3r_22,
          ld3r(v0.V1D(), v1.V1D(), v2.V1D(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3r_23,
          ld3r(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3_0, ld3(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3)))
TEST_NEON(ld3_1, ld3(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3)))
TEST_NEON(ld3_2, ld3(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3)))
TEST_NEON(ld3_3, ld3(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3)))
TEST_NEON(ld3_4, ld3(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3)))
TEST_NEON(ld3_5, ld3(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3)))
TEST_NEON(ld3_6, ld3(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3)))
TEST_NEON(ld3_7,
          ld3(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(ld3_8,
          ld3(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(ld3_9,
          ld3(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(ld3_10,
          ld3(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(ld3_11,
          ld3(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(ld3_12,
          ld3(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(ld3_13,
          ld3(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(ld3_14,
          ld3(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3_15,
          ld3(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3_16,
          ld3(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3_17,
          ld3(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3_18,
          ld3(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3_19,
          ld3(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3_20,
          ld3(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3_21, ld3(v0.B(), v1.B(), v2.B(), 6, MemOperand(x3)))
TEST_NEON(ld3_22, ld3(v0.D(), v1.D(), v2.D(), 1, MemOperand(x3)))
TEST_NEON(ld3_23, ld3(v0.H(), v1.H(), v2.H(), 1, MemOperand(x3)))
TEST_NEON(ld3_24, ld3(v0.S(), v1.S(), v2.S(), 2, MemOperand(x3)))
TEST_NEON(ld3_25, ld3(v0.B(), v1.B(), v2.B(), 7, MemOperand(x3, 3, PostIndex)))
TEST_NEON(ld3_26,
          ld3(v0.B(), v1.B(), v2.B(), 10, MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3_27, ld3(v0.D(), v1.D(), v2.D(), 1, MemOperand(x3, 24, PostIndex)))
TEST_NEON(ld3_28, ld3(v0.D(), v1.D(), v2.D(), 0, MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3_29, ld3(v0.H(), v1.H(), v2.H(), 0, MemOperand(x3, 6, PostIndex)))
TEST_NEON(ld3_30, ld3(v0.H(), v1.H(), v2.H(), 2, MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld3_31, ld3(v0.S(), v1.S(), v2.S(), 2, MemOperand(x3, 12, PostIndex)))
TEST_NEON(ld3_32, ld3(v0.S(), v1.S(), v2.S(), 0, MemOperand(x3, x4, PostIndex)))
TEST_NEON(ld4r_0, ld4r(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4)))
TEST_NEON(ld4r_1,
          ld4r(v0.V16B(), v1.V16B(), v2.V16B(), v3.V16B(), MemOperand(x4)))
TEST_NEON(ld4r_2, ld4r(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4)))
TEST_NEON(ld4r_3, ld4r(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4)))
TEST_NEON(ld4r_4, ld4r(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4)))
TEST_NEON(ld4r_5, ld4r(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4)))
TEST_NEON(ld4r_6, ld4r(v0.V1D(), v1.V1D(), v2.V1D(), v3.V1D(), MemOperand(x4)))
TEST_NEON(ld4r_7, ld4r(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4)))
TEST_NEON(
    ld4r_8,
    ld4r(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4, 4, PostIndex)))
TEST_NEON(ld4r_9,
          ld4r(v0.V16B(),
               v1.V16B(),
               v2.V16B(),
               v3.V16B(),
               MemOperand(x4, 4, PostIndex)))
TEST_NEON(
    ld4r_10,
    ld4r(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4, 8, PostIndex)))
TEST_NEON(
    ld4r_11,
    ld4r(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4, 8, PostIndex)))
TEST_NEON(
    ld4r_12,
    ld4r(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4, 16, PostIndex)))
TEST_NEON(
    ld4r_13,
    ld4r(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4, 16, PostIndex)))
TEST_NEON(
    ld4r_14,
    ld4r(v0.V1D(), v1.V1D(), v2.V1D(), v3.V1D(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(
    ld4r_15,
    ld4r(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(
    ld4r_16,
    ld4r(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(ld4r_17,
          ld4r(v0.V16B(),
               v1.V16B(),
               v2.V16B(),
               v3.V16B(),
               MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld4r_18,
    ld4r(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld4r_19,
    ld4r(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld4r_20,
    ld4r(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld4r_21,
    ld4r(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld4r_22,
    ld4r(v0.V1D(), v1.V1D(), v2.V1D(), v3.V1D(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld4r_23,
    ld4r(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(ld4_0, ld4(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4)))
TEST_NEON(ld4_1,
          ld4(v0.V16B(), v1.V16B(), v2.V16B(), v3.V16B(), MemOperand(x4)))
TEST_NEON(ld4_2, ld4(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4)))
TEST_NEON(ld4_3, ld4(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4)))
TEST_NEON(ld4_4, ld4(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4)))
TEST_NEON(ld4_5, ld4(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4)))
TEST_NEON(ld4_6, ld4(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4)))
TEST_NEON(
    ld4_7,
    ld4(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(ld4_8,
          ld4(v0.V16B(),
              v1.V16B(),
              v2.V16B(),
              v3.V16B(),
              MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    ld4_9,
    ld4(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(
    ld4_10,
    ld4(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    ld4_11,
    ld4(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(
    ld4_12,
    ld4(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    ld4_13,
    ld4(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    ld4_14,
    ld4(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(ld4_15,
          ld4(v0.V16B(),
              v1.V16B(),
              v2.V16B(),
              v3.V16B(),
              MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld4_16,
    ld4(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld4_17,
    ld4(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld4_18,
    ld4(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld4_19,
    ld4(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    ld4_20,
    ld4(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(ld4_21, ld4(v0.B(), v1.B(), v2.B(), v3.B(), 15, MemOperand(x4)))
TEST_NEON(ld4_22, ld4(v0.D(), v1.D(), v2.D(), v3.D(), 1, MemOperand(x4)))
TEST_NEON(ld4_23, ld4(v0.H(), v1.H(), v2.H(), v3.H(), 2, MemOperand(x4)))
TEST_NEON(ld4_24, ld4(v0.S(), v1.S(), v2.S(), v3.S(), 0, MemOperand(x4)))
TEST_NEON(ld4_25,
          ld4(v0.B(), v1.B(), v2.B(), v3.B(), 0, MemOperand(x4, 4, PostIndex)))
TEST_NEON(ld4_26,
          ld4(v0.B(), v1.B(), v2.B(), v3.B(), 0, MemOperand(x4, x5, PostIndex)))
TEST_NEON(ld4_27,
          ld4(v0.D(), v1.D(), v2.D(), v3.D(), 0, MemOperand(x4, 32, PostIndex)))
TEST_NEON(ld4_28,
          ld4(v0.D(), v1.D(), v2.D(), v3.D(), 0, MemOperand(x4, x5, PostIndex)))
TEST_NEON(ld4_29,
          ld4(v0.H(), v1.H(), v2.H(), v3.H(), 1, MemOperand(x4, 8, PostIndex)))
TEST_NEON(ld4_30,
          ld4(v0.H(), v1.H(), v2.H(), v3.H(), 7, MemOperand(x4, x5, PostIndex)))
TEST_NEON(ld4_31,
          ld4(v0.S(), v1.S(), v2.S(), v3.S(), 3, MemOperand(x4, 16, PostIndex)))
TEST_NEON(ld4_32,
          ld4(v0.S(), v1.S(), v2.S(), v3.S(), 1, MemOperand(x4, x5, PostIndex)))
TEST_NEON(ldnp_0, ldnp(d0, d1, MemOperand(x2, -32)))
TEST_NEON(ldnp_1, ldnp(q0, q1, MemOperand(x2, 288)))
TEST_NEON(ldnp_2, ldnp(s0, s1, MemOperand(x2, -180)))
TEST_NEON(ldp_0, ldp(d0, d1, MemOperand(x2, 112)))
TEST_NEON(ldp_1, ldp(d0, d1, MemOperand(x2, 24, PostIndex)))
TEST_NEON(ldp_2, ldp(d0, d1, MemOperand(x2, 192, PreIndex)))
TEST_NEON(ldp_3, ldp(q0, q1, MemOperand(x2, 256)))
TEST_NEON(ldp_4, ldp(q0, q1, MemOperand(x2, -976, PostIndex)))
TEST_NEON(ldp_5, ldp(q0, q1, MemOperand(x2, -976, PreIndex)))
TEST_NEON(ldp_6, ldp(s0, s1, MemOperand(x2, -72)))
TEST_NEON(ldp_7, ldp(s0, s1, MemOperand(x2, 0, PostIndex)))
TEST_NEON(ldp_8, ldp(s0, s1, MemOperand(x2, 144, PreIndex)))
TEST_NEON(ldr_0, ldr(b0, MemOperand(x1, -172, PostIndex)))
TEST_NEON(ldr_1, ldr(b0, MemOperand(x1, 39, PreIndex)))
TEST_NEON(ldr_2, ldr(b0, MemOperand(x1, 1471)))
TEST_NEON(ldr_3, ldr(d0, MemOperand(x1, -240, PostIndex)))
TEST_NEON(ldr_4, ldr(d0, MemOperand(x1, 110, PreIndex)))
TEST_NEON(ldr_5, ldr(d0, MemOperand(x1, 4048)))
TEST_NEON(ldr_6, ldr(h0, MemOperand(x1, -76, PostIndex)))
TEST_NEON(ldr_7, ldr(h0, MemOperand(x1, -124, PreIndex)))
TEST_NEON(ldr_8, ldr(h0, MemOperand(x1, 6230)))
TEST_NEON(ldr_9, ldr(q0, MemOperand(x1, -13, PostIndex)))
TEST_NEON(ldr_10, ldr(q0, MemOperand(x1, 17, PreIndex)))
TEST_NEON(ldr_11, ldr(q0, MemOperand(x1, 11376)))
TEST_NEON(ldr_12, ldr(s0, MemOperand(x1, 148, PostIndex)))
TEST_NEON(ldr_13, ldr(s0, MemOperand(x1, 68, PreIndex)))
TEST_NEON(ldr_14, ldr(s0, MemOperand(x1, 7424)))
TEST_NEON(ldr_15, ldr(d0, 0x35a20))
TEST_NEON(ldr_16, ldr(q0, 0xc067))
TEST_NEON(ldr_17, ldr(s0, 0x1053a))
TEST_NEON(ldr_18, ldr(b0, MemOperand(x1, x2, LSL, 0)))
TEST_NEON(ldr_19, ldr(b0, MemOperand(x1, w2, SXTW, 0)))
TEST_NEON(ldr_20, ldr(d0, MemOperand(x1, w2, SXTW, 3)))
TEST_NEON(ldr_21, ldr(d0, MemOperand(x1, x2, SXTX, 3)))
TEST_NEON(ldr_22, ldr(h0, MemOperand(x1, w2, UXTW, 1)))
TEST_NEON(ldr_23, ldr(h0, MemOperand(x1, x2, SXTX, 0)))
TEST_NEON(ldr_24, ldr(q0, MemOperand(x1, w2, UXTW, 4)))
TEST_NEON(ldr_25, ldr(q0, MemOperand(x1, x2, SXTX, 4)))
TEST_NEON(ldr_26, ldr(s0, MemOperand(x1, w2, SXTW, 2)))
TEST_NEON(ldr_27, ldr(s0, MemOperand(x1, x2, SXTX, 2)))
TEST_NEON(ldur_0, ldur(b0, MemOperand(x1, 153)))
TEST_NEON(ldur_1, ldur(d0, MemOperand(x1, -146)))
TEST_NEON(ldur_2, ldur(h0, MemOperand(x1, -117)))
TEST_NEON(ldur_3, ldur(q0, MemOperand(x1, -81)))
TEST_NEON(ldur_4, ldur(s0, MemOperand(x1, 209)))
TEST_NEON(mla_0, mla(v0.V4H(), v1.V4H(), v2.H(), 1))
TEST_NEON(mla_1, mla(v0.V8H(), v1.V8H(), v2.H(), 7))
TEST_NEON(mla_2, mla(v0.V2S(), v1.V2S(), v2.S(), 0))
TEST_NEON(mla_3, mla(v0.V4S(), v1.V4S(), v2.S(), 3))
TEST_NEON(mla_4, mla(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(mla_5, mla(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(mla_6, mla(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(mla_7, mla(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(mla_8, mla(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(mla_9, mla(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(mls_0, mls(v0.V4H(), v1.V4H(), v2.H(), 3))
TEST_NEON(mls_1, mls(v0.V8H(), v1.V8H(), v2.H(), 5))
TEST_NEON(mls_2, mls(v0.V2S(), v1.V2S(), v2.S(), 0))
TEST_NEON(mls_3, mls(v0.V4S(), v1.V4S(), v2.S(), 1))
TEST_NEON(mls_4, mls(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(mls_5, mls(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(mls_6, mls(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(mls_7, mls(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(mls_8, mls(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(mls_9, mls(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(movi_0, movi(v0.V2D(), 0x0000ff000000ff00))
TEST_NEON(movi_1, movi(d0, 0xffffffff00ffff00))
TEST_NEON(movi_2, movi(v0.V4H(), 0xda, LSL, 8))
TEST_NEON(movi_3, movi(v0.V8H(), 0x32, LSL, 0))
TEST_NEON(movi_4, movi(v0.V2S(), 0xfe, LSL, 0))
TEST_NEON(movi_5, movi(v0.V4S(), 0xf0, LSL, 8))
TEST_NEON(movi_6, movi(v0.V2S(), 0x81, MSL, 8))
TEST_NEON(movi_7, movi(v0.V4S(), 0x7, MSL, 8))
TEST_NEON(movi_8, movi(v0.V8B(), 0x18))
TEST_NEON(movi_9, movi(v0.V16B(), 0x94))
TEST_NEON(mov_0, mov(b0, v1.B(), 5))
TEST_NEON(mov_1, mov(h0, v1.H(), 1))
TEST_NEON(mov_2, mov(s0, v1.S(), 3))
TEST_NEON(mov_3, mov(d0, v1.D(), 0))
TEST_NEON(mov_4, mov(v0.B(), 14, v1.B(), 11))
TEST_NEON(mov_5, mov(v0.H(), 1, v1.H(), 7))
TEST_NEON(mov_6, mov(v0.S(), 1, v1.S(), 1))
TEST_NEON(mov_7, mov(v0.D(), 0, v1.D(), 1))
TEST_NEON(mov_8, mov(v0.B(), 14, w1))
TEST_NEON(mov_9, mov(v0.H(), 7, w1))
TEST_NEON(mov_10, mov(v0.S(), 0, w1))
TEST_NEON(mov_11, mov(v0.D(), 1, x1))
TEST_NEON(mov_12, mov(v0.V8B(), v1.V8B()))
TEST_NEON(mov_13, mov(v0.V16B(), v1.V16B()))
TEST_NEON(mov_14, mov(w0, v1.S(), 3))
TEST_NEON(mov_15, mov(x0, v1.D(), 1))
TEST_NEON(mul_0, mul(v0.V4H(), v1.V4H(), v2.H(), 0))
TEST_NEON(mul_1, mul(v0.V8H(), v1.V8H(), v2.H(), 2))
TEST_NEON(mul_2, mul(v0.V2S(), v1.V2S(), v2.S(), 1))
TEST_NEON(mul_3, mul(v0.V4S(), v1.V4S(), v2.S(), 1))
TEST_NEON(mul_4, mul(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(mul_5, mul(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(mul_6, mul(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(mul_7, mul(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(mul_8, mul(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(mul_9, mul(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(mvni_0, mvni(v0.V4H(), 0xd9, LSL, 8))
TEST_NEON(mvni_1, mvni(v0.V8H(), 0x86, LSL, 0))
TEST_NEON(mvni_2, mvni(v0.V2S(), 0xde, LSL, 16))
TEST_NEON(mvni_3, mvni(v0.V4S(), 0x96, LSL, 24))
TEST_NEON(mvni_4, mvni(v0.V2S(), 0x1e, MSL, 16))
TEST_NEON(mvni_5, mvni(v0.V4S(), 0x9b, MSL, 16))
TEST_NEON(mvn_0, mvn(v0.V8B(), v1.V8B()))
TEST_NEON(mvn_1, mvn(v0.V16B(), v1.V16B()))
TEST_NEON(neg_0, neg(v0.V8B(), v1.V8B()))
TEST_NEON(neg_1, neg(v0.V16B(), v1.V16B()))
TEST_NEON(neg_2, neg(v0.V4H(), v1.V4H()))
TEST_NEON(neg_3, neg(v0.V8H(), v1.V8H()))
TEST_NEON(neg_4, neg(v0.V2S(), v1.V2S()))
TEST_NEON(neg_5, neg(v0.V4S(), v1.V4S()))
TEST_NEON(neg_6, neg(v0.V2D(), v1.V2D()))
TEST_NEON(neg_7, neg(d0, d1))
TEST_NEON(not_0, not_(v0.V8B(), v1.V8B()))
TEST_NEON(not_1, not_(v0.V16B(), v1.V16B()))
TEST_NEON(orn_0, orn(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(orn_1, orn(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(orr_0, orr(v0.V4H(), 0xb1, 0))
TEST_NEON(orr_1, orr(v0.V8H(), 0xe3, 8))
TEST_NEON(orr_2, orr(v0.V2S(), 0x26, 0))
TEST_NEON(orr_3, orr(v0.V4S(), 0x67, 0))
TEST_NEON(orr_4, orr(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(orr_5, orr(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(pmull_0, pmull(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(pmull2_0, pmull2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(pmul_0, pmul(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(pmul_1, pmul(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(raddhn_0, raddhn(v0.V8B(), v1.V8H(), v2.V8H()))
TEST_NEON(raddhn_1, raddhn(v0.V4H(), v1.V4S(), v2.V4S()))
TEST_NEON(raddhn_2, raddhn(v0.V2S(), v1.V2D(), v2.V2D()))
TEST_NEON(raddhn2_0, raddhn2(v0.V16B(), v1.V8H(), v2.V8H()))
TEST_NEON(raddhn2_1, raddhn2(v0.V8H(), v1.V4S(), v2.V4S()))
TEST_NEON(raddhn2_2, raddhn2(v0.V4S(), v1.V2D(), v2.V2D()))
TEST_NEON(rbit_0, rbit(v0.V8B(), v1.V8B()))
TEST_NEON(rbit_1, rbit(v0.V16B(), v1.V16B()))
TEST_NEON(rev16_0, rev16(v0.V8B(), v1.V8B()))
TEST_NEON(rev16_1, rev16(v0.V16B(), v1.V16B()))
TEST_NEON(rev32_0, rev32(v0.V8B(), v1.V8B()))
TEST_NEON(rev32_1, rev32(v0.V16B(), v1.V16B()))
TEST_NEON(rev32_2, rev32(v0.V4H(), v1.V4H()))
TEST_NEON(rev32_3, rev32(v0.V8H(), v1.V8H()))
TEST_NEON(rev64_0, rev64(v0.V8B(), v1.V8B()))
TEST_NEON(rev64_1, rev64(v0.V16B(), v1.V16B()))
TEST_NEON(rev64_2, rev64(v0.V4H(), v1.V4H()))
TEST_NEON(rev64_3, rev64(v0.V8H(), v1.V8H()))
TEST_NEON(rev64_4, rev64(v0.V2S(), v1.V2S()))
TEST_NEON(rev64_5, rev64(v0.V4S(), v1.V4S()))
TEST_NEON(rshrn_0, rshrn(v0.V8B(), v1.V8H(), 6))
TEST_NEON(rshrn_1, rshrn(v0.V4H(), v1.V4S(), 9))
TEST_NEON(rshrn_2, rshrn(v0.V2S(), v1.V2D(), 30))
TEST_NEON(rshrn2_0, rshrn2(v0.V16B(), v1.V8H(), 1))
TEST_NEON(rshrn2_1, rshrn2(v0.V8H(), v1.V4S(), 6))
TEST_NEON(rshrn2_2, rshrn2(v0.V4S(), v1.V2D(), 12))
TEST_NEON(rsubhn_0, rsubhn(v0.V8B(), v1.V8H(), v2.V8H()))
TEST_NEON(rsubhn_1, rsubhn(v0.V4H(), v1.V4S(), v2.V4S()))
TEST_NEON(rsubhn_2, rsubhn(v0.V2S(), v1.V2D(), v2.V2D()))
TEST_NEON(rsubhn2_0, rsubhn2(v0.V16B(), v1.V8H(), v2.V8H()))
TEST_NEON(rsubhn2_1, rsubhn2(v0.V8H(), v1.V4S(), v2.V4S()))
TEST_NEON(rsubhn2_2, rsubhn2(v0.V4S(), v1.V2D(), v2.V2D()))
TEST_NEON(sabal_0, sabal(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(sabal_1, sabal(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(sabal_2, sabal(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(sabal2_0, sabal2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(sabal2_1, sabal2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(sabal2_2, sabal2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(saba_0, saba(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(saba_1, saba(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(saba_2, saba(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(saba_3, saba(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(saba_4, saba(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(saba_5, saba(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sabdl_0, sabdl(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(sabdl_1, sabdl(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(sabdl_2, sabdl(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(sabdl2_0, sabdl2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(sabdl2_1, sabdl2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(sabdl2_2, sabdl2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(sabd_0, sabd(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(sabd_1, sabd(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(sabd_2, sabd(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(sabd_3, sabd(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(sabd_4, sabd(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(sabd_5, sabd(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sadalp_0, sadalp(v0.V4H(), v1.V8B()))
TEST_NEON(sadalp_1, sadalp(v0.V8H(), v1.V16B()))
TEST_NEON(sadalp_2, sadalp(v0.V2S(), v1.V4H()))
TEST_NEON(sadalp_3, sadalp(v0.V4S(), v1.V8H()))
TEST_NEON(sadalp_4, sadalp(v0.V1D(), v1.V2S()))
TEST_NEON(sadalp_5, sadalp(v0.V2D(), v1.V4S()))
TEST_NEON(saddlp_0, saddlp(v0.V4H(), v1.V8B()))
TEST_NEON(saddlp_1, saddlp(v0.V8H(), v1.V16B()))
TEST_NEON(saddlp_2, saddlp(v0.V2S(), v1.V4H()))
TEST_NEON(saddlp_3, saddlp(v0.V4S(), v1.V8H()))
TEST_NEON(saddlp_4, saddlp(v0.V1D(), v1.V2S()))
TEST_NEON(saddlp_5, saddlp(v0.V2D(), v1.V4S()))
TEST_NEON(saddlv_0, saddlv(h0, v1.V8B()))
TEST_NEON(saddlv_1, saddlv(h0, v1.V16B()))
TEST_NEON(saddlv_2, saddlv(s0, v1.V4H()))
TEST_NEON(saddlv_3, saddlv(s0, v1.V8H()))
TEST_NEON(saddlv_4, saddlv(d0, v1.V4S()))
TEST_NEON(saddl_0, saddl(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(saddl_1, saddl(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(saddl_2, saddl(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(saddl2_0, saddl2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(saddl2_1, saddl2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(saddl2_2, saddl2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(saddw_0, saddw(v0.V8H(), v1.V8H(), v2.V8B()))
TEST_NEON(saddw_1, saddw(v0.V4S(), v1.V4S(), v2.V4H()))
TEST_NEON(saddw_2, saddw(v0.V2D(), v1.V2D(), v2.V2S()))
TEST_NEON(saddw2_0, saddw2(v0.V8H(), v1.V8H(), v2.V16B()))
TEST_NEON(saddw2_1, saddw2(v0.V4S(), v1.V4S(), v2.V8H()))
TEST_NEON(saddw2_2, saddw2(v0.V2D(), v1.V2D(), v2.V4S()))
TEST_NEON(shadd_0, shadd(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(shadd_1, shadd(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(shadd_2, shadd(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(shadd_3, shadd(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(shadd_4, shadd(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(shadd_5, shadd(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(shll_0, shll(v0.V8H(), v1.V8B(), 8))
TEST_NEON(shll_1, shll(v0.V4S(), v1.V4H(), 16))
TEST_NEON(shll_2, shll(v0.V2D(), v1.V2S(), 32))
TEST_NEON(shll2_0, shll2(v0.V8H(), v1.V16B(), 8))
TEST_NEON(shll2_1, shll2(v0.V4S(), v1.V8H(), 16))
TEST_NEON(shll2_2, shll2(v0.V2D(), v1.V4S(), 32))
TEST_NEON(shl_0, shl(v0.V8B(), v1.V8B(), 2))
TEST_NEON(shl_1, shl(v0.V16B(), v1.V16B(), 5))
TEST_NEON(shl_2, shl(v0.V4H(), v1.V4H(), 10))
TEST_NEON(shl_3, shl(v0.V8H(), v1.V8H(), 14))
TEST_NEON(shl_4, shl(v0.V2S(), v1.V2S(), 24))
TEST_NEON(shl_5, shl(v0.V4S(), v1.V4S(), 2))
TEST_NEON(shl_6, shl(v0.V2D(), v1.V2D(), 44))
TEST_NEON(shl_7, shl(d0, d1, 7))
TEST_NEON(shrn_0, shrn(v0.V8B(), v1.V8H(), 7))
TEST_NEON(shrn_1, shrn(v0.V4H(), v1.V4S(), 14))
TEST_NEON(shrn_2, shrn(v0.V2S(), v1.V2D(), 20))
TEST_NEON(shrn2_0, shrn2(v0.V16B(), v1.V8H(), 2))
TEST_NEON(shrn2_1, shrn2(v0.V8H(), v1.V4S(), 14))
TEST_NEON(shrn2_2, shrn2(v0.V4S(), v1.V2D(), 18))
TEST_NEON(shsub_0, shsub(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(shsub_1, shsub(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(shsub_2, shsub(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(shsub_3, shsub(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(shsub_4, shsub(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(shsub_5, shsub(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sli_0, sli(v0.V8B(), v1.V8B(), 6))
TEST_NEON(sli_1, sli(v0.V16B(), v1.V16B(), 2))
TEST_NEON(sli_2, sli(v0.V4H(), v1.V4H(), 7))
TEST_NEON(sli_3, sli(v0.V8H(), v1.V8H(), 7))
TEST_NEON(sli_4, sli(v0.V2S(), v1.V2S(), 13))
TEST_NEON(sli_5, sli(v0.V4S(), v1.V4S(), 16))
TEST_NEON(sli_6, sli(v0.V2D(), v1.V2D(), 21))
TEST_NEON(sli_7, sli(d0, d1, 34))
TEST_NEON(smaxp_0, smaxp(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(smaxp_1, smaxp(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(smaxp_2, smaxp(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(smaxp_3, smaxp(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(smaxp_4, smaxp(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(smaxp_5, smaxp(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(smaxv_0, smaxv(b0, v1.V8B()))
TEST_NEON(smaxv_1, smaxv(b0, v1.V16B()))
TEST_NEON(smaxv_2, smaxv(h0, v1.V4H()))
TEST_NEON(smaxv_3, smaxv(h0, v1.V8H()))
TEST_NEON(smaxv_4, smaxv(s0, v1.V4S()))
TEST_NEON(smax_0, smax(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(smax_1, smax(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(smax_2, smax(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(smax_3, smax(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(smax_4, smax(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(smax_5, smax(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sminp_0, sminp(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(sminp_1, sminp(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(sminp_2, sminp(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(sminp_3, sminp(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(sminp_4, sminp(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(sminp_5, sminp(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sminv_0, sminv(b0, v1.V8B()))
TEST_NEON(sminv_1, sminv(b0, v1.V16B()))
TEST_NEON(sminv_2, sminv(h0, v1.V4H()))
TEST_NEON(sminv_3, sminv(h0, v1.V8H()))
TEST_NEON(sminv_4, sminv(s0, v1.V4S()))
TEST_NEON(smin_0, smin(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(smin_1, smin(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(smin_2, smin(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(smin_3, smin(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(smin_4, smin(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(smin_5, smin(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(smlal_0, smlal(v0.V4S(), v1.V4H(), v2.H(), 5))
TEST_NEON(smlal_1, smlal(v0.V2D(), v1.V2S(), v2.S(), 0))
TEST_NEON(smlal2_0, smlal2(v0.V4S(), v1.V8H(), v2.H(), 7))
TEST_NEON(smlal2_1, smlal2(v0.V2D(), v1.V4S(), v2.S(), 1))
TEST_NEON(smlal_2, smlal(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(smlal_3, smlal(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(smlal_4, smlal(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(smlal2_2, smlal2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(smlal2_3, smlal2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(smlal2_4, smlal2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(smlsl_0, smlsl(v0.V4S(), v1.V4H(), v2.H(), 3))
TEST_NEON(smlsl_1, smlsl(v0.V2D(), v1.V2S(), v2.S(), 3))
TEST_NEON(smlsl2_0, smlsl2(v0.V4S(), v1.V8H(), v2.H(), 1))
TEST_NEON(smlsl2_1, smlsl2(v0.V2D(), v1.V4S(), v2.S(), 2))
TEST_NEON(smlsl_2, smlsl(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(smlsl_3, smlsl(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(smlsl_4, smlsl(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(smlsl2_2, smlsl2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(smlsl2_3, smlsl2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(smlsl2_4, smlsl2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(smov_0, smov(w0, v1.B(), 4))
TEST_NEON(smov_1, smov(w0, v1.H(), 1))
TEST_NEON(smov_2, smov(x0, v1.B(), 6))
TEST_NEON(smov_3, smov(x0, v1.H(), 7))
TEST_NEON(smov_4, smov(x0, v1.S(), 2))
TEST_NEON(smull_0, smull(v0.V4S(), v1.V4H(), v2.H(), 6))
TEST_NEON(smull_1, smull(v0.V2D(), v1.V2S(), v2.S(), 3))
TEST_NEON(smull2_0, smull2(v0.V4S(), v1.V8H(), v2.H(), 3))
TEST_NEON(smull2_1, smull2(v0.V2D(), v1.V4S(), v2.S(), 2))
TEST_NEON(smull_2, smull(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(smull_3, smull(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(smull_4, smull(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(smull2_2, smull2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(smull2_3, smull2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(smull2_4, smull2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(sqabs_0, sqabs(v0.V8B(), v1.V8B()))
TEST_NEON(sqabs_1, sqabs(v0.V16B(), v1.V16B()))
TEST_NEON(sqabs_2, sqabs(v0.V4H(), v1.V4H()))
TEST_NEON(sqabs_3, sqabs(v0.V8H(), v1.V8H()))
TEST_NEON(sqabs_4, sqabs(v0.V2S(), v1.V2S()))
TEST_NEON(sqabs_5, sqabs(v0.V4S(), v1.V4S()))
TEST_NEON(sqabs_6, sqabs(v0.V2D(), v1.V2D()))
TEST_NEON(sqabs_7, sqabs(b0, b1))
TEST_NEON(sqabs_8, sqabs(h0, h1))
TEST_NEON(sqabs_9, sqabs(s0, s1))
TEST_NEON(sqabs_10, sqabs(d0, d1))
TEST_NEON(sqadd_0, sqadd(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(sqadd_1, sqadd(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(sqadd_2, sqadd(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(sqadd_3, sqadd(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(sqadd_4, sqadd(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(sqadd_5, sqadd(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sqadd_6, sqadd(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(sqadd_7, sqadd(b0, b1, b2))
TEST_NEON(sqadd_8, sqadd(h0, h1, h2))
TEST_NEON(sqadd_9, sqadd(s0, s1, s2))
TEST_NEON(sqadd_10, sqadd(d0, d1, d2))
TEST_NEON(sqdmlal_0, sqdmlal(v0.V4S(), v1.V4H(), v2.H(), 3))
TEST_NEON(sqdmlal_1, sqdmlal(v0.V2D(), v1.V2S(), v2.S(), 2))
TEST_NEON(sqdmlal2_0, sqdmlal2(v0.V4S(), v1.V8H(), v2.H(), 3))
TEST_NEON(sqdmlal2_1, sqdmlal2(v0.V2D(), v1.V4S(), v2.S(), 0))
TEST_NEON(sqdmlal_2, sqdmlal(s0, h1, v2.H(), 1))
TEST_NEON(sqdmlal_3, sqdmlal(d0, s1, v2.S(), 1))
TEST_NEON(sqdmlal_4, sqdmlal(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(sqdmlal_5, sqdmlal(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(sqdmlal2_2, sqdmlal2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(sqdmlal2_3, sqdmlal2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(sqdmlal_6, sqdmlal(s0, h1, h2))
TEST_NEON(sqdmlal_7, sqdmlal(d0, s1, s2))
TEST_NEON(sqdmlsl_0, sqdmlsl(v0.V4S(), v1.V4H(), v2.H(), 6))
TEST_NEON(sqdmlsl_1, sqdmlsl(v0.V2D(), v1.V2S(), v2.S(), 0))
TEST_NEON(sqdmlsl2_0, sqdmlsl2(v0.V4S(), v1.V8H(), v2.H(), 6))
TEST_NEON(sqdmlsl2_1, sqdmlsl2(v0.V2D(), v1.V4S(), v2.S(), 3))
TEST_NEON(sqdmlsl_2, sqdmlsl(s0, h1, v2.H(), 1))
TEST_NEON(sqdmlsl_3, sqdmlsl(d0, s1, v2.S(), 2))
TEST_NEON(sqdmlsl_4, sqdmlsl(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(sqdmlsl_5, sqdmlsl(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(sqdmlsl2_2, sqdmlsl2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(sqdmlsl2_3, sqdmlsl2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(sqdmlsl_6, sqdmlsl(s0, h1, h2))
TEST_NEON(sqdmlsl_7, sqdmlsl(d0, s1, s2))
TEST_NEON(sqdmulh_0, sqdmulh(v0.V4H(), v1.V4H(), v2.H(), 2))
TEST_NEON(sqdmulh_1, sqdmulh(v0.V8H(), v1.V8H(), v2.H(), 2))
TEST_NEON(sqdmulh_2, sqdmulh(v0.V2S(), v1.V2S(), v2.S(), 1))
TEST_NEON(sqdmulh_3, sqdmulh(v0.V4S(), v1.V4S(), v2.S(), 2))
TEST_NEON(sqdmulh_4, sqdmulh(h0, h1, v2.H(), 2))
TEST_NEON(sqdmulh_5, sqdmulh(s0, s1, v2.S(), 0))
TEST_NEON(sqdmulh_6, sqdmulh(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(sqdmulh_7, sqdmulh(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(sqdmulh_8, sqdmulh(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(sqdmulh_9, sqdmulh(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sqdmulh_10, sqdmulh(h0, h1, h2))
TEST_NEON(sqdmulh_11, sqdmulh(s0, s1, s2))
TEST_NEON(sqdmull_0, sqdmull(v0.V4S(), v1.V4H(), v2.H(), 1))
TEST_NEON(sqdmull_1, sqdmull(v0.V2D(), v1.V2S(), v2.S(), 3))
TEST_NEON(sqdmull2_0, sqdmull2(v0.V4S(), v1.V8H(), v2.H(), 0))
TEST_NEON(sqdmull2_1, sqdmull2(v0.V2D(), v1.V4S(), v2.S(), 3))
TEST_NEON(sqdmull_2, sqdmull(s0, h1, v2.H(), 2))
TEST_NEON(sqdmull_3, sqdmull(d0, s1, v2.S(), 1))
TEST_NEON(sqdmull_4, sqdmull(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(sqdmull_5, sqdmull(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(sqdmull2_2, sqdmull2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(sqdmull2_3, sqdmull2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(sqdmull_6, sqdmull(s0, h1, h2))
TEST_NEON(sqdmull_7, sqdmull(d0, s1, s2))
TEST_NEON(sqneg_0, sqneg(v0.V8B(), v1.V8B()))
TEST_NEON(sqneg_1, sqneg(v0.V16B(), v1.V16B()))
TEST_NEON(sqneg_2, sqneg(v0.V4H(), v1.V4H()))
TEST_NEON(sqneg_3, sqneg(v0.V8H(), v1.V8H()))
TEST_NEON(sqneg_4, sqneg(v0.V2S(), v1.V2S()))
TEST_NEON(sqneg_5, sqneg(v0.V4S(), v1.V4S()))
TEST_NEON(sqneg_6, sqneg(v0.V2D(), v1.V2D()))
TEST_NEON(sqneg_7, sqneg(b0, b1))
TEST_NEON(sqneg_8, sqneg(h0, h1))
TEST_NEON(sqneg_9, sqneg(s0, s1))
TEST_NEON(sqneg_10, sqneg(d0, d1))
TEST_NEON(sqrdmulh_0, sqrdmulh(v0.V4H(), v1.V4H(), v2.H(), 0))
TEST_NEON(sqrdmulh_1, sqrdmulh(v0.V8H(), v1.V8H(), v2.H(), 2))
TEST_NEON(sqrdmulh_2, sqrdmulh(v0.V2S(), v1.V2S(), v2.S(), 3))
TEST_NEON(sqrdmulh_3, sqrdmulh(v0.V4S(), v1.V4S(), v2.S(), 1))
TEST_NEON(sqrdmulh_4, sqrdmulh(h0, h1, v2.H(), 2))
TEST_NEON(sqrdmulh_5, sqrdmulh(s0, s1, v2.S(), 0))
TEST_NEON(sqrdmulh_6, sqrdmulh(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(sqrdmulh_7, sqrdmulh(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(sqrdmulh_8, sqrdmulh(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(sqrdmulh_9, sqrdmulh(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sqrdmulh_10, sqrdmulh(h0, h1, h2))
TEST_NEON(sqrdmulh_11, sqrdmulh(s0, s1, s2))
TEST_NEON(sqrshl_0, sqrshl(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(sqrshl_1, sqrshl(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(sqrshl_2, sqrshl(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(sqrshl_3, sqrshl(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(sqrshl_4, sqrshl(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(sqrshl_5, sqrshl(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sqrshl_6, sqrshl(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(sqrshl_7, sqrshl(b0, b1, b2))
TEST_NEON(sqrshl_8, sqrshl(h0, h1, h2))
TEST_NEON(sqrshl_9, sqrshl(s0, s1, s2))
TEST_NEON(sqrshl_10, sqrshl(d0, d1, d2))
TEST_NEON(sqrshrn_0, sqrshrn(v0.V8B(), v1.V8H(), 1))
TEST_NEON(sqrshrn_1, sqrshrn(v0.V4H(), v1.V4S(), 14))
TEST_NEON(sqrshrn_2, sqrshrn(v0.V2S(), v1.V2D(), 29))
TEST_NEON(sqrshrn2_0, sqrshrn2(v0.V16B(), v1.V8H(), 3))
TEST_NEON(sqrshrn2_1, sqrshrn2(v0.V8H(), v1.V4S(), 11))
TEST_NEON(sqrshrn2_2, sqrshrn2(v0.V4S(), v1.V2D(), 25))
TEST_NEON(sqrshrn_3, sqrshrn(b0, h1, 5))
TEST_NEON(sqrshrn_4, sqrshrn(h0, s1, 4))
TEST_NEON(sqrshrn_5, sqrshrn(s0, d1, 30))
TEST_NEON(sqrshrun_0, sqrshrun(v0.V8B(), v1.V8H(), 3))
TEST_NEON(sqrshrun_1, sqrshrun(v0.V4H(), v1.V4S(), 6))
TEST_NEON(sqrshrun_2, sqrshrun(v0.V2S(), v1.V2D(), 13))
TEST_NEON(sqrshrun2_0, sqrshrun2(v0.V16B(), v1.V8H(), 1))
TEST_NEON(sqrshrun2_1, sqrshrun2(v0.V8H(), v1.V4S(), 7))
TEST_NEON(sqrshrun2_2, sqrshrun2(v0.V4S(), v1.V2D(), 29))
TEST_NEON(sqrshrun_3, sqrshrun(b0, h1, 7))
TEST_NEON(sqrshrun_4, sqrshrun(h0, s1, 13))
TEST_NEON(sqrshrun_5, sqrshrun(s0, d1, 29))
TEST_NEON(sqshlu_0, sqshlu(v0.V8B(), v1.V8B(), 4))
TEST_NEON(sqshlu_1, sqshlu(v0.V16B(), v1.V16B(), 7))
TEST_NEON(sqshlu_2, sqshlu(v0.V4H(), v1.V4H(), 14))
TEST_NEON(sqshlu_3, sqshlu(v0.V8H(), v1.V8H(), 15))
TEST_NEON(sqshlu_4, sqshlu(v0.V2S(), v1.V2S(), 13))
TEST_NEON(sqshlu_5, sqshlu(v0.V4S(), v1.V4S(), 6))
TEST_NEON(sqshlu_6, sqshlu(v0.V2D(), v1.V2D(), 42))
TEST_NEON(sqshlu_7, sqshlu(b0, b1, 3))
TEST_NEON(sqshlu_8, sqshlu(h0, h1, 15))
TEST_NEON(sqshlu_9, sqshlu(s0, s1, 21))
TEST_NEON(sqshlu_10, sqshlu(d0, d1, 15))
TEST_NEON(sqshl_0, sqshl(v0.V8B(), v1.V8B(), 6))
TEST_NEON(sqshl_1, sqshl(v0.V16B(), v1.V16B(), 6))
TEST_NEON(sqshl_2, sqshl(v0.V4H(), v1.V4H(), 8))
TEST_NEON(sqshl_3, sqshl(v0.V8H(), v1.V8H(), 9))
TEST_NEON(sqshl_4, sqshl(v0.V2S(), v1.V2S(), 28))
TEST_NEON(sqshl_5, sqshl(v0.V4S(), v1.V4S(), 27))
TEST_NEON(sqshl_6, sqshl(v0.V2D(), v1.V2D(), 50))
TEST_NEON(sqshl_7, sqshl(b0, b1, 4))
TEST_NEON(sqshl_8, sqshl(h0, h1, 13))
TEST_NEON(sqshl_9, sqshl(s0, s1, 15))
TEST_NEON(sqshl_10, sqshl(d0, d1, 40))
TEST_NEON(sqshl_11, sqshl(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(sqshl_12, sqshl(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(sqshl_13, sqshl(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(sqshl_14, sqshl(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(sqshl_15, sqshl(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(sqshl_16, sqshl(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sqshl_17, sqshl(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(sqshl_18, sqshl(b0, b1, b2))
TEST_NEON(sqshl_19, sqshl(h0, h1, h2))
TEST_NEON(sqshl_20, sqshl(s0, s1, s2))
TEST_NEON(sqshl_21, sqshl(d0, d1, d2))
TEST_NEON(sqshrn_0, sqshrn(v0.V8B(), v1.V8H(), 5))
TEST_NEON(sqshrn_1, sqshrn(v0.V4H(), v1.V4S(), 5))
TEST_NEON(sqshrn_2, sqshrn(v0.V2S(), v1.V2D(), 2))
TEST_NEON(sqshrn2_0, sqshrn2(v0.V16B(), v1.V8H(), 6))
TEST_NEON(sqshrn2_1, sqshrn2(v0.V8H(), v1.V4S(), 10))
TEST_NEON(sqshrn2_2, sqshrn2(v0.V4S(), v1.V2D(), 2))
TEST_NEON(sqshrn_3, sqshrn(b0, h1, 2))
TEST_NEON(sqshrn_4, sqshrn(h0, s1, 8))
TEST_NEON(sqshrn_5, sqshrn(s0, d1, 27))
TEST_NEON(sqshrun_0, sqshrun(v0.V8B(), v1.V8H(), 4))
TEST_NEON(sqshrun_1, sqshrun(v0.V4H(), v1.V4S(), 11))
TEST_NEON(sqshrun_2, sqshrun(v0.V2S(), v1.V2D(), 20))
TEST_NEON(sqshrun2_0, sqshrun2(v0.V16B(), v1.V8H(), 6))
TEST_NEON(sqshrun2_1, sqshrun2(v0.V8H(), v1.V4S(), 5))
TEST_NEON(sqshrun2_2, sqshrun2(v0.V4S(), v1.V2D(), 18))
TEST_NEON(sqshrun_3, sqshrun(b0, h1, 2))
TEST_NEON(sqshrun_4, sqshrun(h0, s1, 10))
TEST_NEON(sqshrun_5, sqshrun(s0, d1, 16))
TEST_NEON(sqsub_0, sqsub(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(sqsub_1, sqsub(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(sqsub_2, sqsub(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(sqsub_3, sqsub(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(sqsub_4, sqsub(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(sqsub_5, sqsub(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sqsub_6, sqsub(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(sqsub_7, sqsub(b0, b1, b2))
TEST_NEON(sqsub_8, sqsub(h0, h1, h2))
TEST_NEON(sqsub_9, sqsub(s0, s1, s2))
TEST_NEON(sqsub_10, sqsub(d0, d1, d2))
TEST_NEON(sqxtn_0, sqxtn(v0.V8B(), v1.V8H()))
TEST_NEON(sqxtn_1, sqxtn(v0.V4H(), v1.V4S()))
TEST_NEON(sqxtn_2, sqxtn(v0.V2S(), v1.V2D()))
TEST_NEON(sqxtn2_0, sqxtn2(v0.V16B(), v1.V8H()))
TEST_NEON(sqxtn2_1, sqxtn2(v0.V8H(), v1.V4S()))
TEST_NEON(sqxtn2_2, sqxtn2(v0.V4S(), v1.V2D()))
TEST_NEON(sqxtn_3, sqxtn(b0, h1))
TEST_NEON(sqxtn_4, sqxtn(h0, s1))
TEST_NEON(sqxtn_5, sqxtn(s0, d1))
TEST_NEON(sqxtun_0, sqxtun(v0.V8B(), v1.V8H()))
TEST_NEON(sqxtun_1, sqxtun(v0.V4H(), v1.V4S()))
TEST_NEON(sqxtun_2, sqxtun(v0.V2S(), v1.V2D()))
TEST_NEON(sqxtun2_0, sqxtun2(v0.V16B(), v1.V8H()))
TEST_NEON(sqxtun2_1, sqxtun2(v0.V8H(), v1.V4S()))
TEST_NEON(sqxtun2_2, sqxtun2(v0.V4S(), v1.V2D()))
TEST_NEON(sqxtun_3, sqxtun(b0, h1))
TEST_NEON(sqxtun_4, sqxtun(h0, s1))
TEST_NEON(sqxtun_5, sqxtun(s0, d1))
TEST_NEON(srhadd_0, srhadd(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(srhadd_1, srhadd(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(srhadd_2, srhadd(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(srhadd_3, srhadd(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(srhadd_4, srhadd(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(srhadd_5, srhadd(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sri_0, sri(v0.V8B(), v1.V8B(), 7))
TEST_NEON(sri_1, sri(v0.V16B(), v1.V16B(), 3))
TEST_NEON(sri_2, sri(v0.V4H(), v1.V4H(), 10))
TEST_NEON(sri_3, sri(v0.V8H(), v1.V8H(), 7))
TEST_NEON(sri_4, sri(v0.V2S(), v1.V2S(), 12))
TEST_NEON(sri_5, sri(v0.V4S(), v1.V4S(), 15))
TEST_NEON(sri_6, sri(v0.V2D(), v1.V2D(), 51))
TEST_NEON(sri_7, sri(d0, d1, 47))
TEST_NEON(srshl_0, srshl(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(srshl_1, srshl(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(srshl_2, srshl(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(srshl_3, srshl(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(srshl_4, srshl(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(srshl_5, srshl(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(srshl_6, srshl(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(srshl_7, srshl(d0, d1, d2))
TEST_NEON(srshr_0, srshr(v0.V8B(), v1.V8B(), 2))
TEST_NEON(srshr_1, srshr(v0.V16B(), v1.V16B(), 3))
TEST_NEON(srshr_2, srshr(v0.V4H(), v1.V4H(), 2))
TEST_NEON(srshr_3, srshr(v0.V8H(), v1.V8H(), 7))
TEST_NEON(srshr_4, srshr(v0.V2S(), v1.V2S(), 25))
TEST_NEON(srshr_5, srshr(v0.V4S(), v1.V4S(), 27))
TEST_NEON(srshr_6, srshr(v0.V2D(), v1.V2D(), 43))
TEST_NEON(srshr_7, srshr(d0, d1, 28))
TEST_NEON(srsra_0, srsra(v0.V8B(), v1.V8B(), 4))
TEST_NEON(srsra_1, srsra(v0.V16B(), v1.V16B(), 2))
TEST_NEON(srsra_2, srsra(v0.V4H(), v1.V4H(), 13))
TEST_NEON(srsra_3, srsra(v0.V8H(), v1.V8H(), 6))
TEST_NEON(srsra_4, srsra(v0.V2S(), v1.V2S(), 4))
TEST_NEON(srsra_5, srsra(v0.V4S(), v1.V4S(), 1))
TEST_NEON(srsra_6, srsra(v0.V2D(), v1.V2D(), 17))
TEST_NEON(srsra_7, srsra(d0, d1, 16))
TEST_NEON(sshll_0, sshll(v0.V8H(), v1.V8B(), 2))
TEST_NEON(sshll_1, sshll(v0.V4S(), v1.V4H(), 4))
TEST_NEON(sshll_2, sshll(v0.V2D(), v1.V2S(), 28))
TEST_NEON(sshll2_0, sshll2(v0.V8H(), v1.V16B(), 6))
TEST_NEON(sshll2_1, sshll2(v0.V4S(), v1.V8H(), 2))
TEST_NEON(sshll2_2, sshll2(v0.V2D(), v1.V4S(), 22))
TEST_NEON(sshl_0, sshl(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(sshl_1, sshl(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(sshl_2, sshl(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(sshl_3, sshl(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(sshl_4, sshl(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(sshl_5, sshl(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sshl_6, sshl(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(sshl_7, sshl(d0, d1, d2))
TEST_NEON(sshr_0, sshr(v0.V8B(), v1.V8B(), 7))
TEST_NEON(sshr_1, sshr(v0.V16B(), v1.V16B(), 1))
TEST_NEON(sshr_2, sshr(v0.V4H(), v1.V4H(), 9))
TEST_NEON(sshr_3, sshr(v0.V8H(), v1.V8H(), 13))
TEST_NEON(sshr_4, sshr(v0.V2S(), v1.V2S(), 19))
TEST_NEON(sshr_5, sshr(v0.V4S(), v1.V4S(), 26))
TEST_NEON(sshr_6, sshr(v0.V2D(), v1.V2D(), 63))
TEST_NEON(sshr_7, sshr(d0, d1, 39))
TEST_NEON(ssra_0, ssra(v0.V8B(), v1.V8B(), 5))
TEST_NEON(ssra_1, ssra(v0.V16B(), v1.V16B(), 7))
TEST_NEON(ssra_2, ssra(v0.V4H(), v1.V4H(), 14))
TEST_NEON(ssra_3, ssra(v0.V8H(), v1.V8H(), 6))
TEST_NEON(ssra_4, ssra(v0.V2S(), v1.V2S(), 12))
TEST_NEON(ssra_5, ssra(v0.V4S(), v1.V4S(), 4))
TEST_NEON(ssra_6, ssra(v0.V2D(), v1.V2D(), 16))
TEST_NEON(ssra_7, ssra(d0, d1, 53))
TEST_NEON(ssubl_0, ssubl(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(ssubl_1, ssubl(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(ssubl_2, ssubl(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(ssubl2_0, ssubl2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(ssubl2_1, ssubl2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(ssubl2_2, ssubl2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(ssubw_0, ssubw(v0.V8H(), v1.V8H(), v2.V8B()))
TEST_NEON(ssubw_1, ssubw(v0.V4S(), v1.V4S(), v2.V4H()))
TEST_NEON(ssubw_2, ssubw(v0.V2D(), v1.V2D(), v2.V2S()))
TEST_NEON(ssubw2_0, ssubw2(v0.V8H(), v1.V8H(), v2.V16B()))
TEST_NEON(ssubw2_1, ssubw2(v0.V4S(), v1.V4S(), v2.V8H()))
TEST_NEON(ssubw2_2, ssubw2(v0.V2D(), v1.V2D(), v2.V4S()))
TEST_NEON(st1_0, st1(v0.V8B(), MemOperand(x1)))
TEST_NEON(st1_1, st1(v0.V16B(), MemOperand(x1)))
TEST_NEON(st1_2, st1(v0.V4H(), MemOperand(x1)))
TEST_NEON(st1_3, st1(v0.V8H(), MemOperand(x1)))
TEST_NEON(st1_4, st1(v0.V2S(), MemOperand(x1)))
TEST_NEON(st1_5, st1(v0.V4S(), MemOperand(x1)))
TEST_NEON(st1_6, st1(v0.V1D(), MemOperand(x1)))
TEST_NEON(st1_7, st1(v0.V2D(), MemOperand(x1)))
TEST_NEON(st1_8, st1(v0.V8B(), v1.V8B(), MemOperand(x2)))
TEST_NEON(st1_9, st1(v0.V16B(), v1.V16B(), MemOperand(x2)))
TEST_NEON(st1_10, st1(v0.V4H(), v1.V4H(), MemOperand(x2)))
TEST_NEON(st1_11, st1(v0.V8H(), v1.V8H(), MemOperand(x2)))
TEST_NEON(st1_12, st1(v0.V2S(), v1.V2S(), MemOperand(x2)))
TEST_NEON(st1_13, st1(v0.V4S(), v1.V4S(), MemOperand(x2)))
TEST_NEON(st1_14, st1(v0.V1D(), v1.V1D(), MemOperand(x2)))
TEST_NEON(st1_15, st1(v0.V2D(), v1.V2D(), MemOperand(x2)))
TEST_NEON(st1_16, st1(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3)))
TEST_NEON(st1_17, st1(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3)))
TEST_NEON(st1_18, st1(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3)))
TEST_NEON(st1_19, st1(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3)))
TEST_NEON(st1_20, st1(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3)))
TEST_NEON(st1_21, st1(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3)))
TEST_NEON(st1_22, st1(v0.V1D(), v1.V1D(), v2.V1D(), MemOperand(x3)))
TEST_NEON(st1_23, st1(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3)))
TEST_NEON(st1_24, st1(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4)))
TEST_NEON(st1_25,
          st1(v0.V16B(), v1.V16B(), v2.V16B(), v3.V16B(), MemOperand(x4)))
TEST_NEON(st1_26, st1(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4)))
TEST_NEON(st1_27, st1(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4)))
TEST_NEON(st1_28, st1(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4)))
TEST_NEON(st1_29, st1(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4)))
TEST_NEON(st1_30, st1(v0.V1D(), v1.V1D(), v2.V1D(), v3.V1D(), MemOperand(x4)))
TEST_NEON(st1_31, st1(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4)))
TEST_NEON(st1_32, st1(v0.V8B(), MemOperand(x1, 8, PostIndex)))
TEST_NEON(st1_33, st1(v0.V16B(), MemOperand(x1, 16, PostIndex)))
TEST_NEON(st1_34, st1(v0.V4H(), MemOperand(x1, 8, PostIndex)))
TEST_NEON(st1_35, st1(v0.V8H(), MemOperand(x1, 16, PostIndex)))
TEST_NEON(st1_36, st1(v0.V2S(), MemOperand(x1, 8, PostIndex)))
TEST_NEON(st1_37, st1(v0.V4S(), MemOperand(x1, 16, PostIndex)))
TEST_NEON(st1_38, st1(v0.V1D(), MemOperand(x1, 8, PostIndex)))
TEST_NEON(st1_39, st1(v0.V2D(), MemOperand(x1, 16, PostIndex)))
TEST_NEON(st1_40, st1(v0.V8B(), v1.V8B(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(st1_41, st1(v0.V16B(), v1.V16B(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(st1_42, st1(v0.V4H(), v1.V4H(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(st1_43, st1(v0.V8H(), v1.V8H(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(st1_44, st1(v0.V2S(), v1.V2S(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(st1_45, st1(v0.V4S(), v1.V4S(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(st1_46, st1(v0.V1D(), v1.V1D(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(st1_47, st1(v0.V2D(), v1.V2D(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(st1_48,
          st1(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(st1_49,
          st1(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(st1_50,
          st1(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(st1_51,
          st1(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(st1_52,
          st1(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(st1_53,
          st1(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(st1_54,
          st1(v0.V1D(), v1.V1D(), v2.V1D(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(st1_55,
          st1(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(
    st1_56,
    st1(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(st1_57,
          st1(v0.V16B(),
              v1.V16B(),
              v2.V16B(),
              v3.V16B(),
              MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    st1_58,
    st1(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(
    st1_59,
    st1(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    st1_60,
    st1(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(
    st1_61,
    st1(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    st1_62,
    st1(v0.V1D(), v1.V1D(), v2.V1D(), v3.V1D(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(
    st1_63,
    st1(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4, 64, PostIndex)))
TEST_NEON(st1_64, st1(v0.V8B(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(st1_65, st1(v0.V16B(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(st1_66, st1(v0.V4H(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(st1_67, st1(v0.V8H(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(st1_68, st1(v0.V2S(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(st1_69, st1(v0.V4S(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(st1_70, st1(v0.V1D(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(st1_71, st1(v0.V2D(), MemOperand(x1, x2, PostIndex)))
TEST_NEON(st1_72, st1(v0.V8B(), v1.V8B(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st1_73, st1(v0.V16B(), v1.V16B(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st1_74, st1(v0.V4H(), v1.V4H(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st1_75, st1(v0.V8H(), v1.V8H(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st1_76, st1(v0.V2S(), v1.V2S(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st1_77, st1(v0.V4S(), v1.V4S(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st1_78, st1(v0.V1D(), v1.V1D(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st1_79, st1(v0.V2D(), v1.V2D(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st1_80,
          st1(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st1_81,
          st1(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st1_82,
          st1(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st1_83,
          st1(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st1_84,
          st1(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st1_85,
          st1(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st1_86,
          st1(v0.V1D(), v1.V1D(), v2.V1D(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st1_87,
          st1(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(
    st1_88,
    st1(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(st1_89,
          st1(v0.V16B(),
              v1.V16B(),
              v2.V16B(),
              v3.V16B(),
              MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    st1_90,
    st1(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    st1_91,
    st1(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    st1_92,
    st1(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    st1_93,
    st1(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    st1_94,
    st1(v0.V1D(), v1.V1D(), v2.V1D(), v3.V1D(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    st1_95,
    st1(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(st1_96, st1(v0.B(), 13, MemOperand(x1)))
TEST_NEON(st1_97, st1(v0.D(), 0, MemOperand(x1)))
TEST_NEON(st1_98, st1(v0.H(), 2, MemOperand(x1)))
TEST_NEON(st1_99, st1(v0.S(), 2, MemOperand(x1)))
TEST_NEON(st1_100, st1(v0.B(), 0, MemOperand(x1, 1, PostIndex)))
TEST_NEON(st1_101, st1(v0.B(), 7, MemOperand(x1, x2, PostIndex)))
TEST_NEON(st1_102, st1(v0.D(), 1, MemOperand(x1, 8, PostIndex)))
TEST_NEON(st1_103, st1(v0.D(), 1, MemOperand(x1, x2, PostIndex)))
TEST_NEON(st1_104, st1(v0.H(), 6, MemOperand(x1, 2, PostIndex)))
TEST_NEON(st1_105, st1(v0.H(), 7, MemOperand(x1, x2, PostIndex)))
TEST_NEON(st1_106, st1(v0.S(), 0, MemOperand(x1, 4, PostIndex)))
TEST_NEON(st1_107, st1(v0.S(), 2, MemOperand(x1, x2, PostIndex)))
TEST_NEON(st2_0, st2(v0.V8B(), v1.V8B(), MemOperand(x2)))
TEST_NEON(st2_1, st2(v0.V16B(), v1.V16B(), MemOperand(x2)))
TEST_NEON(st2_2, st2(v0.V4H(), v1.V4H(), MemOperand(x2)))
TEST_NEON(st2_3, st2(v0.V8H(), v1.V8H(), MemOperand(x2)))
TEST_NEON(st2_4, st2(v0.V2S(), v1.V2S(), MemOperand(x2)))
TEST_NEON(st2_5, st2(v0.V4S(), v1.V4S(), MemOperand(x2)))
TEST_NEON(st2_6, st2(v0.V2D(), v1.V2D(), MemOperand(x2)))
TEST_NEON(st2_7, st2(v0.V8B(), v1.V8B(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(st2_8, st2(v0.V16B(), v1.V16B(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(st2_9, st2(v0.V4H(), v1.V4H(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(st2_10, st2(v0.V8H(), v1.V8H(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(st2_11, st2(v0.V2S(), v1.V2S(), MemOperand(x2, 16, PostIndex)))
TEST_NEON(st2_12, st2(v0.V4S(), v1.V4S(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(st2_13, st2(v0.V2D(), v1.V2D(), MemOperand(x2, 32, PostIndex)))
TEST_NEON(st2_14, st2(v0.V8B(), v1.V8B(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st2_15, st2(v0.V16B(), v1.V16B(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st2_16, st2(v0.V4H(), v1.V4H(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st2_17, st2(v0.V8H(), v1.V8H(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st2_18, st2(v0.V2S(), v1.V2S(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st2_19, st2(v0.V4S(), v1.V4S(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st2_20, st2(v0.V2D(), v1.V2D(), MemOperand(x2, x3, PostIndex)))
TEST_NEON(st2_21, st2(v0.B(), v1.B(), 0, MemOperand(x2)))
TEST_NEON(st2_22, st2(v0.D(), v1.D(), 1, MemOperand(x2)))
TEST_NEON(st2_23, st2(v0.H(), v1.H(), 5, MemOperand(x2)))
TEST_NEON(st2_24, st2(v0.S(), v1.S(), 1, MemOperand(x2)))
TEST_NEON(st2_25, st2(v0.B(), v1.B(), 13, MemOperand(x2, 2, PostIndex)))
TEST_NEON(st2_26, st2(v0.B(), v1.B(), 14, MemOperand(x2, x3, PostIndex)))
TEST_NEON(st2_27, st2(v0.D(), v1.D(), 1, MemOperand(x2, 16, PostIndex)))
TEST_NEON(st2_28, st2(v0.D(), v1.D(), 1, MemOperand(x2, x3, PostIndex)))
TEST_NEON(st2_29, st2(v0.H(), v1.H(), 2, MemOperand(x2, 4, PostIndex)))
TEST_NEON(st2_30, st2(v0.H(), v1.H(), 4, MemOperand(x2, x3, PostIndex)))
TEST_NEON(st2_31, st2(v0.S(), v1.S(), 0, MemOperand(x2, 8, PostIndex)))
TEST_NEON(st2_32, st2(v0.S(), v1.S(), 0, MemOperand(x2, x3, PostIndex)))
TEST_NEON(st3_0, st3(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3)))
TEST_NEON(st3_1, st3(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3)))
TEST_NEON(st3_2, st3(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3)))
TEST_NEON(st3_3, st3(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3)))
TEST_NEON(st3_4, st3(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3)))
TEST_NEON(st3_5, st3(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3)))
TEST_NEON(st3_6, st3(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3)))
TEST_NEON(st3_7,
          st3(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(st3_8,
          st3(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(st3_9,
          st3(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(st3_10,
          st3(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(st3_11,
          st3(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3, 24, PostIndex)))
TEST_NEON(st3_12,
          st3(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(st3_13,
          st3(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3, 48, PostIndex)))
TEST_NEON(st3_14,
          st3(v0.V8B(), v1.V8B(), v2.V8B(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st3_15,
          st3(v0.V16B(), v1.V16B(), v2.V16B(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st3_16,
          st3(v0.V4H(), v1.V4H(), v2.V4H(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st3_17,
          st3(v0.V8H(), v1.V8H(), v2.V8H(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st3_18,
          st3(v0.V2S(), v1.V2S(), v2.V2S(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st3_19,
          st3(v0.V4S(), v1.V4S(), v2.V4S(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st3_20,
          st3(v0.V2D(), v1.V2D(), v2.V2D(), MemOperand(x3, x4, PostIndex)))
TEST_NEON(st3_21, st3(v0.B(), v1.B(), v2.B(), 1, MemOperand(x3)))
TEST_NEON(st3_22, st3(v0.D(), v1.D(), v2.D(), 1, MemOperand(x3)))
TEST_NEON(st3_23, st3(v0.H(), v1.H(), v2.H(), 7, MemOperand(x3)))
TEST_NEON(st3_24, st3(v0.S(), v1.S(), v2.S(), 2, MemOperand(x3)))
TEST_NEON(st3_25, st3(v0.B(), v1.B(), v2.B(), 6, MemOperand(x3, 3, PostIndex)))
TEST_NEON(st3_26,
          st3(v0.B(), v1.B(), v2.B(), 12, MemOperand(x3, x4, PostIndex)))
TEST_NEON(st3_27, st3(v0.D(), v1.D(), v2.D(), 0, MemOperand(x3, 24, PostIndex)))
TEST_NEON(st3_28, st3(v0.D(), v1.D(), v2.D(), 0, MemOperand(x3, x4, PostIndex)))
TEST_NEON(st3_29, st3(v0.H(), v1.H(), v2.H(), 0, MemOperand(x3, 6, PostIndex)))
TEST_NEON(st3_30, st3(v0.H(), v1.H(), v2.H(), 4, MemOperand(x3, x4, PostIndex)))
TEST_NEON(st3_31, st3(v0.S(), v1.S(), v2.S(), 3, MemOperand(x3, 12, PostIndex)))
TEST_NEON(st3_32, st3(v0.S(), v1.S(), v2.S(), 2, MemOperand(x3, x4, PostIndex)))
TEST_NEON(st4_0, st4(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4)))
TEST_NEON(st4_1,
          st4(v0.V16B(), v1.V16B(), v2.V16B(), v3.V16B(), MemOperand(x4)))
TEST_NEON(st4_2, st4(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4)))
TEST_NEON(st4_3, st4(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4)))
TEST_NEON(st4_4, st4(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4)))
TEST_NEON(st4_5, st4(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4)))
TEST_NEON(st4_6, st4(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4)))
TEST_NEON(
    st4_7,
    st4(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(st4_8,
          st4(v0.V16B(),
              v1.V16B(),
              v2.V16B(),
              v3.V16B(),
              MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    st4_9,
    st4(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(
    st4_10,
    st4(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    st4_11,
    st4(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4, 32, PostIndex)))
TEST_NEON(
    st4_12,
    st4(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    st4_13,
    st4(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4, 64, PostIndex)))
TEST_NEON(
    st4_14,
    st4(v0.V8B(), v1.V8B(), v2.V8B(), v3.V8B(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(st4_15,
          st4(v0.V16B(),
              v1.V16B(),
              v2.V16B(),
              v3.V16B(),
              MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    st4_16,
    st4(v0.V4H(), v1.V4H(), v2.V4H(), v3.V4H(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    st4_17,
    st4(v0.V8H(), v1.V8H(), v2.V8H(), v3.V8H(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    st4_18,
    st4(v0.V2S(), v1.V2S(), v2.V2S(), v3.V2S(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    st4_19,
    st4(v0.V4S(), v1.V4S(), v2.V4S(), v3.V4S(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(
    st4_20,
    st4(v0.V2D(), v1.V2D(), v2.V2D(), v3.V2D(), MemOperand(x4, x5, PostIndex)))
TEST_NEON(st4_21, st4(v0.B(), v1.B(), v2.B(), v3.B(), 1, MemOperand(x4)))
TEST_NEON(st4_22, st4(v0.D(), v1.D(), v2.D(), v3.D(), 0, MemOperand(x4)))
TEST_NEON(st4_23, st4(v0.H(), v1.H(), v2.H(), v3.H(), 6, MemOperand(x4)))
TEST_NEON(st4_24, st4(v0.S(), v1.S(), v2.S(), v3.S(), 0, MemOperand(x4)))
TEST_NEON(st4_25,
          st4(v0.B(), v1.B(), v2.B(), v3.B(), 4, MemOperand(x4, 4, PostIndex)))
TEST_NEON(st4_26,
          st4(v0.B(), v1.B(), v2.B(), v3.B(), 4, MemOperand(x4, x5, PostIndex)))
TEST_NEON(st4_27,
          st4(v0.D(), v1.D(), v2.D(), v3.D(), 1, MemOperand(x4, 32, PostIndex)))
TEST_NEON(st4_28,
          st4(v0.D(), v1.D(), v2.D(), v3.D(), 0, MemOperand(x4, x5, PostIndex)))
TEST_NEON(st4_29,
          st4(v0.H(), v1.H(), v2.H(), v3.H(), 0, MemOperand(x4, 8, PostIndex)))
TEST_NEON(st4_30,
          st4(v0.H(), v1.H(), v2.H(), v3.H(), 7, MemOperand(x4, x5, PostIndex)))
TEST_NEON(st4_31,
          st4(v0.S(), v1.S(), v2.S(), v3.S(), 0, MemOperand(x4, 16, PostIndex)))
TEST_NEON(st4_32,
          st4(v0.S(), v1.S(), v2.S(), v3.S(), 3, MemOperand(x4, x5, PostIndex)))
TEST_NEON(stnp_0, stnp(d0, d1, MemOperand(x2, 304)))
TEST_NEON(stnp_1, stnp(q0, q1, MemOperand(x2, 480)))
TEST_NEON(stnp_2, stnp(s0, s1, MemOperand(x2, -12)))
TEST_NEON(stp_0, stp(d0, d1, MemOperand(x2, 168)))
TEST_NEON(stp_1, stp(d0, d1, MemOperand(x2, -376, PostIndex)))
TEST_NEON(stp_2, stp(d0, d1, MemOperand(x2, 296, PreIndex)))
TEST_NEON(stp_3, stp(q0, q1, MemOperand(x2, -80)))
TEST_NEON(stp_4, stp(q0, q1, MemOperand(x2, -768, PostIndex)))
TEST_NEON(stp_5, stp(q0, q1, MemOperand(x2, -288, PreIndex)))
TEST_NEON(stp_6, stp(s0, s1, MemOperand(x2, -256)))
TEST_NEON(stp_7, stp(s0, s1, MemOperand(x2, 208, PostIndex)))
TEST_NEON(stp_8, stp(s0, s1, MemOperand(x2, -4, PreIndex)))
TEST_NEON(str_0, str(b0, MemOperand(x1, -45, PostIndex)))
TEST_NEON(str_1, str(b0, MemOperand(x1, -154, PreIndex)))
TEST_NEON(str_2, str(b0, MemOperand(x1, 992)))
TEST_NEON(str_3, str(d0, MemOperand(x1, -181, PostIndex)))
TEST_NEON(str_4, str(d0, MemOperand(x1, 91, PreIndex)))
TEST_NEON(str_5, str(d0, MemOperand(x1, 32672)))
TEST_NEON(str_6, str(h0, MemOperand(x1, -5, PostIndex)))
TEST_NEON(str_7, str(h0, MemOperand(x1, 213, PreIndex)))
TEST_NEON(str_8, str(h0, MemOperand(x1, 6406)))
TEST_NEON(str_9, str(q0, MemOperand(x1, -87, PostIndex)))
TEST_NEON(str_10, str(q0, MemOperand(x1, 198, PreIndex)))
TEST_NEON(str_11, str(q0, MemOperand(x1, 56032)))
TEST_NEON(str_12, str(s0, MemOperand(x1, -81, PostIndex)))
TEST_NEON(str_13, str(s0, MemOperand(x1, -126, PreIndex)))
TEST_NEON(str_14, str(s0, MemOperand(x1, 15692)))
TEST_NEON(str_15, str(b0, MemOperand(x1, x2, LSL, 0)))
TEST_NEON(str_16, str(b0, MemOperand(x1, w2, SXTW, 0)))
TEST_NEON(str_17, str(d0, MemOperand(x1, w2, SXTW, 0)))
TEST_NEON(str_18, str(d0, MemOperand(x1, x2, LSL, 0)))
TEST_NEON(str_19, str(h0, MemOperand(x1, w2, UXTW, 1)))
TEST_NEON(str_20, str(h0, MemOperand(x1, x2, SXTX, 1)))
TEST_NEON(str_21, str(q0, MemOperand(x1, w2, SXTW, 4)))
TEST_NEON(str_22, str(q0, MemOperand(x1, x2, SXTX, 4)))
TEST_NEON(str_23, str(s0, MemOperand(x1, w2, UXTW, 0)))
TEST_NEON(str_24, str(s0, MemOperand(x1, x2, SXTX, 2)))
TEST_NEON(stur_0, stur(b0, MemOperand(x1, 83)))
TEST_NEON(stur_1, stur(d0, MemOperand(x1, 22)))
TEST_NEON(stur_2, stur(h0, MemOperand(x1, -236)))
TEST_NEON(stur_3, stur(q0, MemOperand(x1, 13)))
TEST_NEON(stur_4, stur(s0, MemOperand(x1, 23)))
TEST_NEON(subhn_0, subhn(v0.V8B(), v1.V8H(), v2.V8H()))
TEST_NEON(subhn_1, subhn(v0.V4H(), v1.V4S(), v2.V4S()))
TEST_NEON(subhn_2, subhn(v0.V2S(), v1.V2D(), v2.V2D()))
TEST_NEON(subhn2_0, subhn2(v0.V16B(), v1.V8H(), v2.V8H()))
TEST_NEON(subhn2_1, subhn2(v0.V8H(), v1.V4S(), v2.V4S()))
TEST_NEON(subhn2_2, subhn2(v0.V4S(), v1.V2D(), v2.V2D()))
TEST_NEON(sub_0, sub(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(sub_1, sub(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(sub_2, sub(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(sub_3, sub(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(sub_4, sub(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(sub_5, sub(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(sub_6, sub(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(sub_7, sub(d0, d1, d2))
TEST_NEON(suqadd_0, suqadd(v0.V8B(), v1.V8B()))
TEST_NEON(suqadd_1, suqadd(v0.V16B(), v1.V16B()))
TEST_NEON(suqadd_2, suqadd(v0.V4H(), v1.V4H()))
TEST_NEON(suqadd_3, suqadd(v0.V8H(), v1.V8H()))
TEST_NEON(suqadd_4, suqadd(v0.V2S(), v1.V2S()))
TEST_NEON(suqadd_5, suqadd(v0.V4S(), v1.V4S()))
TEST_NEON(suqadd_6, suqadd(v0.V2D(), v1.V2D()))
TEST_NEON(suqadd_7, suqadd(b0, b1))
TEST_NEON(suqadd_8, suqadd(h0, h1))
TEST_NEON(suqadd_9, suqadd(s0, s1))
TEST_NEON(suqadd_10, suqadd(d0, d1))
TEST_NEON(sxtl_0, sxtl(v0.V8H(), v1.V8B()))
TEST_NEON(sxtl_1, sxtl(v0.V4S(), v1.V4H()))
TEST_NEON(sxtl_2, sxtl(v0.V2D(), v1.V2S()))
TEST_NEON(sxtl2_0, sxtl2(v0.V8H(), v1.V16B()))
TEST_NEON(sxtl2_1, sxtl2(v0.V4S(), v1.V8H()))
TEST_NEON(sxtl2_2, sxtl2(v0.V2D(), v1.V4S()))
TEST_NEON(tbl_0, tbl(v0.V8B(), v1.V16B(), v2.V8B()))
TEST_NEON(tbl_1, tbl(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(tbl_2, tbl(v0.V8B(), v1.V16B(), v2.V16B(), v3.V8B()))
TEST_NEON(tbl_3, tbl(v0.V16B(), v1.V16B(), v2.V16B(), v3.V16B()))
TEST_NEON(tbl_4, tbl(v0.V8B(), v1.V16B(), v2.V16B(), v3.V16B(), v4.V8B()))
TEST_NEON(tbl_5, tbl(v0.V16B(), v1.V16B(), v2.V16B(), v3.V16B(), v4.V16B()))
TEST_NEON(tbl_6,
          tbl(v0.V8B(), v1.V16B(), v2.V16B(), v3.V16B(), v4.V16B(), v5.V8B()))
TEST_NEON(tbl_7,
          tbl(v0.V16B(), v1.V16B(), v2.V16B(), v3.V16B(), v4.V16B(), v5.V16B()))
TEST_NEON(tbx_0, tbx(v0.V8B(), v1.V16B(), v2.V8B()))
TEST_NEON(tbx_1, tbx(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(tbx_2, tbx(v0.V8B(), v1.V16B(), v2.V16B(), v3.V8B()))
TEST_NEON(tbx_3, tbx(v0.V16B(), v1.V16B(), v2.V16B(), v3.V16B()))
TEST_NEON(tbx_4, tbx(v0.V8B(), v1.V16B(), v2.V16B(), v3.V16B(), v4.V8B()))
TEST_NEON(tbx_5, tbx(v0.V16B(), v1.V16B(), v2.V16B(), v3.V16B(), v4.V16B()))
TEST_NEON(tbx_6,
          tbx(v0.V8B(), v1.V16B(), v2.V16B(), v3.V16B(), v4.V16B(), v5.V8B()))
TEST_NEON(tbx_7,
          tbx(v0.V16B(), v1.V16B(), v2.V16B(), v3.V16B(), v4.V16B(), v5.V16B()))
TEST_NEON(trn1_0, trn1(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(trn1_1, trn1(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(trn1_2, trn1(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(trn1_3, trn1(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(trn1_4, trn1(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(trn1_5, trn1(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(trn1_6, trn1(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(trn2_0, trn2(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(trn2_1, trn2(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(trn2_2, trn2(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(trn2_3, trn2(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(trn2_4, trn2(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(trn2_5, trn2(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(trn2_6, trn2(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(uabal_0, uabal(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(uabal_1, uabal(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(uabal_2, uabal(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(uabal2_0, uabal2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(uabal2_1, uabal2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(uabal2_2, uabal2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(uaba_0, uaba(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(uaba_1, uaba(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(uaba_2, uaba(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(uaba_3, uaba(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(uaba_4, uaba(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(uaba_5, uaba(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(uabdl_0, uabdl(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(uabdl_1, uabdl(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(uabdl_2, uabdl(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(uabdl2_0, uabdl2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(uabdl2_1, uabdl2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(uabdl2_2, uabdl2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(uabd_0, uabd(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(uabd_1, uabd(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(uabd_2, uabd(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(uabd_3, uabd(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(uabd_4, uabd(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(uabd_5, uabd(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(uadalp_0, uadalp(v0.V4H(), v1.V8B()))
TEST_NEON(uadalp_1, uadalp(v0.V8H(), v1.V16B()))
TEST_NEON(uadalp_2, uadalp(v0.V2S(), v1.V4H()))
TEST_NEON(uadalp_3, uadalp(v0.V4S(), v1.V8H()))
TEST_NEON(uadalp_4, uadalp(v0.V1D(), v1.V2S()))
TEST_NEON(uadalp_5, uadalp(v0.V2D(), v1.V4S()))
TEST_NEON(uaddlp_0, uaddlp(v0.V4H(), v1.V8B()))
TEST_NEON(uaddlp_1, uaddlp(v0.V8H(), v1.V16B()))
TEST_NEON(uaddlp_2, uaddlp(v0.V2S(), v1.V4H()))
TEST_NEON(uaddlp_3, uaddlp(v0.V4S(), v1.V8H()))
TEST_NEON(uaddlp_4, uaddlp(v0.V1D(), v1.V2S()))
TEST_NEON(uaddlp_5, uaddlp(v0.V2D(), v1.V4S()))
TEST_NEON(uaddlv_0, uaddlv(h0, v1.V8B()))
TEST_NEON(uaddlv_1, uaddlv(h0, v1.V16B()))
TEST_NEON(uaddlv_2, uaddlv(s0, v1.V4H()))
TEST_NEON(uaddlv_3, uaddlv(s0, v1.V8H()))
TEST_NEON(uaddlv_4, uaddlv(d0, v1.V4S()))
TEST_NEON(uaddl_0, uaddl(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(uaddl_1, uaddl(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(uaddl_2, uaddl(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(uaddl2_0, uaddl2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(uaddl2_1, uaddl2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(uaddl2_2, uaddl2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(uaddw_0, uaddw(v0.V8H(), v1.V8H(), v2.V8B()))
TEST_NEON(uaddw_1, uaddw(v0.V4S(), v1.V4S(), v2.V4H()))
TEST_NEON(uaddw_2, uaddw(v0.V2D(), v1.V2D(), v2.V2S()))
TEST_NEON(uaddw2_0, uaddw2(v0.V8H(), v1.V8H(), v2.V16B()))
TEST_NEON(uaddw2_1, uaddw2(v0.V4S(), v1.V4S(), v2.V8H()))
TEST_NEON(uaddw2_2, uaddw2(v0.V2D(), v1.V2D(), v2.V4S()))
TEST_NEON(uhadd_0, uhadd(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(uhadd_1, uhadd(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(uhadd_2, uhadd(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(uhadd_3, uhadd(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(uhadd_4, uhadd(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(uhadd_5, uhadd(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(uhsub_0, uhsub(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(uhsub_1, uhsub(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(uhsub_2, uhsub(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(uhsub_3, uhsub(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(uhsub_4, uhsub(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(uhsub_5, uhsub(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(umaxp_0, umaxp(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(umaxp_1, umaxp(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(umaxp_2, umaxp(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(umaxp_3, umaxp(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(umaxp_4, umaxp(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(umaxp_5, umaxp(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(umaxv_0, umaxv(b0, v1.V8B()))
TEST_NEON(umaxv_1, umaxv(b0, v1.V16B()))
TEST_NEON(umaxv_2, umaxv(h0, v1.V4H()))
TEST_NEON(umaxv_3, umaxv(h0, v1.V8H()))
TEST_NEON(umaxv_4, umaxv(s0, v1.V4S()))
TEST_NEON(umax_0, umax(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(umax_1, umax(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(umax_2, umax(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(umax_3, umax(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(umax_4, umax(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(umax_5, umax(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(uminp_0, uminp(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(uminp_1, uminp(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(uminp_2, uminp(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(uminp_3, uminp(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(uminp_4, uminp(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(uminp_5, uminp(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(uminv_0, uminv(b0, v1.V8B()))
TEST_NEON(uminv_1, uminv(b0, v1.V16B()))
TEST_NEON(uminv_2, uminv(h0, v1.V4H()))
TEST_NEON(uminv_3, uminv(h0, v1.V8H()))
TEST_NEON(uminv_4, uminv(s0, v1.V4S()))
TEST_NEON(umin_0, umin(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(umin_1, umin(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(umin_2, umin(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(umin_3, umin(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(umin_4, umin(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(umin_5, umin(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(umlal_0, umlal(v0.V4S(), v1.V4H(), v2.H(), 1))
TEST_NEON(umlal_1, umlal(v0.V2D(), v1.V2S(), v2.S(), 2))
TEST_NEON(umlal2_0, umlal2(v0.V4S(), v1.V8H(), v2.H(), 6))
TEST_NEON(umlal2_1, umlal2(v0.V2D(), v1.V4S(), v2.S(), 0))
TEST_NEON(umlal_2, umlal(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(umlal_3, umlal(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(umlal_4, umlal(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(umlal2_2, umlal2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(umlal2_3, umlal2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(umlal2_4, umlal2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(umlsl_0, umlsl(v0.V4S(), v1.V4H(), v2.H(), 0))
TEST_NEON(umlsl_1, umlsl(v0.V2D(), v1.V2S(), v2.S(), 3))
TEST_NEON(umlsl2_0, umlsl2(v0.V4S(), v1.V8H(), v2.H(), 1))
TEST_NEON(umlsl2_1, umlsl2(v0.V2D(), v1.V4S(), v2.S(), 1))
TEST_NEON(umlsl_2, umlsl(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(umlsl_3, umlsl(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(umlsl_4, umlsl(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(umlsl2_2, umlsl2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(umlsl2_3, umlsl2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(umlsl2_4, umlsl2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(umov_0, umov(w0, v1.B(), 4))
TEST_NEON(umov_1, umov(w0, v1.H(), 3))
TEST_NEON(umov_2, umov(w0, v1.S(), 0))
TEST_NEON(umov_3, umov(x0, v1.D(), 1))
TEST_NEON(umull_0, umull(v0.V4S(), v1.V4H(), v2.H(), 0))
TEST_NEON(umull_1, umull(v0.V2D(), v1.V2S(), v2.S(), 1))
TEST_NEON(umull2_0, umull2(v0.V4S(), v1.V8H(), v2.H(), 6))
TEST_NEON(umull2_1, umull2(v0.V2D(), v1.V4S(), v2.S(), 3))
TEST_NEON(umull_2, umull(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(umull_3, umull(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(umull_4, umull(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(umull2_2, umull2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(umull2_3, umull2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(umull2_4, umull2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(uqadd_0, uqadd(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(uqadd_1, uqadd(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(uqadd_2, uqadd(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(uqadd_3, uqadd(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(uqadd_4, uqadd(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(uqadd_5, uqadd(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(uqadd_6, uqadd(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(uqadd_7, uqadd(b0, b1, b2))
TEST_NEON(uqadd_8, uqadd(h0, h1, h2))
TEST_NEON(uqadd_9, uqadd(s0, s1, s2))
TEST_NEON(uqadd_10, uqadd(d0, d1, d2))
TEST_NEON(uqrshl_0, uqrshl(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(uqrshl_1, uqrshl(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(uqrshl_2, uqrshl(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(uqrshl_3, uqrshl(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(uqrshl_4, uqrshl(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(uqrshl_5, uqrshl(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(uqrshl_6, uqrshl(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(uqrshl_7, uqrshl(b0, b1, b2))
TEST_NEON(uqrshl_8, uqrshl(h0, h1, h2))
TEST_NEON(uqrshl_9, uqrshl(s0, s1, s2))
TEST_NEON(uqrshl_10, uqrshl(d0, d1, d2))
TEST_NEON(uqrshrn_0, uqrshrn(v0.V8B(), v1.V8H(), 5))
TEST_NEON(uqrshrn_1, uqrshrn(v0.V4H(), v1.V4S(), 4))
TEST_NEON(uqrshrn_2, uqrshrn(v0.V2S(), v1.V2D(), 23))
TEST_NEON(uqrshrn2_0, uqrshrn2(v0.V16B(), v1.V8H(), 4))
TEST_NEON(uqrshrn2_1, uqrshrn2(v0.V8H(), v1.V4S(), 5))
TEST_NEON(uqrshrn2_2, uqrshrn2(v0.V4S(), v1.V2D(), 11))
TEST_NEON(uqrshrn_3, uqrshrn(b0, h1, 4))
TEST_NEON(uqrshrn_4, uqrshrn(h0, s1, 4))
TEST_NEON(uqrshrn_5, uqrshrn(s0, d1, 7))
TEST_NEON(uqshl_0, uqshl(v0.V8B(), v1.V8B(), 7))
TEST_NEON(uqshl_1, uqshl(v0.V16B(), v1.V16B(), 2))
TEST_NEON(uqshl_2, uqshl(v0.V4H(), v1.V4H(), 4))
TEST_NEON(uqshl_3, uqshl(v0.V8H(), v1.V8H(), 4))
TEST_NEON(uqshl_4, uqshl(v0.V2S(), v1.V2S(), 1))
TEST_NEON(uqshl_5, uqshl(v0.V4S(), v1.V4S(), 2))
TEST_NEON(uqshl_6, uqshl(v0.V2D(), v1.V2D(), 28))
TEST_NEON(uqshl_7, uqshl(b0, b1, 6))
TEST_NEON(uqshl_8, uqshl(h0, h1, 15))
TEST_NEON(uqshl_9, uqshl(s0, s1, 21))
TEST_NEON(uqshl_10, uqshl(d0, d1, 24))
TEST_NEON(uqshl_11, uqshl(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(uqshl_12, uqshl(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(uqshl_13, uqshl(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(uqshl_14, uqshl(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(uqshl_15, uqshl(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(uqshl_16, uqshl(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(uqshl_17, uqshl(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(uqshl_18, uqshl(b0, b1, b2))
TEST_NEON(uqshl_19, uqshl(h0, h1, h2))
TEST_NEON(uqshl_20, uqshl(s0, s1, s2))
TEST_NEON(uqshl_21, uqshl(d0, d1, d2))
TEST_NEON(uqshrn_0, uqshrn(v0.V8B(), v1.V8H(), 6))
TEST_NEON(uqshrn_1, uqshrn(v0.V4H(), v1.V4S(), 1))
TEST_NEON(uqshrn_2, uqshrn(v0.V2S(), v1.V2D(), 7))
TEST_NEON(uqshrn2_0, uqshrn2(v0.V16B(), v1.V8H(), 3))
TEST_NEON(uqshrn2_1, uqshrn2(v0.V8H(), v1.V4S(), 9))
TEST_NEON(uqshrn2_2, uqshrn2(v0.V4S(), v1.V2D(), 20))
TEST_NEON(uqshrn_3, uqshrn(b0, h1, 7))
TEST_NEON(uqshrn_4, uqshrn(h0, s1, 11))
TEST_NEON(uqshrn_5, uqshrn(s0, d1, 17))
TEST_NEON(uqsub_0, uqsub(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(uqsub_1, uqsub(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(uqsub_2, uqsub(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(uqsub_3, uqsub(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(uqsub_4, uqsub(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(uqsub_5, uqsub(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(uqsub_6, uqsub(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(uqsub_7, uqsub(b0, b1, b2))
TEST_NEON(uqsub_8, uqsub(h0, h1, h2))
TEST_NEON(uqsub_9, uqsub(s0, s1, s2))
TEST_NEON(uqsub_10, uqsub(d0, d1, d2))
TEST_NEON(uqxtn_0, uqxtn(v0.V8B(), v1.V8H()))
TEST_NEON(uqxtn_1, uqxtn(v0.V4H(), v1.V4S()))
TEST_NEON(uqxtn_2, uqxtn(v0.V2S(), v1.V2D()))
TEST_NEON(uqxtn2_0, uqxtn2(v0.V16B(), v1.V8H()))
TEST_NEON(uqxtn2_1, uqxtn2(v0.V8H(), v1.V4S()))
TEST_NEON(uqxtn2_2, uqxtn2(v0.V4S(), v1.V2D()))
TEST_NEON(uqxtn_3, uqxtn(b0, h1))
TEST_NEON(uqxtn_4, uqxtn(h0, s1))
TEST_NEON(uqxtn_5, uqxtn(s0, d1))
TEST_NEON(urecpe_0, urecpe(v0.V2S(), v1.V2S()))
TEST_NEON(urecpe_1, urecpe(v0.V4S(), v1.V4S()))
TEST_NEON(urhadd_0, urhadd(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(urhadd_1, urhadd(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(urhadd_2, urhadd(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(urhadd_3, urhadd(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(urhadd_4, urhadd(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(urhadd_5, urhadd(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(urshl_0, urshl(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(urshl_1, urshl(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(urshl_2, urshl(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(urshl_3, urshl(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(urshl_4, urshl(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(urshl_5, urshl(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(urshl_6, urshl(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(urshl_7, urshl(d0, d1, d2))
TEST_NEON(urshr_0, urshr(v0.V8B(), v1.V8B(), 4))
TEST_NEON(urshr_1, urshr(v0.V16B(), v1.V16B(), 5))
TEST_NEON(urshr_2, urshr(v0.V4H(), v1.V4H(), 11))
TEST_NEON(urshr_3, urshr(v0.V8H(), v1.V8H(), 4))
TEST_NEON(urshr_4, urshr(v0.V2S(), v1.V2S(), 27))
TEST_NEON(urshr_5, urshr(v0.V4S(), v1.V4S(), 21))
TEST_NEON(urshr_6, urshr(v0.V2D(), v1.V2D(), 8))
TEST_NEON(urshr_7, urshr(d0, d1, 11))
TEST_NEON(ursqrte_0, ursqrte(v0.V2S(), v1.V2S()))
TEST_NEON(ursqrte_1, ursqrte(v0.V4S(), v1.V4S()))
TEST_NEON(ursra_0, ursra(v0.V8B(), v1.V8B(), 3))
TEST_NEON(ursra_1, ursra(v0.V16B(), v1.V16B(), 6))
TEST_NEON(ursra_2, ursra(v0.V4H(), v1.V4H(), 12))
TEST_NEON(ursra_3, ursra(v0.V8H(), v1.V8H(), 7))
TEST_NEON(ursra_4, ursra(v0.V2S(), v1.V2S(), 6))
TEST_NEON(ursra_5, ursra(v0.V4S(), v1.V4S(), 6))
TEST_NEON(ursra_6, ursra(v0.V2D(), v1.V2D(), 26))
TEST_NEON(ursra_7, ursra(d0, d1, 20))
TEST_NEON(ushll_0, ushll(v0.V8H(), v1.V8B(), 6))
TEST_NEON(ushll_1, ushll(v0.V4S(), v1.V4H(), 9))
TEST_NEON(ushll_2, ushll(v0.V2D(), v1.V2S(), 21))
TEST_NEON(ushll2_0, ushll2(v0.V8H(), v1.V16B(), 7))
TEST_NEON(ushll2_1, ushll2(v0.V4S(), v1.V8H(), 15))
TEST_NEON(ushll2_2, ushll2(v0.V2D(), v1.V4S(), 14))
TEST_NEON(ushl_0, ushl(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(ushl_1, ushl(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(ushl_2, ushl(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(ushl_3, ushl(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(ushl_4, ushl(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(ushl_5, ushl(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(ushl_6, ushl(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(ushl_7, ushl(d0, d1, d2))
TEST_NEON(ushr_0, ushr(v0.V8B(), v1.V8B(), 1))
TEST_NEON(ushr_1, ushr(v0.V16B(), v1.V16B(), 1))
TEST_NEON(ushr_2, ushr(v0.V4H(), v1.V4H(), 5))
TEST_NEON(ushr_3, ushr(v0.V8H(), v1.V8H(), 4))
TEST_NEON(ushr_4, ushr(v0.V2S(), v1.V2S(), 1))
TEST_NEON(ushr_5, ushr(v0.V4S(), v1.V4S(), 24))
TEST_NEON(ushr_6, ushr(v0.V2D(), v1.V2D(), 50))
TEST_NEON(ushr_7, ushr(d0, d1, 30))
TEST_NEON(usqadd_0, usqadd(v0.V8B(), v1.V8B()))
TEST_NEON(usqadd_1, usqadd(v0.V16B(), v1.V16B()))
TEST_NEON(usqadd_2, usqadd(v0.V4H(), v1.V4H()))
TEST_NEON(usqadd_3, usqadd(v0.V8H(), v1.V8H()))
TEST_NEON(usqadd_4, usqadd(v0.V2S(), v1.V2S()))
TEST_NEON(usqadd_5, usqadd(v0.V4S(), v1.V4S()))
TEST_NEON(usqadd_6, usqadd(v0.V2D(), v1.V2D()))
TEST_NEON(usqadd_7, usqadd(b0, b1))
TEST_NEON(usqadd_8, usqadd(h0, h1))
TEST_NEON(usqadd_9, usqadd(s0, s1))
TEST_NEON(usqadd_10, usqadd(d0, d1))
TEST_NEON(usra_0, usra(v0.V8B(), v1.V8B(), 6))
TEST_NEON(usra_1, usra(v0.V16B(), v1.V16B(), 4))
TEST_NEON(usra_2, usra(v0.V4H(), v1.V4H(), 9))
TEST_NEON(usra_3, usra(v0.V8H(), v1.V8H(), 3))
TEST_NEON(usra_4, usra(v0.V2S(), v1.V2S(), 12))
TEST_NEON(usra_5, usra(v0.V4S(), v1.V4S(), 14))
TEST_NEON(usra_6, usra(v0.V2D(), v1.V2D(), 27))
TEST_NEON(usra_7, usra(d0, d1, 54))
TEST_NEON(usubl_0, usubl(v0.V8H(), v1.V8B(), v2.V8B()))
TEST_NEON(usubl_1, usubl(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_NEON(usubl_2, usubl(v0.V2D(), v1.V2S(), v2.V2S()))
TEST_NEON(usubl2_0, usubl2(v0.V8H(), v1.V16B(), v2.V16B()))
TEST_NEON(usubl2_1, usubl2(v0.V4S(), v1.V8H(), v2.V8H()))
TEST_NEON(usubl2_2, usubl2(v0.V2D(), v1.V4S(), v2.V4S()))
TEST_NEON(usubw_0, usubw(v0.V8H(), v1.V8H(), v2.V8B()))
TEST_NEON(usubw_1, usubw(v0.V4S(), v1.V4S(), v2.V4H()))
TEST_NEON(usubw_2, usubw(v0.V2D(), v1.V2D(), v2.V2S()))
TEST_NEON(usubw2_0, usubw2(v0.V8H(), v1.V8H(), v2.V16B()))
TEST_NEON(usubw2_1, usubw2(v0.V4S(), v1.V4S(), v2.V8H()))
TEST_NEON(usubw2_2, usubw2(v0.V2D(), v1.V2D(), v2.V4S()))
TEST_NEON(uxtl_0, uxtl(v0.V8H(), v1.V8B()))
TEST_NEON(uxtl_1, uxtl(v0.V4S(), v1.V4H()))
TEST_NEON(uxtl_2, uxtl(v0.V2D(), v1.V2S()))
TEST_NEON(uxtl2_0, uxtl2(v0.V8H(), v1.V16B()))
TEST_NEON(uxtl2_1, uxtl2(v0.V4S(), v1.V8H()))
TEST_NEON(uxtl2_2, uxtl2(v0.V2D(), v1.V4S()))
TEST_NEON(uzp1_0, uzp1(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(uzp1_1, uzp1(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(uzp1_2, uzp1(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(uzp1_3, uzp1(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(uzp1_4, uzp1(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(uzp1_5, uzp1(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(uzp1_6, uzp1(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(uzp2_0, uzp2(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(uzp2_1, uzp2(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(uzp2_2, uzp2(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(uzp2_3, uzp2(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(uzp2_4, uzp2(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(uzp2_5, uzp2(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(uzp2_6, uzp2(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(xtn_0, xtn(v0.V8B(), v1.V8H()))
TEST_NEON(xtn_1, xtn(v0.V4H(), v1.V4S()))
TEST_NEON(xtn_2, xtn(v0.V2S(), v1.V2D()))
TEST_NEON(xtn2_0, xtn2(v0.V16B(), v1.V8H()))
TEST_NEON(xtn2_1, xtn2(v0.V8H(), v1.V4S()))
TEST_NEON(xtn2_2, xtn2(v0.V4S(), v1.V2D()))
TEST_NEON(zip1_0, zip1(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(zip1_1, zip1(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(zip1_2, zip1(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(zip1_3, zip1(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(zip1_4, zip1(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(zip1_5, zip1(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(zip1_6, zip1(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_NEON(zip2_0, zip2(v0.V8B(), v1.V8B(), v2.V8B()))
TEST_NEON(zip2_1, zip2(v0.V16B(), v1.V16B(), v2.V16B()))
TEST_NEON(zip2_2, zip2(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_NEON(zip2_3, zip2(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_NEON(zip2_4, zip2(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_NEON(zip2_5, zip2(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_NEON(zip2_6, zip2(v0.V2D(), v1.V2D(), v2.V2D()))

#define TEST_RCPC(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kRCpc), RCpc_##NAME, ASM)
TEST_RCPC(ldapr_0, ldapr(w0, MemOperand(x1, 0)))
TEST_RCPC(ldapr_1, ldapr(x0, MemOperand(x1, 0)))
TEST_RCPC(ldaprb_0, ldaprb(w0, MemOperand(x1, 0)))
TEST_RCPC(ldaprh_0, ldaprh(w0, MemOperand(x1, 0)))

#define TEST_CRC32(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kCRC32), CRC32_##NAME, ASM)
TEST_CRC32(crc32b_0, crc32b(w0, w1, w2))
TEST_CRC32(crc32h_0, crc32h(w0, w1, w2))
TEST_CRC32(crc32w_0, crc32w(w0, w1, w2))
TEST_CRC32(crc32x_0, crc32x(w0, w1, x2))
TEST_CRC32(crc32cb_0, crc32cb(w0, w1, w2))
TEST_CRC32(crc32ch_0, crc32ch(w0, w1, w2))
TEST_CRC32(crc32cw_0, crc32cw(w0, w1, w2))
TEST_CRC32(crc32cx_0, crc32cx(w0, w1, x2))

#define TEST_DCPOP(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kDCPoP), DCPoP_##NAME, ASM)
TEST_DCPOP(dc_0, dc(CVAP, x0))

#define TEST_FLAGM(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kFlagM), FlagM_##NAME, ASM)
TEST_FLAGM(cfinv_0, cfinv())
TEST_FLAGM(rmif_0, rmif(x0, 52, NVFlag))
TEST_FLAGM(setf16_0, setf16(w0))
TEST_FLAGM(setf8_0, setf8(w0))

#define TEST_PAUTH(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kPAuth), PAuth_##NAME, ASM)
TEST_PAUTH(autda_0, autda(x0, x1))
TEST_PAUTH(autdza_0, autdza(x0))
TEST_PAUTH(autdb_0, autdb(x0, x1))
TEST_PAUTH(autdzb_0, autdzb(x0))
TEST_PAUTH(autia1716_0, autia1716())
TEST_PAUTH(autiasp_0, autiasp())
TEST_PAUTH(autiaz_0, autiaz())
TEST_PAUTH(autia_0, autia(x0, x1))
TEST_PAUTH(autiza_0, autiza(x0))
TEST_PAUTH(autib1716_0, autib1716())
TEST_PAUTH(autibsp_0, autibsp())
TEST_PAUTH(autibz_0, autibz())
TEST_PAUTH(autib_0, autib(x0, x1))
TEST_PAUTH(autizb_0, autizb(x0))
TEST_PAUTH(blraaz_0, blraaz(x0))
TEST_PAUTH(blraa_0, blraa(x0, x1))
TEST_PAUTH(blrabz_0, blrabz(x0))
TEST_PAUTH(blrab_0, blrab(x0, x1))
TEST_PAUTH(braaz_0, braaz(x0))
TEST_PAUTH(braa_0, braa(x0, x1))
TEST_PAUTH(brabz_0, brabz(x0))
TEST_PAUTH(brab_0, brab(x0, x1))
TEST_PAUTH(ldraa_0, ldraa(x0, MemOperand(x1, -48, PreIndex)))
TEST_PAUTH(ldraa_1, ldraa(x0, MemOperand(x1, -1808)))
TEST_PAUTH(ldrab_0, ldrab(x0, MemOperand(x1, 1544, PreIndex)))
TEST_PAUTH(ldrab_1, ldrab(x0, MemOperand(x1, 1512)))
TEST_PAUTH(pacda_0, pacda(x0, x1))
TEST_PAUTH(pacdza_0, pacdza(x0))
TEST_PAUTH(pacdb_0, pacdb(x0, x1))
TEST_PAUTH(pacdzb_0, pacdzb(x0))
TEST_PAUTH(pacia1716_0, pacia1716())
TEST_PAUTH(paciasp_0, paciasp())
TEST_PAUTH(paciaz_0, paciaz())
TEST_PAUTH(pacia_0, pacia(x0, x1))
TEST_PAUTH(paciza_0, paciza(x0))
TEST_PAUTH(pacib1716_0, pacib1716())
TEST_PAUTH(pacibsp_0, pacibsp())
TEST_PAUTH(pacibz_0, pacibz())
TEST_PAUTH(pacib_0, pacib(x0, x1))
TEST_PAUTH(pacizb_0, pacizb(x0))
TEST_PAUTH(retaa_0, retaa())
TEST_PAUTH(retab_0, retab())
TEST_PAUTH(xpacd_0, xpacd(x0))
TEST_PAUTH(xpaci_0, xpaci(x0))
TEST_PAUTH(xpaclri_0, xpaclri())

#define TEST_AXFLAG(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kAXFlag), AXFlag_##NAME, ASM)
TEST_AXFLAG(axflag_0, axflag())
TEST_AXFLAG(xaflag_0, xaflag())

#define TEST_ATOMICS(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kAtomics), Atomics_##NAME, ASM)
TEST_ATOMICS(casal_0, casal(w0, w1, MemOperand(x2, 0)))
TEST_ATOMICS(casal_1, casal(x0, x1, MemOperand(x2, 0)))
TEST_ATOMICS(casa_0, casa(w0, w1, MemOperand(x2, 0)))
TEST_ATOMICS(casa_1, casa(x0, x1, MemOperand(x2, 0)))
TEST_ATOMICS(casl_0, casl(w0, w1, MemOperand(x2, 0)))
TEST_ATOMICS(casl_1, casl(x0, x1, MemOperand(x2, 0)))
TEST_ATOMICS(cas_0, cas(w0, w1, MemOperand(x2, 0)))
TEST_ATOMICS(cas_1, cas(x0, x1, MemOperand(x2, 0)))
TEST_ATOMICS(casab_0, casab(w0, w1, MemOperand(x2, 0)))
TEST_ATOMICS(casalb_0, casalb(w0, w1, MemOperand(x2, 0)))
TEST_ATOMICS(casb_0, casb(w0, w1, MemOperand(x2, 0)))
TEST_ATOMICS(caslb_0, caslb(w0, w1, MemOperand(x2, 0)))
TEST_ATOMICS(casah_0, casah(w0, w1, MemOperand(x2, 0)))
TEST_ATOMICS(casalh_0, casalh(w0, w1, MemOperand(x2, 0)))
TEST_ATOMICS(cash_0, cash(w0, w1, MemOperand(x2, 0)))
TEST_ATOMICS(caslh_0, caslh(w0, w1, MemOperand(x2, 0)))
TEST_ATOMICS(caspal_0, caspal(w0, w1, w2, w3, MemOperand(x4, 0)))
TEST_ATOMICS(caspal_1, caspal(x0, x1, x2, x3, MemOperand(x4, 0)))
TEST_ATOMICS(caspa_0, caspa(w0, w1, w2, w3, MemOperand(x4, 0)))
TEST_ATOMICS(caspa_1, caspa(x0, x1, x2, x3, MemOperand(x4, 0)))
TEST_ATOMICS(caspl_0, caspl(w0, w1, w2, w3, MemOperand(x4, 0)))
TEST_ATOMICS(caspl_1, caspl(x0, x1, x2, x3, MemOperand(x4, 0)))
TEST_ATOMICS(casp_0, casp(w0, w1, w2, w3, MemOperand(x4, 0)))
TEST_ATOMICS(casp_1, casp(x0, x1, x2, x3, MemOperand(x4, 0)))
TEST_ATOMICS(ldaddal_0, ldaddal(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldaddal_1, ldaddal(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldadda_0, ldadda(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldadda_1, ldadda(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldaddl_0, ldaddl(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldaddl_1, ldaddl(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldadd_0, ldadd(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldadd_1, ldadd(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldaddab_0, ldaddab(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldaddalb_0, ldaddalb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldaddb_0, ldaddb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldaddlb_0, ldaddlb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldaddah_0, ldaddah(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldaddalh_0, ldaddalh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldaddh_0, ldaddh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldaddlh_0, ldaddlh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldclral_0, ldclral(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldclral_1, ldclral(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldclra_0, ldclra(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldclra_1, ldclra(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldclrl_0, ldclrl(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldclrl_1, ldclrl(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldclr_0, ldclr(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldclr_1, ldclr(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldclrab_0, ldclrab(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldclralb_0, ldclralb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldclrb_0, ldclrb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldclrlb_0, ldclrlb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldclrah_0, ldclrah(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldclralh_0, ldclralh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldclrh_0, ldclrh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldclrlh_0, ldclrlh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldeoral_0, ldeoral(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldeoral_1, ldeoral(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldeora_0, ldeora(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldeora_1, ldeora(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldeorl_0, ldeorl(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldeorl_1, ldeorl(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldeor_0, ldeor(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldeor_1, ldeor(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldeorab_0, ldeorab(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldeoralb_0, ldeoralb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldeorb_0, ldeorb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldeorlb_0, ldeorlb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldeorah_0, ldeorah(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldeoralh_0, ldeoralh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldeorh_0, ldeorh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldeorlh_0, ldeorlh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsetal_0, ldsetal(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsetal_1, ldsetal(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldseta_0, ldseta(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldseta_1, ldseta(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldsetl_0, ldsetl(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsetl_1, ldsetl(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldset_0, ldset(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldset_1, ldset(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldsetab_0, ldsetab(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsetalb_0, ldsetalb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsetb_0, ldsetb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsetlb_0, ldsetlb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsetah_0, ldsetah(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsetalh_0, ldsetalh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldseth_0, ldseth(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsetlh_0, ldsetlh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxal_0, ldsmaxal(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxal_1, ldsmaxal(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxa_0, ldsmaxa(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxa_1, ldsmaxa(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxl_0, ldsmaxl(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxl_1, ldsmaxl(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldsmax_0, ldsmax(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmax_1, ldsmax(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxab_0, ldsmaxab(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxalb_0, ldsmaxalb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxb_0, ldsmaxb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxlb_0, ldsmaxlb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxah_0, ldsmaxah(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxalh_0, ldsmaxalh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxh_0, ldsmaxh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmaxlh_0, ldsmaxlh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsminal_0, ldsminal(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsminal_1, ldsminal(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldsmina_0, ldsmina(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmina_1, ldsmina(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldsminl_0, ldsminl(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsminl_1, ldsminl(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldsmin_0, ldsmin(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsmin_1, ldsmin(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldsminab_0, ldsminab(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsminalb_0, ldsminalb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsminb_0, ldsminb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsminlb_0, ldsminlb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsminah_0, ldsminah(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsminalh_0, ldsminalh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsminh_0, ldsminh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldsminlh_0, ldsminlh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumaxal_0, ldumaxal(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumaxal_1, ldumaxal(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldumaxa_0, ldumaxa(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumaxa_1, ldumaxa(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldumaxl_0, ldumaxl(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumaxl_1, ldumaxl(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldumax_0, ldumax(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumax_1, ldumax(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldumaxab_0, ldumaxab(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumaxalb_0, ldumaxalb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumaxb_0, ldumaxb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumaxlb_0, ldumaxlb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumaxah_0, ldumaxah(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumaxalh_0, ldumaxalh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumaxh_0, ldumaxh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumaxlh_0, ldumaxlh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(lduminal_0, lduminal(w0, w1, MemOperand(x2)))
TEST_ATOMICS(lduminal_1, lduminal(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldumina_0, ldumina(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumina_1, ldumina(x0, x1, MemOperand(x2)))
TEST_ATOMICS(lduminl_0, lduminl(w0, w1, MemOperand(x2)))
TEST_ATOMICS(lduminl_1, lduminl(x0, x1, MemOperand(x2)))
TEST_ATOMICS(ldumin_0, ldumin(w0, w1, MemOperand(x2)))
TEST_ATOMICS(ldumin_1, ldumin(x0, x1, MemOperand(x2)))
TEST_ATOMICS(lduminab_0, lduminab(w0, w1, MemOperand(x2)))
TEST_ATOMICS(lduminalb_0, lduminalb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(lduminb_0, lduminb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(lduminlb_0, lduminlb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(lduminah_0, lduminah(w0, w1, MemOperand(x2)))
TEST_ATOMICS(lduminalh_0, lduminalh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(lduminh_0, lduminh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(lduminlh_0, lduminlh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(staddb_0, staddb(w0, MemOperand(x1)))
TEST_ATOMICS(staddlb_0, staddlb(w0, MemOperand(x1)))
TEST_ATOMICS(staddh_0, staddh(w0, MemOperand(x1)))
TEST_ATOMICS(staddlh_0, staddlh(w0, MemOperand(x1)))
TEST_ATOMICS(staddl_0, staddl(w0, MemOperand(x1)))
TEST_ATOMICS(staddl_1, staddl(x0, MemOperand(x1)))
TEST_ATOMICS(stadd_0, stadd(w0, MemOperand(x1)))
TEST_ATOMICS(stadd_1, stadd(x0, MemOperand(x1)))
TEST_ATOMICS(stclrb_0, stclrb(w0, MemOperand(x1)))
TEST_ATOMICS(stclrlb_0, stclrlb(w0, MemOperand(x1)))
TEST_ATOMICS(stclrh_0, stclrh(w0, MemOperand(x1)))
TEST_ATOMICS(stclrlh_0, stclrlh(w0, MemOperand(x1)))
TEST_ATOMICS(stclrl_0, stclrl(w0, MemOperand(x1)))
TEST_ATOMICS(stclrl_1, stclrl(x0, MemOperand(x1)))
TEST_ATOMICS(stclr_0, stclr(w0, MemOperand(x1)))
TEST_ATOMICS(stclr_1, stclr(x0, MemOperand(x1)))
TEST_ATOMICS(steorb_0, steorb(w0, MemOperand(x1)))
TEST_ATOMICS(steorlb_0, steorlb(w0, MemOperand(x1)))
TEST_ATOMICS(steorh_0, steorh(w0, MemOperand(x1)))
TEST_ATOMICS(steorlh_0, steorlh(w0, MemOperand(x1)))
TEST_ATOMICS(steorl_0, steorl(w0, MemOperand(x1)))
TEST_ATOMICS(steorl_1, steorl(x0, MemOperand(x1)))
TEST_ATOMICS(steor_0, steor(w0, MemOperand(x1)))
TEST_ATOMICS(steor_1, steor(x0, MemOperand(x1)))
TEST_ATOMICS(stsetb_0, stsetb(w0, MemOperand(x1)))
TEST_ATOMICS(stsetlb_0, stsetlb(w0, MemOperand(x1)))
TEST_ATOMICS(stseth_0, stseth(w0, MemOperand(x1)))
TEST_ATOMICS(stsetlh_0, stsetlh(w0, MemOperand(x1)))
TEST_ATOMICS(stsetl_0, stsetl(w0, MemOperand(x1)))
TEST_ATOMICS(stsetl_1, stsetl(x0, MemOperand(x1)))
TEST_ATOMICS(stset_0, stset(w0, MemOperand(x1)))
TEST_ATOMICS(stset_1, stset(x0, MemOperand(x1)))
TEST_ATOMICS(stsmaxb_0, stsmaxb(w0, MemOperand(x1)))
TEST_ATOMICS(stsmaxlb_0, stsmaxlb(w0, MemOperand(x1)))
TEST_ATOMICS(stsmaxh_0, stsmaxh(w0, MemOperand(x1)))
TEST_ATOMICS(stsmaxlh_0, stsmaxlh(w0, MemOperand(x1)))
TEST_ATOMICS(stsmaxl_0, stsmaxl(w0, MemOperand(x1)))
TEST_ATOMICS(stsmaxl_1, stsmaxl(x0, MemOperand(x1)))
TEST_ATOMICS(stsmax_0, stsmax(w0, MemOperand(x1)))
TEST_ATOMICS(stsmax_1, stsmax(x0, MemOperand(x1)))
TEST_ATOMICS(stsminb_0, stsminb(w0, MemOperand(x1)))
TEST_ATOMICS(stsminlb_0, stsminlb(w0, MemOperand(x1)))
TEST_ATOMICS(stsminh_0, stsminh(w0, MemOperand(x1)))
TEST_ATOMICS(stsminlh_0, stsminlh(w0, MemOperand(x1)))
TEST_ATOMICS(stsminl_0, stsminl(w0, MemOperand(x1)))
TEST_ATOMICS(stsminl_1, stsminl(x0, MemOperand(x1)))
TEST_ATOMICS(stsmin_0, stsmin(w0, MemOperand(x1)))
TEST_ATOMICS(stsmin_1, stsmin(x0, MemOperand(x1)))
TEST_ATOMICS(stumaxb_0, stumaxb(w0, MemOperand(x1)))
TEST_ATOMICS(stumaxlb_0, stumaxlb(w0, MemOperand(x1)))
TEST_ATOMICS(stumaxh_0, stumaxh(w0, MemOperand(x1)))
TEST_ATOMICS(stumaxlh_0, stumaxlh(w0, MemOperand(x1)))
TEST_ATOMICS(stumaxl_0, stumaxl(w0, MemOperand(x1)))
TEST_ATOMICS(stumaxl_1, stumaxl(x0, MemOperand(x1)))
TEST_ATOMICS(stumax_0, stumax(w0, MemOperand(x1)))
TEST_ATOMICS(stumax_1, stumax(x0, MemOperand(x1)))
TEST_ATOMICS(stuminb_0, stuminb(w0, MemOperand(x1)))
TEST_ATOMICS(stuminlb_0, stuminlb(w0, MemOperand(x1)))
TEST_ATOMICS(stuminh_0, stuminh(w0, MemOperand(x1)))
TEST_ATOMICS(stuminlh_0, stuminlh(w0, MemOperand(x1)))
TEST_ATOMICS(stuminl_0, stuminl(w0, MemOperand(x1)))
TEST_ATOMICS(stuminl_1, stuminl(x0, MemOperand(x1)))
TEST_ATOMICS(stumin_0, stumin(w0, MemOperand(x1)))
TEST_ATOMICS(stumin_1, stumin(x0, MemOperand(x1)))
TEST_ATOMICS(swpal_0, swpal(w0, w1, MemOperand(x2)))
TEST_ATOMICS(swpal_1, swpal(x0, x1, MemOperand(x2)))
TEST_ATOMICS(swpa_0, swpa(w0, w1, MemOperand(x2)))
TEST_ATOMICS(swpa_1, swpa(x0, x1, MemOperand(x2)))
TEST_ATOMICS(swpl_0, swpl(w0, w1, MemOperand(x2)))
TEST_ATOMICS(swpl_1, swpl(x0, x1, MemOperand(x2)))
TEST_ATOMICS(swp_0, swp(w0, w1, MemOperand(x2)))
TEST_ATOMICS(swp_1, swp(x0, x1, MemOperand(x2)))
TEST_ATOMICS(swpab_0, swpab(w0, w1, MemOperand(x2)))
TEST_ATOMICS(swpalb_0, swpalb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(swpb_0, swpb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(swplb_0, swplb(w0, w1, MemOperand(x2)))
TEST_ATOMICS(swpah_0, swpah(w0, w1, MemOperand(x2)))
TEST_ATOMICS(swpalh_0, swpalh(w0, w1, MemOperand(x2)))
TEST_ATOMICS(swph_0, swph(w0, w1, MemOperand(x2)))
TEST_ATOMICS(swplh_0, swplh(w0, w1, MemOperand(x2)))

#define TEST_FP_NEON(NAME, ASM)                                    \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kFP, CPUFeatures::kNEON), \
                FP_NEON_##NAME,                                    \
                ASM)
TEST_FP_NEON(fabd_0, fabd(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fabd_1, fabd(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fabd_2, fabd(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fabd_3, fabd(s0, s1, s2))
TEST_FP_NEON(fabd_4, fabd(d0, d1, d2))
TEST_FP_NEON(fabs_0, fabs(v0.V2S(), v1.V2S()))
TEST_FP_NEON(fabs_1, fabs(v0.V4S(), v1.V4S()))
TEST_FP_NEON(fabs_2, fabs(v0.V2D(), v1.V2D()))
TEST_FP_NEON(facge_0, facge(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(facge_1, facge(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(facge_2, facge(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(facge_3, facge(s0, s1, s2))
TEST_FP_NEON(facge_4, facge(d0, d1, d2))
TEST_FP_NEON(facgt_0, facgt(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(facgt_1, facgt(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(facgt_2, facgt(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(facgt_3, facgt(s0, s1, s2))
TEST_FP_NEON(facgt_4, facgt(d0, d1, d2))
TEST_FP_NEON(faddp_0, faddp(s0, v1.V2S()))
TEST_FP_NEON(faddp_1, faddp(d0, v1.V2D()))
TEST_FP_NEON(faddp_2, faddp(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(faddp_3, faddp(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(faddp_4, faddp(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fadd_0, fadd(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fadd_1, fadd(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fadd_2, fadd(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fcmeq_0, fcmeq(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fcmeq_1, fcmeq(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fcmeq_2, fcmeq(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fcmeq_3, fcmeq(s0, s1, s2))
TEST_FP_NEON(fcmeq_4, fcmeq(d0, d1, d2))
TEST_FP_NEON(fcmeq_5, fcmeq(v0.V2S(), v1.V2S(), 0.0))
TEST_FP_NEON(fcmeq_6, fcmeq(v0.V4S(), v1.V4S(), 0.0))
TEST_FP_NEON(fcmeq_7, fcmeq(v0.V2D(), v1.V2D(), 0.0))
TEST_FP_NEON(fcmeq_8, fcmeq(s0, s1, 0.0))
TEST_FP_NEON(fcmeq_9, fcmeq(d0, d1, 0.0))
TEST_FP_NEON(fcmge_0, fcmge(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fcmge_1, fcmge(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fcmge_2, fcmge(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fcmge_3, fcmge(s0, s1, s2))
TEST_FP_NEON(fcmge_4, fcmge(d0, d1, d2))
TEST_FP_NEON(fcmge_5, fcmge(v0.V2S(), v1.V2S(), 0.0))
TEST_FP_NEON(fcmge_6, fcmge(v0.V4S(), v1.V4S(), 0.0))
TEST_FP_NEON(fcmge_7, fcmge(v0.V2D(), v1.V2D(), 0.0))
TEST_FP_NEON(fcmge_8, fcmge(s0, s1, 0.0))
TEST_FP_NEON(fcmge_9, fcmge(d0, d1, 0.0))
TEST_FP_NEON(fcmgt_0, fcmgt(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fcmgt_1, fcmgt(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fcmgt_2, fcmgt(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fcmgt_3, fcmgt(s0, s1, s2))
TEST_FP_NEON(fcmgt_4, fcmgt(d0, d1, d2))
TEST_FP_NEON(fcmgt_5, fcmgt(v0.V2S(), v1.V2S(), 0.0))
TEST_FP_NEON(fcmgt_6, fcmgt(v0.V4S(), v1.V4S(), 0.0))
TEST_FP_NEON(fcmgt_7, fcmgt(v0.V2D(), v1.V2D(), 0.0))
TEST_FP_NEON(fcmgt_8, fcmgt(s0, s1, 0.0))
TEST_FP_NEON(fcmgt_9, fcmgt(d0, d1, 0.0))
TEST_FP_NEON(fcmle_0, fcmle(v0.V2S(), v1.V2S(), 0.0))
TEST_FP_NEON(fcmle_1, fcmle(v0.V4S(), v1.V4S(), 0.0))
TEST_FP_NEON(fcmle_2, fcmle(v0.V2D(), v1.V2D(), 0.0))
TEST_FP_NEON(fcmle_3, fcmle(s0, s1, 0.0))
TEST_FP_NEON(fcmle_4, fcmle(d0, d1, 0.0))
TEST_FP_NEON(fcmlt_0, fcmlt(v0.V2S(), v1.V2S(), 0.0))
TEST_FP_NEON(fcmlt_1, fcmlt(v0.V4S(), v1.V4S(), 0.0))
TEST_FP_NEON(fcmlt_2, fcmlt(v0.V2D(), v1.V2D(), 0.0))
TEST_FP_NEON(fcmlt_3, fcmlt(s0, s1, 0.0))
TEST_FP_NEON(fcmlt_4, fcmlt(d0, d1, 0.0))
TEST_FP_NEON(fcvtas_0, fcvtas(v0.V2S(), v1.V2S()))
TEST_FP_NEON(fcvtas_1, fcvtas(v0.V4S(), v1.V4S()))
TEST_FP_NEON(fcvtas_2, fcvtas(v0.V2D(), v1.V2D()))
TEST_FP_NEON(fcvtas_3, fcvtas(s0, s1))
TEST_FP_NEON(fcvtas_4, fcvtas(d0, d1))
TEST_FP_NEON(fcvtau_0, fcvtau(v0.V2S(), v1.V2S()))
TEST_FP_NEON(fcvtau_1, fcvtau(v0.V4S(), v1.V4S()))
TEST_FP_NEON(fcvtau_2, fcvtau(v0.V2D(), v1.V2D()))
TEST_FP_NEON(fcvtau_3, fcvtau(s0, s1))
TEST_FP_NEON(fcvtau_4, fcvtau(d0, d1))
TEST_FP_NEON(fcvtl_0, fcvtl(v0.V4S(), v1.V4H()))
TEST_FP_NEON(fcvtl_1, fcvtl(v0.V2D(), v1.V2S()))
TEST_FP_NEON(fcvtl2_0, fcvtl2(v0.V4S(), v1.V8H()))
TEST_FP_NEON(fcvtl2_1, fcvtl2(v0.V2D(), v1.V4S()))
TEST_FP_NEON(fcvtms_0, fcvtms(v0.V2S(), v1.V2S()))
TEST_FP_NEON(fcvtms_1, fcvtms(v0.V4S(), v1.V4S()))
TEST_FP_NEON(fcvtms_2, fcvtms(v0.V2D(), v1.V2D()))
TEST_FP_NEON(fcvtms_3, fcvtms(s0, s1))
TEST_FP_NEON(fcvtms_4, fcvtms(d0, d1))
TEST_FP_NEON(fcvtmu_0, fcvtmu(v0.V2S(), v1.V2S()))
TEST_FP_NEON(fcvtmu_1, fcvtmu(v0.V4S(), v1.V4S()))
TEST_FP_NEON(fcvtmu_2, fcvtmu(v0.V2D(), v1.V2D()))
TEST_FP_NEON(fcvtmu_3, fcvtmu(s0, s1))
TEST_FP_NEON(fcvtmu_4, fcvtmu(d0, d1))
TEST_FP_NEON(fcvtns_0, fcvtns(v0.V2S(), v1.V2S()))
TEST_FP_NEON(fcvtns_1, fcvtns(v0.V4S(), v1.V4S()))
TEST_FP_NEON(fcvtns_2, fcvtns(v0.V2D(), v1.V2D()))
TEST_FP_NEON(fcvtns_3, fcvtns(s0, s1))
TEST_FP_NEON(fcvtns_4, fcvtns(d0, d1))
TEST_FP_NEON(fcvtnu_0, fcvtnu(v0.V2S(), v1.V2S()))
TEST_FP_NEON(fcvtnu_1, fcvtnu(v0.V4S(), v1.V4S()))
TEST_FP_NEON(fcvtnu_2, fcvtnu(v0.V2D(), v1.V2D()))
TEST_FP_NEON(fcvtnu_3, fcvtnu(s0, s1))
TEST_FP_NEON(fcvtnu_4, fcvtnu(d0, d1))
TEST_FP_NEON(fcvtn_0, fcvtn(v0.V4H(), v1.V4S()))
TEST_FP_NEON(fcvtn_1, fcvtn(v0.V2S(), v1.V2D()))
TEST_FP_NEON(fcvtn2_0, fcvtn2(v0.V8H(), v1.V4S()))
TEST_FP_NEON(fcvtn2_1, fcvtn2(v0.V4S(), v1.V2D()))
TEST_FP_NEON(fcvtps_0, fcvtps(v0.V2S(), v1.V2S()))
TEST_FP_NEON(fcvtps_1, fcvtps(v0.V4S(), v1.V4S()))
TEST_FP_NEON(fcvtps_2, fcvtps(v0.V2D(), v1.V2D()))
TEST_FP_NEON(fcvtps_3, fcvtps(s0, s1))
TEST_FP_NEON(fcvtps_4, fcvtps(d0, d1))
TEST_FP_NEON(fcvtpu_0, fcvtpu(v0.V2S(), v1.V2S()))
TEST_FP_NEON(fcvtpu_1, fcvtpu(v0.V4S(), v1.V4S()))
TEST_FP_NEON(fcvtpu_2, fcvtpu(v0.V2D(), v1.V2D()))
TEST_FP_NEON(fcvtpu_3, fcvtpu(s0, s1))
TEST_FP_NEON(fcvtpu_4, fcvtpu(d0, d1))
TEST_FP_NEON(fcvtxn_0, fcvtxn(v0.V2S(), v1.V2D()))
TEST_FP_NEON(fcvtxn2_0, fcvtxn2(v0.V4S(), v1.V2D()))
TEST_FP_NEON(fcvtxn_1, fcvtxn(s0, d1))
TEST_FP_NEON(fcvtzs_0, fcvtzs(v0.V2S(), v1.V2S(), 5))
TEST_FP_NEON(fcvtzs_1, fcvtzs(v0.V4S(), v1.V4S(), 5))
TEST_FP_NEON(fcvtzs_2, fcvtzs(v0.V2D(), v1.V2D(), 5))
TEST_FP_NEON(fcvtzs_3, fcvtzs(s0, s1, 5))
TEST_FP_NEON(fcvtzs_4, fcvtzs(d0, d1, 5))
TEST_FP_NEON(fcvtzs_5, fcvtzs(v0.V2S(), v1.V2S()))
TEST_FP_NEON(fcvtzs_6, fcvtzs(v0.V4S(), v1.V4S()))
TEST_FP_NEON(fcvtzs_7, fcvtzs(v0.V2D(), v1.V2D()))
TEST_FP_NEON(fcvtzs_8, fcvtzs(s0, s1))
TEST_FP_NEON(fcvtzs_9, fcvtzs(d0, d1))
TEST_FP_NEON(fcvtzu_0, fcvtzu(v0.V2S(), v1.V2S(), 5))
TEST_FP_NEON(fcvtzu_1, fcvtzu(v0.V4S(), v1.V4S(), 5))
TEST_FP_NEON(fcvtzu_2, fcvtzu(v0.V2D(), v1.V2D(), 5))
TEST_FP_NEON(fcvtzu_3, fcvtzu(s0, s1, 5))
TEST_FP_NEON(fcvtzu_4, fcvtzu(d0, d1, 5))
TEST_FP_NEON(fcvtzu_5, fcvtzu(v0.V2S(), v1.V2S()))
TEST_FP_NEON(fcvtzu_6, fcvtzu(v0.V4S(), v1.V4S()))
TEST_FP_NEON(fcvtzu_7, fcvtzu(v0.V2D(), v1.V2D()))
TEST_FP_NEON(fcvtzu_8, fcvtzu(s0, s1))
TEST_FP_NEON(fcvtzu_9, fcvtzu(d0, d1))
TEST_FP_NEON(fdiv_0, fdiv(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fdiv_1, fdiv(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fdiv_2, fdiv(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fmaxnmp_0, fmaxnmp(s0, v1.V2S()))
TEST_FP_NEON(fmaxnmp_1, fmaxnmp(d0, v1.V2D()))
TEST_FP_NEON(fmaxnmp_2, fmaxnmp(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fmaxnmp_3, fmaxnmp(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fmaxnmp_4, fmaxnmp(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fmaxnmv_0, fmaxnmv(s0, v1.V4S()))
TEST_FP_NEON(fmaxnm_0, fmaxnm(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fmaxnm_1, fmaxnm(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fmaxnm_2, fmaxnm(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fmaxp_0, fmaxp(s0, v1.V2S()))
TEST_FP_NEON(fmaxp_1, fmaxp(d0, v1.V2D()))
TEST_FP_NEON(fmaxp_2, fmaxp(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fmaxp_3, fmaxp(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fmaxp_4, fmaxp(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fmaxv_0, fmaxv(s0, v1.V4S()))
TEST_FP_NEON(fmax_0, fmax(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fmax_1, fmax(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fmax_2, fmax(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fminnmp_0, fminnmp(s0, v1.V2S()))
TEST_FP_NEON(fminnmp_1, fminnmp(d0, v1.V2D()))
TEST_FP_NEON(fminnmp_2, fminnmp(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fminnmp_3, fminnmp(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fminnmp_4, fminnmp(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fminnmv_0, fminnmv(s0, v1.V4S()))
TEST_FP_NEON(fminnm_0, fminnm(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fminnm_1, fminnm(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fminnm_2, fminnm(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fminp_0, fminp(s0, v1.V2S()))
TEST_FP_NEON(fminp_1, fminp(d0, v1.V2D()))
TEST_FP_NEON(fminp_2, fminp(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fminp_3, fminp(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fminp_4, fminp(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fminv_0, fminv(s0, v1.V4S()))
TEST_FP_NEON(fmin_0, fmin(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fmin_1, fmin(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fmin_2, fmin(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(fmla_0, fmla(v0.V2S(), v1.V2S(), v2.S(), 3))
TEST_FP_NEON(fmla_1, fmla(v0.V4S(), v1.V4S(), v2.S(), 2))
TEST_FP_NEON(fmla_2, fmla(v0.V2D(), v1.V2D(), v2.D(), 0))
TEST_FP_NEON(fmla_3, fmla(s0, s1, v2.S(), 1))
TEST_FP_NEON(fmla_4, fmla(d0, d1, v2.D(), 0))
TEST_FP_NEON(fmla_5, fmla(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fmla_6, fmla(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fmls_0, fmls(v0.V2S(), v1.V2S(), v2.S(), 3))
TEST_FP_NEON(fmls_1, fmls(v0.V4S(), v1.V4S(), v2.S(), 1))
TEST_FP_NEON(fmls_2, fmls(v0.V2D(), v1.V2D(), v2.D(), 1))
TEST_FP_NEON(fmls_3, fmls(s0, s1, v2.S(), 3))
TEST_FP_NEON(fmls_4, fmls(d0, d1, v2.D(), 0))
TEST_FP_NEON(fmls_5, fmls(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fmls_6, fmls(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fmov_0, fmov(v0.V2D(), -0.96875))
TEST_FP_NEON(fmov_1, fmov(v0.V2S(), 3.875f))
TEST_FP_NEON(fmov_2, fmov(v0.V4S(), 0.1328125f))
TEST_FP_NEON(fmov_3, fmov(x0, v1.D(), 1))
TEST_FP_NEON(fmov_4, fmov(v0.D(), 1, x1))
TEST_FP_NEON(fmulx_0, fmulx(v0.V2S(), v1.V2S(), v2.S(), 1))
TEST_FP_NEON(fmulx_1, fmulx(v0.V4S(), v1.V4S(), v2.S(), 2))
TEST_FP_NEON(fmulx_2, fmulx(v0.V2D(), v1.V2D(), v2.D(), 0))
TEST_FP_NEON(fmulx_3, fmulx(s0, s1, v2.S(), 2))
TEST_FP_NEON(fmulx_4, fmulx(d0, d1, v2.D(), 0))
TEST_FP_NEON(fmulx_5, fmulx(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fmulx_6, fmulx(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fmulx_7, fmulx(s0, s1, s2))
TEST_FP_NEON(fmulx_8, fmulx(d0, d1, d2))
TEST_FP_NEON(fmul_0, fmul(v0.V2S(), v1.V2S(), v2.S(), 3))
TEST_FP_NEON(fmul_1, fmul(v0.V4S(), v1.V4S(), v2.S(), 2))
TEST_FP_NEON(fmul_2, fmul(v0.V2D(), v1.V2D(), v2.D(), 1))
TEST_FP_NEON(fmul_3, fmul(s0, s1, v2.S(), 1))
TEST_FP_NEON(fmul_4, fmul(d0, d1, v2.D(), 1))
TEST_FP_NEON(fmul_5, fmul(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fmul_6, fmul(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fneg_0, fneg(v0.V2S(), v1.V2S()))
TEST_FP_NEON(fneg_1, fneg(v0.V4S(), v1.V4S()))
TEST_FP_NEON(fneg_2, fneg(v0.V2D(), v1.V2D()))
TEST_FP_NEON(frecpe_0, frecpe(v0.V2S(), v1.V2S()))
TEST_FP_NEON(frecpe_1, frecpe(v0.V4S(), v1.V4S()))
TEST_FP_NEON(frecpe_2, frecpe(v0.V2D(), v1.V2D()))
TEST_FP_NEON(frecpe_3, frecpe(s0, s1))
TEST_FP_NEON(frecpe_4, frecpe(d0, d1))
TEST_FP_NEON(frecps_0, frecps(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(frecps_1, frecps(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(frecps_2, frecps(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(frecps_3, frecps(s0, s1, s2))
TEST_FP_NEON(frecps_4, frecps(d0, d1, d2))
TEST_FP_NEON(frecpx_0, frecpx(s0, s1))
TEST_FP_NEON(frecpx_1, frecpx(d0, d1))
TEST_FP_NEON(frinta_0, frinta(v0.V2S(), v1.V2S()))
TEST_FP_NEON(frinta_1, frinta(v0.V4S(), v1.V4S()))
TEST_FP_NEON(frinta_2, frinta(v0.V2D(), v1.V2D()))
TEST_FP_NEON(frinti_0, frinti(v0.V2S(), v1.V2S()))
TEST_FP_NEON(frinti_1, frinti(v0.V4S(), v1.V4S()))
TEST_FP_NEON(frinti_2, frinti(v0.V2D(), v1.V2D()))
TEST_FP_NEON(frintm_0, frintm(v0.V2S(), v1.V2S()))
TEST_FP_NEON(frintm_1, frintm(v0.V4S(), v1.V4S()))
TEST_FP_NEON(frintm_2, frintm(v0.V2D(), v1.V2D()))
TEST_FP_NEON(frintn_0, frintn(v0.V2S(), v1.V2S()))
TEST_FP_NEON(frintn_1, frintn(v0.V4S(), v1.V4S()))
TEST_FP_NEON(frintn_2, frintn(v0.V2D(), v1.V2D()))
TEST_FP_NEON(frintp_0, frintp(v0.V2S(), v1.V2S()))
TEST_FP_NEON(frintp_1, frintp(v0.V4S(), v1.V4S()))
TEST_FP_NEON(frintp_2, frintp(v0.V2D(), v1.V2D()))
TEST_FP_NEON(frintx_0, frintx(v0.V2S(), v1.V2S()))
TEST_FP_NEON(frintx_1, frintx(v0.V4S(), v1.V4S()))
TEST_FP_NEON(frintx_2, frintx(v0.V2D(), v1.V2D()))
TEST_FP_NEON(frintz_0, frintz(v0.V2S(), v1.V2S()))
TEST_FP_NEON(frintz_1, frintz(v0.V4S(), v1.V4S()))
TEST_FP_NEON(frintz_2, frintz(v0.V2D(), v1.V2D()))
TEST_FP_NEON(frsqrte_0, frsqrte(v0.V2S(), v1.V2S()))
TEST_FP_NEON(frsqrte_1, frsqrte(v0.V4S(), v1.V4S()))
TEST_FP_NEON(frsqrte_2, frsqrte(v0.V2D(), v1.V2D()))
TEST_FP_NEON(frsqrte_3, frsqrte(s0, s1))
TEST_FP_NEON(frsqrte_4, frsqrte(d0, d1))
TEST_FP_NEON(frsqrts_0, frsqrts(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(frsqrts_1, frsqrts(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(frsqrts_2, frsqrts(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(frsqrts_3, frsqrts(s0, s1, s2))
TEST_FP_NEON(frsqrts_4, frsqrts(d0, d1, d2))
TEST_FP_NEON(fsqrt_0, fsqrt(v0.V2S(), v1.V2S()))
TEST_FP_NEON(fsqrt_1, fsqrt(v0.V4S(), v1.V4S()))
TEST_FP_NEON(fsqrt_2, fsqrt(v0.V2D(), v1.V2D()))
TEST_FP_NEON(fsub_0, fsub(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_FP_NEON(fsub_1, fsub(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_FP_NEON(fsub_2, fsub(v0.V2D(), v1.V2D(), v2.V2D()))
TEST_FP_NEON(scvtf_0, scvtf(v0.V2S(), v1.V2S(), 5))
TEST_FP_NEON(scvtf_1, scvtf(v0.V4S(), v1.V4S(), 5))
TEST_FP_NEON(scvtf_2, scvtf(v0.V2D(), v1.V2D(), 5))
TEST_FP_NEON(scvtf_3, scvtf(s0, s1, 5))
TEST_FP_NEON(scvtf_4, scvtf(d0, d1, 5))
TEST_FP_NEON(scvtf_5, scvtf(v0.V2S(), v1.V2S()))
TEST_FP_NEON(scvtf_6, scvtf(v0.V4S(), v1.V4S()))
TEST_FP_NEON(scvtf_7, scvtf(v0.V2D(), v1.V2D()))
TEST_FP_NEON(scvtf_8, scvtf(s0, s1))
TEST_FP_NEON(scvtf_9, scvtf(d0, d1))
TEST_FP_NEON(ucvtf_0, ucvtf(v0.V2S(), v1.V2S(), 5))
TEST_FP_NEON(ucvtf_1, ucvtf(v0.V4S(), v1.V4S(), 5))
TEST_FP_NEON(ucvtf_2, ucvtf(v0.V2D(), v1.V2D(), 5))
TEST_FP_NEON(ucvtf_3, ucvtf(s0, s1, 5))
TEST_FP_NEON(ucvtf_4, ucvtf(d0, d1, 5))
TEST_FP_NEON(ucvtf_5, ucvtf(v0.V2S(), v1.V2S()))
TEST_FP_NEON(ucvtf_6, ucvtf(v0.V4S(), v1.V4S()))
TEST_FP_NEON(ucvtf_7, ucvtf(v0.V2D(), v1.V2D()))
TEST_FP_NEON(ucvtf_8, ucvtf(s0, s1))
TEST_FP_NEON(ucvtf_9, ucvtf(d0, d1))

#define TEST_FP_NEON_FRINT(NAME, ASM)                            \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kFP,                    \
                            CPUFeatures::kNEON,                  \
                            CPUFeatures::kFrintToFixedSizedInt), \
                FP_NEON_##NAME,                                  \
                ASM)
TEST_FP_NEON_FRINT(frint32x_0, frint32x(v0.V2S(), v1.V2S()))
TEST_FP_NEON_FRINT(frint32x_1, frint32x(v0.V4S(), v1.V4S()))
TEST_FP_NEON_FRINT(frint32x_2, frint32x(v0.V2D(), v1.V2D()))
TEST_FP_NEON_FRINT(frint32z_0, frint32z(v0.V2S(), v1.V2S()))
TEST_FP_NEON_FRINT(frint32z_1, frint32z(v0.V4S(), v1.V4S()))
TEST_FP_NEON_FRINT(frint32z_2, frint32z(v0.V2D(), v1.V2D()))
TEST_FP_NEON_FRINT(frint64x_0, frint64x(v0.V2S(), v1.V2S()))
TEST_FP_NEON_FRINT(frint64x_1, frint64x(v0.V4S(), v1.V4S()))
TEST_FP_NEON_FRINT(frint64x_2, frint64x(v0.V2D(), v1.V2D()))
TEST_FP_NEON_FRINT(frint64z_0, frint64z(v0.V2S(), v1.V2S()))
TEST_FP_NEON_FRINT(frint64z_1, frint64z(v0.V4S(), v1.V4S()))
TEST_FP_NEON_FRINT(frint64z_2, frint64z(v0.V2D(), v1.V2D()))

#define TEST_FP_JSCVT(NAME, ASM)                                    \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kFP, CPUFeatures::kJSCVT), \
                FP_JSCVT_##NAME,                                    \
                ASM)
TEST_FP_JSCVT(fjcvtzs_0, fjcvtzs(w0, d1))

#define TEST_RDM_NEON(NAME, ASM)                                    \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kRDM, CPUFeatures::kNEON), \
                RDM_NEON_##NAME,                                    \
                ASM)
TEST_RDM_NEON(sqrdmlah_0, sqrdmlah(v0.V4H(), v1.V4H(), v2.H(), 5))
TEST_RDM_NEON(sqrdmlah_1, sqrdmlah(v0.V8H(), v1.V8H(), v2.H(), 4))
TEST_RDM_NEON(sqrdmlah_2, sqrdmlah(v0.V2S(), v1.V2S(), v2.S(), 3))
TEST_RDM_NEON(sqrdmlah_3, sqrdmlah(v0.V4S(), v1.V4S(), v2.S(), 0))
TEST_RDM_NEON(sqrdmlah_4, sqrdmlah(h0, h1, v2.H(), 5))
TEST_RDM_NEON(sqrdmlah_5, sqrdmlah(s0, s1, v2.S(), 1))
TEST_RDM_NEON(sqrdmlah_6, sqrdmlah(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_RDM_NEON(sqrdmlah_7, sqrdmlah(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_RDM_NEON(sqrdmlah_8, sqrdmlah(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_RDM_NEON(sqrdmlah_9, sqrdmlah(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_RDM_NEON(sqrdmlah_10, sqrdmlah(h0, h1, h2))
TEST_RDM_NEON(sqrdmlah_11, sqrdmlah(s0, s1, s2))
TEST_RDM_NEON(sqrdmlsh_0, sqrdmlsh(v0.V4H(), v1.V4H(), v2.H(), 5))
TEST_RDM_NEON(sqrdmlsh_1, sqrdmlsh(v0.V8H(), v1.V8H(), v2.H(), 5))
TEST_RDM_NEON(sqrdmlsh_2, sqrdmlsh(v0.V2S(), v1.V2S(), v2.S(), 2))
TEST_RDM_NEON(sqrdmlsh_3, sqrdmlsh(v0.V4S(), v1.V4S(), v2.S(), 1))
TEST_RDM_NEON(sqrdmlsh_4, sqrdmlsh(h0, h1, v2.H(), 6))
TEST_RDM_NEON(sqrdmlsh_5, sqrdmlsh(s0, s1, v2.S(), 1))
TEST_RDM_NEON(sqrdmlsh_6, sqrdmlsh(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_RDM_NEON(sqrdmlsh_7, sqrdmlsh(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_RDM_NEON(sqrdmlsh_8, sqrdmlsh(v0.V2S(), v1.V2S(), v2.V2S()))
TEST_RDM_NEON(sqrdmlsh_9, sqrdmlsh(v0.V4S(), v1.V4S(), v2.V4S()))
TEST_RDM_NEON(sqrdmlsh_10, sqrdmlsh(h0, h1, h2))
TEST_RDM_NEON(sqrdmlsh_11, sqrdmlsh(s0, s1, s2))

#define TEST_FP_FPHALF(NAME, ASM)                                    \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kFP, CPUFeatures::kFPHalf), \
                FP_FPHalf_##NAME,                                    \
                ASM)
TEST_FP_FPHALF(fabs_0, fabs(h0, h1))
TEST_FP_FPHALF(fadd_0, fadd(h0, h1, h2))
TEST_FP_FPHALF(fccmpe_0, fccmpe(h0, h1, CFlag, cc))
TEST_FP_FPHALF(fccmp_0, fccmp(h0, h1, NCFlag, lt))
TEST_FP_FPHALF(fcmpe_0, fcmpe(h0, 0.0))
TEST_FP_FPHALF(fcmpe_1, fcmpe(h0, h1))
TEST_FP_FPHALF(fcmp_0, fcmp(h0, 0.0))
TEST_FP_FPHALF(fcmp_1, fcmp(h0, h1))
TEST_FP_FPHALF(fcsel_0, fcsel(h0, h1, h2, cc))
TEST_FP_FPHALF(fcvtas_0, fcvtas(w0, h1))
TEST_FP_FPHALF(fcvtas_1, fcvtas(x0, h1))
TEST_FP_FPHALF(fcvtau_0, fcvtau(w0, h1))
TEST_FP_FPHALF(fcvtau_1, fcvtau(x0, h1))
TEST_FP_FPHALF(fcvtms_0, fcvtms(w0, h1))
TEST_FP_FPHALF(fcvtms_1, fcvtms(x0, h1))
TEST_FP_FPHALF(fcvtmu_0, fcvtmu(w0, h1))
TEST_FP_FPHALF(fcvtmu_1, fcvtmu(x0, h1))
TEST_FP_FPHALF(fcvtns_0, fcvtns(w0, h1))
TEST_FP_FPHALF(fcvtns_1, fcvtns(x0, h1))
TEST_FP_FPHALF(fcvtnu_0, fcvtnu(w0, h1))
TEST_FP_FPHALF(fcvtnu_1, fcvtnu(x0, h1))
TEST_FP_FPHALF(fcvtps_0, fcvtps(w0, h1))
TEST_FP_FPHALF(fcvtps_1, fcvtps(x0, h1))
TEST_FP_FPHALF(fcvtpu_0, fcvtpu(w0, h1))
TEST_FP_FPHALF(fcvtpu_1, fcvtpu(x0, h1))
TEST_FP_FPHALF(fcvtzs_0, fcvtzs(w0, h1, 5))
TEST_FP_FPHALF(fcvtzs_1, fcvtzs(x0, h1, 5))
TEST_FP_FPHALF(fcvtzs_2, fcvtzs(w0, h1))
TEST_FP_FPHALF(fcvtzs_3, fcvtzs(x0, h1))
TEST_FP_FPHALF(fcvtzu_0, fcvtzu(w0, h1, 5))
TEST_FP_FPHALF(fcvtzu_1, fcvtzu(x0, h1, 5))
TEST_FP_FPHALF(fcvtzu_2, fcvtzu(w0, h1))
TEST_FP_FPHALF(fcvtzu_3, fcvtzu(x0, h1))
TEST_FP_FPHALF(fdiv_0, fdiv(h0, h1, h2))
TEST_FP_FPHALF(fmadd_0, fmadd(h0, h1, h2, h3))
TEST_FP_FPHALF(fmaxnm_0, fmaxnm(h0, h1, h2))
TEST_FP_FPHALF(fmax_0, fmax(h0, h1, h2))
TEST_FP_FPHALF(fminnm_0, fminnm(h0, h1, h2))
TEST_FP_FPHALF(fmin_0, fmin(h0, h1, h2))
TEST_FP_FPHALF(fmov_0, fmov(h0, h1))
TEST_FP_FPHALF(fmov_1, fmov(w0, h1))
TEST_FP_FPHALF(fmov_2, fmov(x0, h1))
TEST_FP_FPHALF(fmov_3, fmov(h0, w1))
TEST_FP_FPHALF(fmov_4, fmov(h0, x1))
TEST_FP_FPHALF(fmov_5, fmov(h0, Float16(0.390625)))
TEST_FP_FPHALF(fmsub_0, fmsub(h0, h1, h2, h3))
TEST_FP_FPHALF(fmul_0, fmul(h0, h1, h2))
TEST_FP_FPHALF(fneg_0, fneg(h0, h1))
TEST_FP_FPHALF(fnmadd_0, fnmadd(h0, h1, h2, h3))
TEST_FP_FPHALF(fnmsub_0, fnmsub(h0, h1, h2, h3))
TEST_FP_FPHALF(fnmul_0, fnmul(h0, h1, h2))
TEST_FP_FPHALF(frinta_0, frinta(h0, h1))
TEST_FP_FPHALF(frinti_0, frinti(h0, h1))
TEST_FP_FPHALF(frintm_0, frintm(h0, h1))
TEST_FP_FPHALF(frintn_0, frintn(h0, h1))
TEST_FP_FPHALF(frintp_0, frintp(h0, h1))
TEST_FP_FPHALF(frintx_0, frintx(h0, h1))
TEST_FP_FPHALF(frintz_0, frintz(h0, h1))
TEST_FP_FPHALF(fsqrt_0, fsqrt(h0, h1))
TEST_FP_FPHALF(fsub_0, fsub(h0, h1, h2))
TEST_FP_FPHALF(scvtf_0, scvtf(h0, w1, 5))
TEST_FP_FPHALF(scvtf_1, scvtf(h0, x1, 5))
TEST_FP_FPHALF(scvtf_2, scvtf(h0, w1))
TEST_FP_FPHALF(scvtf_3, scvtf(h0, x1))
TEST_FP_FPHALF(ucvtf_0, ucvtf(h0, w1, 5))
TEST_FP_FPHALF(ucvtf_1, ucvtf(h0, x1, 5))
TEST_FP_FPHALF(ucvtf_2, ucvtf(h0, w1))
TEST_FP_FPHALF(ucvtf_3, ucvtf(h0, x1))

#define TEST_LOREGIONS(NAME, ASM) \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kLORegions), LORegions_##NAME, ASM)
TEST_LOREGIONS(ldlar_0, ldlar(w0, MemOperand(x1, 0)))
TEST_LOREGIONS(ldlar_1, ldlar(x0, MemOperand(x1, 0)))
TEST_LOREGIONS(ldlarb_0, ldlarb(w0, MemOperand(x1, 0)))
TEST_LOREGIONS(ldlarh_0, ldlarh(w0, MemOperand(x1, 0)))
TEST_LOREGIONS(stllr_0, stllr(w0, MemOperand(x1, 0)))
TEST_LOREGIONS(stllr_1, stllr(x0, MemOperand(x1, 0)))
TEST_LOREGIONS(stllrb_0, stllrb(w0, MemOperand(x1, 0)))
TEST_LOREGIONS(stllrh_0, stllrh(w0, MemOperand(x1, 0)))

#define TEST_FP_FCMA_NEON(NAME, ASM)             \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kFP,    \
                            CPUFeatures::kFcma,  \
                            CPUFeatures::kNEON), \
                FP_Fcma_NEON_##NAME,             \
                ASM)
TEST_FP_FCMA_NEON(fcadd_0, fcadd(v0.V2S(), v1.V2S(), v2.V2S(), 270))
TEST_FP_FCMA_NEON(fcadd_1, fcadd(v0.V4S(), v1.V4S(), v2.V4S(), 90))
TEST_FP_FCMA_NEON(fcadd_2, fcadd(v0.V2D(), v1.V2D(), v2.V2D(), 270))
TEST_FP_FCMA_NEON(fcmla_0, fcmla(v0.V4S(), v1.V4S(), v2.S(), 0, 180))
TEST_FP_FCMA_NEON(fcmla_1, fcmla(v0.V2S(), v1.V2S(), v2.V2S(), 90))
TEST_FP_FCMA_NEON(fcmla_2, fcmla(v0.V4S(), v1.V4S(), v2.V4S(), 90))
TEST_FP_FCMA_NEON(fcmla_3, fcmla(v0.V2D(), v1.V2D(), v2.V2D(), 90))

#define TEST_RCPC_RCPCIMM(NAME, ASM)                                    \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kRCpc, CPUFeatures::kRCpcImm), \
                RCpc_RCpcImm_##NAME,                                    \
                ASM)
TEST_RCPC_RCPCIMM(ldapurb_0, ldapurb(w0, MemOperand(x1, 20)))
TEST_RCPC_RCPCIMM(ldapurh_0, ldapurh(w0, MemOperand(x1, 194)))
TEST_RCPC_RCPCIMM(ldapursb_0, ldapursb(w0, MemOperand(x1, -27)))
TEST_RCPC_RCPCIMM(ldapursb_1, ldapursb(x0, MemOperand(x1, -64)))
TEST_RCPC_RCPCIMM(ldapursh_0, ldapursh(w0, MemOperand(x1, 180)))
TEST_RCPC_RCPCIMM(ldapursh_1, ldapursh(x0, MemOperand(x1, -212)))
TEST_RCPC_RCPCIMM(ldapursw_0, ldapursw(x0, MemOperand(x1, -196)))
TEST_RCPC_RCPCIMM(ldapur_0, ldapur(w0, MemOperand(x1, -96)))
TEST_RCPC_RCPCIMM(ldapur_1, ldapur(x0, MemOperand(x1, -112)))
TEST_RCPC_RCPCIMM(stlurb_0, stlurb(w0, MemOperand(x1, -233)))
TEST_RCPC_RCPCIMM(stlurh_0, stlurh(w0, MemOperand(x1, -147)))
TEST_RCPC_RCPCIMM(stlur_0, stlur(w0, MemOperand(x1, 40)))
TEST_RCPC_RCPCIMM(stlur_1, stlur(x0, MemOperand(x1, 209)))

#define TEST_NEON_DOTPRODUCT(NAME, ASM)                                    \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kNEON, CPUFeatures::kDotProduct), \
                NEON_DotProduct_##NAME,                                    \
                ASM)
TEST_NEON_DOTPRODUCT(sdot_0, sdot(v0.V2S(), v1.V8B(), v2.S4B(), 1))
TEST_NEON_DOTPRODUCT(sdot_1, sdot(v0.V4S(), v1.V16B(), v2.S4B(), 1))
TEST_NEON_DOTPRODUCT(sdot_2, sdot(v0.V2S(), v1.V8B(), v2.V8B()))
TEST_NEON_DOTPRODUCT(sdot_3, sdot(v0.V4S(), v1.V16B(), v2.V16B()))
TEST_NEON_DOTPRODUCT(udot_0, udot(v0.V2S(), v1.V8B(), v2.S4B(), 0))
TEST_NEON_DOTPRODUCT(udot_1, udot(v0.V4S(), v1.V16B(), v2.S4B(), 1))
TEST_NEON_DOTPRODUCT(udot_2, udot(v0.V2S(), v1.V8B(), v2.V8B()))
TEST_NEON_DOTPRODUCT(udot_3, udot(v0.V4S(), v1.V16B(), v2.V16B()))

#define TEST_FP_NEON_NEONHALF(NAME, ASM)             \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kFP,        \
                            CPUFeatures::kNEON,      \
                            CPUFeatures::kNEONHalf), \
                FP_NEON_NEONHalf_##NAME,             \
                ASM)
TEST_FP_NEON_NEONHALF(fabd_0, fabd(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fabd_1, fabd(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fabd_2, fabd(h0, h1, h2))
TEST_FP_NEON_NEONHALF(fabs_0, fabs(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(fabs_1, fabs(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(facge_0, facge(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(facge_1, facge(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(facge_2, facge(h0, h1, h2))
TEST_FP_NEON_NEONHALF(facgt_0, facgt(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(facgt_1, facgt(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(facgt_2, facgt(h0, h1, h2))
TEST_FP_NEON_NEONHALF(faddp_0, faddp(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(faddp_1, faddp(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fadd_0, fadd(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fadd_1, fadd(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fcmeq_0, fcmeq(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fcmeq_1, fcmeq(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fcmeq_2, fcmeq(h0, h1, h2))
TEST_FP_NEON_NEONHALF(fcmeq_3, fcmeq(v0.V4H(), v1.V4H(), 0.0))
TEST_FP_NEON_NEONHALF(fcmeq_4, fcmeq(v0.V8H(), v1.V8H(), 0.0))
TEST_FP_NEON_NEONHALF(fcmeq_5, fcmeq(h0, h1, 0.0))
TEST_FP_NEON_NEONHALF(fcmge_0, fcmge(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fcmge_1, fcmge(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fcmge_2, fcmge(h0, h1, h2))
TEST_FP_NEON_NEONHALF(fcmge_3, fcmge(v0.V4H(), v1.V4H(), 0.0))
TEST_FP_NEON_NEONHALF(fcmge_4, fcmge(v0.V8H(), v1.V8H(), 0.0))
TEST_FP_NEON_NEONHALF(fcmge_5, fcmge(h0, h1, 0.0))
TEST_FP_NEON_NEONHALF(fcmgt_0, fcmgt(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fcmgt_1, fcmgt(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fcmgt_2, fcmgt(h0, h1, h2))
TEST_FP_NEON_NEONHALF(fcmgt_3, fcmgt(v0.V4H(), v1.V4H(), 0.0))
TEST_FP_NEON_NEONHALF(fcmgt_4, fcmgt(v0.V8H(), v1.V8H(), 0.0))
TEST_FP_NEON_NEONHALF(fcmgt_5, fcmgt(h0, h1, 0.0))
TEST_FP_NEON_NEONHALF(fcmle_0, fcmle(v0.V4H(), v1.V4H(), 0.0))
TEST_FP_NEON_NEONHALF(fcmle_1, fcmle(v0.V8H(), v1.V8H(), 0.0))
TEST_FP_NEON_NEONHALF(fcmle_2, fcmle(h0, h1, 0.0))
TEST_FP_NEON_NEONHALF(fcmlt_0, fcmlt(v0.V4H(), v1.V4H(), 0.0))
TEST_FP_NEON_NEONHALF(fcmlt_1, fcmlt(v0.V8H(), v1.V8H(), 0.0))
TEST_FP_NEON_NEONHALF(fcmlt_2, fcmlt(h0, h1, 0.0))
TEST_FP_NEON_NEONHALF(fcvtas_0, fcvtas(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(fcvtas_1, fcvtas(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(fcvtas_2, fcvtas(h0, h1))
TEST_FP_NEON_NEONHALF(fcvtau_0, fcvtau(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(fcvtau_1, fcvtau(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(fcvtau_2, fcvtau(h0, h1))
TEST_FP_NEON_NEONHALF(fcvtms_0, fcvtms(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(fcvtms_1, fcvtms(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(fcvtms_2, fcvtms(h0, h1))
TEST_FP_NEON_NEONHALF(fcvtmu_0, fcvtmu(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(fcvtmu_1, fcvtmu(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(fcvtmu_2, fcvtmu(h0, h1))
TEST_FP_NEON_NEONHALF(fcvtns_0, fcvtns(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(fcvtns_1, fcvtns(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(fcvtns_2, fcvtns(h0, h1))
TEST_FP_NEON_NEONHALF(fcvtnu_0, fcvtnu(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(fcvtnu_1, fcvtnu(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(fcvtnu_2, fcvtnu(h0, h1))
TEST_FP_NEON_NEONHALF(fcvtps_0, fcvtps(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(fcvtps_1, fcvtps(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(fcvtps_2, fcvtps(h0, h1))
TEST_FP_NEON_NEONHALF(fcvtpu_0, fcvtpu(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(fcvtpu_1, fcvtpu(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(fcvtpu_2, fcvtpu(h0, h1))
TEST_FP_NEON_NEONHALF(fcvtzs_0, fcvtzs(v0.V4H(), v1.V4H(), 5))
TEST_FP_NEON_NEONHALF(fcvtzs_1, fcvtzs(v0.V8H(), v1.V8H(), 5))
TEST_FP_NEON_NEONHALF(fcvtzs_2, fcvtzs(h0, h1, 5))
TEST_FP_NEON_NEONHALF(fcvtzs_3, fcvtzs(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(fcvtzs_4, fcvtzs(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(fcvtzs_5, fcvtzs(h0, h1))
TEST_FP_NEON_NEONHALF(fcvtzu_0, fcvtzu(v0.V4H(), v1.V4H(), 5))
TEST_FP_NEON_NEONHALF(fcvtzu_1, fcvtzu(v0.V8H(), v1.V8H(), 5))
TEST_FP_NEON_NEONHALF(fcvtzu_2, fcvtzu(h0, h1, 5))
TEST_FP_NEON_NEONHALF(fcvtzu_3, fcvtzu(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(fcvtzu_4, fcvtzu(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(fcvtzu_5, fcvtzu(h0, h1))
TEST_FP_NEON_NEONHALF(fdiv_0, fdiv(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fdiv_1, fdiv(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fmaxnmp_0, fmaxnmp(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fmaxnmp_1, fmaxnmp(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fmaxnmv_0, fmaxnmv(h0, v1.V4H()))
TEST_FP_NEON_NEONHALF(fmaxnmv_1, fmaxnmv(h0, v1.V8H()))
TEST_FP_NEON_NEONHALF(fmaxnm_0, fmaxnm(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fmaxnm_1, fmaxnm(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fmaxp_0, fmaxp(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fmaxp_1, fmaxp(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fmax_0, fmax(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fmax_1, fmax(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fminnmp_0, fminnmp(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fminnmp_1, fminnmp(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fminnmv_0, fminnmv(h0, v1.V4H()))
TEST_FP_NEON_NEONHALF(fminnmv_1, fminnmv(h0, v1.V8H()))
TEST_FP_NEON_NEONHALF(fminnm_0, fminnm(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fminnm_1, fminnm(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fminp_0, fminp(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fminp_1, fminp(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fmin_0, fmin(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fmin_1, fmin(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fmla_0, fmla(v0.V4H(), v1.V4H(), v2.H(), 4))
TEST_FP_NEON_NEONHALF(fmla_1, fmla(v0.V8H(), v1.V8H(), v2.H(), 2))
TEST_FP_NEON_NEONHALF(fmla_2, fmla(h0, h1, v2.H(), 5))
TEST_FP_NEON_NEONHALF(fmla_3, fmla(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fmla_4, fmla(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fmls_0, fmls(v0.V4H(), v1.V4H(), v2.H(), 0))
TEST_FP_NEON_NEONHALF(fmls_1, fmls(v0.V8H(), v1.V8H(), v2.H(), 0))
TEST_FP_NEON_NEONHALF(fmls_2, fmls(h0, h1, v2.H(), 0))
TEST_FP_NEON_NEONHALF(fmls_3, fmls(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fmls_4, fmls(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fmov_0, fmov(v0.V4H(), Float16(18.0)))
TEST_FP_NEON_NEONHALF(fmov_1, fmov(v0.V8H(), Float16(-0.2421875)))
TEST_FP_NEON_NEONHALF(fmulx_0, fmulx(v0.V4H(), v1.V4H(), v2.H(), 6))
TEST_FP_NEON_NEONHALF(fmulx_1, fmulx(v0.V8H(), v1.V8H(), v2.H(), 2))
TEST_FP_NEON_NEONHALF(fmulx_2, fmulx(h0, h1, v2.H(), 4))
TEST_FP_NEON_NEONHALF(fmulx_3, fmulx(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fmulx_4, fmulx(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fmulx_5, fmulx(h0, h1, h2))
TEST_FP_NEON_NEONHALF(fmul_0, fmul(v0.V4H(), v1.V4H(), v2.H(), 3))
TEST_FP_NEON_NEONHALF(fmul_1, fmul(v0.V8H(), v1.V8H(), v2.H(), 1))
TEST_FP_NEON_NEONHALF(fmul_2, fmul(h0, h1, v2.H(), 1))
TEST_FP_NEON_NEONHALF(fmul_3, fmul(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fmul_4, fmul(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(fneg_0, fneg(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(fneg_1, fneg(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(frecpe_0, frecpe(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(frecpe_1, frecpe(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(frecpe_2, frecpe(h0, h1))
TEST_FP_NEON_NEONHALF(frecps_0, frecps(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(frecps_1, frecps(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(frecps_2, frecps(h0, h1, h2))
TEST_FP_NEON_NEONHALF(frecpx_0, frecpx(h0, h1))
TEST_FP_NEON_NEONHALF(frinta_0, frinta(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(frinta_1, frinta(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(frinti_0, frinti(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(frinti_1, frinti(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(frintm_0, frintm(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(frintm_1, frintm(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(frintn_0, frintn(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(frintn_1, frintn(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(frintp_0, frintp(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(frintp_1, frintp(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(frintx_0, frintx(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(frintx_1, frintx(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(frintz_0, frintz(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(frintz_1, frintz(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(frsqrte_0, frsqrte(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(frsqrte_1, frsqrte(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(frsqrte_2, frsqrte(h0, h1))
TEST_FP_NEON_NEONHALF(frsqrts_0, frsqrts(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(frsqrts_1, frsqrts(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(frsqrts_2, frsqrts(h0, h1, h2))
TEST_FP_NEON_NEONHALF(fsqrt_0, fsqrt(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(fsqrt_1, fsqrt(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(fsub_0, fsub(v0.V4H(), v1.V4H(), v2.V4H()))
TEST_FP_NEON_NEONHALF(fsub_1, fsub(v0.V8H(), v1.V8H(), v2.V8H()))
TEST_FP_NEON_NEONHALF(scvtf_0, scvtf(v0.V4H(), v1.V4H(), 5))
TEST_FP_NEON_NEONHALF(scvtf_1, scvtf(v0.V8H(), v1.V8H(), 5))
TEST_FP_NEON_NEONHALF(scvtf_2, scvtf(h0, h1, 5))
TEST_FP_NEON_NEONHALF(scvtf_3, scvtf(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(scvtf_4, scvtf(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(scvtf_5, scvtf(h0, h1))
TEST_FP_NEON_NEONHALF(ucvtf_0, ucvtf(v0.V4H(), v1.V4H(), 5))
TEST_FP_NEON_NEONHALF(ucvtf_1, ucvtf(v0.V8H(), v1.V8H(), 5))
TEST_FP_NEON_NEONHALF(ucvtf_2, ucvtf(h0, h1, 5))
TEST_FP_NEON_NEONHALF(ucvtf_3, ucvtf(v0.V4H(), v1.V4H()))
TEST_FP_NEON_NEONHALF(ucvtf_4, ucvtf(v0.V8H(), v1.V8H()))
TEST_FP_NEON_NEONHALF(ucvtf_5, ucvtf(h0, h1))

#define TEST_PAUTH_PAUTHGENERIC(NAME, ASM)                                    \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kPAuth, CPUFeatures::kPAuthGeneric), \
                PAuth_PAuthGeneric_##NAME,                                    \
                ASM)
TEST_PAUTH_PAUTHGENERIC(pacga_0, pacga(x0, x1, x2))

#define TEST_FP_FHM_NEON_NEONHALF(NAME, ASM)         \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kFP,        \
                            CPUFeatures::kFHM,       \
                            CPUFeatures::kNEON,      \
                            CPUFeatures::kNEONHalf), \
                FP_FHM_NEON_NEONHalf_##NAME,         \
                ASM)
TEST_FP_FHM_NEON_NEONHALF(fmlal2_0, fmlal2(v0.V4S(), v1.V4H(), v2.H(), 6))
TEST_FP_FHM_NEON_NEONHALF(fmlal_0, fmlal(v0.V4S(), v1.V4H(), v2.H(), 0))
TEST_FP_FHM_NEON_NEONHALF(fmlal2_1, fmlal2(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_FP_FHM_NEON_NEONHALF(fmlal_1, fmlal(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_FP_FHM_NEON_NEONHALF(fmlsl2_0, fmlsl2(v0.V4S(), v1.V4H(), v2.H(), 2))
TEST_FP_FHM_NEON_NEONHALF(fmlsl_0, fmlsl(v0.V4S(), v1.V4H(), v2.H(), 2))
TEST_FP_FHM_NEON_NEONHALF(fmlsl2_1, fmlsl2(v0.V4S(), v1.V4H(), v2.V4H()))
TEST_FP_FHM_NEON_NEONHALF(fmlsl_1, fmlsl(v0.V4S(), v1.V4H(), v2.V4H()))

#define TEST_FP_FCMA_NEON_NEONHALF(NAME, ASM)        \
  TEST_TEMPLATE(CPUFeatures(CPUFeatures::kFP,        \
                            CPUFeatures::kFcma,      \
                            CPUFeatures::kNEON,      \
                            CPUFeatures::kNEONHalf), \
                FP_Fcma_NEON_NEONHalf_##NAME,        \
                ASM)
TEST_FP_FCMA_NEON_NEONHALF(fcadd_0, fcadd(v0.V4H(), v1.V4H(), v2.V4H(), 90))
TEST_FP_FCMA_NEON_NEONHALF(fcadd_1, fcadd(v0.V8H(), v1.V8H(), v2.V8H(), 90))
TEST_FP_FCMA_NEON_NEONHALF(fcmla_0, fcmla(v0.V4H(), v1.V4H(), v2.H(), 0, 0))
TEST_FP_FCMA_NEON_NEONHALF(fcmla_1, fcmla(v0.V8H(), v1.V8H(), v2.H(), 2, 180))
TEST_FP_FCMA_NEON_NEONHALF(fcmla_2, fcmla(v0.V4H(), v1.V4H(), v2.V4H(), 180))
TEST_FP_FCMA_NEON_NEONHALF(fcmla_3, fcmla(v0.V8H(), v1.V8H(), v2.V8H(), 0))

}  // namespace aarch64
}  // namespace vixl
