import numpy as np

# This file contains reference geometries, energies, and gradients
#  for comparing QC program results. It is placed in QCEngine but is also
#  used directly by QCDB and Psi4. New or filled-out systems+modelchems
#  welcome, but it's good to start a discussion on GitHub to check that
#  its downstream roles not impinged.

std_refs = ["rhf", "uhf", "rohf"]

std_molecules = {
    "hf": """
                H
                F 1 0.917
              """,
    # "bh3p": """
    #            1 2
    #            B     0.10369114     0.00000000     0.00000000
    #            H    -1.13269886     0.00000000     0.00000000
    #            H     3.00000000     0.37149000     0.00000000
    #            H     3.00000000    -0.37149000     0.00000000
    #          """,
    "bh3p": """
                1 2
                B     0.00000000     0.00000000     0.10369114
                H     0.00000000     0.00000000    -1.13269886
                H     0.00000000    -0.37149000     3.00000000
                H     0.00000000     0.37149000     3.00000000
              """,
    "h2o": """
                O
                H 1 R
                H 1 R 2 A

                R=0.958
                A=104.5
              """,
    "nh2": """
                0 2
                N
                H 1 R
                H 1 R 2 A

                R=1.008
                A=105.0
              """,
    "h2o-xyz": """
 # R=0.958 A=104.5
 O                  0.000000000000     0.000000000000    -0.124038860300
 H                  0.000000000000    -1.431430901356     0.984293362719
 H                  0.000000000000     1.431430901356     0.984293362719
 units au
              """,
    "nh2-xyz": """
 # R=1.008 #A=105.0
 0 2
 N   0.000000000000000   0.000000000000000  -0.145912918634892
 H   0.000000000000000  -1.511214298139000   1.013682596946108
 H   0.000000000000000   1.511214298139000   1.013682596946108
 units au
              """,
    "hf-xyz": """
        H    0.          0.         -1.64558411
        F    0.          0.          0.08729475
        units bohr
               """,
}
std_molecules["bh3p-xyz"] = std_molecules["bh3p"]

_std_generics = {
    "hf_cc-pvdz_ae": (19, 19, 5, 5),
    "hf_cc-pvdz_fc": (19, 19, 5, 5),
    "bh3p_cc-pvdz_ae": (29, 29, 4, 3),
    "bh3p_cc-pvdz_fc": (29, 29, 4, 3),
    "h2o_aug-cc-pvdz_ae": (41, 41, 5, 5),
    "h2o_aug-cc-pvdz_fc": (41, 41, 5, 5),
    "nh2_aug-cc-pvdz_ae": (41, 41, 5, 4),
    "nh2_aug-cc-pvdz_fc": (41, 41, 5, 4),
    "h2o_cfour-qz2p_ae": (48, 48, 5, 5),
    "h2o_cfour-qz2p_fc": (48, 48, 5, 5),
    "nh2_cfour-qz2p_ae": (48, 48, 5, 4),
    "nh2_cfour-qz2p_fc": (48, 48, 5, 4),
}
_std_generics = {
    k: dict(zip(["N BASIS FUNCTIONS", "N MOLECULAR ORBITALS", "N ALPHA ELECTRONS", "N BETA ELECTRONS"], v))
    for k, v in _std_generics.items()
}

_scf_hf_dz_pk_rhf = -100.01941126902270
_scf_bh3p_dz_pk_uhf = -25.94513842869638
_scf_bh3p_dz_pk_rohf = -25.943614318546

_scf_hf_dz_df_rhf = -100.019400605629
_scf_bh3p_dz_df_uhf = -25.945130559147
_scf_bh3p_dz_df_rohf = -25.943606522029

_scf_hf_dz_cd_rhf = -100.01939270219628
_scf_bh3p_dz_cd_uhf = -25.94511891510799
_scf_bh3p_dz_cd_rohf = -25.943595251664313


_scf_h2o_qz2p_pk_rhf = -76.0627484601
_scf_nh2_qz2p_pk_uhf = -55.5893469688
_scf_nh2_qz2p_pk_rohf = -55.5847372601

_scf_h2o_qz2p_df_rhf = -76.06274142753659
_scf_nh2_qz2p_df_uhf = -55.58934323208402
_scf_nh2_qz2p_df_rohf = -55.58473319013903

_scf_h2o_qz2p_cd_rhf = -76.06277445978574
_scf_nh2_qz2p_cd_uhf = -55.58934916135869
_scf_nh2_qz2p_cd_rohf = -55.58473942870229


_scf_h2o_adz_pk_rhf = -76.0413815332
_scf_nh2_adz_pk_uhf = -55.57513805247548
_scf_nh2_adz_pk_rohf = -55.570724348574

_scf_h2o_adz_df_rhf = -76.04136132628614
_scf_nh2_adz_df_uhf = -55.57512538464817
_scf_nh2_adz_df_rohf = -55.57071142443952

_scf_h2o_adz_cd_rhf = -76.04132169763341
_scf_nh2_adz_cd_uhf = -55.57506886675886
_scf_nh2_adz_cd_rohf = -55.57065536578708


# fmt: off
_grad_scf_hf_dz_pk_rhf = np.array([0.0, 0.0, -0.019789748774, -0.0, -0.0, 0.019789748774]).reshape((-1, 3))
_grad_scf_bh3p_dz_pk_uhf = np.array([ 0.0, 0.0, 0.018283596898, 0.0, 0.0, -0.016914487538, -0.0, 0.004869782481, -0.00068455468, 0.0, -0.004869782481, -0.00068455468,]).reshape((-1, 3))
_grad_scf_bh3p_dz_pk_rohf = np.array([ 0.0, 0.0, 0.022062218595, 0.0, 0.0, -0.02050835054, -0.0, 0.004706239974, -0.000776934027, 0.0, -0.004706239974, -0.000776934027,]).reshape((-1, 3))

_grad_scf_hf_dz_df_rhf = np.array([0.0, -0.0, -0.0197887676, -0.0, 0.0, 0.0197887676]).reshape((-1, 3))
_grad_scf_bh3p_dz_df_uhf = np.array([-0.0, 0.0, 0.018279955719, 0.0, 0.0, -0.016906923007, -0.0, 0.004865311963, -0.000686516356, 0.0, -0.004865311963, -0.000686516356,]).reshape((-1, 3))
_grad_scf_bh3p_dz_df_rohf = np.array([-0.0, 0.0, 0.022058469913, 0.0, 0.0, -0.020500364247, -0.0, 0.004701465609, -0.000779052833, 0.0, -0.004701465609, -0.000779052833,]).reshape((-1, 3))

_grad_scf_hf_dz_cd_rhf = np.array([0.0, 0.0, -0.019796742926, 0.0, 0.0, 0.019796742926]).reshape((-1, 3))
_grad_scf_bh3p_dz_cd_uhf = np.array([ 0.0, 0.0, 0.018280917408, 0.0, 0.0, -0.016907239597, 0.0, 0.004874331571, -0.000686838905, 0.0, -0.004874331571, -0.000686838905,]).reshape((-1, 3))
_grad_scf_bh3p_dz_cd_rohf = np.array([ 0.0, 0.0, 0.022058379283, 0.0, 0.0, -0.020499895346, 0.0, 0.004710759277, -0.000779241969, 0.0, -0.004710759277, -0.000779241969,]).reshape((-1, 3))


_grad_scf_h2o_qz2p_pk_rhf = np.array([-0.0, 0.0, -0.026426043127, 0.0, -0.013943334328, 0.013213021564, -0.0, 0.013943334328, 0.013213021564]).reshape((-1, 3))
_grad_scf_nh2_qz2p_pk_uhf = np.array([0.0, 0.0, 0.001550839727, -0.0, 0.000092404341, -0.000775419863, 0.0, -0.000092404341, -0.000775419863]).reshape((-1, 3))
_grad_scf_nh2_qz2p_pk_rohf = np.array([-0.0, 0.0, 0.000683766715, 0.0, -0.000526016129, -0.000341883358, -0.0, 0.000526016129, -0.000341883358]).reshape((-1, 3))

_grad_scf_h2o_qz2p_df_rhf = np.array([0.0, 0.0, -0.026454963718, 0.0, -0.013950706431, 0.013227481859, -0.0, 0.013950706431, 0.013227481859]).reshape((-1, 3))
_grad_scf_nh2_qz2p_df_uhf = np.array([-0.0, 0.0, 0.001532526876, -0.0, 0.000082065307, -0.000766263438, 0.0, -0.000082065307, -0.000766263438]).reshape((-1, 3))
_grad_scf_nh2_qz2p_df_rohf = np.array([0.0, 0.0, 0.000665139571, 0.0, -0.000536617267, -0.000332569786, -0.0, 0.000536617267, -0.000332569786]).reshape((-1, 3))

_grad_scf_h2o_qz2p_cd_rhf = np.array([0.0, 0.0, -0.026440883169, 0.0, -0.013949099753, 0.013220441584, 0.0, 0.013949099753, 0.013220441584]).reshape((-1, 3))
_grad_scf_nh2_qz2p_cd_uhf = np.array([0.0, 0.0, 0.001549105775, 0.0, 0.000091952809, -0.000774552887, 0.0, -0.000091952809, -0.000774552887]).reshape((-1, 3))
_grad_scf_nh2_qz2p_cd_rohf = np.array([0.0, 0.0, 0.000681888827, 0.0, -0.000526508263, -0.000340944414, 0.0, 0.000526508263, -0.000340944414]).reshape((-1, 3))


_grad_scf_h2o_adz_pk_rhf = np.array([-0.0, 0.0, -0.021391822417, 0.0, -0.011259127342, 0.010695911208, -0.0, 0.011259127342, 0.010695911208]).reshape((-1, 3))
_grad_scf_nh2_adz_pk_uhf = np.array([-0.0, 0.0, 0.008498123792, -0.0, 0.004518040488, -0.004249061896, 0.0, -0.004518040488, -0.004249061896]).reshape((-1, 3))
_grad_scf_nh2_adz_pk_rohf = np.array([-0.0, 0.0, 0.007678819724, -0.0, 0.003856927713, -0.003839409862, 0.0, -0.003856927713, -0.003839409862]).reshape((-1, 3))

_grad_scf_h2o_adz_df_rhf = np.array([0.0, 0.0, -0.021402714298, 0.0, -0.011255158091, 0.010701357149, -0.0, 0.011255158091, 0.010701357149]).reshape((-1, 3))
_grad_scf_nh2_adz_df_uhf = np.array([0.0, 0.0, 0.00848503512, -0.0, 0.004519001873, -0.00424251756, 0.0, -0.004519001873, -0.00424251756]).reshape((-1, 3))
_grad_scf_nh2_adz_df_rohf = np.array([0.0, 0.0, 0.007664957774, -0.0, 0.003857593701, -0.003832478887, 0.0, -0.003857593701, -0.003832478887]).reshape((-1, 3))

_grad_scf_h2o_adz_cd_rhf = np.array([0.0, 0.0, -0.021374103228, 0.0, -0.011265854947, 0.010687051614, 0.0, 0.011265854947, 0.010687051614]).reshape((-1, 3))
_grad_scf_nh2_adz_cd_uhf = np.array([0.0, 0.0, 0.00848026807, 0.0, 0.004537635752, -0.004240134035, 0.0, -0.004537635752, -0.004240134035]).reshape((-1, 3))
_grad_scf_nh2_adz_cd_rohf = np.array([0.0, 0.0, 0.007661397585, 0.0, 0.003876803846, -0.003830698792, 0.0, -0.003876803846, -0.003830698792]).reshape((-1, 3))
# fmt: on


# fmt: off
_hess_scf_hf_dz_pk_rhf = np.array([
    0.011420157, 0., 0., -0.011420157, 0., 0.,
    0., 0.011420157, 0., 0., -0.011420157, 0.,
    0., 0., 0.6371334086, 0., 0., -0.6371334086,
    -0.011420157, 0., 0., 0.011420157, 0., 0.,
    0., -0.011420157, 0., 0., 0.011420157, 0.,
    0., 0., -0.6371334086, 0., 0., 0.6371334086]).reshape((6, 6))
_hess_scf_bh3p_dz_pk_uhf = np.array([
    0.0051661616, 0., 0., -0.0059617366, 0., 0., 0.0003977875, 0., 0., 0.0003977875, 0., 0.,
    0., 0.005028448, 0., 0., -0.0057802433, 0., 0., 0.0003758976, 0.0004726863, 0., 0.0003758976, -0.0004726863,
    0., 0., 0.2162990046, 0., 0., -0.2130428571, 0., 0.0001869025, -0.0016280737, 0., -0.0001869025, -0.0016280737,
    -0.0059617366, 0., 0., 0.0063439881, 0., 0., -0.0001911257, 0., 0., -0.0001911257, 0., 0.,
    0., -0.0057802433, 0., 0., 0.0062399336, 0., 0., -0.0002298451, 0.0001287182, 0., -0.0002298451, -0.0001287182,
    0., 0., -0.2130428571, 0., 0., 0.213169144, 0., 0.000042202, -0.0000631434, 0., -0.000042202, -0.0000631434,
    0.0003977875, 0., 0., -0.0001911257, 0., 0., -0.0035717661, 0., 0., 0.0033651042, 0., 0.,
    0., 0.0003758976, 0.0001869025, 0., -0.0002298451, 0.000042202, 0., 0.4101678639, -0.0004152545, 0., -0.4103139164, 0.00018615,
    0., 0.0004726863, -0.0016280737, 0., 0.0001287182, -0.0000631434, 0., -0.0004152545, -0.0000642125, 0., -0.00018615, 0.0017554296,
    0.0003977875, 0., 0., -0.0001911257, 0., 0., 0.0033651042, 0., 0., -0.0035717661, 0., 0.,
    0., 0.0003758976, -0.0001869025, 0., -0.0002298451, -0.000042202, 0., -0.4103139164, -0.00018615, 0., 0.4101678639, 0.0004152545,
    0., -0.0004726863, -0.0016280737, 0., -0.0001287182, -0.0000631434, 0., 0.00018615, 0.0017554296, 0., 0.0004152545, -0.0000642125]).reshape((12, 12))
_hess_scf_bh3p_dz_pk_rohf = np.array([
    0.0065592076, 0., 0., -0.0074218605, 0., 0., 0.0004313264, 0., 0., 0.0004313264, 0., 0.,
    0., 0.006418851, 0., 0., -0.0072356386, 0., 0., 0.0004083938, 0.0004886847, 0., 0.0004083938, -0.0004886847,
    0., 0., 0.2263725717, 0., 0., -0.2227777355, 0., -0.0000454089, -0.0017974181, 0., 0.0000454089, -0.0017974181,
    -0.0074218605, 0., 0., 0.0078274642, 0., 0., -0.0002028018, 0., 0., -0.0002028018, 0., 0.,
    0., -0.0072356386, 0., 0., 0.007720208, 0., 0., -0.0002422847, 0.0001293422, 0., -0.0002422847, -0.0001293422,
    0., 0., -0.2227777355, 0., 0., 0.2229077847, 0., -0.0000149119, -0.0000650246, 0., 0.0000149119, -0.0000650246,
    0.0004313264, 0., 0., -0.0002028018, 0., 0., -0.003466217, 0., 0., 0.0032376924, 0., 0.,
    0., 0.0004083938, -0.0000454089, 0., -0.0002422847, -0.0000149119, 0., 0.4104056572, -0.0002788531, 0., -0.4105717664, 0.0003391738,
    0., 0.0004886847, -0.0017974181, 0., 0.0001293422, -0.0000650246, 0., -0.0002788531, 0.0002037173, 0., -0.0003391738, 0.0016587254,
    0.0004313264, 0., 0., -0.0002028018, 0., 0., 0.0032376924, 0., 0., -0.003466217, 0., 0.,
    0., 0.0004083938, 0.0000454089, 0., -0.0002422847, 0.0000149119, 0., -0.4105717664, -0.0003391738, 0., 0.4104056572, 0.0002788531,
    0., -0.0004886847, -0.0017974181, 0., -0.0001293422, -0.0000650246, 0., 0.0003391738, 0.0016587254, 0., 0.0002788531, 0.0002037173]).reshape((12, 12))

_hess_scf_hf_dz_df_rhf = np.array([
                         0.011419589945, -0.            ,  0.            , -0.011419589946,  0.            , -0.            ,
                        -0.            ,  0.011419589946, -0.            ,  0.            , -0.011419589946,  0.            ,
                         0.            , -0.            ,  0.637098996637, -0.            ,  0.            , -0.637098996645,
                        -0.011419589946,  0.            , -0.            ,  0.011419589947, -0.            ,  0.            ,
                         0.            , -0.011419589946,  0.            , -0.            ,  0.011419589946, -0.            ,
                        -0.            ,  0.            , -0.637098996645,  0.            , -0.            ,  0.63709899664 ]).reshape((6, 6))
_hess_scf_bh3p_dz_df_uhf = np.array([
                         0.005157358561,  0.            , -0.            , -0.005955101752, -0.            ,  0.            ,  0.000398871596,
                         -0.            , -0.            ,  0.000398871596,  0.            ,  0.            ,
                         0.            ,  0.005015888751,  0.            , -0.            , -0.005772282862, -0.            ,  0.,
                          0.000378197056,  0.000465416538, -0.            ,  0.000378197056, -0.000465416538,
                        -0.            ,  0.            ,  0.216295663219,  0.            , -0.            , -0.213040064004, -0.,
                          0.000185641893, -0.001627799608,  0.            , -0.000185641893, -0.001627799609,
                        -0.005955101752, -0.            ,  0.            ,  0.006338370045,  0.            , -0.            , -0.000191634146,
                          0.            ,  0.            , -0.000191634146, -0.            , -0.            ,
                        -0.            , -0.005772282862, -0.            ,  0.            ,  0.006233942057,  0.            , -0.,
                         -0.000230829597,  0.000131807767,  0.            , -0.000230829598, -0.000131807767,
                         0.            , -0.            , -0.213040064004, -0.            ,  0.            ,  0.213165507196,  0.,
                          0.000042542535, -0.000062721596, -0.            , -0.000042542535, -0.000062721596,
                         0.000398871596,  0.            , -0.            , -0.000191634146, -0.            ,  0.            , -0.003568868648,
                          0.            ,  0.            ,  0.003361631199, -0.            , -0.            ,
                        -0.            ,  0.000378197056,  0.000185641893,  0.            , -0.000230829597,  0.000042542535,  0.,
                          0.410158196459, -0.000412704367, -0.            , -0.410305563917,  0.000184519939,
                        -0.            ,  0.000465416538, -0.001627799608,  0.            ,  0.000131807767, -0.000062721596,  0.,
                         -0.000412704367, -0.000072529233, -0.            , -0.000184519939,  0.001763050438,
                         0.000398871596, -0.            ,  0.            , -0.000191634146,  0.            , -0.            ,  0.003361631199,
                         -0.            , -0.            , -0.003568868648,  0.            ,  0.            ,
                         0.            ,  0.000378197056, -0.000185641893, -0.            , -0.000230829598, -0.000042542535, -0.,
                         -0.410305563917, -0.000184519939,  0.            ,  0.410158196459,  0.000412704367,
                         0.            , -0.000465416538, -0.001627799609, -0.            , -0.000131807767, -0.000062721596, -0.,
                          0.000184519939,  0.001763050438,  0.            ,  0.000412704367, -0.000072529233]).reshape((12, 12))
_hess_scf_bh3p_dz_df_rohf = np.array([
                        0.006549722571,  0.            ,  0.            , -0.007414731669,  0.            , 0.            ,  0.000432504549,
                        0.            ,  0.            ,  0.000432504549,  0.            ,  0.            ,
                        0.            ,  0.006404685466,  0.            ,  0.            , -0.007227207838, 0.            ,  0.,
                        0.000411261186,  0.000481228539,  0.            ,  0.000411261186, -0.000481228539,
                        0.            ,  0.            ,  0.226368773958,  0.            ,  0.            ,-0.222774139159,  0.,
                       -0.000047153809, -0.001797317399,  0.            ,  0.000047153809, -0.001797317399,
                       -0.007414731669,  0.            ,  0.            ,  0.007821444417,  0.            , 0.            , -0.000203356374,
                        0.            ,  0.            , -0.000203356374,  0.            ,  0.            ,
                        0.            , -0.007227207838,  0.            ,  0.            ,  0.007714708106, 0.            ,  0.,
                       -0.000243750134,  0.00013251408 ,  0.            , -0.000243750134, -0.00013251408 ,
                        0.            ,  0.            , -0.222774139159,  0.            ,  0.            , 0.222903015477,  0.,
                       -0.000014309509, -0.000064438159,  0.            ,  0.000014309509, -0.000064438159,
                        0.000432504549,  0.            ,  0.            , -0.000203356374,  0.            , 0.            , -0.00346312684,
                        0.            ,  0.            ,  0.003233978665,  0.            ,  0.            ,
                        0.            ,  0.000411261186, -0.000047153809,  0.            , -0.000243750134,-0.000014309509,  0.,
                        0.410395982746, -0.00027613965 ,  0.            , -0.410563493798,  0.000337602969,
                        0.            ,  0.000481228539, -0.001797317399,  0.            ,  0.00013251408 ,-0.000064438159,  0.,
                       -0.00027613965 ,  0.000195370438,  0.            , -0.000337602969,  0.00166638512 ,
                        0.000432504549,  0.            ,  0.            , -0.000203356374,  0.            , 0.            ,  0.003233978665,
                        0.            ,  0.            , -0.00346312684 ,  0.            ,  0.            ,
                        0.            ,  0.000411261186,  0.000047153809,  0.            , -0.000243750134, 0.000014309509,  0.,
                       -0.410563493798, -0.000337602969,  0.            ,  0.410395982746,  0.00027613965 ,
                        0.            , -0.000481228539, -0.001797317399,  0.            , -0.00013251408 ,-0.000064438159,  0.,
                        0.000337602969,  0.00166638512 ,  0.            ,  0.00027613965 ,  0.000195370438]).reshape((12, 12))

_hess_scf_hf_dz_cd_rhf = np.zeros(36).reshape((6, 6))
_hess_scf_bh3p_dz_cd_uhf = np.zeros(144).reshape((12, 12))
_hess_scf_bh3p_dz_cd_rohf = np.zeros(144).reshape((12, 12))


_hess_scf_h2o_qz2p_pk_rhf = np.array([
    0.0238430708, 0., 0., -0.0119215354, 0., 0., -0.0119215354, 0., 0.,
    0., 0.7093110919, 0., 0., -0.3546555459, 0.2653730246, 0., -0.3546555459, -0.2653730246,
    0., 0., 0.4985784094, 0., 0.2027912096, -0.2492892047, 0., -0.2027912096, -0.2492892047,
    -0.0119215354, 0., 0., 0.0108311859, 0., 0., 0.0010903495, 0., 0.,
    0., -0.3546555459, 0.2027912096, 0., 0.3886078743, -0.2340821171, 0., -0.0339523283, 0.0312909075,
    0., 0.2653730246, -0.2492892047, 0., -0.2340821171, 0.2322518953, 0., -0.0312909075, 0.0170373094,
    -0.0119215354, 0., 0., 0.0010903495, 0., 0., 0.0108311859, 0., 0.,
    0., -0.3546555459, -0.2027912096, 0., -0.0339523283, -0.0312909075, 0., 0.3886078743, 0.2340821171,
    0., -0.2653730246, -0.2492892047, 0., 0.0312909075, 0.0170373094, 0., 0.2340821171, 0.2322518953]).reshape((9, 9))
_hess_scf_nh2_qz2p_pk_uhf=np.array([
    -0.0013373967, 0., 0., 0.0006686983, 0., 0., 0.0006686983, 0., 0.,
    0., 0.6034171677, 0., 0., -0.3017085838, 0.2320222491, 0., -0.3017085838, -0.2320222491,
    0., 0., 0.4139303729, 0., 0.1753720372, -0.2069651864, 0., -0.1753720372, -0.2069651864,
    0.0006686983, 0., 0., -0.000364922, 0., 0., -0.0003037763, 0., 0.,
    0., -0.3017085838, 0.1753720372, 0., 0.3327449782, -0.2036971432, 0., -0.0310363944, 0.0283251059,
    0., 0.2320222491, -0.2069651864, 0., -0.2036971432, 0.1924704871, 0., -0.0283251059, 0.0144946993,
    0.0006686983, 0., 0., -0.0003037763, 0., 0., -0.000364922, 0., 0.,
    0., -0.3017085838, -0.1753720372, 0., -0.0310363944, -0.0283251059, 0., 0.3327449782, 0.2036971432,
    0., -0.2320222491, -0.2069651864, 0., 0.0283251059, 0.0144946993, 0., 0.2036971432, 0.1924704871]).reshape((9, 9))
_hess_scf_nh2_qz2p_pk_rohf=np.array([
    -0.0005896607, 0., 0., 0.0002948303, 0., 0., 0.0002948303, 0., 0.,
    0., 0.6051456253, 0., 0., -0.3025728127, 0.2323985163, 0., -0.3025728127, -0.2323985163,
    0., 0., 0.4152409466, 0., 0.1756925779, -0.2076204733, 0., -0.1756925779, -0.2076204733,
    0.0002948303, 0., 0., 0.0000266223, 0., 0., -0.0003214527, 0., 0.,
    0., -0.3025728127, 0.1756925779, 0., 0.3336853184, -0.2040455471, 0., -0.0311125058, 0.0283529691,
    0., 0.2323985163, -0.2076204733, 0., -0.2040455471, 0.1931471008, 0., -0.0283529691, 0.0144733724,
    0.0002948303, 0., 0., -0.0003214527, 0., 0., 0.0000266223, 0., 0.,
    0., -0.3025728127, -0.1756925779, 0., -0.0311125058, -0.0283529691, 0., 0.3336853184, 0.2040455471,
    0., -0.2323985163, -0.2076204733, 0., 0.0283529691, 0.0144733724, 0., 0.2040455471, 0.1931471008]).reshape((9, 9))

_hess_scf_h2o_qz2p_df_rhf = np.array([
                        0.023869160976,  0.            ,  0.            , -0.011934580488, -0.            ,  0.            , -0.011934580489,
                       -0.            , -0.            ,
                        0.            ,  0.709278145474, -0.000000000002, -0.            , -0.35463907274 ,  0.265350165815, -0.,
                       -0.354639072734, -0.265350165813,
                        0.            , -0.000000000002,  0.498586860727,  0.            ,  0.202756793814, -0.249293430373, -0.,
                       -0.202756793806, -0.249293430373,
                       -0.011934580488, -0.            ,  0.            ,  0.010840285263,  0.            , -0.            ,  0.001094295225,
                        0.            ,  0.            ,
                       -0.            , -0.35463907274 ,  0.202756793814,  0.            ,  0.388608483063, -0.23405347982 , -0.,
                       -0.033969410325,  0.031296685995,
                        0.            ,  0.265350165815, -0.249293430373, -0.            , -0.23405347982 ,  0.232247735371, -0.,
                       -0.031296685997,  0.01704569501 ,
                       -0.011934580489, -0.            , -0.            ,  0.001094295225, -0.            , -0.            ,  0.010840285264,
                        0.            ,  0.            ,
                       -0.            , -0.354639072734, -0.202756793806,  0.            , -0.033969410325, -0.031296685997,  0.,
                        0.38860848306 ,  0.234053479815,
                       -0.            , -0.265350165813, -0.249293430373,  0.            ,  0.031296685995,  0.01704569501 ,  0.,
                        0.234053479815,  0.232247735369]).reshape((9, 9))
_hess_scf_nh2_qz2p_df_uhf = np.array([
                         -0.001321632112,  0.            , 0.            ,  0.000660816056,  0.            , -0.            ,  0.000660816056,
                         -0.            , -0.            ,
                          0.            ,  0.603376756297,-0.000000000003,  0.            , -0.301688378153,  0.232000689906, -0.,
                         -0.301688378145, -0.232000689902,
                          0.            , -0.000000000003, 0.413952361231, -0.            ,  0.17533315624 , -0.206976180617, -0.,
                         -0.17533315624 , -0.206976180614,
                          0.000660816056,  0.            ,-0.            , -0.000357558133, -0.            ,  0.            , -0.000303257923,
                          0.            ,  0.            ,
                          0.            , -0.301688378153, 0.17533315624 , -0.            ,  0.332733856055, -0.203666923072, -0.,
                         -0.031045477904,  0.028333766831,
                         -0.            ,  0.232000689906,-0.206976180617,  0.            , -0.203666923072,  0.192471131065,  0.,
                         -0.028333766832,  0.014505049551,
                          0.000660816056, -0.            ,-0.            , -0.000303257923, -0.            ,  0.            , -0.000357558133,
                          0.            ,  0.            ,
                         -0.            , -0.301688378145,-0.17533315624 ,  0.            , -0.031045477904, -0.028333766832,  0.,
                          0.33273385605 ,  0.203666923073,
                         -0.            , -0.232000689902, -0.206976180614,  0.           ,   0.028333766831,  0.014505049551,  0.,
                          0.203666923073,  0.192471131063]).reshape((9, 9))
_hess_scf_nh2_qz2p_df_rohf = np.array([
                       -0.000573597702,  0.            , 0.            ,  0.000286798851,  0.            ,  0.            ,  0.000286798851,
                        0.            ,  0.            ,
                        0.            ,  0.605105430726, 0.            ,  0.            , -0.302552715363,  0.232376931706,  0.,
                       -0.302552715363, -0.232376931706,
                        0.            ,  0.            , 0.415262555438,  0.            ,  0.175653363465, -0.207631277719,  0.,
                       -0.175653363465, -0.207631277719,
                        0.000286798851,  0.            , 0.            ,  0.00003414546 ,  0.            ,  0.            , -0.000320944311,
                        0.            ,  0.            ,
                        0.            , -0.302552715363, 0.175653363465,  0.            ,  0.333674638104, -0.204015147586,  0.,
                       -0.03112192274 ,  0.02836178412 ,
                        0.            ,  0.232376931706,-0.207631277719,  0.            , -0.204015147586,  0.193147728967,  0.,
                       -0.02836178412 ,  0.014483548752,
                        0.000286798851,  0.            , 0.            , -0.000320944311,  0.            ,  0.            ,  0.00003414546,
                        0.            ,  0.            ,
                        0.            , -0.302552715363,-0.175653363465,  0.            , -0.03112192274 , -0.02836178412 ,  0.,
                        0.333674638104,  0.204015147586,
                        0.            , -0.232376931706,-0.207631277719,  0.            ,  0.02836178412 ,  0.014483548752,  0.,
                        0.204015147586,  0.193147728967]).reshape((9, 9))

_hess_scf_h2o_qz2p_cd_rhf = np.zeros(81).reshape((9, 9))
_hess_scf_nh2_qz2p_cd_uhf = np.zeros(81).reshape((9, 9))
_hess_scf_nh2_qz2p_cd_rohf = np.zeros(81).reshape((9, 9))


_hess_scf_h2o_adz_pk_rhf = np.array([
    0.0193009116, 0., 0., -0.0096504557, 0., 0., -0.0096504557, 0., 0.,
    0., 0.7259570062, 0., 0., -0.362978503, 0.2735758043, 0., -0.362978503, -0.2735758043,
    0., 0., 0.507233142, 0., 0.2101340878, -0.253616571, 0., -0.2101340878, -0.253616571,
    -0.0096504557, 0., 0., 0.0087580503, 0., 0., 0.0008924054, 0., 0.,
    0., -0.362978503, 0.2101340878, 0., 0.3955011984, -0.2418549461, 0., -0.0325226952, 0.0317208583,
    0., 0.2735758043, -0.253616571, 0., -0.2418549461, 0.236653618, 0., -0.0317208583, 0.016962953,
    -0.0096504557, 0., 0., 0.0008924054, 0., 0., 0.0087580503, 0., 0.,
    0., -0.362978503, -0.2101340878, 0., -0.0325226952, -0.0317208583, 0., 0.3955011984, 0.2418549461,
    0., -0.2735758043, -0.253616571, 0., 0.0317208583, 0.016962953, 0., 0.2418549461, 0.236653618 ]).reshape((9, 9))
_hess_scf_nh2_adz_pk_uhf = np.array([
    -0.0073285241, 0., 0., 0.0036642621, 0., 0., 0.0036642621, 0., 0.,
    0., 0.6248676289, 0., 0., -0.3124338145, 0.2425505849, 0., -0.3124338145, -0.2425505849,
    0., 0., 0.4213971779, 0., 0.185122888, -0.210698589, 0., -0.185122888, -0.210698589,
    0.0036642621, 0., 0., -0.0033269688, 0., 0., -0.0003372933, 0., 0.,
    0., -0.3124338145, 0.185122888, 0., 0.3427670403, -0.2138367365, 0., -0.0303332259, 0.0287138485,
    0., 0.2425505849, -0.210698589, 0., -0.2138367365, 0.1969122616, 0., -0.0287138485, 0.0137863273,
    0.0036642621, 0., 0., -0.0003372933, 0., 0., -0.0033269688, 0., 0.,
    0., -0.3124338145, -0.185122888, 0., -0.0303332259, -0.0287138485, 0., 0.3427670403, 0.2138367365,
    0., -0.2425505849, -0.210698589, 0., 0.0287138485, 0.0137863273, 0., 0.2138367365, 0.1969122616]).reshape((9, 9))
_hess_scf_nh2_adz_pk_rohf = np.array([
    -0.006621982, 0., 0., 0.003310991, 0., 0., 0.003310991, 0., 0.,
    0., 0.6266642672, 0., 0., -0.3133321336, 0.2429688151, 0., -0.3133321336, -0.2429688151,
    0., 0., 0.4229135704, 0., 0.1853731231, -0.2114567853, 0., -0.1853731231, -0.2114567853,
    0.003310991, 0., 0., -0.0029315977, 0., 0., -0.0003793933, 0., 0.,
    0., -0.3133321336, 0.1853731231, 0., 0.3437839161, -0.2141709691, 0., -0.0304517825, 0.028797846,
    0., 0.2429688151, -0.2114567853, 0., -0.2141709691, 0.1976705549, 0., -0.028797846, 0.0137862303,
    0.003310991, 0., 0., -0.0003793933, 0., 0., -0.0029315977, 0., 0.,
    0., -0.3133321336, -0.1853731231, 0., -0.0304517825, -0.028797846, 0., 0.3437839161, 0.2141709691,
    0., -0.2429688151, -0.2114567853, 0., 0.028797846, 0.0137862303, 0., 0.2141709691, 0.1976705549]).reshape((9, 9))

_hess_scf_h2o_adz_df_rhf = np.array([
                        0.019310737009, -0.            , -0.            , -0.009655368505, -0.            ,  0.            , -0.009655368503,
                          0.            ,  0.            ,
                        -0.            ,  0.725875223511,  0.000000000003, -0.            , -0.362937611655,  0.273540335334,  0.,
                         -0.362937611676, -0.273540335273,
                        -0.            ,  0.000000000003,  0.507198979725,  0.            ,  0.210065178279, -0.253599489912,  0.,
                         -0.210065178221, -0.253599489856,
                        -0.009655368505, -0.            ,  0.            ,  0.008759121798,  0.            , -0.            ,  0.000896246713,
                         -0.            , -0.            ,
                        -0.            , -0.362937611655,  0.210065178279,  0.            ,  0.395481274657, -0.241802756803,  0.,
                         -0.032543663037,  0.031737578554,
                         0.            ,  0.273540335334, -0.253599489912, -0.            , -0.241802756803,  0.236629960127, -0.,
                         -0.031737578562,  0.016969529729,
                        -0.009655368503,  0.            ,  0.            ,  0.000896246713,  0.            , -0.            ,  0.008759121792,
                         -0.            , -0.            ,
                         0.            , -0.362937611676, -0.210065178221, -0.            , -0.032543663037, -0.031737578562, -0.,
                          0.395481274584,  0.241802756768,
                         0.            , -0.273540335273, -0.253599489856, -0.            ,  0.031737578554,  0.016969529729, -0.,
                          0.241802756768,  0.236629960119]).reshape((9, 9))
_hess_scf_nh2_adz_df_uhf = np.array([
                       -0.007317259569,  0.            , 0.            ,  0.00365862978 ,  0.            , -0.            ,  0.003658629786,
                         -0.            , -0.            ,
                         0.            ,  0.62484772987 , -0.000000000011,  0.            , -0.312423864907,  0.242538620551, -0.,
                         -0.312423864936, -0.242538620553,
                         0.            , -0.000000000011, 0.421394753004, -0.            ,  0.185083431455, -0.210697376525, -0.,
                         -0.185083431454, -0.210697376488,
                         0.00365862978 ,  0.            ,-0.            , -0.003324468914, -0.            ,  0.            , -0.000334160866,
                          0.            ,  0.            ,
                         0.            , -0.312423864907, 0.185083431455, -0.            ,  0.342764539209, -0.21381102597 , -0.,
                         -0.030340674329,  0.028727594562,
                        -0.            ,  0.242538620551,-0.210697376525,  0.            , -0.21381102597 ,  0.196906745018,  0.,
                         -0.02872759456 ,  0.013790631506,
                         0.003658629786, -0.            ,-0.            , -0.000334160866, -0.            ,  0.            , -0.003324468915,
                          0.            ,  0.            ,
                        -0.            , -0.312423864936,-0.185083431454,  0.            , -0.030340674329, -0.02872759456 ,  0.,
                          0.342764539113,  0.213811026012,
                        -0.            , -0.242538620553,-0.210697376488,  0.            ,  0.028727594562,  0.013790631506,  0.,
                          0.213811026012,  0.196906744992]).reshape((9, 9))
_hess_scf_nh2_adz_df_rohf = np.array([
                       -0.006610027359,  0.            ,  0.            ,  0.003305013679,  0.            ,  0.            ,  0.003305013679, 0.            ,  0.            ,
                        0.            ,  0.626646193304,  0.            ,  0.            , -0.313323096652,  0.242957299808,  0., -0.313323096652, -0.242957299808,
                        0.            ,  0.            ,  0.422913366944,  0.            ,  0.185333436236, -0.211456683472,  0., -0.185333436236, -0.211456683472,
                        0.003305013679,  0.            ,  0.            , -0.002928829428,  0.            ,  0.            , -0.000376184252, 0.            ,  0.            ,
                        0.            , -0.313323096652,  0.185333436236,  0.            ,  0.343783148838, -0.214145368022,  0., -0.030460052185,  0.028811931786,
                        0.            ,  0.242957299808, -0.211456683472,  0.            , -0.214145368022,  0.197665865008,  0., -0.028811931786,  0.013790818464,
                        0.003305013679,  0.            ,  0.            , -0.000376184252,  0.            ,  0.            , -0.002928829428, 0.            ,  0.            ,
                        0.            , -0.313323096652, -0.185333436236,  0.            , -0.030460052185, -0.028811931786,  0., 0.343783148838,  0.214145368022,
                        0.            , -0.242957299808, -0.211456683472,  0.            ,  0.028811931786,  0.013790818464,  0., 0.214145368022,  0.197665865008]).reshape((9, 9))

_hess_scf_h2o_adz_cd_rhf = np.zeros(81).reshape((9, 9))
_hess_scf_nh2_adz_cd_uhf = np.zeros(81).reshape((9, 9))
_hess_scf_nh2_adz_cd_rohf = np.zeros(81).reshape((9, 9))


# DFT FUNCTIONAL portion of B2PLYP from psi 99,590
_b2plyp_hf_dz_pk_rhf = -100.3005792821600437
_b2plyp_bh3p_dz_pk_uhf =  -26.0528813461710449
_b2plyp_bh3p_dz_pk_rohf = 0.0

_b2plyp_hf_dz_df_rhf = -100.3005907504988699
_b2plyp_bh3p_dz_df_uhf = -26.0528839219733257
_b2plyp_bh3p_dz_df_rohf = 0.0

_b2plyp_hf_dz_cd_rhf = -100.3006117916440019
_b2plyp_bh3p_dz_cd_uhf = -26.0528827170103980
_b2plyp_bh3p_dz_cd_rohf = 0.0


_b2plyp_h2o_qz2p_pk_rhf = -76.3306993470827706
_b2plyp_nh2_qz2p_pk_uhf =  -55.7984427240855254
_b2plyp_nh2_qz2p_pk_rohf = 0.0

_b2plyp_h2o_qz2p_df_rhf = -76.3307137247329877
_b2plyp_nh2_qz2p_df_uhf = -55.7984551557867974
_b2plyp_nh2_qz2p_df_rohf = 0.0

_b2plyp_h2o_qz2p_cd_rhf = -76.3307381934528735
_b2plyp_nh2_qz2p_cd_uhf = -55.7984547742191594
_b2plyp_nh2_qz2p_cd_rohf = 0.0


_b2plyp_h2o_adz_pk_rhf = -76.3084021997237016
_b2plyp_nh2_adz_pk_uhf = -55.7829154325298688
_b2plyp_nh2_adz_pk_rohf = 0.0

_b2plyp_h2o_adz_df_rhf = -76.3084028180061580
_b2plyp_nh2_adz_df_uhf = -55.7829186059061328
_b2plyp_nh2_adz_df_rohf = 0.0

_b2plyp_h2o_adz_cd_rhf = -76.3083803442428916
_b2plyp_nh2_adz_cd_uhf = -55.7828904397309557
_b2plyp_nh2_adz_cd_rohf = 0.0
# fmt: on

# Rarely, reference values by conventional algorithms are _not_ available, while density-fitted or Cholesky-decomposed values _are_ available.
#   This deprives DF/CD of a sanity check versus CONV values, so:
#   (1) make every effort to hunt one down from other programs, reference implementations, etc.
#   (2) if an inferior but ballpark value is available (e.g., 3-pt findif gradient or value with a small correction (<1e-4) missing), include it with notes on caveats.
#   (3) if all else fails, add the QCVariable to _std_suite as `_knownmissing`
#       (a) CONV-AE-CONV and CONV-FC-CONV blocks only
#       (b) may have to adjust logic in compute_derived_qcvars
#       (c) only use if it'll show up in checks (e.g., DF checks SAME-SPIN for uhf but not rhf, so CONV only needs _knownmissing for uhf)
#   (4) revisit and add good CONV refs if opportunity arises.
#
#   Current `_knownmissing`s:
#       * "OLCCD SAME-SPIN CORRELATION ENERGY" in CONV-FC-CONV for uhf/rohf
#       * "OLCCD TOTAL GRADIENT" in CONV-FC-CONV
#       * "O(T) CORRECTION ENERGY in CONV-AE-CONV and CONV-FC-CONV
#       * "A-O(T) CORRECTION ENERGY" in CONV-AE-CONV and CONV-FC-CONV
#       * "OMP2/OMP2.5/OMP3/OREMP2 REFERENCE CORRECTION ENERGY" in CONV-FC-CONV for rhf/uhf/rohf
#       * "OMP2/OMP2.5/OMP3/OREMP2 CORRELATION ENERGY" in CONV-FC-CONV for rhf/uhf/rohf
#       * "OMP2/OMP2.5/OMP3/OREMP2 TOTAL GRADIENT" in CONV-FC-CONV for rhf/uhf/rohf
#       * "OMP2/OMP2.5/OMP3/OREMP2 SAME-SPIN CORRELATION ENERGY" in CONV-FC-CONV for uhf/rohf
_knownmissing = "KnownMissing"

_std_suite = [
    # <<<  CONV-AE-CONV  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.203781911950,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05427697023782003,
            "MP2 TOTAL GRADIENT": np.array(
                [  # fnocc findif-5 ae pk+conv
                    0.0000000000,
                    0.0000000000,
                    0.0028193375,
                    0.0000000000,
                    0.0000000000,
                    -0.0028193375,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [-0.00162697, 0.0, 0.0, 0.00162697, 0.0, 0.0],
                    [0.0, -0.00162697, 0.0, 0.0, 0.00162697, 0.0],
                    [0.0, 0.0, 0.64124876, 0.0, 0.0, -0.64124876],
                    [0.00162697, 0.0, 0.0, -0.00162697, 0.0, 0.0],
                    [0.0, 0.00162697, 0.0, 0.0, -0.00162697, 0.0],
                    [0.0, 0.0, -0.64124876, 0.0, 0.0, 0.64124876],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.20647996,  # fnocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04989037,  # fnocc
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000917944631, 0.0, 0.0, 0.000917944631]).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.20864512,  # vcc
            "MP4(T) CORRECTION ENERGY": -0.002048407683,  # vcc
            "CISD CORRELATION ENERGY": -0.20226573645347,  # vcc
            "QCISD CORRELATION ENERGY": -0.20892771089382,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00182210,  # vcc
            "FCI CORRELATION ENERGY": -0.21117389325,  # detci
            "REMP2 CORRELATION ENERGY": -0.20840575987435,  # occ, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.04954400816816,  # occ, tight
            "LCCD CORRELATION ENERGY": -0.2099060277,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.048339903547,  # fnocc
            "LCCD TOTAL GRADIENT": np.array([0.0, -0.0, 0.002203664242, -0.0, 0.0, -0.002203664242]).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2107436391,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.048460183760,  # fnocc
            "CEPA(1) CORRELATION ENERGY": -0.2083463452,  # fnocc
            "CEPA(1) SINGLES ENERGY": 0.0,
            "CEPA(1) SAME-SPIN CORRELATION ENERGY": -0.0480716539,  # fnocc
            "CEPA(3) CORRELATION ENERGY": -0.2067017395,  # fnocc
            "CEPA(3) SINGLES ENERGY": 0.0,
            "CEPA(3) SAME-SPIN CORRELATION ENERGY": -0.0477774156,  # fnocc
            "ACPF CORRELATION ENERGY": -0.2089157318,  # fnocc
            "ACPF SINGLES ENERGY": 0.0,
            "ACPF SAME-SPIN CORRELATION ENERGY": -0.0481691534,  # fnocc
            "AQCC CORRELATION ENERGY": -0.2073512040,  # fnocc
            "AQCC SINGLES ENERGY": 0.0,
            "AQCC SAME-SPIN CORRELATION ENERGY": -0.0479164828,  # fnocc
            "CCD CORRELATION ENERGY": -0.20815176752849,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.04851867,  # vcc
            "CCD TOTAL GRADIENT": np.array([[0.0, 0.0, 0.0008368526], [0.0, 0.0, -0.0008368526]]),  # vcc
            "CCD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-4.82892500e-04, 0.00000000e00, 0.00000000e00, 4.82892500e-04, 0.00000000e00, 0.00000000e00],
                    [0.00000000e00, -4.82892500e-04, 0.00000000e00, 0.00000000e00, 4.82892500e-04, 0.00000000e00],
                    [0.00000000e00, 0.00000000e00, 6.40299636e-01, 0.00000000e00, 0.00000000e00, -6.40299636e-01],
                    [4.82892500e-04, 0.00000000e00, 0.00000000e00, -4.82892500e-04, 0.00000000e00, 0.00000000e00],
                    [0.00000000e00, 4.82892500e-04, 0.00000000e00, 0.00000000e00, -4.82892500e-04, 0.00000000e00],
                    [0.00000000e00, 0.00000000e00, -6.40299636e-01, 0.00000000e00, 0.00000000e00, 6.40299636e-01],
                ]
            ),
            "BCCD CORRELATION ENERGY": -0.20860360,  # ccenergy
            "CC2 CORRELATION ENERGY": -0.204522923018627,  # ccenergy
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.004123981053, 0.0, 0.0, -0.004123981053]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.208743643,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04857419039,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.001989217717,
                    0.0,
                    0.0,
                    -0.001989217717,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.00114793, 0.0, 0.0, 0.00114793, 0.0, 0.0],
                    [0.0, -0.00114793, 0.0, 0.0, 0.00114793, 0.0],
                    [0.0, 0.0, 0.6378403, 0.0, 0.0, -0.6378403],
                    [0.00114793, 0.0, 0.0, -0.00114793, 0.0, 0.0],
                    [0.0, 0.00114793, 0.0, 0.0, -0.00114793, 0.0],
                    [0.0, 0.0, -0.6378403, 0.0, 0.0, 0.6378403],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.002058455537,  # vcc
            "(T) CORRECTION ENERGY": -0.0019363896542312043,
            "CCSD(T) TOTAL GRADIENT": np.array([[0.0, 0.0, 0.00308437284379], [0.0, 0.0, -0.00308437284379]]),  # ncc fd
            "CCSD(T) TOTAL HESSIAN": np.array(  # ncc fd
                [
                    [-0.00177990234, 0.0, 0.0, 0.00177990234, 0.0, 0.0],
                    [0.0, -0.00177990234, 0.0, 0.0, 0.00177990234, 0.0],
                    [0.0, 0.0, 0.636483940298, 0.0, 0.0, -0.636483940298],
                    [0.00177990234, 0.0, 0.0, -0.00177990234, 0.0, 0.0],
                    [0.0, 0.00177990234, 0.0, 0.0, -0.00177990234, 0.0],
                    [0.0, 0.0, -0.636483940298, 0.0, 0.0, 0.636483940298],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.001961066509136,  # ncc
            "A-CCSD(T) TOTAL GRADIENT": np.array(  # ncc
                [
                    [0.000000000000000, 0.000000000000000, 0.003071874425935],
                    [0.000000000000000, 0.000000000000000, -0.003071874425935],
                ]
            ),
            "B(T) CORRECTION ENERGY": -0.00208496,  # ccenergy
            "CC3 CORRELATION ENERGY": -0.210769121626813,  # ccenergy
            "CCSDT-1A CORRELATION ENERGY": -0.210768179766,  # ecc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[0.0, 0.0, 0.00316187041423], [0.0, 0.0, -0.00316187041423]]
            ),
            "CCSDT-1A TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.001824798014, 0.0, 0.0, 0.001824798014, 0.0, 0.0],
                    [0.0, -0.001824795113, 0.0, 0.0, 0.001824795113, 0.0],
                    [0.0, 0.0, 0.63641316654, 0.0, 0.0, -0.63641316654],
                    [0.001824798014, 0.0, 0.0, -0.001824798014, 0.0, 0.0],
                    [0.0, 0.001824795113, 0.0, 0.0, -0.001824795113, 0.0],
                    [0.0, 0.0, -0.63641316654, 0.0, 0.0, 0.63641316654],
                ]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.210771725787,  # ecc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[0.0, 0.0, 0.00316311457309], [0.0, 0.0, -0.00316311457309]]
            ),
            "CCSDT-1B TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.001825508871, 0.000000011606, 0.000000016683, 0.001825508871, -0.000000011606, -0.000000016683],
                    [0.000000011606, -0.001825503068, 0.000000007254, -0.000000011606, 0.001825503068, -0.000000007254],
                    [0.000000016683, 0.000000007254, 0.636412451331, -0.000000016683, -0.000000007254, -0.636412451331],
                    [0.001825508871, -0.000000011606, -0.000000016683, -0.001825508871, 0.000000011606, 0.000000016683],
                    [-0.000000011606, 0.001825503068, -0.000000007254, 0.000000011606, -0.001825503068, 0.000000007254],
                    [-0.000000016683, -0.000000007254, -0.636412451331, 0.000000016683, 0.000000007254, 0.636412451331],
                ]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.210638072036,  # ecc
            "CCSDT-2 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[-0.0, 0.0, 0.00310773649518], [0.0, 0.0, -0.00310773649518]]
            ),
            "CCSDT-2 TOTAL HESSIAN": np.array(  # ecc by psi fd
                [
                    [-0.001793373798, -0.000000001451, -0.000000029015, 0.001793373798, 0.000000001451, 0.000000029015],
                    [-0.000000001451, -0.001793373798, -0.000000029015, 0.000000001451, 0.001793373798, 0.000000029015],
                    [-0.000000029015, -0.000000029015, 0.636438446923, 0.000000029015, 0.000000029015, -0.636438446923],
                    [0.001793373798, 0.000000001451, 0.000000029015, -0.001793373798, -0.000000001451, -0.000000029015],
                    [0.000000001451, 0.001793373798, 0.000000029015, -0.000000001451, -0.001793373798, -0.000000029015],
                    [0.000000029015, 0.000000029015, -0.636438446923, -0.000000029015, -0.000000029015, 0.636438446923],
                ]
            ),
            "CCSDT-3 CORRELATION ENERGY": -0.21063680423440,  # vcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[0.0, 0.0, 0.00310517589721], [0.0, 0.0, -0.00310517589721]]
            ),
            "CCSDT-3 TOTAL HESSIAN": np.array(  # ecc
                [
                    [-0.00179205, 0.0, 0.0, 0.00179205, 0.0, 0.0],
                    [0.0, -0.00179205, 0.0, 0.0, 0.00179205, 0.0],
                    [0.0, 0.0, 0.63647126, 0.0, 0.0, -0.63647126],
                    [0.00179205, 0.0, 0.0, -0.00179205, 0.0, 0.0],
                    [0.0, 0.00179205, 0.0, 0.0, -0.00179205, 0.0],
                    [0.0, 0.0, -0.63647126, 0.0, 0.0, 0.63647126],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.210768503167476,  # ncc
            "CCSDT TOTAL GRADIENT": np.array(
                [
                    0.000000000000000,
                    0.000000000000000,
                    0.003174888578646,
                    0.000000000000000,
                    0.000000000000000,
                    -0.003174888578646,
                ]
            ).reshape((-1, 3)),
            "CCSDT TOTAL HESSIAN": np.array(  # ecc
                [
                    [-0.00183215, 0.0, 0.0, 0.00183215, 0.0, 0.0],
                    [0.0, -0.00183215, 0.0, 0.0, 0.00183215, 0.0],
                    [0.0, 0.0, 0.63633545, 0.0, 0.0, -0.63633545],
                    [0.00183215, 0.0, 0.0, -0.00183215, 0.0, 0.0],
                    [0.0, 0.00183215, 0.0, 0.0, -0.00183215, 0.0],
                    [0.0, 0.0, -0.63633545, 0.0, 0.0, 0.63633545],
                ]
            ),
            "[Q] CORRECTION ENERGY": -0.000380822242155,  # ncc
            "(Q) CORRECTION ENERGY": -0.000414550980182,  # ncc
            "CCSDT(Q) TOTAL GRADIENT": np.array(
                [  # ncc
                    0.000000000000000,
                    0.000000000000000,
                    0.003539304813171,
                    0.000000000000000,
                    0.000000000000000,
                    -0.00353930481317,
                ]
            ).reshape((-1, 3)),
            "CCSDTQ CORRELATION ENERGY": -0.211162705496338,  # ncc
            "CCSDTQ TOTAL GRADIENT": np.array(
                [
                    0.000000000000000,
                    0.000000000000000,
                    0.003507320893113,
                    0.000000000000000,
                    0.000000000000000,
                    -0.003507320893113,
                ]
            ).reshape((-1, 3)),
            "OMP2 REFERENCE CORRECTION ENERGY": 0.000704890964,  # occ, tight
            "OMP2 CORRELATION ENERGY": -0.204465719970,  # occ, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.054616313597,  # occ, tight
            "OMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [0.0, 0.0, 0.004107528173, 0.0, 0.0, -0.004107528173]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.000571728736,  # occ, tight
            "OMP2.5 CORRELATION ENERGY": -0.205686837008,  # occ, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.052301161413,  # occ, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # occ, tight
                [0.0, 0.000000000000, 0.002017001118, 0.0, 0.000000000000, -0.002017001118]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000467222482,  # occ, tight
            "OMP3 CORRELATION ENERGY": -0.206935131375,  # occ, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.050007707662,  # occ, tight
            "OMP3 TOTAL GRADIENT": np.array(  # occ, tight
                [0.0, 0.0, -0.000007413433, 0.0, 0.0, 0.000007413433]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.000553304064,  # occ, tight
            "OREMP2 CORRELATION ENERGY": -0.208942935871,  # occ, tight
            "OREMP2 SAME-SPIN CORRELATION ENERGY": -0.049683397802,  # occ, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [0.0, 0.000000000000, 0.002830089047, 0.0, 0.000000000000, -0.002830089047]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0005522939,  # p4n
            "OLCCD CORRELATION ENERGY": -0.2104417743,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0484443079,  # occ
            "OLCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.00339205449, 0.0, 0.0, -0.00339205449]).reshape(  # occ
                (-1, 3)
            ),
            "OCCD REFERENCE CORRECTION ENERGY": 0.000511277,  # qchem
            "OCCD CORRELATION ENERGY": -0.208649453,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem, rearranged
                [0.00000000, 0.00000000, 0.00184797, 0, 0, -0.00184797]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "SVWN TOTAL ENERGY": -99.97965143812243,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.019336350645, -0.0, 0.0, -0.019336301692]
            ).reshape((-1, 3)),
            "SVWN TOTAL HESSIAN": np.array(  # psi v1.8 99,590
                [
                    -0.011158529195,
                    -0.0,
                    -0.0,
                    0.011158509954,
                    -0.0,
                    -0.0,
                    -0.0,
                    -0.011158529195,
                    -0.0,
                    0.0,
                    0.011158509954,
                    0.0,
                    -0.0,
                    -0.0,
                    0.642213454497,
                    0.0,
                    0.0,
                    -0.642213457165,
                    0.011158509954,
                    0.0,
                    0.0,
                    -0.011155887562,
                    -0.0,
                    -0.0,
                    -0.0,
                    0.011158509954,
                    0.0,
                    -0.0,
                    -0.011155887564,
                    -0.0,
                    -0.0,
                    0.0,
                    -0.642213457165,
                    -0.0,
                    -0.0,
                    0.642216280292,
                ]
            ).reshape((6, 6)),
            "PBE TOTAL ENERGY": -100.33517315116806,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array([[0.0, 0.0, 0.020107103338], [0.0, 0.0, -0.020107128125]]),  # psi 99,590
            "B3LYP TOTAL ENERGY": -100.43544624005466,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [[-2.37578116e-16, -5.15211396e-17, 1.13731953e-02], [-5.23661679e-14, 5.41065792e-15, -1.13730866e-02]]
            ),
            "B3LYP5 TOTAL ENERGY": -100.39768534808519,  # psi 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # psi 99,590
                [[-6.94208991e-16, 1.87550833e-16, 1.16101107e-02], [3.49213371e-14, 3.53945060e-15, -1.16100020e-02]]
            ),
            "WB97X TOTAL ENERGY": -100.41398805061500,  # psi 500,1202
            "WB97X TOTAL GRADIENT": np.array(  # psi 500,1202, findif-5
                [0.0, 0.0, 0.006570841474, 0.0, 0.0, -0.006570658806]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_pk_rhf,
            "B2PLYP TOTAL ENERGY": -100.364597842761,  # nwc 99,590
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.2218977246,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05669988343022163,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.009624481085,
                    0.0,
                    0.005505796371,
                    -0.004812240542,
                    0.0,
                    -0.005505796371,
                    -0.004812240542,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [
                        -8.68375090e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.34187540e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.34187540e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.18925148e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.59462574e-01,
                        2.81687501e-01,
                        0.00000000e00,
                        -3.59462574e-01,
                        -2.81687501e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.67076274e-01,
                        0.00000000e00,
                        2.16194016e-01,
                        -2.33538137e-01,
                        0.00000000e00,
                        -2.16194016e-01,
                        -2.33538137e-01,
                    ],
                    [
                        4.34187540e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.09411690e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.47758500e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.59462574e-01,
                        2.16194016e-01,
                        0.00000000e00,
                        3.87745363e-01,
                        -2.48940759e-01,
                        0.00000000e00,
                        -2.82827890e-02,
                        3.27467430e-02,
                    ],
                    [
                        0.00000000e00,
                        2.81687501e-01,
                        -2.33538137e-01,
                        0.00000000e00,
                        -2.48940759e-01,
                        2.23898773e-01,
                        0.00000000e00,
                        -3.27467430e-02,
                        9.63936430e-03,
                    ],
                    [
                        4.34187540e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.47758500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -4.09411690e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.59462574e-01,
                        -2.16194016e-01,
                        0.00000000e00,
                        -2.82827890e-02,
                        -3.27467430e-02,
                        0.00000000e00,
                        3.87745363e-01,
                        2.48940759e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.81687501e-01,
                        -2.33538137e-01,
                        0.00000000e00,
                        3.27467430e-02,
                        9.63936430e-03,
                        0.00000000e00,
                        2.48940759e-01,
                        2.23898773e-01,
                    ],
                ]
            ),  # cfour
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.051851951707,  # fnocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.226431141430,  # fnocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.003997280108, 0.0, 0.00255475829, -0.001998640054, 0.0, -0.00255475829, -0.001998640054]
            ).reshape(-1, 3),
            "MP4(SDQ) CORRELATION ENERGY": -0.229617624405,  # vcc
            "MP4(T) CORRECTION ENERGY": -0.005637388222,  # vcc
            "CISD CORRELATION ENERGY": -0.21978965712829,  # vcc
            "QCISD CORRELATION ENERGY": -0.22998871354660,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.0048836279,  # vcc
            "REMP2 CORRELATION ENERGY": -0.229445317607,  # occ, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.051382139941,  # occ, tight
            "LCCD CORRELATION ENERGY": -0.2318870702,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.049937236558,  # fnocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.008553750217, -0.0, 0.005238229687, -0.004276875109, 0.0, -0.005238229687, -0.004276875109]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2341051403,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.050442387759,  # fnocc
            "CEPA(1) CORRELATION ENERGY": -0.2299219110,  # fnocc
            "CEPA(1) SINGLES ENERGY": 0.0,
            "CEPA(1) SAME-SPIN CORRELATION ENERGY": -0.0497747018,  # fnocc
            "CEPA(3) CORRELATION ENERGY": -0.227111702,  # fnocc
            "CEPA(3) SINGLES ENERGY": 0.0,
            "CEPA(3) SAME-SPIN CORRELATION ENERGY": -0.0492934875,  # fnocc
            "ACPF CORRELATION ENERGY": -0.2308608117,  # fnocc
            "ACPF SINGLES ENERGY": 0.0,
            "ACPF SAME-SPIN CORRELATION ENERGY": -0.0499379269,  # fnocc
            "AQCC CORRELATION ENERGY": -0.2281621411,  # fnocc
            "AQCC SINGLES ENERGY": 0.0,
            "AQCC SAME-SPIN CORRELATION ENERGY": -0.0495127294,  # fnocc
            "CCD CORRELATION ENERGY": -0.22802083552559,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.0499178,  # vcc
            "CCD TOTAL GRADIENT": np.array(
                [[0.0, 0.0, 0.00617203], [0.0, 0.00392094, -0.00308601], [0.0, -0.00392094, -0.00308601]]
            ),
            "CCD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.0055683485, 0.0, 0.0, 0.0027841742, 0.0, 0.0, 0.0027841742, 0.0, 0.0],
                    [0.0, 0.7149628559, 0.0, 0.0, -0.357481428, 0.2789479625, 0.0, -0.357481428, -0.2789479625],
                    [0.0, 0.0, 0.4751691519, 0.0, 0.215124084, -0.237584576, 0.0, -0.215124084, -0.237584576],
                    [0.0027841742, 0.0, 0.0, -0.0027615491, 0.0, 0.0, -0.0000226252, 0.0, 0.0],
                    [
                        0.0,
                        -0.357481428,
                        0.215124084,
                        0.0,
                        0.3876970703,
                        -0.2470360126,
                        0.0,
                        -0.0302156423,
                        0.0319119498,
                    ],
                    [
                        0.0,
                        0.2789479625,
                        -0.237584576,
                        0.0,
                        -0.2470360126,
                        0.2254150323,
                        0.0,
                        -0.0319119498,
                        0.0121695437,
                    ],
                    [0.0027841742, 0.0, 0.0, -0.0000226252, 0.0, 0.0, -0.0027615491, 0.0, 0.0],
                    [
                        0.0,
                        -0.357481428,
                        -0.215124084,
                        0.0,
                        -0.0302156423,
                        -0.0319119498,
                        0.0,
                        0.3876970703,
                        0.2470360126,
                    ],
                    [
                        0.0,
                        -0.2789479625,
                        -0.237584576,
                        0.0,
                        0.0319119498,
                        0.0121695437,
                        0.0,
                        0.2470360126,
                        0.2254150323,
                    ],
                ]
            ),
            "BCCD CORRELATION ENERGY": -0.228985653284813,  # ccenergy
            "CC2 CORRELATION ENERGY": -0.224173737389996,  # ccenergy
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.011903811069, 0.0, 0.00673003665, -0.005951905535, 0.0, -0.00673003665, -0.005951905535]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.2294105794,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.050177977945205,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.007512595487,
                    0.0,
                    0.004613769715,
                    -0.003756297743,
                    0.0,
                    -0.004613769715,
                    -0.003756297743,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        -6.77828820e-03,
                        0.00000000e00,
                        0.00000000e00,
                        3.38914410e-03,
                        0.00000000e00,
                        0.00000000e00,
                        3.38914410e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.12577187e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.56288594e-01,
                        2.78492260e-01,
                        0.00000000e00,
                        -3.56288594e-01,
                        -2.78492260e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.72789375e-01,
                        0.00000000e00,
                        2.14574581e-01,
                        -2.36394688e-01,
                        0.00000000e00,
                        -2.14574581e-01,
                        -2.36394688e-01,
                    ],
                    [
                        3.38914410e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -3.30616560e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -8.29785000e-05,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.56288594e-01,
                        2.14574581e-01,
                        0.00000000e00,
                        3.86544885e-01,
                        -2.46533421e-01,
                        0.00000000e00,
                        -3.02562911e-02,
                        3.19588394e-02,
                    ],
                    [
                        0.00000000e00,
                        2.78492260e-01,
                        -2.36394688e-01,
                        0.00000000e00,
                        -2.46533421e-01,
                        2.24401624e-01,
                        0.00000000e00,
                        -3.19588394e-02,
                        1.19930640e-02,
                    ],
                    [
                        3.38914410e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -8.29785000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        -3.30616560e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.56288594e-01,
                        -2.14574581e-01,
                        0.00000000e00,
                        -3.02562911e-02,
                        -3.19588394e-02,
                        0.00000000e00,
                        3.86544885e-01,
                        2.46533421e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.78492260e-01,
                        -2.36394688e-01,
                        0.00000000e00,
                        3.19588394e-02,
                        1.19930640e-02,
                        0.00000000e00,
                        2.46533421e-01,
                        2.24401624e-01,
                    ],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.005640585707,  # vcc
            "(T) CORRECTION ENERGY": -0.00523856,
            "CCSD(T) TOTAL GRADIENT": np.array(  # ncc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.01019219792989],
                    [0.00000000000000, 0.00604927147843, -0.00509609896494],
                    [0.00000000000000, -0.00604927147843, -0.00509609896494],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # ncc fd
                [
                    [
                        -9.19597780e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.59798890e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.59798890e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.10457601e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.55228801e-01,
                        2.78607515e-01,
                        0.00000000e00,
                        -3.55228801e-01,
                        -2.78607515e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.68590076e-01,
                        0.00000000e00,
                        2.14495554e-01,
                        -2.34295038e-01,
                        0.00000000e00,
                        -2.14495554e-01,
                        -2.34295038e-01,
                    ],
                    [
                        4.59798890e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.41199179e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.85997115e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.55228801e-01,
                        2.14495554e-01,
                        0.00000000e00,
                        3.85055647e-01,
                        -2.46551535e-01,
                        0.00000000e00,
                        -2.98268463e-02,
                        3.20559807e-02,
                    ],
                    [
                        0.00000000e00,
                        2.78607515e-01,
                        -2.34295038e-01,
                        0.00000000e00,
                        -2.46551535e-01,
                        2.22895271e-01,
                        0.00000000e00,
                        -3.20559807e-02,
                        1.13997664e-02,
                    ],
                    [
                        4.59798890e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.85997115e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -4.41199179e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.55228801e-01,
                        -2.14495554e-01,
                        0.00000000e00,
                        -2.98268463e-02,
                        -3.20559807e-02,
                        0.00000000e00,
                        3.85055647e-01,
                        2.46551535e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.78607515e-01,
                        -2.34295038e-01,
                        0.00000000e00,
                        3.20559807e-02,
                        1.13997664e-02,
                        0.00000000e00,
                        2.46551535e-01,
                        2.22895271e-01,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.005236187785,  # ecc
            "A-CCSD(T) TOTAL GRADIENT": np.array(  # ncc
                [
                    [0.000000000000000, 0.000000000000000, 0.010120884870498],
                    [0.000000000000000, 0.006023996226962, -0.005060442435251],
                    [0.000000000000000, -0.006023996226962, -0.005060442435251],
                ]
            ),
            "B(T) CORRECTION ENERGY": -0.005680766924884,  # ccenergy
            "CC3 CORRELATION ENERGY": -0.235030687907142,  # ccenergy
            "CCSDT-1A CORRELATION ENERGY": -0.23500915078065532,  # mrcc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01040697979689],
                    [0.00000000000000, 0.00619487901010, -0.00520348989845],
                    [0.00000000000000, -0.00619487901010, -0.00520348989845],
                ]
            ),
            "CCSDT-1A TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.00938924, 0.0, 0.0, 0.00469462, 0.0, 0.0, 0.00469462, 0.0, 0.0],
                    [0.0, 0.71035902, 0.0, 0.0, -0.35517951, 0.27864419, 0.0, -0.35517951, -0.27864419],
                    [0.0, 0.0, 0.46816235, 0.0, 0.21453598, -0.23408118, 0.0, -0.21453598, -0.23408118],
                    [0.00469462, 0.0, 0.0, -0.00451106, 0.0, 0.0, -0.00018357, 0.0, 0.0],
                    [0.0, -0.35517951, 0.21453598, 0.0, 0.38493736, -0.24659008, 0.0, -0.02975785, 0.0320541],
                    [0.0, 0.27864419, -0.23408118, 0.0, -0.24659008, 0.2227516, 0.0, -0.0320541, 0.01132957],
                    [0.00469462, 0.0, 0.0, -0.00018357, 0.0, 0.0, -0.00451106, 0.0, 0.0],
                    [0.0, -0.35517951, -0.21453598, 0.0, -0.02975785, -0.0320541, 0.0, 0.38493736, 0.24659008],
                    [0.0, -0.27864419, -0.23408118, 0.0, 0.0320541, 0.01132957, 0.0, 0.24659008, 0.2227516],
                ]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.23501318627899082,  # mrcc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01040244518005],
                    [0.00000000000000, 0.00619405837854, -0.00520122259002],
                    [0.00000000000000, -0.00619405837854, -0.00520122259002],
                ]
            ),
            "CCSDT-1B TOTAL HESSIAN": np.array(  # ecc by psi fd
                [
                    [-0.009385784958, 0.0, 0.0, 0.004692892479, 0.0, 0.0, 0.004692892479, 0.0, 0.0],
                    [
                        0.0,
                        0.710344410436,
                        0.0,
                        0.0,
                        -0.355172205218,
                        0.278637075614,
                        0.0,
                        -0.355172205218,
                        -0.278637075614,
                    ],
                    [
                        0.0,
                        0.0,
                        0.468154078949,
                        0.0,
                        0.214541783785,
                        -0.234077039475,
                        0.0,
                        -0.214541783785,
                        -0.234077039475,
                    ],
                    [0.004692892479, 0.0, 0.0, -0.004510041295, 0.0, 0.0, -0.000182851184, 0.0, 0.0],
                    [
                        0.0,
                        -0.355172205218,
                        0.214541783785,
                        0.0,
                        0.384930286936,
                        -0.2465894297,
                        0.0,
                        -0.029758081719,
                        0.032047645914,
                    ],
                    [
                        0.0,
                        0.278637075614,
                        -0.234077039475,
                        0.0,
                        -0.2465894297,
                        0.222747174147,
                        0.0,
                        -0.032047645914,
                        0.011329865328,
                    ],
                    [0.004692892479, 0.0, 0.0, -0.000182851184, 0.0, 0.0, -0.004510041295, 0.0, 0.0],
                    [
                        0.0,
                        -0.355172205218,
                        -0.214541783785,
                        0.0,
                        -0.029758081719,
                        -0.032047645914,
                        0.0,
                        0.384930286936,
                        0.2465894297,
                    ],
                    [
                        0.0,
                        -0.278637075614,
                        -0.234077039475,
                        0.0,
                        0.032047645914,
                        0.011329865328,
                        0.0,
                        0.2465894297,
                        0.222747174147,
                    ],
                ]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.234514404732,  # ecc
            "CCSDT-2 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01007036070231],
                    [0.00000000000000, 0.00600051212420, -0.00503518035116],
                    [0.00000000000000, -0.00600051212420, -0.00503518035116],
                ]
            ),
            "CCSDT-2 TOTAL HESSIAN": np.array(  # ecc by psi fd
                [
                    [-0.009086018755, 0.0, 0.0, 0.004543009378, 0.0, 0.0, 0.004543009378, 0.0, 0.0],
                    [
                        0.0,
                        0.710438905643,
                        0.0,
                        0.0,
                        -0.355219452821,
                        0.278557716256,
                        0.0,
                        -0.355219452821,
                        -0.278557716256,
                    ],
                    [
                        0.0,
                        0.0,
                        0.468711648157,
                        0.0,
                        0.214551521677,
                        -0.234355824078,
                        0.0,
                        -0.214551521677,
                        -0.234355824078,
                    ],
                    [0.004543009378, 0.0, 0.0, -0.004367495061, 0.0, 0.0, -0.000175514317, 0.0, 0.0],
                    [
                        0.0,
                        -0.355219452821,
                        0.214551521677,
                        0.0,
                        0.385072138348,
                        -0.246554618967,
                        0.0,
                        -0.029852685527,
                        0.032003097289,
                    ],
                    [
                        0.0,
                        0.278557716256,
                        -0.234355824078,
                        0.0,
                        -0.246554618967,
                        0.222923456244,
                        0.0,
                        -0.032003097289,
                        0.011432367835,
                    ],
                    [0.004543009378, 0.0, 0.0, -0.000175514317, 0.0, 0.0, -0.004367495061, 0.0, 0.0],
                    [
                        0.0,
                        -0.355219452821,
                        -0.214551521677,
                        0.0,
                        -0.029852685527,
                        -0.032003097289,
                        0.0,
                        0.385072138348,
                        0.246554618967,
                    ],
                    [
                        0.0,
                        -0.278557716256,
                        -0.234355824078,
                        0.0,
                        0.032003097289,
                        0.011432367835,
                        0.0,
                        0.246554618967,
                        0.222923456244,
                    ],
                ]
            ),
            "CCSDT-3 CORRELATION ENERGY": -0.2345361459852029,  # mrcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01009389495078],
                    [0.00000000000000, 0.00601318997564, -0.00504694747539],
                    [0.00000000000000, -0.00601318997564, -0.00504694747539],
                ]
            ),
            "CCSDT-3 TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.00910672, 0.0, 0.0, 0.00455336, 0.0, 0.0, 0.00455336, 0.0, 0.0],
                    [0.0, 0.71040265, 0.0, 0.0, -0.35520132, 0.27855168, 0.0, -0.35520132, -0.27855168],
                    [0.0, 0.0, 0.46867889, 0.0, 0.21454552, -0.23433945, 0.0, -0.21454552, -0.23433945],
                    [0.00455336, 0.0, 0.0, -0.00437695, 0.0, 0.0, -0.00017641, 0.0, 0.0],
                    [0.0, -0.35520132, 0.21454552, 0.0, 0.38506038, -0.2465486, 0.0, -0.02985906, 0.03200308],
                    [0.0, 0.27855168, -0.23433945, 0.0, -0.2465486, 0.22290841, 0.0, -0.03200308, 0.01143103],
                    [0.00455336, 0.0, 0.0, -0.00017641, 0.0, 0.0, -0.00437695, 0.0, 0.0],
                    [0.0, -0.35520132, -0.21454552, 0.0, -0.02985906, -0.03200308, 0.0, 0.38506038, 0.2465486],
                    [0.0, -0.27855168, -0.23433945, 0.0, 0.03200308, 0.01143103, 0.0, 0.2465486, 0.22290841],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.234880733453953,  # ncc
            "CCSDT TOTAL GRADIENT": np.array(
                [
                    0.000000000000000,
                    0.000000000000000,
                    0.010281901133138,
                    0.000000000000000,
                    0.006116030816197,
                    -0.005140950566570,
                    0.000000000000000,
                    -0.006116030816197,
                    -0.005140950566570,
                ]
            ).reshape((-1, 3)),
            "CCSDT TOTAL HESSIAN": np.array(  # ecc
                [
                    [
                        -9.27691040e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.63845520e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.63845520e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.10153874e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.55076937e-01,
                        2.78521415e-01,
                        0.00000000e00,
                        -3.55076937e-01,
                        -2.78521415e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.68513289e-01,
                        0.00000000e00,
                        2.14488040e-01,
                        -2.34256644e-01,
                        0.00000000e00,
                        -2.14488040e-01,
                        -2.34256644e-01,
                    ],
                    [
                        4.63845520e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.45556240e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.82892800e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.55076937e-01,
                        2.14488040e-01,
                        0.00000000e00,
                        3.84940871e-01,
                        -2.46504728e-01,
                        0.00000000e00,
                        -2.98639346e-02,
                        3.20166873e-02,
                    ],
                    [
                        0.00000000e00,
                        2.78521415e-01,
                        -2.34256644e-01,
                        0.00000000e00,
                        -2.46504728e-01,
                        2.22819148e-01,
                        0.00000000e00,
                        -3.20166873e-02,
                        1.14374966e-02,
                    ],
                    [
                        4.63845520e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.82892800e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -4.45556240e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.55076937e-01,
                        -2.14488040e-01,
                        0.00000000e00,
                        -2.98639346e-02,
                        -3.20166873e-02,
                        0.00000000e00,
                        3.84940871e-01,
                        2.46504728e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.78521415e-01,
                        -2.34256644e-01,
                        0.00000000e00,
                        3.20166873e-02,
                        1.14374966e-02,
                        0.00000000e00,
                        2.46504728e-01,
                        2.22819148e-01,
                    ],
                ]
            ),
            "[Q] CORRECTION ENERGY": -0.000291847368718,  # ncc
            "(Q) CORRECTION ENERGY": -0.000507547985814,  # ncc
            "CCSDT(Q) TOTAL GRADIENT": np.array(
                [  # ncc
                    0.000000000000000,
                    0.000000000000000,
                    0.010822433278893,
                    0.000000000000000,
                    0.006408928209897,
                    -0.005411216639448,
                    0.000000000000000,
                    -0.006408928209897,
                    -0.005411216639448,
                ]
            ).reshape((-1, 3)),
            "CCSDTQ CORRELATION ENERGY": -0.235338854850175,  # ncc
            "OMP2 REFERENCE CORRECTION ENERGY": 0.002352035679,  # occ, tight
            "OMP2 CORRELATION ENERGY": -0.224125328536,  # occ, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.058141813805,  # occ, tight
            "OMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [0.0, 0.0, 0.012093662469, 0.0, 0.006798848063, -0.006046831235, 0.0, -0.006798848063, -0.006046831235]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001534835679,  # occ, tight
            "OMP2.5 CORRELATION ENERGY": -0.225651726817,  # occ, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.055105451161,  # occ, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.008280964210,
                    0.0,
                    0.004780030747,
                    -0.004140482105,
                    0.0,
                    -0.004780030747,
                    -0.004140482105,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000953570304,  # occ, tight
            "OMP3 CORRELATION ENERGY": -0.227372086899,  # occ, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.052239106558,  # occ, tight
            "OMP3 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.004803850255,
                    0.0,
                    0.002953922342,
                    -0.002401925128,
                    0.0,
                    -0.002953922342,
                    -0.002401925128,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.00128167136578,  # occ, tight
            "OREMP2 CORRELATION ENERGY": -0.23069139385821,  # occ, tight
            "OREMP2 SAME-SPIN CORRELATION ENERGY": -0.05189818473696,  # occ, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    -0.000000000000,
                    0.009103029917,
                    0.0,
                    0.005405690204,
                    -0.004551514958,
                    0.0,
                    -0.005405690204,
                    -0.004551514958,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0011895155,  # p4n
            "OLCCD CORRELATION ENERGY": -0.2330452995,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0503175223,  # occ
            "OLCCD TOTAL GRADIENT": np.array(  # occ
                [0.0, 0.0, 0.009755099696, -0.0, 0.005854889163, -0.004877549848, 0.0, -0.005854889163, -0.004877549848]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.001123344,  # qchem
            "OCCD CORRELATION ENERGY": -0.229118050,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem, rearranged
                [0.0, 0.0, 0.00727331, 0.0, 0.00447444, -0.00363665, 0.0, -0.00447444, -0.00363665]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "SVWN TOTAL ENERGY": -76.07571659552373,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, -0.0, 0.01699540452, -0.0, 0.012801478421, -0.008497882106, 0.0, -0.012801478421, -0.008497882106]
            ).reshape((-1, 3)),
            "SVWN TOTAL HESSIAN": np.array(  # psi v1.8 99,590
                [
                    -0.015282137109,
                    0.0,
                    -0.0,
                    0.007667256256,
                    0.0,
                    -0.0,
                    0.007667256256,
                    0.0,
                    -0.0,
                    0.0,
                    0.714169890673,
                    0.000000000001,
                    -0.0,
                    -0.357096356198,
                    0.282430177213,
                    -0.0,
                    -0.357096356198,
                    -0.282430177213,
                    -0.0,
                    0.000000000001,
                    0.45653162932,
                    0.0,
                    0.223223468116,
                    -0.228270944088,
                    -0.0,
                    -0.223223468116,
                    -0.228270944088,
                    0.007667256256,
                    -0.0,
                    0.0,
                    -0.008305196866,
                    0.0,
                    0.0,
                    0.000637932765,
                    -0.0,
                    0.0,
                    0.0,
                    -0.357096356198,
                    0.223223468116,
                    0.0,
                    0.385967088688,
                    -0.252826811188,
                    -0.0,
                    -0.028870808826,
                    0.029603353817,
                    -0.0,
                    0.282430177213,
                    -0.228270944088,
                    0.0,
                    -0.252826811188,
                    0.219004239874,
                    0.0,
                    -0.029603353817,
                    0.009266633228,
                    0.007667256256,
                    -0.0,
                    -0.0,
                    0.000637932765,
                    -0.0,
                    0.0,
                    -0.008305196866,
                    -0.0,
                    0.0,
                    0.0,
                    -0.357096356198,
                    -0.223223468116,
                    -0.0,
                    -0.028870808826,
                    -0.029603353817,
                    -0.0,
                    0.385967088688,
                    0.252826811188,
                    -0.0,
                    -0.282430177213,
                    -0.228270944088,
                    0.0,
                    0.029603353817,
                    0.009266633228,
                    0.0,
                    0.252826811188,
                    0.219004239874,
                ]
            ).reshape((9, 9)),
            "PBE TOTAL ENERGY": -76.35896461348528,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-0.0, 0.0, 0.018069712075],
                    [-0.0, 0.011568971307, -0.009035203776],
                    [0.0, -0.011568971307, -0.009035203776],
                ]
            ),
            "B3LYP TOTAL ENERGY": -76.4445487826922090,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-1.77493704e-30, 0.00000000e00, 7.45592796e-03],
                    [-1.87490767e-19, 6.12391320e-03, -3.72819302e-03],
                    [1.87490767e-19, -6.12391320e-03, -3.72819302e-03],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -76.40749597292017,  # psi 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.00000000e00, -7.88860905e-31, 7.74045385e-03],
                    [-1.93603325e-19, 6.32356448e-03, -3.87045600e-03],
                    [1.93603325e-19, -6.32356448e-03, -3.87045600e-03],
                ]
            ),
            "WB97X TOTAL ENERGY": -76.42162677735320,  # psi 500,1202
            "WB97X TOTAL GRADIENT": np.array(  # psi 500,1202, findif-5
                [0.0, -0.0, 0.0028243495, -0.0, 0.002950472033, -0.001411682043, 0.0, -0.002950472033, -0.001411682043]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_adz_pk_rhf,
            "B2PLYP TOTAL ENERGY": -76.380406192274,  # nwc 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.2701916672,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06530131,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    -0.000531535533,
                    0.0,
                    -0.000960201925,
                    0.000265767766,
                    0.0,
                    0.000960201925,
                    0.000265767766,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [
                        4.79582300e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.39791100e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.39791100e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.99657262e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.49828631e-01,
                        2.70680600e-01,
                        0.00000000e00,
                        -3.49828631e-01,
                        -2.70680600e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.58837997e-01,
                        0.00000000e00,
                        2.05924494e-01,
                        -2.29418999e-01,
                        0.00000000e00,
                        -2.05924494e-01,
                        -2.29418999e-01,
                    ],
                    [
                        -2.39791100e-04,
                        0.00000000e00,
                        0.00000000e00,
                        4.55295000e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.15503800e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.49828631e-01,
                        2.05924494e-01,
                        0.00000000e00,
                        3.80569607e-01,
                        -2.38302547e-01,
                        0.00000000e00,
                        -3.07409760e-02,
                        3.23780531e-02,
                    ],
                    [
                        0.00000000e00,
                        2.70680600e-01,
                        -2.29418999e-01,
                        0.00000000e00,
                        -2.38302547e-01,
                        2.19836556e-01,
                        0.00000000e00,
                        -3.23780531e-02,
                        9.58244310e-03,
                    ],
                    [
                        -2.39791100e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.15503800e-04,
                        0.00000000e00,
                        0.00000000e00,
                        4.55295000e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.49828631e-01,
                        -2.05924494e-01,
                        0.00000000e00,
                        -3.07409760e-02,
                        -3.23780531e-02,
                        0.00000000e00,
                        3.80569607e-01,
                        2.38302547e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.70680600e-01,
                        -2.29418999e-01,
                        0.00000000e00,
                        3.23780531e-02,
                        9.58244310e-03,
                        0.00000000e00,
                        2.38302547e-01,
                        2.19836556e-01,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.27294451,  # fnocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.05952936,  # fnocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.007003320116, 0.0, -0.004485227443, 0.003501660058, 0.0, 0.004485227443, 0.003501660058]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.27601976,  # vcc
            "MP4(T) CORRECTION ENERGY": -0.007944754373,  # vcc
            "CISD CORRELATION ENERGY": -0.26425382513286,  # vcc
            "QCISD CORRELATION ENERGY": -0.27614913924585,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00704158,  # vcc
            "REMP2 CORRELATION ENERGY": -0.276444865873,  # occ, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.059323770883,  # occ, tight
            "LCCD CORRELATION ENERGY": -0.2786913134,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.057792990490,  # fnocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.002309010869, 0.0, -0.0017745445, 0.001154505435, -0.0, 0.0017745445, 0.001154505435]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2808517417,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.058297242512,  # fnocc
            "CEPA(1) CORRELATION ENERGY": -0.27639235,  # fnocc
            "CEPA(1) SINGLES ENERGY": 0.0,
            "CEPA(1) SAME-SPIN CORRELATION ENERGY": -0.05761465,  # fnocc
            "CEPA(3) CORRELATION ENERGY": -0.27334861,  # fnocc
            "CEPA(3) SINGLES ENERGY": 0.0,
            "CEPA(3) SAME-SPIN CORRELATION ENERGY": -0.05710750,  # fnocc
            "ACPF CORRELATION ENERGY": -0.27709825,  # fnocc
            "ACPF SINGLES ENERGY": 0.0,
            "ACPF SAME-SPIN CORRELATION ENERGY": -0.05773653,  # fnocc
            "AQCC CORRELATION ENERGY": -0.27396970,  # fnocc
            "AQCC SINGLES ENERGY": 0.0,
            "AQCC SAME-SPIN CORRELATION ENERGY": -0.05725788,  # fnocc
            "CCD CORRELATION ENERGY": -0.27417224324851,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.05767874,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc
                [[0.0, 0.0, -0.0047556235], [0.0, -0.0030400734, 0.0023778117], [0.0, 0.0030400734, 0.0023778117]]
            ),
            "CCD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        4.29117820e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.14558910e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.14558910e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.94523343e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.47261672e-01,
                        2.67217473e-01,
                        0.00000000e00,
                        -3.47261672e-01,
                        -2.67217473e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.66482117e-01,
                        0.00000000e00,
                        2.04034191e-01,
                        -2.33241059e-01,
                        0.00000000e00,
                        -2.04034191e-01,
                        -2.33241059e-01,
                    ],
                    [
                        -2.14558910e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.13479200e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.07971000e-05,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.47261672e-01,
                        2.04034191e-01,
                        0.00000000e00,
                        3.79908779e-01,
                        -2.35625839e-01,
                        0.00000000e00,
                        -3.26471076e-02,
                        3.15916340e-02,
                    ],
                    [
                        0.00000000e00,
                        2.67217473e-01,
                        -2.33241059e-01,
                        0.00000000e00,
                        -2.35625839e-01,
                        2.21133541e-01,
                        0.00000000e00,
                        -3.15916340e-02,
                        1.21075177e-02,
                    ],
                    [
                        -2.14558910e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.07971000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        2.13479200e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.47261672e-01,
                        -2.04034191e-01,
                        0.00000000e00,
                        -3.26471076e-02,
                        -3.15916340e-02,
                        0.00000000e00,
                        3.79908779e-01,
                        2.35625839e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.67217473e-01,
                        -2.33241059e-01,
                        0.00000000e00,
                        3.15916340e-02,
                        1.21075177e-02,
                        0.00000000e00,
                        2.35625839e-01,
                        2.21133541e-01,
                    ],
                ]
            ),
            "BCCD CORRELATION ENERGY": -0.27532829,  # ccenergy
            "CC2 CORRELATION ENERGY": -0.272546060672979,  # ccenergy
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.001726445285, 0.0, 0.000194402307, -0.000863222642, 0.0, -0.000194402307, -0.000863222642]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.275705491773,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.058006927914493,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    -0.003374258422,
                    0.0,
                    -0.002334452569,
                    0.001687129211,
                    0.0,
                    0.002334452569,
                    0.001687129211,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        3.04444750e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.52222380e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.52222380e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.92271308e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.46135654e-01,
                        2.66828227e-01,
                        0.00000000e00,
                        -3.46135654e-01,
                        -2.66828227e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.64211859e-01,
                        0.00000000e00,
                        2.03582577e-01,
                        -2.32105930e-01,
                        0.00000000e00,
                        -2.03582577e-01,
                        -2.32105930e-01,
                    ],
                    [
                        -1.52222380e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.57653820e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -5.43145000e-05,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.46135654e-01,
                        2.03582577e-01,
                        0.00000000e00,
                        3.78818316e-01,
                        -2.35205402e-01,
                        0.00000000e00,
                        -3.26826626e-02,
                        3.16228251e-02,
                    ],
                    [
                        0.00000000e00,
                        2.66828227e-01,
                        -2.32105930e-01,
                        0.00000000e00,
                        -2.35205402e-01,
                        2.20168635e-01,
                        0.00000000e00,
                        -3.16228251e-02,
                        1.19372948e-02,
                    ],
                    [
                        -1.52222380e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -5.43145000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        1.57653820e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.46135654e-01,
                        -2.03582577e-01,
                        0.00000000e00,
                        -3.26826626e-02,
                        -3.16228251e-02,
                        0.00000000e00,
                        3.78818316e-01,
                        2.35205402e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.66828227e-01,
                        -2.32105930e-01,
                        0.00000000e00,
                        3.16228251e-02,
                        1.19372948e-02,
                        0.00000000e00,
                        2.35205402e-01,
                        2.20168635e-01,
                    ],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.007532613647,  # vcc
            "(T) CORRECTION ENERGY": -0.007263596331,
            "CCSD(T) TOTAL GRADIENT": np.array(  # ncc fd
                [
                    [0.00000000000000, 0.00000000000000, -0.00018184005110],
                    [0.00000000000000, -0.00073407755645, 0.00009092002555],
                    [0.00000000000000, 0.00073407755645, 0.00009092002555],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # ncc fd
                [
                    [0.000164043137, 0.0, 0.0, -0.000082021569, 0.0, 0.0, -0.000082021569, 0.0, 0.0],
                    [
                        0.0,
                        0.690399868053,
                        0.0,
                        0.0,
                        -0.345199934026,
                        0.26721874523,
                        0.0,
                        -0.345199934026,
                        -0.26721874523,
                    ],
                    [
                        0.0,
                        0.0,
                        0.459903012519,
                        0.0,
                        0.203684212528,
                        -0.229951506259,
                        0.0,
                        -0.203684212528,
                        -0.229951506259,
                    ],
                    [-0.000082021569, 0.0, 0.0, 0.000297412235, 0.0, 0.0, -0.000215390667, 0.0, 0.0],
                    [
                        0.0,
                        -0.345199934026,
                        0.203684212528,
                        0.0,
                        0.377546525502,
                        -0.235451478879,
                        0.0,
                        -0.032346591476,
                        0.031767266351,
                    ],
                    [
                        0.0,
                        0.26721874523,
                        -0.229951506259,
                        0.0,
                        -0.235451478879,
                        0.218683624079,
                        0.0,
                        -0.031767266351,
                        0.011267882181,
                    ],
                    [-0.000082021569, 0.0, 0.0, -0.000215390667, 0.0, 0.0, 0.000297412235, 0.0, 0.0],
                    [
                        0.0,
                        -0.345199934026,
                        -0.203684212528,
                        0.0,
                        -0.032346591476,
                        -0.031767266351,
                        0.0,
                        0.377546525502,
                        0.235451478879,
                    ],
                    [
                        0.0,
                        -0.26721874523,
                        -0.229951506259,
                        0.0,
                        0.031767266351,
                        0.011267882181,
                        0.0,
                        0.235451478879,
                        0.218683624079,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.007181434345419,  # ncc
            "A-CCSD(T) TOTAL GRADIENT": np.array(  # ncc
                [
                    [0.000000000000000, 0.000000000000000, -0.000263235380777],
                    [0.000000000000000, -0.000769830346953, 0.000131617690376],
                    [0.000000000000000, 0.000769830346953, 0.000131617690376],
                ]
            ),
            "B(T) CORRECTION ENERGY": -0.00766044,  # ccenergy
            "CC3 CORRELATION ENERGY": -0.283326604729141,  # ccenergy
            "CCSDT-1A CORRELATION ENERGY": -0.283301014194,  # ecc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.00003469032178],
                    [0.00000000000000, -0.00061071465370, -0.00001734516089],
                    [0.00000000000000, 0.00061071465370, -0.00001734516089],
                ]
            ),
            "CCSDT-1A TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.000031470364, 0.0, 0.0, 0.000015735182, 0.0, 0.0, 0.000015735182, 0.0, 0.0],
                    [
                        0.0,
                        0.690357139667,
                        0.0,
                        0.0,
                        -0.345178569834,
                        0.267277548335,
                        0.0,
                        -0.345178569834,
                        -0.267277548335,
                    ],
                    [
                        0.0,
                        0.0,
                        0.459591068441,
                        0.0,
                        0.203726352234,
                        -0.229795534221,
                        0.0,
                        -0.203726352234,
                        -0.229795534221,
                    ],
                    [0.000015735182, 0.0, 0.0, 0.000205401177, 0.0, 0.0, -0.000221136359, 0.0, 0.0],
                    [
                        0.0,
                        -0.345178569834,
                        0.203726352234,
                        0.0,
                        0.377473575693,
                        -0.235501950285,
                        0.0,
                        -0.03229500586,
                        0.031775598051,
                    ],
                    [
                        0.0,
                        0.267277548335,
                        -0.229795534221,
                        0.0,
                        -0.235501950285,
                        0.218585504687,
                        0.0,
                        -0.031775598051,
                        0.011210029534,
                    ],
                    [0.000015735182, 0.0, 0.0, -0.000221136359, 0.0, 0.0, 0.000205401177, 0.0, 0.0],
                    [
                        0.0,
                        -0.345178569834,
                        -0.203726352234,
                        0.0,
                        -0.03229500586,
                        -0.031775598051,
                        0.0,
                        0.377473575693,
                        0.235501950285,
                    ],
                    [
                        0.0,
                        -0.267277548335,
                        -0.229795534221,
                        0.0,
                        0.031775598051,
                        0.011210029534,
                        0.0,
                        0.235501950285,
                        0.218585504687,
                    ],
                ]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.283304708235,  # ecc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.00003781932904],
                    [0.00000000000000, -0.00060845267210, -0.00001890966452],
                    [0.00000000000000, 0.00060845267210, -0.00001890966452],
                ]
            ),
            "CCSDT-1B TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.000034308612, 0.0, 0.0, 0.000017154306, 0.0, 0.0, 0.000017154306, 0.0, 0.0],
                    [
                        0.0,
                        0.69034722208,
                        0.0,
                        0.0,
                        -0.34517361104,
                        0.267274828449,
                        0.0,
                        -0.34517361104,
                        -0.267274828449,
                    ],
                    [
                        0.0,
                        0.0,
                        0.459575612743,
                        0.0,
                        0.203725689988,
                        -0.229787806371,
                        0.0,
                        -0.203725689988,
                        -0.229787806371,
                    ],
                    [0.000017154306, 0.0, 0.0, 0.000203904093, 0.0, 0.0, -0.000221058399, 0.0, 0.0],
                    [
                        0.0,
                        -0.34517361104,
                        0.203725689988,
                        0.0,
                        0.377466284086,
                        -0.235500259218,
                        0.0,
                        -0.032292673046,
                        0.03177456923,
                    ],
                    [
                        0.0,
                        0.267274828449,
                        -0.229787806371,
                        0.0,
                        -0.235500259218,
                        0.218579787679,
                        0.0,
                        -0.03177456923,
                        0.011208018692,
                    ],
                    [0.000017154306, 0.0, 0.0, -0.000221058399, 0.0, 0.0, 0.000203904093, 0.0, 0.0],
                    [
                        0.0,
                        -0.34517361104,
                        -0.203725689988,
                        0.0,
                        -0.032292673046,
                        -0.03177456923,
                        0.0,
                        0.377466284086,
                        0.235500259218,
                    ],
                    [
                        0.0,
                        -0.267274828449,
                        -0.229787806371,
                        0.0,
                        0.03177456923,
                        0.011208018692,
                        0.0,
                        0.235500259218,
                        0.218579787679,
                    ],
                ]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.282650715222,  # ecc
            "CCSDT-2 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, -0.00033837753770],
                    [0.00000000000000, -0.00081713978219, 0.00016918876885],
                    [0.00000000000000, 0.00081713978219, 0.00016918876885],
                ]
            ),
            "CCSDT-2 TOTAL HESSIAN": np.array(  # ecc by psi fd
                [
                    [0.000305098093, 0.0, 0.0, -0.000152549047, 0.0, 0.0, -0.000152549047, 0.0, 0.0],
                    [
                        0.0,
                        0.690395599827,
                        0.0,
                        0.0,
                        -0.345197799913,
                        0.267162476654,
                        0.0,
                        -0.345197799913,
                        -0.267162476654,
                    ],
                    [
                        0.0,
                        0.0,
                        0.460115900113,
                        0.0,
                        0.203699000404,
                        -0.230057950057,
                        0.0,
                        -0.203699000404,
                        -0.230057950057,
                    ],
                    [-0.000152549047, 0.0, 0.0, 0.000361653884, 0.0, 0.0, -0.000209104837, 0.0, 0.0],
                    [
                        0.0,
                        -0.345197799913,
                        0.203699000404,
                        0.0,
                        0.377579483066,
                        -0.235430738529,
                        0.0,
                        -0.032381683152,
                        0.031731738125,
                    ],
                    [
                        0.0,
                        0.267162476654,
                        -0.230057950057,
                        0.0,
                        -0.235430738529,
                        0.218744030684,
                        0.0,
                        -0.031731738125,
                        0.011313919373,
                    ],
                    [-0.000152549047, 0.0, 0.0, -0.000209104837, 0.0, 0.0, 0.000361653884, 0.0, 0.0],
                    [
                        0.0,
                        -0.345197799913,
                        -0.203699000404,
                        0.0,
                        -0.032381683152,
                        -0.031731738125,
                        0.0,
                        0.377579483066,
                        0.235430738529,
                    ],
                    [
                        0.0,
                        -0.267162476654,
                        -0.230057950057,
                        0.0,
                        0.031731738125,
                        0.011313919373,
                        0.0,
                        0.235430738529,
                        0.218744030684,
                    ],
                ]
            ),
            "CCSDT-3 CORRELATION ENERGY": -0.28267436870285,  # vcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, -0.00031801749656],
                    [0.00000000000000, -0.00080724235455, 0.00015900874828],
                    [0.00000000000000, 0.00080724235455, 0.00015900874828],
                ]
            ),
            "CCSDT-3 TOTAL HESSIAN": np.array(  # mrcc
                [
                    [
                        2.86755305e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.43377653e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.43377653e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.90367936e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.45183968e-01,
                        2.67158588e-01,
                        0.00000000e00,
                        -3.45183968e-01,
                        -2.67158588e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.60092567e-01,
                        0.00000000e00,
                        2.03699151e-01,
                        -2.30046284e-01,
                        0.00000000e00,
                        -2.03699151e-01,
                        -2.30046284e-01,
                    ],
                    [
                        -1.43377653e-04,
                        0.00000000e00,
                        0.00000000e00,
                        3.53604923e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.10227270e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45183968e-01,
                        2.03699151e-01,
                        0.00000000e00,
                        3.77567061e-01,
                        -2.35428870e-01,
                        0.00000000e00,
                        -3.23830932e-02,
                        3.17297185e-02,
                    ],
                    [
                        0.00000000e00,
                        2.67158588e-01,
                        -2.30046284e-01,
                        0.00000000e00,
                        -2.35428870e-01,
                        2.18733436e-01,
                        0.00000000e00,
                        -3.17297185e-02,
                        1.13128480e-02,
                    ],
                    [
                        -1.43377653e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.10227270e-04,
                        0.00000000e00,
                        0.00000000e00,
                        3.53604923e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45183968e-01,
                        -2.03699151e-01,
                        0.00000000e00,
                        -3.23830932e-02,
                        -3.17297185e-02,
                        0.00000000e00,
                        3.77567061e-01,
                        2.35428870e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.67158588e-01,
                        -2.30046284e-01,
                        0.00000000e00,
                        3.17297185e-02,
                        1.13128480e-02,
                        0.00000000e00,
                        2.35428870e-01,
                        2.18733436e-01,
                    ],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.283033246143265,  # ncc
            "CCSDT TOTAL GRADIENT": np.array(
                [
                    0.000000000000000,
                    0.000000000000000,
                    -0.000217363788816,
                    0.000000000000000,
                    -0.000742442561705,
                    0.000108681894407,
                    0.000000000000000,
                    0.000742442561705,
                    0.000108681894407,
                ]
            ).reshape((-1, 3)),
            "CCSDT TOTAL HESSIAN": np.array(  # ecc
                [
                    [
                        1.96117400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -9.80587000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        -9.80587000e-05,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.90148921e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.45074461e-01,
                        2.67109270e-01,
                        0.00000000e00,
                        -3.45074461e-01,
                        -2.67109270e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.60001543e-01,
                        0.00000000e00,
                        2.03664977e-01,
                        -2.30000771e-01,
                        0.00000000e00,
                        -2.03664977e-01,
                        -2.30000771e-01,
                    ],
                    [
                        -9.80587000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        3.08365200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.10306500e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45074461e-01,
                        2.03664977e-01,
                        0.00000000e00,
                        3.77473316e-01,
                        -2.35387123e-01,
                        0.00000000e00,
                        -3.23988558e-02,
                        3.17221463e-02,
                    ],
                    [
                        0.00000000e00,
                        2.67109270e-01,
                        -2.30000771e-01,
                        0.00000000e00,
                        -2.35387123e-01,
                        2.18668768e-01,
                        0.00000000e00,
                        -3.17221463e-02,
                        1.13320029e-02,
                    ],
                    [
                        -9.80587000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        -2.10306500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        3.08365200e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45074461e-01,
                        -2.03664977e-01,
                        0.00000000e00,
                        -3.23988558e-02,
                        -3.17221463e-02,
                        0.00000000e00,
                        3.77473316e-01,
                        2.35387123e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.67109270e-01,
                        -2.30000771e-01,
                        0.00000000e00,
                        3.17221463e-02,
                        1.13320029e-02,
                        0.00000000e00,
                        2.35387123e-01,
                        2.18668768e-01,
                    ],
                ]
            ),
            "[Q] CORRECTION ENERGY": -0.000163881984112,  # ncc
            "(Q) CORRECTION ENERGY": -0.000383414612913,  # ncc
            "CCSDT(Q) TOTAL GRADIENT": np.array(
                [
                    0.000000000000000,
                    0.000000000000000,
                    0.000216022413761,
                    0.000000000000000,
                    -0.000511432746023,
                    -0.000108011206892,
                    0.000000000000000,
                    0.000511432746023,
                    -0.000108011206892,
                ]
            ).reshape((-1, 3)),
            "CCSDTQ CORRELATION ENERGY": -0.283376955140291,  # ncc
            "OMP2 REFERENCE CORRECTION ENERGY": 0.002513901693,  # occ, tight
            "OMP2 CORRELATION ENERGY": -0.272599646070,  # occ, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.066843168345,  # occ, tight
            "OMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.001853244534,
                    0.0,
                    0.000249902550,
                    -0.000926622267,
                    0.0,
                    -0.000249902550,
                    -0.000926622267,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001691543451,  # occ, tight
            "OMP2.5 CORRELATION ENERGY": -0.273211228305,  # occ, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.063331759367,  # occ, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    -0.002291101232,
                    0.0,
                    -0.001990147396,
                    0.001145550616,
                    0.0,
                    0.001990147396,
                    0.001145550616,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001090059672,  # occ, tight
            "OMP3 CORRELATION ENERGY": -0.274015056808,  # occ, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.059986220382,  # occ, tight
            "OMP3 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    -0.006148152213,
                    0.0,
                    -0.004066707268,
                    0.003074076107,
                    0.0,
                    0.004066707268,
                    0.003074076107,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001444328268,  # occ, tight
            "OREMP2 CORRELATION ENERGY": -0.277848169449,  # occ, tight
            "OREMP2 SAME-SPIN CORRELATION ENERGY": -0.059932090515,  # occ, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    -0.001616411571,
                    0.0,
                    -0.001508635375,
                    0.000808205786,
                    0.0,
                    0.001508635375,
                    0.000808205786,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0013521561,  # p4n
            "OLCCD CORRELATION ENERGY": -0.2800053174,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0582676514,  # occ
            "OLCCD TOTAL GRADIENT": np.array(  # occ
                [0.0, 0.0, -0.001040099489, 0.0, -0.001127979233, 0.000520049745, -0.0, 0.001127979233, 0.000520049745]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.001279242,  # qchem
            "OCCD CORRELATION ENERGY": -0.275418269,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem, rearranged
                [0, 0, -0.00363692, 0, -0.00246929, 0.00181846, 0, 0.00246929, 0.00181846]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "SVWN TOTAL ENERGY": -76.10229674046080,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.01116011934, -0.0, 0.009507405365, -0.005580350828, 0.0, -0.009507405365, -0.005580350828]
            ).reshape((-1, 3)),
            "SVWN TOTAL HESSIAN": np.array(  # psi v1.8 99,590
                [
                    -0.010015602973,
                    -0.000000000001,
                    0.000000000002,
                    0.005034918743,
                    0.0,
                    -0.0,
                    0.005034918743,
                    -0.0,
                    0.0,
                    -0.000000000001,
                    0.699200093422,
                    -0.000000000002,
                    -0.0,
                    -0.349611743856,
                    0.274596771974,
                    -0.0,
                    -0.349611743856,
                    -0.274596771974,
                    0.000000000002,
                    -0.000000000002,
                    0.449624843917,
                    -0.0,
                    0.216250731357,
                    -0.224817271283,
                    -0.0,
                    -0.216250731357,
                    -0.224817271283,
                    0.005034918743,
                    -0.0,
                    -0.0,
                    -0.005839372129,
                    -0.0,
                    0.0,
                    0.000803489613,
                    0.0,
                    0.0,
                    0.0,
                    -0.349611743856,
                    0.216250731357,
                    -0.0,
                    0.379671911068,
                    -0.245424063175,
                    0.0,
                    -0.030060195243,
                    0.029173021276,
                    -0.0,
                    0.274596771974,
                    -0.224817271283,
                    0.0,
                    -0.245424063175,
                    0.215395274362,
                    0.0,
                    -0.029173021276,
                    0.009421810131,
                    0.005034918743,
                    -0.0,
                    -0.0,
                    0.000803489613,
                    0.0,
                    0.0,
                    -0.005839372129,
                    0.0,
                    -0.0,
                    -0.0,
                    -0.349611743856,
                    -0.216250731357,
                    0.0,
                    -0.030060195243,
                    -0.029173021276,
                    0.0,
                    0.379671911068,
                    0.245424063175,
                    0.0,
                    -0.274596771974,
                    -0.224817271283,
                    0.0,
                    0.029173021276,
                    0.009421810131,
                    -0.0,
                    0.245424063175,
                    0.215395274362,
                ]
            ).reshape((9, 9)),
            "PBE TOTAL ENERGY": -76.38209089347507,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-0.000000000000, -0.000000000000, 0.012770562551],
                    [-0.000000000000, 0.008604125876, -0.006385673964],
                    [0.000000000000, -0.008604125876, -0.006385673964],
                ]
            ),
            "B3LYP TOTAL ENERGY": -76.4684173811156711,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.00000000e00, 1.57772181e-30, 2.02268338e-03],
                    [-9.31840056e-20, 3.04362060e-03, -1.01162676e-03],
                    [9.31840056e-20, -3.04362060e-03, -1.01162676e-03],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -76.43134091764595,  # psi 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-7.88860905e-31, 7.88860905e-31, 2.30650526e-03],
                    [-9.91741439e-20, 3.23927336e-03, -1.15353738e-03],
                    [9.91741439e-20, -3.23927336e-03, -1.15353738e-03],
                ]
            ),
            "WB97X TOTAL ENERGY": -76.44423395697297,  # psi 500,1202
            "WB97X TOTAL GRADIENT": np.array(  # psi 500,1202, findif-5
                [0.0, 0.0, -0.002561607305, 0.0, -0.000034257733, 0.00128137288, -0.0, 0.000034257733, 0.00128137288]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_qz2p_pk_rhf,
            "B2PLYP TOTAL ENERGY": -76.416806713650,  # nwc 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.05948928003552,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001918693775,
            "MP2 TOTAL GRADIENT": np.array(
                [  # occ findif-5 ae pk+conv
                    0.00000000000000,
                    0.00000000000000,
                    0.01250561195911,
                    0.00000000000000,
                    0.00000000000000,
                    -0.01206536529299,
                    0.00000000000000,
                    0.01033165380573,
                    -0.00022012333306,
                    0.00000000000000,
                    -0.01033165380573,
                    -0.00022012333306,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [
                        3.44512200e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.01573440e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.85306200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.85306200e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        3.35480990e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -3.87467360e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.59931900e-04,
                        4.32569100e-04,
                        0.00000000e00,
                        2.59931900e-04,
                        -4.32569100e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.15837144e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.13789867e-01,
                        0.00000000e00,
                        1.11748040e-03,
                        -1.02363850e-03,
                        0.00000000e00,
                        -1.11748040e-03,
                        -1.02363850e-03,
                    ],
                    [
                        -4.01573440e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.35926330e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.71764400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.71764400e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.87467360e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.27499890e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.00162700e-04,
                        8.11818000e-05,
                        0.00000000e00,
                        -2.00162700e-04,
                        -8.11818000e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.13789867e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.13964588e-01,
                        0.00000000e00,
                        1.98755000e-05,
                        -8.73610000e-05,
                        0.00000000e00,
                        -1.98755000e-05,
                        -8.73610000e-05,
                    ],
                    [
                        2.85306200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.71764400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -7.41535090e-03,
                        0.00000000e00,
                        0.00000000e00,
                        7.30180910e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.59931900e-04,
                        1.11748040e-03,
                        0.00000000e00,
                        -2.00162700e-04,
                        1.98755000e-05,
                        0.00000000e00,
                        4.11358061e-01,
                        -8.25553300e-04,
                        0.00000000e00,
                        -4.11417830e-01,
                        -3.11802500e-04,
                    ],
                    [
                        0.00000000e00,
                        4.32569100e-04,
                        -1.02363850e-03,
                        0.00000000e00,
                        8.11818000e-05,
                        -8.73610000e-05,
                        0.00000000e00,
                        -8.25553300e-04,
                        -4.66526560e-03,
                        0.00000000e00,
                        3.11802500e-04,
                        5.77626520e-03,
                    ],
                    [
                        2.85306200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.71764400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        7.30180910e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -7.41535090e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.59931900e-04,
                        -1.11748040e-03,
                        0.00000000e00,
                        -2.00162700e-04,
                        -1.98755000e-05,
                        0.00000000e00,
                        -4.11417830e-01,
                        3.11802500e-04,
                        0.00000000e00,
                        4.11358061e-01,
                        8.25553300e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.32569100e-04,
                        -1.02363850e-03,
                        0.00000000e00,
                        -8.11818000e-05,
                        -8.73610000e-05,
                        0.00000000e00,
                        -3.11802500e-04,
                        5.77626520e-03,
                        0.00000000e00,
                        8.25553300e-04,
                        -4.66526560e-03,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.07481556,  # occ
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00235692,  # occ
            "MP3 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.0093167821,
                    0.0,
                    0.0,
                    -0.00908998793,
                    0.0,
                    0.012725715572,
                    -0.000113397085,
                    0.0,
                    -0.012725715572,
                    -0.000113397085,
                ]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.07927852,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.000248958942,  # vcc only
            "CISD CORRELATION ENERGY": -0.08014448073265,  # vcc
            "QCISD CORRELATION ENERGY": -0.08218197897917,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00062359,  # vcc only
            "REMP2 CORRELATION ENERGY": -0.07803997531851,  # occ, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.00233934205346,  # occ, tight
            "LCCD CORRELATION ENERGY": -0.0834347185,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0024003297,  # occ
            "LCCD TOTAL GRADIENT": np.array(
                [
                    -0.0,
                    0.0,
                    0.004644430473,
                    0.0,
                    0.0,
                    -0.004561891704,
                    -0.0,
                    0.015262723486,
                    -0.000041269384,
                    0.0,
                    -0.015262723486,
                    -0.000041269384,
                ]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.0848110820,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.0024351,  # vcc
            "CCD CORRELATION ENERGY": -0.08113776652663,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.00235010,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc
                [
                    [0.0, 0.0, 0.0064056106],
                    [0.0, 0.0, -0.0062159539],
                    [0.0, 0.0144496921, -0.0000948283],
                    [0.0, -0.0144496921, -0.0000948283],
                ]
            ),
            "CCD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        1.21745460e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.67345390e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.27999700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.27999700e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.18297710e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.56709910e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.92061000e-04,
                        4.57116300e-04,
                        0.00000000e00,
                        1.92061000e-04,
                        -4.57116300e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.08191655e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.06455377e-01,
                        0.00000000e00,
                        1.17641080e-03,
                        -8.68139000e-04,
                        0.00000000e00,
                        -1.17641080e-03,
                        -8.68139000e-04,
                    ],
                    [
                        -1.67345390e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.96873420e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.47647400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.47647400e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -1.56709910e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.90289150e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.67896600e-04,
                        4.83546000e-05,
                        0.00000000e00,
                        -1.67896600e-04,
                        -4.83546000e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.06455377e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.06585356e-01,
                        0.00000000e00,
                        1.26821500e-04,
                        -6.49859000e-05,
                        0.00000000e00,
                        -1.26821500e-04,
                        -6.49859000e-05,
                    ],
                    [
                        2.27999700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.47647400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.03317422e-02,
                        0.00000000e00,
                        0.00000000e00,
                        1.02513900e-02,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.92061000e-04,
                        1.17641080e-03,
                        0.00000000e00,
                        -1.67896600e-04,
                        1.26821500e-04,
                        0.00000000e00,
                        4.08407145e-01,
                        -9.04348400e-04,
                        0.00000000e00,
                        -4.08431309e-01,
                        -3.98877500e-04,
                    ],
                    [
                        0.00000000e00,
                        4.57116300e-04,
                        -8.68139000e-04,
                        0.00000000e00,
                        4.83546000e-05,
                        -6.49859000e-05,
                        0.00000000e00,
                        -9.04348400e-04,
                        -7.77410400e-03,
                        0.00000000e00,
                        3.98877500e-04,
                        8.70722880e-03,
                    ],
                    [
                        2.27999700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.47647400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.02513900e-02,
                        0.00000000e00,
                        0.00000000e00,
                        -1.03317422e-02,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.92061000e-04,
                        -1.17641080e-03,
                        0.00000000e00,
                        -1.67896600e-04,
                        -1.26821500e-04,
                        0.00000000e00,
                        -4.08431309e-01,
                        3.98877500e-04,
                        0.00000000e00,
                        4.08407145e-01,
                        9.04348400e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.57116300e-04,
                        -8.68139000e-04,
                        0.00000000e00,
                        -4.83546000e-05,
                        -6.49859000e-05,
                        0.00000000e00,
                        -3.98877500e-04,
                        8.70722880e-03,
                        0.00000000e00,
                        9.04348400e-04,
                        -7.77410400e-03,
                    ],
                ]
            ),
            "BCCD CORRELATION ENERGY": -0.08217317,  # ccenergy
            "CC2 CORRELATION ENERGY": -0.059830885446002,  # ccenergy
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [
                    0.0,
                    0.0,
                    0.012101218207,
                    0.0,
                    0.0,
                    -0.011694433668,
                    0.0,
                    0.010554775214,
                    -0.000203392269,
                    0.0,
                    -0.010554775214,
                    -0.000203392269,
                ]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.08217287869,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.002377557359,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.005209606766,
                    0.0,
                    0.0,
                    -0.005071403517,
                    0.0,
                    0.014880198292,
                    -0.000069101625,
                    0.0,
                    -0.014880198292,
                    -0.000069101625,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        7.74047000e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.20954470e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.17748900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.17748900e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.31670700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.10104560e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.84687400e-04,
                        4.38316100e-04,
                        0.00000000e00,
                        1.84687400e-04,
                        -4.38316100e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.06125475e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.04450486e-01,
                        0.00000000e00,
                        1.15794910e-03,
                        -8.37494800e-04,
                        0.00000000e00,
                        -1.15794910e-03,
                        -8.37494800e-04,
                    ],
                    [
                        -1.20954470e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.49705710e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.43756200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.43756200e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -1.10104560e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.43042210e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.64688300e-04,
                        5.23087000e-05,
                        0.00000000e00,
                        -1.64688300e-04,
                        -5.23087000e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.04450486e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.04563594e-01,
                        0.00000000e00,
                        1.62442000e-04,
                        -5.65541000e-05,
                        0.00000000e00,
                        -1.62442000e-04,
                        -5.65541000e-05,
                    ],
                    [
                        2.17748900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.43756200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.06352124e-02,
                        0.00000000e00,
                        0.00000000e00,
                        1.05612197e-02,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.84687400e-04,
                        1.15794910e-03,
                        0.00000000e00,
                        -1.64688300e-04,
                        1.62442000e-04,
                        0.00000000e00,
                        4.07393462e-01,
                        -9.05508000e-04,
                        0.00000000e00,
                        -4.07413462e-01,
                        -4.14883200e-04,
                    ],
                    [
                        0.00000000e00,
                        4.38316100e-04,
                        -8.37494800e-04,
                        0.00000000e00,
                        5.23087000e-05,
                        -5.65541000e-05,
                        0.00000000e00,
                        -9.05508000e-04,
                        -8.15157550e-03,
                        0.00000000e00,
                        4.14883200e-04,
                        9.04562440e-03,
                    ],
                    [
                        2.17748900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.43756200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.05612197e-02,
                        0.00000000e00,
                        0.00000000e00,
                        -1.06352124e-02,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.84687400e-04,
                        -1.15794910e-03,
                        0.00000000e00,
                        -1.64688300e-04,
                        -1.62442000e-04,
                        0.00000000e00,
                        -4.07413462e-01,
                        4.14883200e-04,
                        0.00000000e00,
                        4.07393462e-01,
                        9.05508000e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.38316100e-04,
                        -8.37494800e-04,
                        0.00000000e00,
                        -5.23087000e-05,
                        -5.65541000e-05,
                        0.00000000e00,
                        -4.14883200e-04,
                        9.04562440e-03,
                        0.00000000e00,
                        9.05508000e-04,
                        -8.15157550e-03,
                    ],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.000630302926,  # vcc
            "(T) CORRECTION ENERGY": -0.00062614,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.00469857108970],
                    [0.00000000000000, 0.00000000000000, -0.00464327797336],
                    [0.00000000000000, 0.01496002471717, -0.00002764655817],
                    [0.00000000000000, -0.01496002471717, -0.00002764655817],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # vcc fd
                [
                    [
                        0.000605645748,
                        0.0,
                        0.0,
                        -0.001025922218,
                        0.0,
                        0.0,
                        0.000210138235,
                        0.0,
                        0.0,
                        0.000210138235,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.000566396166,
                        0.0,
                        0.0,
                        -0.000920471148,
                        0.0,
                        0.0,
                        0.000177037491,
                        0.000434687941,
                        0.0,
                        0.000177037491,
                        -0.000434687941,
                    ],
                    [
                        0.0,
                        0.0,
                        0.205974349398,
                        0.0,
                        0.0,
                        -0.20443915744,
                        0.0,
                        0.001282151675,
                        -0.000767595979,
                        0.0,
                        -0.001282151675,
                        -0.000767595979,
                    ],
                    [
                        -0.001025922218,
                        0.0,
                        0.0,
                        0.001313504777,
                        0.0,
                        0.0,
                        -0.000143791279,
                        0.0,
                        0.0,
                        -0.000143791279,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.000920471148,
                        0.0,
                        0.0,
                        0.001248510511,
                        0.0,
                        0.0,
                        -0.000164019682,
                        0.000048912565,
                        0.0,
                        -0.000164019682,
                        -0.000048912565,
                    ],
                    [
                        0.0,
                        0.0,
                        -0.20443915744,
                        0.0,
                        0.0,
                        0.204583205785,
                        0.0,
                        0.000137691376,
                        -0.000072024172,
                        0.0,
                        -0.000137691376,
                        -0.000072024172,
                    ],
                    [
                        0.000210138235,
                        0.0,
                        0.0,
                        -0.000143791279,
                        0.0,
                        0.0,
                        -0.010688228885,
                        0.0,
                        0.0,
                        0.01062188193,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.000177037491,
                        0.001282151675,
                        0.0,
                        -0.000164019682,
                        0.000137691376,
                        0.0,
                        0.407323404956,
                        -0.000951721779,
                        0.0,
                        -0.407336422766,
                        -0.000468121273,
                    ],
                    [
                        0.0,
                        0.000434687941,
                        -0.000767595979,
                        0.0,
                        0.000048912565,
                        -0.000072024172,
                        0.0,
                        -0.000951721779,
                        -0.008268538433,
                        0.0,
                        0.000468121273,
                        0.009108158584,
                    ],
                    [
                        0.000210138235,
                        0.0,
                        0.0,
                        -0.000143791279,
                        0.0,
                        0.0,
                        0.01062188193,
                        0.0,
                        0.0,
                        -0.010688228885,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.000177037491,
                        -0.001282151675,
                        0.0,
                        -0.000164019682,
                        -0.000137691376,
                        0.0,
                        -0.407336422766,
                        0.000468121273,
                        0.0,
                        0.407323404956,
                        0.000951721779,
                    ],
                    [
                        0.0,
                        -0.000434687941,
                        -0.000767595979,
                        0.0,
                        -0.000048912565,
                        -0.000072024172,
                        0.0,
                        -0.000468121273,
                        0.009108158584,
                        0.0,
                        0.000951721779,
                        -0.008268538433,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.000609324,  # mrcc only
            "B(T) CORRECTION ENERGY": -0.00063370,  # ccenergy
            "CC3 CORRELATION ENERGY": -0.082800948490126,  # ccenergy
            "CCSDT-1A CORRELATION ENERGY": -0.082798380297,  # ecc
            "CCSDT-1B CORRELATION ENERGY": -0.082796300732,  # ecc
            "CCSDT-2 CORRELATION ENERGY": -0.082795702684,  # ecc
            "CCSDT-3 CORRELATION ENERGY": -0.08280038799307,  # vcc
            "CCSDT CORRELATION ENERGY": -0.08311156413461,  # vcc
            "OMP2 REFERENCE CORRECTION ENERGY": 0.000395359637,  # occ, tight
            "OMP2 CORRELATION ENERGY": -0.059864579866,  # occ, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.001971155898,  # occ, tight
            "OMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.012017558262,
                    0.0,
                    0.000000000000,
                    -0.011623172725,
                    0.0,
                    0.010549443964,
                    -0.000197192768,
                    0.0,
                    -0.010549443964,
                    -0.000197192768,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.000564671059,  # occ, tight
            "OMP2.5 CORRELATION ENERGY": -0.067680196985,  # occ, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.002198682496,  # occ, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.010260851431,
                    0.0,
                    0.000000000000,
                    -0.009981279417,
                    0.0,
                    0.011818192519,
                    -0.000139786007,
                    0.0,
                    -0.011818192519,
                    -0.000139786007,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000777674538,  # occ, tight
            "OMP3 CORRELATION ENERGY": -0.075531256945,  # occ, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.002426022161,  # occ, tight
            "OMP3 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.008456348171,
                    0.0,
                    0.000000000000,
                    -0.008291902042,
                    0.0,
                    0.013097750811,
                    -0.000082223065,
                    0.0,
                    -0.013097750811,
                    -0.000082223065,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001061018532,  # occ, tight
            "OREMP2 CORRELATION ENERGY": -0.078996001732,  # occ, tight
            "OREMP2 SAME-SPIN CORRELATION ENERGY": -0.002398210306,  # occ, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.005898073051,
                    0.0,
                    0.000000000000,
                    -0.005790147152,
                    0.0,
                    0.014384641403,
                    -0.000053962950,
                    0.0,
                    -0.014384641403,
                    -0.000053962950,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0014842084,  # p4n
            "OLCCD CORRELATION ENERGY": -0.0847413506,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0024486744,  # occ
            "OLCCD TOTAL GRADIENT": np.array(  # occ
                [
                    0.0,
                    0.0,
                    0.002952428656,
                    0.0,
                    0.0,
                    -0.002937832654,
                    -0.0,
                    0.015746258127,
                    -0.000007298001,
                    0.0,
                    -0.015746258127,
                    -0.000007298001,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.001165795,  # qchem
            "OCCD CORRELATION ENERGY": -0.082188930,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem, rearranged
                [0, 0, 0.00520110, 0, 0, -0.00505850, 0.0, 0.01487527, -0.00007130, 0.0, -0.01487527, -0.00007130]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "SVWN TOTAL ENERGY": -25.88281835226915,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    0.0,
                    -0.0,
                    -0.007517701008,
                    0.0,
                    -0.0,
                    0.00226814353,
                    -0.0,
                    0.036127961709,
                    0.002624776187,
                    0.0,
                    -0.036127961709,
                    0.002624776187,
                ]
            ).reshape((-1, 3)),
            "SVWN TOTAL HESSIAN": np.array(  # psi 99,590, findif-5
                [
                    -0.001865509938,
                    0.0,
                    0.0,
                    0.002269947766,
                    0.0,
                    0.0,
                    -0.000202218914,
                    0.0,
                    0.0,
                    -0.000202218914,
                    0.0,
                    0.0,
                    0.0,
                    -0.002096501204,
                    0.0,
                    0.0,
                    0.002406884872,
                    0.0,
                    0.0,
                    -0.000155191834,
                    -0.000138790593,
                    0.0,
                    -0.000155191834,
                    0.000138790593,
                    0.0,
                    0.0,
                    0.209856775843,
                    0.0,
                    0.0,
                    -0.211646788845,
                    0.0,
                    0.003612344032,
                    0.000895006501,
                    0.0,
                    -0.003612344032,
                    0.000895006501,
                    0.002269947766,
                    0.0,
                    0.0,
                    -0.001881222548,
                    0.0,
                    0.0,
                    -0.000194362609,
                    0.0,
                    0.0,
                    -0.000194362609,
                    0.0,
                    0.0,
                    0.0,
                    0.002406884872,
                    0.0,
                    0.0,
                    -0.00196262459,
                    0.0,
                    0.0,
                    -0.000222130141,
                    0.000081041368,
                    0.0,
                    -0.000222130141,
                    -0.000081041368,
                    0.0,
                    0.0,
                    -0.211646788845,
                    0.0,
                    0.0,
                    0.211526661243,
                    0.0,
                    0.000036018441,
                    0.000060063801,
                    0.0,
                    -0.000036018441,
                    0.000060063801,
                    -0.000202218914,
                    0.0,
                    0.0,
                    -0.000194362609,
                    0.0,
                    0.0,
                    -0.025533348395,
                    0.0,
                    0.0,
                    0.025929929918,
                    0.0,
                    0.0,
                    0.0,
                    -0.000155191834,
                    0.003612344032,
                    0.0,
                    -0.000222130141,
                    0.000036018441,
                    0.0,
                    0.408474646099,
                    -0.001795306624,
                    0.0,
                    -0.408097324124,
                    -0.001853055849,
                    0.0,
                    -0.000138790593,
                    0.000895006501,
                    0.0,
                    0.000081041368,
                    0.000060063801,
                    0.0,
                    -0.001795306624,
                    -0.026299434591,
                    0.0,
                    0.001853055849,
                    0.025344364289,
                    -0.000202218914,
                    0.0,
                    0.0,
                    -0.000194362609,
                    0.0,
                    0.0,
                    0.025929929918,
                    0.0,
                    0.0,
                    -0.025533348395,
                    0.0,
                    0.0,
                    0.0,
                    -0.000155191834,
                    -0.003612344032,
                    0.0,
                    -0.000222130141,
                    -0.000036018441,
                    0.0,
                    -0.408097324124,
                    0.001853055849,
                    0.0,
                    0.408474646099,
                    0.001795306624,
                    0.0,
                    0.000138790593,
                    0.000895006501,
                    0.0,
                    -0.000081041368,
                    0.000060063801,
                    0.0,
                    -0.001853055849,
                    0.025344364289,
                    0.0,
                    0.001795306624,
                    -0.026299434591,
                ]
            ).reshape((12, 12)),
            "PBE TOTAL ENERGY": -26.04681636799003,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-0.000000000000, -0.000000000000, -0.002915908298],
                    [0.000000000000, 0.000000000000, -0.000179663088],
                    [-0.000000000000, 0.026137498638, 0.001547791345],
                    [0.000000000000, -0.026137498638, 0.001547791345],
                ]
            ),
            "B3LYP TOTAL ENERGY": -26.1140057832643038,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [1.97215226e-31, 0.00000000e00, 3.32365409e-03],
                    [0.00000000e00, -1.05926147e-32, -5.83320285e-03],
                    [-5.87137508e-19, 1.91773664e-02, 1.25496529e-03],
                    [5.87137508e-19, -1.91773664e-02, 1.25496529e-03],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -26.0893142594464784,  # psi 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.00000000e00, 9.86076132e-31, 3.05278616e-03],
                    [1.23259516e-32, 0.00000000e00, -5.57401307e-03],
                    [-5.95843028e-19, 1.94617102e-02, 1.26080391e-03],
                    [5.95843028e-19, -1.94617102e-02, 1.26080391e-03],
                ]
            ),
            "WB97X TOTAL ENERGY": -26.09949538872584,  # psi 500,1202
            "WB97X TOTAL GRADIENT": np.array(  # psi 500,1202, findif-5
                [
                    0.0,
                    0.0,
                    0.003387202757,
                    0.0,
                    -0.0,
                    -0.006573452287,
                    -0.0,
                    0.01815316913,
                    0.001592986157,
                    0.0,
                    -0.01815316913,
                    0.001592986157,
                ]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_bh3p_dz_pk_uhf,
            "B2PLYP TOTAL ENERGY": -26.072947373723,  # nwc 99,590
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.15485993330517828,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03520162545964887,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.025490652204,
                    0.0,
                    0.013491755791,
                    -0.012745326102,
                    0.0,
                    -0.013491755791,
                    -0.012745326102,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [-0.02198237, 0.0, 0.0, 0.01099118, 0.0, 0.0, 0.01099118, 0.0, 0.0],
                    [0.0, 0.62517441, 0.0, 0.0, -0.31258721, 0.24829043, 0.0, -0.31258721, -0.24829043],
                    [0.0, 0.0, 0.39893991, 0.0, 0.19093143, -0.19946995, 0.0, -0.19093143, -0.19946995],
                    [0.01099118, 0.0, 0.0, -0.00995947, 0.0, 0.0, -0.00103171, 0.0, 0.0],
                    [0.0, -0.31258721, 0.19093143, 0.0, 0.33968697, -0.21961093, 0.0, -0.02709976, 0.0286795],
                    [0.0, 0.24829043, -0.19946995, 0.0, -0.21961093, 0.19053107, 0.0, -0.0286795, 0.00893888],
                    [0.01099118, 0.0, 0.0, -0.00103171, 0.0, 0.0, -0.00995947, 0.0, 0.0],
                    [0.0, -0.31258721, -0.19093143, 0.0, -0.02709976, -0.0286795, 0.0, 0.33968697, 0.21961093],
                    [0.0, -0.24829043, -0.19946995, 0.0, 0.0286795, 0.00893888, 0.0, 0.21961093, 0.19053107],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.17091690,  # occ
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03533349,  # occ
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.027091204409, 0.0, 0.014414444742, -0.013545602205, 0.0, -0.014414444742, -0.013545602205]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.17311842,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.003268429219,  # vcc only
            "CISD CORRELATION ENERGY": -0.16787805277043,  # vcc
            "QCISD CORRELATION ENERGY": -0.17409647936869,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00371845,  # vcc only
            "REMP2 CORRELATION ENERGY": -0.172501515758,  # occ, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.034548274152,  # occ, tight
            "LCCD CORRELATION ENERGY": -0.1770086091,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0341268118,  # occ
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.030699807076, 0.0, 0.016626789229, -0.015349903538, 0.0, -0.016626789229, -0.015349903538]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.1786081472,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.0344530,  # vcc
            "CCD CORRELATION ENERGY": -0.17274172367784,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.0337442,  # vcc
            "CCD TOTAL GRADIENT": np.array(
                [[0.0, 0.0, 0.02815376], [0.0, 0.01520857, -0.01407688], [0.0, -0.01520857, -0.01407688]]
            ),  # fcc findif in psi
            "CCD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.02427911, 0.0, 0.0, 0.01213956, 0.0, 0.0, 0.01213956, 0.0, 0.0],
                    [0.0, 0.61891369, 0.0, 0.0, -0.30945684, 0.2467696, 0.0, -0.30945684, -0.2467696],
                    [0.0, 0.0, 0.39863476, 0.0, 0.19000027, -0.19931738, 0.0, -0.19000027, -0.19931738],
                    [0.01213956, 0.0, 0.0, -0.01110176, 0.0, 0.0, -0.0010378, 0.0, 0.0],
                    [0.0, -0.30945684, 0.19000027, 0.0, 0.3374568, -0.21838493, 0.0, -0.02799995, 0.02838466],
                    [0.0, 0.2467696, -0.19931738, 0.0, -0.21838493, 0.18930325, 0.0, -0.02838466, 0.01001413],
                    [0.01213956, 0.0, 0.0, -0.0010378, 0.0, 0.0, -0.01110176, 0.0, 0.0],
                    [0.0, -0.30945684, -0.19000027, 0.0, -0.02799995, -0.02838466, 0.0, 0.3374568, 0.21838493],
                    [0.0, -0.2467696, -0.19931738, 0.0, 0.02838466, 0.01001413, 0.0, 0.21838493, 0.18930325],
                ]
            ),
            "BCCD CORRELATION ENERGY": -0.173674381451717,  # ccenergy
            "CC2 CORRELATION ENERGY": -0.156061046320285,  # ccenergy
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.026664454971, 0.0, 0.014117073057, -0.013332227485, 0.0, -0.014117073057, -0.013332227485]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.17387203707017695,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.033935818857082,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.029278727285,
                    0.0,
                    0.015813927533,
                    -0.014639363642,
                    0.0,
                    -0.015813927533,
                    -0.014639363642,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.02524909, 0.0, 0.0, 0.01262454, 0.0, 0.0, 0.01262454, 0.0, 0.0],
                    [0.0, 0.61582469, 0.0, 0.0, -0.30791234, 0.2459566, 0.0, -0.30791234, -0.2459566],
                    [0.0, 0.0, 0.39625423, 0.0, 0.18929354, -0.19812712, 0.0, -0.18929354, -0.19812712],
                    [0.01262454, 0.0, 0.0, -0.01154446, 0.0, 0.0, -0.00108008, 0.0, 0.0],
                    [0.0, -0.30791234, 0.18929354, 0.0, 0.3360106, -0.21762507, 0.0, -0.02809826, 0.02833153],
                    [0.0, 0.2459566, -0.19812712, 0.0, -0.21762507, 0.18819594, 0.0, -0.02833153, 0.00993118],
                    [0.01262454, 0.0, 0.0, -0.00108008, 0.0, 0.0, -0.01154446, 0.0, 0.0],
                    [0.0, -0.30791234, -0.18929354, 0.0, -0.02809826, -0.02833153, 0.0, 0.3360106, 0.21762507],
                    [0.0, -0.2459566, -0.19812712, 0.0, 0.02833153, 0.00993118, 0.0, 0.21762507, 0.18819594],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.003990106867,  # vcc
            "(T) CORRECTION ENERGY": -0.00384378,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.03116000949177],
                    [0.00000000000000, 0.01681030698770, -0.01558000474588],
                    [0.00000000000000, -0.01681030698770, -0.01558000474588],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # vcc fd
                [
                    [-0.026871482612, 0.0, 0.0, 0.013435741306, 0.0, 0.0, 0.013435741306, 0.0, 0.0],
                    [
                        0.0,
                        0.614216172202,
                        0.0,
                        0.0,
                        -0.307108086101,
                        0.245961925921,
                        0.0,
                        -0.307108086101,
                        -0.245961925921,
                    ],
                    [
                        0.0,
                        0.0,
                        0.39318444945,
                        0.0,
                        0.189186035018,
                        -0.196592224725,
                        0.0,
                        -0.189186035018,
                        -0.196592224725,
                    ],
                    [0.013435741306, 0.0, 0.0, -0.012279754441, 0.0, 0.0, -0.001155986865, 0.0, 0.0],
                    [
                        0.0,
                        -0.307108086101,
                        0.189186035018,
                        0.0,
                        0.334882090526,
                        -0.217573980469,
                        0.0,
                        -0.027774004425,
                        0.028387945452,
                    ],
                    [
                        0.0,
                        0.245961925921,
                        -0.196592224725,
                        0.0,
                        -0.217573980469,
                        0.18710085378,
                        0.0,
                        -0.028387945452,
                        0.009491370945,
                    ],
                    [0.013435741306, 0.0, 0.0, -0.001155986865, 0.0, 0.0, -0.012279754441, 0.0, 0.0],
                    [
                        0.0,
                        -0.307108086101,
                        -0.189186035018,
                        0.0,
                        -0.027774004425,
                        -0.028387945452,
                        0.0,
                        0.334882090526,
                        0.217573980469,
                    ],
                    [
                        0.0,
                        -0.245961925921,
                        -0.196592224725,
                        0.0,
                        0.028387945452,
                        0.009491370945,
                        0.0,
                        0.217573980469,
                        0.18710085378,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.00376395919,  # mrcc only
            "B(T) CORRECTION ENERGY": -0.004031201058265,  # ccenergy
            "CC3 CORRELATION ENERGY": -0.177840345513789,  # ccenergy
            "CCSDT-1A CORRELATION ENERGY": -0.17782828147829832,  # mrcc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.0311852], [0.0, 0.01683584, -0.0155926], [0.0, -0.01683584, -0.0155926]]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.17782563826176556,  # mrcc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.03117878], [0.0, 0.01683259, -0.01558939], [0.0, -0.01683259, -0.01558939]]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.177555974326,  # ecc
            "CCSDT-3 CORRELATION ENERGY": -0.17757120863770837,  # mrcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.03102835], [0.0, 0.01674637, -0.01551417], [0.0, -0.01674637, -0.01551417]]
            ),
            "CCSDT CORRELATION ENERGY": -0.17817199774450,  # vcc
            "OMP2 REFERENCE CORRECTION ENERGY": 0.001367125249,  # occ, tight
            "OMP2 CORRELATION ENERGY": -0.156193539287,  # occ, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.035974458829,  # occ, tight
            "OMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.026928654402,
                    0.0,
                    0.014250914503,
                    -0.013464327201,
                    0.0,
                    -0.014250914503,
                    -0.013464327201,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001173901427,  # occ, tight
            "OMP2.5 CORRELATION ENERGY": -0.164044987798,  # occ, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.035847752938,  # occ, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.027494393497,
                    0.0,
                    0.014585953475,
                    -0.013747196748,
                    0.0,
                    -0.014585953475,
                    -0.013747196748,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001024209316,  # occ, tight
            "OMP3 CORRELATION ENERGY": -0.171933638522,  # occ, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.035747352644,  # occ, tight
            "OMP3 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.028120207496,
                    0.0,
                    0.014956911575,
                    -0.014060103748,
                    0.0,
                    -0.014956911575,
                    -0.014060103748,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.00111233720168,  # occ, tight
            "OREMP2 CORRELATION ENERGY": -0.17359939909712,  # occ, tight
            "OREMP2 SAME-SPIN CORRELATION ENERGY": -0.03496212583967,  # occ, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.030225286483,
                    0.0,
                    0.016238459426,
                    -0.015112643242,
                    0.0,
                    -0.016238459426,
                    -0.015112643242,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0011118724,  # p4n
            "OLCCD CORRELATION ENERGY": -0.1781057943,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0344689234,  # occ
            "OLCCD TOTAL GRADIENT": np.array(  # occ
                [0.0, 0.0, 0.031877236853, -0.0, 0.017265501329, -0.015938618427, 0.0, -0.017265501329, -0.015938618427]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.000987794,  # qchem
            "OCCD CORRELATION ENERGY": -0.173720242,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem, rearranged
                [0.0, 0.0, 0.02915872, 0.0, 0.01575094, -0.01457936, 0.0, -0.01575094, -0.01457936]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "SVWN TOTAL ENERGY": -55.57342920898311,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    0.0,
                    -0.0,
                    0.035459696882,
                    -0.0,
                    0.020922069258,
                    -0.017729858581,
                    0.0,
                    -0.020922069258,
                    -0.017729858581,
                ]
            ).reshape((-1, 3)),
            "SVWN TOTAL HESSIAN": np.array(  # psi v1.9 99,590
                [
                    -0.03056359,
                    0.0,
                    -0.0,
                    0.01528971,
                    -0.0,
                    -0.0,
                    0.01528971,
                    -0.0,
                    -0.0,
                    0.0,
                    0.61365123,
                    -0.0,
                    0.0,
                    -0.30683213,
                    0.24717277,
                    0.0,
                    -0.30683213,
                    -0.24717277,
                    -0.0,
                    -0.0,
                    0.37850767,
                    0.0,
                    0.19398671,
                    -0.18925551,
                    0.0,
                    -0.19398671,
                    -0.18925551,
                    0.01528971,
                    0.0,
                    0.0,
                    -0.01456645,
                    0.0,
                    -0.0,
                    -0.00072257,
                    -0.0,
                    0.0,
                    -0.0,
                    -0.30683213,
                    0.19398671,
                    0.0,
                    0.3324625,
                    -0.22057959,
                    -0.0,
                    -0.02563036,
                    0.02659303,
                    -0.0,
                    0.24717277,
                    -0.18925551,
                    -0.0,
                    -0.22057959,
                    0.18253678,
                    0.0,
                    -0.02659303,
                    0.00671886,
                    0.01528971,
                    0.0,
                    0.0,
                    -0.00072257,
                    -0.0,
                    0.0,
                    -0.01456645,
                    0.0,
                    0.0,
                    -0.0,
                    -0.30683213,
                    -0.19398671,
                    -0.0,
                    -0.02563036,
                    -0.02659303,
                    0.0,
                    0.3324625,
                    0.22057959,
                    -0.0,
                    -0.24717277,
                    -0.18925551,
                    0.0,
                    0.02659303,
                    0.00671886,
                    0.0,
                    0.22057959,
                    0.18253678,
                ]
            ).reshape((9, 9)),
            "PBE TOTAL ENERGY": -55.81473169701641,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-0.000000000000, 0.000000000000, 0.035829930902],
                    [-0.000000000000, 0.019748673246, -0.017915051582],
                    [0.000000000000, -0.019748673246, -0.017915051582],
                ]
            ),
            "B3LYP TOTAL ENERGY": -55.8900885216505259,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.00000000e00, 5.91645679e-31, 2.68092690e-02],
                    [-4.57342645e-19, 1.49379444e-02, -1.34046506e-02],
                    [4.57342645e-19, -1.49379444e-02, -1.34046506e-02],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -55.8573454463695782,  # psi 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.00000000e00, 1.97215226e-30, 2.70819655e-02],
                    [-4.63472198e-19, 1.51381508e-02, -1.35409990e-02],
                    [4.63472198e-19, -1.51381508e-02, -1.35409990e-02],
                ]
            ),
            "WB97X TOTAL ENERGY": -55.86921249027672,  # psi 500,1202
            "WB97X TOTAL GRADIENT": np.array(  # psi 500,1202, findif-5
                [0.0, -0.0, 0.024476615666, -0.0, 0.01345957865, -0.012237852106, 0.0, -0.01345957865, -0.012237852106]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_adz_pk_uhf,
            "B2PLYP TOTAL ENERGY": -55.83362937,  # nwc 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.195530391293,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04161633,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.013731673196,
                    0.0,
                    0.005352105826,
                    -0.006865836598,
                    0.0,
                    -0.005352105826,
                    -0.006865836598,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [-0.01184178, 0.0, 0.0, 0.00592089, 0.0, 0.0, 0.00592089, 0.0, 0.0],
                    [0.0, 0.59647691, 0.0, 0.0, -0.29823846, 0.23338967, 0.0, -0.29823846, -0.23338967],
                    [0.0, 0.0, 0.38876074, 0.0, 0.17693874, -0.19438037, 0.0, -0.17693874, -0.19438037],
                    [0.00592089, 0.0, 0.0, -0.00473124, 0.0, 0.0, -0.00118965, 0.0, 0.0],
                    [0.0, -0.29823846, 0.17693874, 0.0, 0.32637561, -0.20516421, 0.0, -0.02813715, 0.02822547],
                    [0.0, 0.23338967, -0.19438037, 0.0, -0.20516421, 0.18496249, 0.0, -0.02822547, 0.00941788],
                    [0.00592089, 0.0, 0.0, -0.00118965, 0.0, 0.0, -0.00473124, 0.0, 0.0],
                    [0.0, -0.29823846, -0.17693874, 0.0, -0.02813715, -0.02822547, 0.0, 0.32637561, 0.20516421],
                    [0.0, -0.23338967, -0.19438037, 0.0, 0.02822547, 0.00941788, 0.0, 0.20516421, 0.18496249],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.21084677,  # occ
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04130360,  # occ
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.014436297067, 0.0, 0.005622513341, -0.007218148534, 0.0, -0.005622513341, -0.007218148534]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.21275859,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.004646780601,  # vcc only
            "CISD CORRELATION ENERGY": -0.20576009250440,  # vcc
            "QCISD CORRELATION ENERGY": -0.21351003301667,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00505349,  # vcc only
            "REMP2 CORRELATION ENERGY": -0.212452538490,  # occ, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.040608279810,  # occ, tight
            "LCCD CORRELATION ENERGY": -0.2167878305,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0401306050,  # occ
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.018147039011, 0.0, 0.007823362162, -0.009073519505, 0.0, -0.007823362162, -0.009073519505]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2185061347,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.0404661,  # vcc
            "CCD CORRELATION ENERGY": -0.2119966470870,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.0396775,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc
                [[0.0, 0.0, 0.015575676], [0.0, 0.0064826668, -0.007787838], [0.0, -0.0064826668, -0.007787838]]
            ),
            "CCD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.01343195, 0.0, 0.0, 0.00671598, 0.0, 0.0, 0.00671598, 0.0, 0.0],
                    [0.0, 0.58883226, 0.0, 0.0, -0.29441613, 0.23106683, 0.0, -0.29441613, -0.23106683],
                    [0.0, 0.0, 0.38825029, 0.0, 0.17519478, -0.19412514, 0.0, -0.17519478, -0.19412514],
                    [0.00671598, 0.0, 0.0, -0.00550282, 0.0, 0.0, -0.00121316, 0.0, 0.0],
                    [0.0, -0.29441613, 0.17519478, 0.0, 0.3235041, -0.20313082, 0.0, -0.02908797, 0.02793601],
                    [0.0, 0.23106683, -0.19412514, 0.0, -0.20313082, 0.1835696, 0.0, -0.02793601, 0.01055554],
                    [0.00671598, 0.0, 0.0, -0.00121316, 0.0, 0.0, -0.00550282, 0.0, 0.0],
                    [0.0, -0.29441613, -0.17519478, 0.0, -0.02908797, -0.02793601, 0.0, 0.3235041, 0.20313082],
                    [0.0, -0.23106683, -0.19412514, 0.0, 0.02793601, 0.01055554, 0.0, 0.20313082, 0.1835696],
                ]
            ),
            "BCCD CORRELATION ENERGY": -0.21309874,  # ccenergy
            "CC2 CORRELATION ENERGY": -0.196947901479342,  # ccenergy
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.015143356579, 0.0, 0.006071754877, -0.00757167829, 0.0, -0.006071754877, -0.00757167829]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.213298055172,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.039907245914335,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.016842165003,
                    0.0,
                    0.007150136873,
                    -0.008421082502,
                    0.0,
                    -0.007150136873,
                    -0.008421082502,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.01452417, 0.0, 0.0, 0.00726209, 0.0, 0.0, 0.00726209, 0.0, 0.0],
                    [0.0, 0.58571417, 0.0, 0.0, -0.29285708, 0.23028954, 0.0, -0.29285708, -0.23028954],
                    [0.0, 0.0, 0.38565345, 0.0, 0.17448698, -0.19282672, 0.0, -0.17448698, -0.19282672],
                    [0.00726209, 0.0, 0.0, -0.00599674, 0.0, 0.0, -0.00126535, 0.0, 0.0],
                    [0.0, -0.29285708, 0.17448698, 0.0, 0.32200974, -0.20238826, 0.0, -0.02915266, 0.02790128],
                    [0.0, 0.23028954, -0.19282672, 0.0, -0.20238826, 0.18240136, 0.0, -0.02790128, 0.01042537],
                    [0.00726209, 0.0, 0.0, -0.00126535, 0.0, 0.0, -0.00599674, 0.0, 0.0],
                    [0.0, -0.29285708, -0.17448698, 0.0, -0.02915266, -0.02790128, 0.0, 0.32200974, 0.20238826],
                    [0.0, -0.23028954, -0.19282672, 0.0, 0.02790128, 0.01042537, 0.0, 0.20238826, 0.18240136],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.005298735528,  # vcc
            "(T) CORRECTION ENERGY": -0.00516659,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.01904285993124],
                    [0.00000000000000, 0.00824911606964, -0.00952142996562],
                    [0.00000000000000, -0.00824911606964, -0.00952142996562],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # vcc fd
                [
                    [-0.01642193739, 0.0, 0.0, 0.008210968695, 0.0, 0.0, 0.008210968695, 0.0, 0.0],
                    [
                        0.0,
                        0.584003042573,
                        0.0,
                        0.0,
                        -0.292001521287,
                        0.230361256906,
                        0.0,
                        -0.292001521287,
                        -0.230361256906,
                    ],
                    [
                        0.0,
                        0.0,
                        0.38222875445,
                        0.0,
                        0.174417714376,
                        -0.191114377225,
                        0.0,
                        -0.174417714376,
                        -0.191114377225,
                    ],
                    [0.008210968695, 0.0, 0.0, -0.006834745201, 0.0, 0.0, -0.001376223494, 0.0, 0.0],
                    [
                        0.0,
                        -0.292001521287,
                        0.174417714376,
                        0.0,
                        0.320849017325,
                        -0.202389485641,
                        0.0,
                        -0.028847496039,
                        0.027971771265,
                    ],
                    [
                        0.0,
                        0.230361256906,
                        -0.191114377225,
                        0.0,
                        -0.202389485641,
                        0.181208918903,
                        0.0,
                        -0.027971771265,
                        0.009905458321,
                    ],
                    [0.008210968695, 0.0, 0.0, -0.001376223494, 0.0, 0.0, -0.006834745201, 0.0, 0.0],
                    [
                        0.0,
                        -0.292001521287,
                        -0.174417714376,
                        0.0,
                        -0.028847496039,
                        -0.027971771265,
                        0.0,
                        0.320849017325,
                        0.202389485641,
                    ],
                    [
                        0.0,
                        -0.230361256906,
                        -0.191114377225,
                        0.0,
                        0.027971771265,
                        0.009905458321,
                        0.0,
                        0.202389485641,
                        0.181208918903,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.0050643207,  # mrcc only
            "B(T) CORRECTION ENERGY": -0.00537543,  # ccenergy
            "CC3 CORRELATION ENERGY": -0.218616407232958,  # ccenergy
            "CCSDT-1A CORRELATION ENERGY": -0.218598725534,  # ecc
            "CCSDT-1B CORRELATION ENERGY": -0.218597572977,  # ecc
            "CCSDT-2 CORRELATION ENERGY": -0.218253889761,  # ecc
            "CCSDT-3 CORRELATION ENERGY": -0.21827269362849,  # vcc
            "CCSDT CORRELATION ENERGY": -0.21884856037681,  # vcc
            "OMP2 REFERENCE CORRECTION ENERGY": 0.001570623297,  # occ, tight
            "OMP2 CORRELATION ENERGY": -0.197071770029,  # occ, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.042459313035,  # occ, tight
            "OMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.015314078302,
                    0.0,
                    0.006156840975,
                    -0.007657039151,
                    0.0,
                    -0.006156840975,
                    -0.007657039151,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001350812375,  # occ, tight
            "OMP2.5 CORRELATION ENERGY": -0.204522615491,  # occ, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.042086148458,  # occ, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.015411835251,
                    0.0,
                    0.006162364663,
                    -0.007705917626,
                    0.0,
                    -0.006162364663,
                    -0.007705917626,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001177767799,  # occ, tight
            "OMP3 CORRELATION ENERGY": -0.212015174087,  # occ, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.041743491888,  # occ, tight
            "OMP3 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.015569868835,
                    0.0,
                    0.006203313670,
                    -0.007784934417,
                    0.0,
                    -0.006203313670,
                    -0.007784934417,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001287772670,  # occ, tight
            "OREMP2 CORRELATION ENERGY": -0.213723933328,  # occ, tight
            "OREMP2 SAME-SPIN CORRELATION ENERGY": -0.041063372010,  # occ, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.017901979677,
                    0.0,
                    0.007592988824,
                    -0.008950989839,
                    0.0,
                    -0.007592988824,
                    -0.008950989839,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0012856903,  # p4n
            "OLCCD CORRELATION ENERGY": -0.2180560836,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0405122800,  # occ
            "OLCCD TOTAL GRADIENT": np.array(  # occ
                [
                    -0.0,
                    0.0,
                    0.019472272298,
                    -0.0,
                    0.008522161738,
                    -0.009736136149,
                    0.0,
                    -0.008522161738,
                    -0.009736136149,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.001155093,  # qchem
            "OCCD CORRELATION ENERGY": -0.213139955,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem, rearranged
                [0, 0, 0.01669862, 0, 0.00707632, -0.00834931, 0, -0.00707632, -0.00834931],
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "SVWN TOTAL ENERGY": -55.59258481155869,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    0.0,
                    -0.0,
                    0.028874430294,
                    -0.0,
                    0.016569686304,
                    -0.014437276469,
                    0.0,
                    -0.016569686304,
                    -0.014437276469,
                ]
            ).reshape((-1, 3)),
            "SVWN TOTAL HESSIAN": np.array(  # psi v1.9 99,590
                [
                    -0.02488189,
                    -0.0,
                    0.0,
                    0.01245024,
                    0.0,
                    -0.0,
                    0.01245024,
                    0.0,
                    0.0,
                    -0.0,
                    0.59347392,
                    -0.0,
                    0.0,
                    -0.29674373,
                    0.23725288,
                    -0.0,
                    -0.29674373,
                    -0.23725288,
                    0.0,
                    -0.0,
                    0.37093753,
                    0.0,
                    0.18502874,
                    -0.18547044,
                    0.0,
                    -0.18502874,
                    -0.18547044,
                    0.01245024,
                    0.0,
                    0.0,
                    -0.01170876,
                    -0.0,
                    -0.0,
                    -0.00074289,
                    0.0,
                    0.0,
                    0.0,
                    -0.29674373,
                    0.18502874,
                    -0.0,
                    0.32275381,
                    -0.21114139,
                    -0.0,
                    -0.02601004,
                    0.02611207,
                    -0.0,
                    0.23725288,
                    -0.18547044,
                    -0.0,
                    -0.21114139,
                    0.17827787,
                    -0.0,
                    -0.02611207,
                    0.00719218,
                    0.01245024,
                    -0.0,
                    0.0,
                    -0.00074289,
                    -0.0,
                    -0.0,
                    -0.01170876,
                    -0.0,
                    -0.0,
                    0.0,
                    -0.29674373,
                    -0.18502874,
                    0.0,
                    -0.02601004,
                    -0.02611207,
                    -0.0,
                    0.32275381,
                    0.21114139,
                    0.0,
                    -0.23725288,
                    -0.18547044,
                    0.0,
                    0.02611207,
                    0.00719218,
                    -0.0,
                    0.21114139,
                    0.17827787,
                ]
            ).reshape((9, 9)),
            "PBE TOTAL ENERGY": -55.83097248811423,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.000000000000, 0.000000000000, 0.029614373344],
                    [-0.000000000000, 0.015644026348, -0.014807315411],
                    [0.000000000000, -0.015644026348, -0.014807315411],
                ]
            ),
            "B3LYP TOTAL ENERGY": -55.9072559501566744,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.00000000e00, 3.15544362e-30, 1.99929859e-02],
                    [-3.19390336e-19, 1.04320800e-02, -9.99655055e-03],
                    [3.19390336e-19, -1.04320800e-02, -9.99655055e-03],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -55.8744982972056121,  # psi 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-7.88860905e-31, 0.00000000e00, 2.02715602e-02],
                    [-3.25473520e-19, 1.06307719e-02, -1.01358376e-02],
                    [3.25473520e-19, -1.06307719e-02, -1.01358376e-02],
                ]
            ),
            "WB97X TOTAL ENERGY": -55.88527882823288,  # psi 500,1202
            "WB97X TOTAL GRADIENT": np.array(  # psi 500,1202, findif-5
                [
                    0.0,
                    -0.0,
                    0.017275816568,
                    -0.0,
                    0.008857017471,
                    -0.008637144304,
                    0.0,
                    -0.008857017471,
                    -0.008637144304,
                ]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_qz2p_pk_uhf,
            "B2PLYP TOTAL ENERGY": -55.861074179189,  # nwc 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "CISD CORRELATION ENERGY": -0.08129169589155,  # vcc, sd == sc
            "LCCSD CORRELATION ENERGY": -0.08620498,  # vcc nyi != p4n, sd == sc
            "LCCSD OPPOSITE-SPIN CORRELATION ENERGY": -0.08239000,  # vcc nyi, sd == sc
            "CCD CORRELATION ENERGY": -0.08115920440309,  # vcc nyi == tce, sd == sc
            "CCSD CORRELATION ENERGY": -0.08357160616,  # sd == sc
            "CCSD SINGLES ENERGY": -0.0011743271,  # sd == sd
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.00244892164,  # sd == sc
            "CCSD TOTAL GRADIENT": np.array(  # sd == sc
                [
                    0.0,
                    0.0,
                    0.005568141758,
                    0.0,
                    0.0,
                    -0.005430974166,
                    0.0,
                    0.014884143028,
                    -0.000068583796,
                    0.0,
                    -0.014884143028,
                    -0.000068583796,
                ]
            ).reshape((-1, 3)),
            "CCSDT CORRELATION ENERGY": -0.08463562959121,  # vcc,  sd == sc
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.060478115157,
            "MP2 SINGLES ENERGY": -0.000694049865,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.002004909679,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    # switches sign from unkn ref
                    0.000000000000000,
                    0.000000000000000,
                    0.013594741747853,
                    0.000000000000000,
                    0.000000000000000,
                    -0.013127629532095,
                    0.000000000000000,
                    0.010308255599051,
                    -0.000233556107879,
                    0.000000000000000,
                    -0.010308255599051,
                    -0.000233556107879,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # vcc by psi fd
                [
                    [
                        3.87098460e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.45378253e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.91398962e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.91398962e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        3.77896598e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.31078241e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.65908217e-04,
                        4.34462348e-04,
                        0.00000000e00,
                        2.65908217e-04,
                        -4.34462348e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.17564784e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.15517211e-01,
                        0.00000000e00,
                        1.14220963e-03,
                        -1.02378661e-03,
                        0.00000000e00,
                        -1.14220963e-03,
                        -1.02378661e-03,
                    ],
                    [
                        -4.45378253e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.80247034e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74343907e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74343907e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -4.31078241e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.71699407e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.03105828e-04,
                        8.16927983e-05,
                        0.00000000e00,
                        -2.03105828e-04,
                        -8.16927983e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.15517211e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.15737900e-01,
                        0.00000000e00,
                        -4.73512607e-05,
                        -1.10344685e-04,
                        0.00000000e00,
                        4.73512607e-05,
                        -1.10344685e-04,
                    ],
                    [
                        2.91398962e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74343907e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -7.40034143e-03,
                        0.00000000e00,
                        0.00000000e00,
                        7.28328638e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.65908217e-04,
                        1.14220963e-03,
                        0.00000000e00,
                        -2.03105828e-04,
                        -4.73512607e-05,
                        0.00000000e00,
                        4.11395088e-01,
                        -8.05506756e-04,
                        0.00000000e00,
                        -4.11457890e-01,
                        -2.89351610e-04,
                    ],
                    [
                        0.00000000e00,
                        4.34462348e-04,
                        -1.02378661e-03,
                        0.00000000e00,
                        8.16927983e-05,
                        -1.10344685e-04,
                        0.00000000e00,
                        -8.05506756e-04,
                        -4.62479868e-03,
                        0.00000000e00,
                        2.89351610e-04,
                        5.75892997e-03,
                    ],
                    [
                        2.91398962e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74343907e-04,
                        0.00000000e00,
                        0.00000000e00,
                        7.28328638e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -7.40034143e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.65908217e-04,
                        -1.14220963e-03,
                        0.00000000e00,
                        -2.03105828e-04,
                        4.73512607e-05,
                        0.00000000e00,
                        -4.11457890e-01,
                        2.89351610e-04,
                        0.00000000e00,
                        4.11395088e-01,
                        8.05506756e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.34462348e-04,
                        -1.02378661e-03,
                        0.00000000e00,
                        -8.16927983e-05,
                        -1.10344685e-04,
                        0.00000000e00,
                        -2.89351610e-04,
                        5.75892997e-03,
                        0.00000000e00,
                        8.05506756e-04,
                        -4.62479868e-03,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.075906990477,  # cfour
            "MP3 SINGLES ENERGY": -0.000916783719,  # cfour
            "MP3 TOTAL GRADIENT": np.array(  # vcc by psi fd
                [
                    [0.0, 0.0, 0.010131197027],
                    [0.0, 0.0, -0.009902977129],
                    [0.0, 0.012728956474, -0.000114109949],
                    [0.0, -0.012728956474, -0.000114109949],
                ]
            ),
            "MP4(SDQ) CORRELATION ENERGY": -0.08034256,  # vcc only, sc
            "MP4(T) CORRECTION ENERGY": -0.000301596734,  # vcc only, sc
            "ZAPT2 CORRELATION ENERGY": -0.059970897396607,  # detci (ZAPT uses special canonicalization but grouping here with sc)
            "CISD CORRELATION ENERGY": -0.08129169589155,  # vcc, sc
            # "CISD CORRELATION ENERGY": -0.08142433,  # detci != vcc ???
            "LCCD CORRELATION ENERGY": -0.0834094914,  # p4n (likely sd since tce matches but leaving here until detangled)
            # "LCCSD CORRELATION ENERGY": -0.0861427228,  # p4n (was uncommented)
            "LCCSD CORRELATION ENERGY": -0.08620498,  # vcc nyi != p4n
            "LCCSD OPPOSITE-SPIN CORRELATION ENERGY": -0.08239000,  # vcc nyi
            "CCD CORRELATION ENERGY": -0.08115920440309,  # vcc nyi == tce, sd == sc
            "BCCD CORRELATION ENERGY": -0.08369728,  # ccenergy
            "CC2 CORRELATION ENERGY": -0.061327748089029,  # ccenergy only, sc
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [
                    0.0,
                    0.0,
                    0.012124989222,
                    0.0,
                    0.0,
                    -0.011720065418,
                    0.0,
                    0.010555684617,
                    -0.000202461902,
                    0.0,
                    -0.010555684617,
                    -0.000202461902,
                ]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.08357160616,
            "CCSD SINGLES ENERGY": -0.0011743271,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.00244892164,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.005568141758,
                    0.0,
                    0.0,
                    -0.005430974166,
                    0.0,
                    0.014884143028,
                    -0.000068583796,
                    0.0,
                    -0.014884143028,
                    -0.000068583796,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.000713766189,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.00481175567113],
                    [0.00000000000000, 0.00000000000000, -0.00475864799255],
                    [0.00000000000000, 0.01496337127168, -0.00002655383929],
                    [0.00000000000000, -0.01496337127168, -0.00002655383929],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.000699810766,  # mrcc only, sc
            "B(T) CORRECTION ENERGY": -0.00063370,  # ccenergy
            "CC3 CORRELATION ENERGY": -0.084278230953036,  # ccenergy only, sc
            "CCSDT CORRELATION ENERGY": -0.08463562959121,  # vcc
            "OMP2 REFERENCE CORRECTION ENERGY": -0.001128749117,  # occ, tight
            "OMP2 CORRELATION ENERGY": -0.061388690115,  # occ, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.001971155774,  # occ, tight
            "OMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.012017560624,
                    0.0,
                    0.000000000000,
                    -0.011623175532,
                    0.0,
                    0.010549441086,
                    -0.000197192546,
                    0.0,
                    -0.010549441086,
                    -0.000197192546,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.000959439798,  # occ, tight
            "OMP2.5 CORRELATION ENERGY": -0.069204306750,  # occ, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.002198682531,  # occ, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.010260857036,
                    0.0,
                    0.000000000000,
                    -0.009981284641,
                    0.0,
                    0.011818192175,
                    -0.000139786197,
                    0.0,
                    -0.011818192175,
                    -0.000139786197,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": -0.000746435584,  # occ, tight
            "OMP3 CORRELATION ENERGY": -0.077055367311,  # occ, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.002426022153,  # occ, tight
            "OMP3 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.008456347254,
                    0.0,
                    0.000000000000,
                    -0.008291901023,
                    0.0,
                    0.013097750617,
                    -0.000082223115,
                    0.0,
                    -0.013097750617,
                    -0.000082223115,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.000463092111,  # occ, tight
            "OREMP2 CORRELATION ENERGY": -0.080520111623,  # occ, tight
            "OREMP2 SAME-SPIN CORRELATION ENERGY": -0.002398210414,  # occ, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.005898074368,
                    0.0,
                    0.000000000000,
                    -0.005790150595,
                    0.0,
                    0.014384643442,
                    -0.000053961887,
                    0.0,
                    -0.014384643442,
                    -0.000053961887,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": -0.0000399018,  # p4n
            "OLCCD CORRELATION ENERGY": -0.0862654609,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0024486744,  # occ
            "OLCCD TOTAL GRADIENT": np.array(  # occ
                [
                    -0.0,
                    0.0,
                    0.002952428137,
                    0.0,
                    0.0,
                    -0.002937832352,
                    -0.0,
                    0.015746258328,
                    -0.000007297893,
                    0.0,
                    -0.015746258328,
                    -0.000007297893,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": -0.000358562,  # qchem
            "OCCD CORRELATION ENERGY": -0.083713042,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem 3-pt findif
                [
                    0.0,
                    0.0,
                    5.200665059079133e-03,
                    0.0,
                    0.0,
                    -5.057359056039087e-03,
                    0.0,
                    1.487654179061328e-02,
                    -7.165259590676168e-05,
                    0.0,
                    -1.487654176147304e-02,
                    -7.165259308673899e-05,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "PBE TOTAL ENERGY": -26.046275584237,  # nwc 99,590
            "PBE TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [7.84827311e-37, -2.13273843e-19, -1.92100000e-03],
                    [4.63705881e-37, -1.26010313e-19, -1.13500000e-03],
                    [-1.12670315e-35, 2.60500000e-02, 1.52800000e-03],
                    [1.00184983e-35, -2.60500000e-02, 1.52800000e-03],
                ]
            ),
            "B3LYP TOTAL ENERGY": -26.113222717203,  # nwc 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-1.96104690e-36, 5.32907052e-19, 4.80000000e-03],
                    [2.94483876e-36, -8.00248756e-19, -7.20800000e-03],
                    [-8.25273903e-36, 1.89960000e-02, 1.20400000e-03],
                    [7.26894717e-36, -1.89960000e-02, 1.20400000e-03],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -26.088528482841,  # nwc 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-1.85523208e-36, 5.04152275e-19, 4.54100000e-03],
                    [2.84351800e-36, -7.72715225e-19, -6.96000000e-03],
                    [-8.37121895e-36, 1.92800000e-02, 1.21000000e-03],
                    [7.38252447e-36, -1.92800000e-02, 1.21000000e-03],
                ]
            ),
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "CISD CORRELATION ENERGY": -0.17186174301557,  # vcc, sd == sc
            "LCCSD CORRELATION ENERGY": -0.18308771,  # vcc nyi != p4n, sd == sc
            "LCCSD OPPOSITE-SPIN CORRELATION ENERGY": -0.14511930,  # vcc nyi, sd == sc
            "CCD CORRELATION ENERGY": -0.17475513572639,  # vcc nyi == tce, sd == sc
            "CCSD CORRELATION ENERGY": -0.178236032911,  # == sc
            "CCSD SINGLES ENERGY": -0.00327524740575,  # == sc
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.033982707798170,  # == sc
            "CCSD TOTAL GRADIENT": np.array(  # == sc
                [
                    0.0,
                    0.0,
                    0.029273628227,
                    0.0,
                    0.015808308241,
                    -0.014636814114,
                    0.0,
                    -0.015808308241,
                    -0.014636814114,
                ]
            ).reshape((-1, 3)),
            "CCSDT CORRELATION ENERGY": -0.18258437583017,  # vcc, sd == sc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.15949744108346664,
            "MP2 SINGLES ENERGY": -0.0028296307982793997,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03541709278508698,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.025609525826,
                    0.0,
                    0.013506941035,
                    -0.012804762913,
                    0.0,
                    -0.013506941035,
                    -0.012804762913,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # vcc by psi fd
                [
                    [-0.02208471, 0.0, 0.0, 0.01104236, 0.0, 0.0, 0.01104236, 0.0, 0.0],
                    [0.0, 0.62483805, 0.0, 0.0, -0.31241902, 0.24820063, 0.0, -0.31241902, -0.24820063],
                    [0.0, 0.0, 0.39861928, 0.0, 0.19070358, -0.19930964, 0.0, -0.19070358, -0.19930964],
                    [0.01104236, 0.0, 0.0, -0.00999006, 0.0, 0.0, -0.0010523, 0.0, 0.0],
                    [0.0, -0.31241902, 0.19070358, 0.0, 0.33943262, -0.2194521, 0.0, -0.0270136, 0.02874852],
                    [0.0, 0.24820063, -0.19930964, 0.0, -0.2194521, 0.19041148, 0.0, -0.02874852, 0.00889816],
                    [0.01104236, 0.0, 0.0, -0.0010523, 0.0, 0.0, -0.00999006, 0.0, 0.0],
                    [0.0, -0.31241902, -0.19070358, 0.0, -0.0270136, -0.02874852, 0.0, 0.33943262, 0.2194521],
                    [0.0, -0.24820063, -0.19930964, 0.0, 0.02874852, 0.00889816, 0.0, 0.2194521, 0.19041148],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.17571623051,  # cfour
            "MP3 SINGLES ENERGY": -0.003236109562,  # cfour
            "MP3 TOTAL GRADIENT": np.array(  # vcc by psi fd
                [
                    [0.00000000000000, 0.00000000000000, 0.02731056234441],
                    [0.00000000000000, 0.01453084545981, -0.01365528117220],
                    [0.00000000000000, -0.01453084545981, -0.01365528117220],
                ]
            ),
            "MP4(SDQ) CORRELATION ENERGY": -0.17767316,  # vcc only, sc
            "MP4(T) CORRECTION ENERGY": -0.003373917694,  # vcc only, sc
            "ZAPT2 CORRELATION ENERGY": -0.15875585,  # gamess
            "CISD CORRELATION ENERGY": -0.17186174301557,  # vcc, sc
            # "CISD CORRELATION ENERGY": -0.1723668643052676,  # detci != vcc ???
            "LCCD CORRELATION ENERGY": -0.1791714105,  # p4n (likely sd since tce matches but leaving here until detangled)
            # "LCCSD CORRELATION ENERGY": -0.1830545845,  # p4n (was uncommented)
            "LCCSD CORRELATION ENERGY": -0.18308771,  # vcc nyi != p4n
            "LCCSD OPPOSITE-SPIN CORRELATION ENERGY": -0.14511930,  # vcc nyi
            "CCD CORRELATION ENERGY": -0.17475513572639,  # vcc nyi == tce, sd=sc
            "BCCD CORRELATION ENERGY": -0.178088085354507,  # ccenergy
            "CC2 CORRELATION ENERGY": -0.160653595161691,  # ccenergy only, sc
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.026788245572, 0.0, 0.014198292126, -0.013394122786, 0.0, -0.014198292126, -0.013394122786]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.178236032911,
            "CCSD SINGLES ENERGY": -0.00327524740575,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.033982707798170,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.029273628227,
                    0.0,
                    0.015808308241,
                    -0.014636814114,
                    0.0,
                    -0.015808308241,
                    -0.014636814114,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.003901085777,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.03117381407549],
                    [0.00000000000000, 0.01681897058340, -0.01558690703775],
                    [0.00000000000000, -0.01681897058340, -0.0155869070377],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.18258437583017,  # vcc
            "A-(T) CORRECTION ENERGY": -0.003817334300,  # mrcc only, sc
            "B(T) CORRECTION ENERGY": -0.004031201058265,  # ccenergy
            "CC3 CORRELATION ENERGY": -0.182256450138473,  # ccenergy only, sc
            "OMP2 REFERENCE CORRECTION ENERGY": -0.003046592318,  # occ, tight
            "OMP2 CORRELATION ENERGY": -0.160607243189,  # occ, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.035974454782,  # occ, tight
            "OMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.026928663612,
                    0.0,
                    0.014250907985,
                    -0.013464331806,
                    0.0,
                    -0.014250907985,
                    -0.013464331806,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.003239801580,  # occ, tight
            "OMP2.5 CORRELATION ENERGY": -0.168458692138,  # occ, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.035847753329,  # occ, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.027494386758,
                    0.0,
                    0.014585950370,
                    -0.013747193379,
                    0.0,
                    -0.014585950370,
                    -0.013747193379,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": -0.003389494582,  # occ, tight
            "OMP3 CORRELATION ENERGY": -0.176347342433,  # occ, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.035747352645,  # occ, tight
            "OMP3 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.028120207297,
                    0.0,
                    0.014956911348,
                    -0.014060103648,
                    0.0,
                    -0.014956911348,
                    -0.014060103648,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.003301366872,  # occ, tight
            "OREMP2 CORRELATION ENERGY": -0.178013102573,  # occ, tight
            "OREMP2 SAME-SPIN CORRELATION ENERGY": -0.034962125667,  # occ, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.030225286965,
                    0.0,
                    0.016238460656,
                    -0.015112643483,
                    0.0,
                    -0.016238460656,
                    -0.015112643483,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": -0.0033018315,  # p4n
            "OLCCD CORRELATION ENERGY": -0.1825194982,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0344689234,  # occ
            "OLCCD TOTAL GRADIENT": np.array(  # occ
                [
                    -0.0,
                    0.0,
                    0.031877236593,
                    -0.0,
                    0.017265501452,
                    -0.015938618296,
                    0.0,
                    -0.017265501452,
                    -0.015938618296,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": -0.003426081,  # qchem
            "OCCD CORRELATION ENERGY": -0.178133946,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem 3-pt findif
                [
                    0.0,
                    2.629008122312371e-10,
                    2.915992176610871e-02,
                    0.0,
                    1.575113356366842e-02,
                    -1.457995768916476e-02,
                    0.0,
                    -1.575113354590485e-02,
                    -1.457995755771435e-02,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "PBE TOTAL ENERGY": -55.813318056978,  # nwc 99,590
            "PBE TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-2.94106602e-19, 2.94106602e-19, 3.56590000e-02],
                    [1.47049177e-19, 1.96000000e-02, -1.78290000e-02],
                    [1.47049177e-19, -1.96000000e-02, -1.78290000e-02],
                ]
            ),
            "B3LYP TOTAL ENERGY": -55.888559899085,  # nwc 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-2.19134533e-19, 2.19134533e-19, 2.65690000e-02],
                    [1.09571390e-19, 1.47410000e-02, -1.32850000e-02],
                    [1.09571390e-19, -1.47410000e-02, -1.32850000e-02],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -55.855798924171,  # nwc 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-2.21336683e-19, 2.21336683e-19, 2.68360000e-02],
                    [1.10668341e-19, 1.49380000e-02, -1.34180000e-02],
                    [1.10668341e-19, -1.49380000e-02, -1.34180000e-02],
                ]
            ),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "CISD CORRELATION ENERGY": -0.20988111766307,  # vcc, sd == sc
            "LCCSD CORRELATION ENERGY": -0.22316121,  # vcc nyi != p4n ???, sd == sc
            "LCCSD OPPOSITE-SPIN CORRELATION ENERGY": -0.17910640,  # vcc nyi, sd == sc
            "CCD CORRELATION ENERGY": -0.21415436056607,  # vcc nyi == tce, sd == sc
            "CCSD CORRELATION ENERGY": -0.217849506326,  # sd == sc
            "CCSD SINGLES ENERGY": -0.00338286103325,  # sd == sc
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.039891470497466,  # sd == sc
            "CCSD TOTAL GRADIENT": np.array(  # sd == sc
                [
                    0.0,
                    0.0,
                    0.016833254665,
                    0.0,
                    0.007144029475,
                    -0.008416627332,
                    0.0,
                    -0.007144029475,
                    -0.008416627332,
                ]
            ).reshape((-1, 3)),
            "CCSDT CORRELATION ENERGY": -0.22345631762464,  # vcc,  sd == sc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.2005395272,
            "MP2 SINGLES ENERGY": -0.00298375,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04178535,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.0138883429,
                    0.0,
                    0.005389090661,
                    -0.00694417145,
                    0.0,
                    -0.005389090661,
                    -0.00694417145,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # vcc by psi fd
                [
                    [-0.01197655, 0.0, 0.0, 0.00598828, 0.0, 0.0, 0.00598828, 0.0, 0.0],
                    [0.0, 0.59600884, 0.0, 0.0, -0.29800442, 0.23326184, 0.0, -0.29800442, -0.23326184],
                    [0.0, 0.0, 0.38838238, 0.0, 0.17666871, -0.19419119, 0.0, -0.17666871, -0.19419119],
                    [0.00598828, 0.0, 0.0, -0.0047771, 0.0, 0.0, -0.00121118, 0.0, 0.0],
                    [0.0, -0.29800442, 0.17666871, 0.0, 0.32604541, -0.20496528, 0.0, -0.02804099, 0.02829657],
                    [0.0, 0.23326184, -0.19419119, 0.0, -0.20496528, 0.18480667, 0.0, -0.02829657, 0.00938452],
                    [0.00598828, 0.0, 0.0, -0.00121118, 0.0, 0.0, -0.0047771, 0.0, 0.0],
                    [0.0, -0.29800442, -0.17666871, 0.0, -0.02804099, -0.02829657, 0.0, 0.32604541, 0.20496528],
                    [0.0, -0.23326184, -0.19419119, 0.0, 0.02829657, 0.00938452, 0.0, 0.20496528, 0.18480667],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.215858882347,  # cfour
            "MP3 SINGLES ENERGY": -0.003348697377,  # cfour
            "MP3 TOTAL GRADIENT": np.array(  # vcc by fd psi
                [
                    [0.0, 0.0, 0.014669237518],
                    [0.0, 0.005743113908, -0.007334618759],
                    [0.0, -0.005743113908, -0.007334618759],
                ]
            ),
            "MP4(SDQ) CORRELATION ENERGY": -0.21748938,  # vcc only, sc
            "MP4(T) CORRECTION ENERGY": -0.004777946072,  # vcc only, sc
            "ZAPT2 CORRELATION ENERGY": -0.19975428,  # gamess
            "CISD CORRELATION ENERGY": -0.20988111766307,  # vcc
            # "CISD CORRELATION ENERGY": -0.21038651,  # detci != vcc ???
            "LCCD CORRELATION ENERGY": -0.2191039411,  # p4n (likely sd since tce matches but leaving here until detangled)
            # "LCCSD CORRELATION ENERGY": -0.2231241199,  # p4n  (was uncommented)
            "LCCSD CORRELATION ENERGY": -0.22316121,  # vcc nyi != p4n ???
            "LCCSD OPPOSITE-SPIN CORRELATION ENERGY": -0.17910640,  # vcc nyi
            "CCD CORRELATION ENERGY": -0.21415436056607,  # vcc nyi == tce, sd == sc
            "BCCD CORRELATION ENERGY": -0.21770845,  # ccenergy
            "CC2 CORRELATION ENERGY": -0.201748666827081,  # ccenergy only, sc
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.015270065209, 0.0, 0.006152718989, -0.007635032604, 0.0, -0.006152718989, -0.007635032604]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.217849506326,
            "CCSD SINGLES ENERGY": -0.00338286103325,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.039891470497466,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.016833254665,
                    0.0,
                    0.007144029475,
                    -0.008416627332,
                    0.0,
                    -0.007144029475,
                    -0.008416627332,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.005233938447,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.01905797562047],
                    [0.00000000000000, 0.00825727647494, -0.00952898781024],
                    [0.00000000000000, -0.00825727647494, -0.00952898781024],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.005126584825,  # mrcc only, sc
            "B(T) CORRECTION ENERGY": -0.00537543,  # ccenergy
            "CC3 CORRELATION ENERGY": -0.223229153563849,  # ccenergy only, sc
            "CCSDT CORRELATION ENERGY": -0.22345631762464,  # vcc
            "OMP2 REFERENCE CORRECTION ENERGY": -0.003039092557,  # occ, tight
            "OMP2 CORRELATION ENERGY": -0.201681478150,  # occ, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.042459312118,  # occ, tight
            "OMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.015314087546,
                    0.0,
                    0.006156841239,
                    -0.007657043773,
                    0.0,
                    -0.006156841239,
                    -0.007657043773,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.003258895747,  # occ, tight
            "OMP2.5 CORRELATION ENERGY": -0.209132323529,  # occ, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.042086148395,  # occ, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.015411833442,
                    0.0,
                    0.006162364110,
                    -0.007705916721,
                    0.0,
                    -0.006162364110,
                    -0.007705916721,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": -0.003431940343,  # occ, tight
            "OMP3 CORRELATION ENERGY": -0.216624882230,  # occ, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.041743491868,  # occ, tight
            "OMP3 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.015569867435,
                    0.0,
                    0.006203313291,
                    -0.007784933717,
                    0.0,
                    -0.006203313291,
                    -0.007784933717,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.003321935448,  # occ, tight
            "OREMP2 CORRELATION ENERGY": -0.218333639848,  # occ, tight
            "OREMP2 SAME-SPIN CORRELATION ENERGY": -0.041063372384,  # occ, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # occ, tight
                [
                    0.0,
                    0.000000000000,
                    0.017901994172,
                    0.0,
                    0.007592995571,
                    -0.008950997086,
                    0.0,
                    -0.007592995571,
                    -0.008950997086,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": -0.0033240178,  # p4n
            "OLCCD CORRELATION ENERGY": -0.2226657917,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0405122800,  # occ
            "OLCCD TOTAL GRADIENT": np.array(  # occ
                [0.0, 0.0, 0.019472272302, -0.0, 0.008522161726, -0.009736136151, 0.0, -0.008522161726, -0.009736136151]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": -0.00345534,  # qchem
            "OCCD CORRELATION ENERGY": -0.217749658,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem 3-pt findif
                [
                    0.0,
                    2.948752353404416e-10,
                    1.669883480914791e-02,
                    0.0,
                    7.076450128096212e-03,
                    -8.349414756025908e-03,
                    0.0,
                    -7.076449627163584e-03,
                    -8.349414585495651e-03,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "PBE TOTAL ENERGY": -55.829414170216,  # nwc 99,590
            "PBE TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-2.43250961e-19, 2.43250961e-19, 2.94930000e-02],
                    [1.21621356e-19, 1.55340000e-02, -1.47460000e-02],
                    [1.21621356e-19, -1.55340000e-02, -1.47460000e-02],
                ]
            ),
            "B3LYP TOTAL ENERGY": -55.905608800925,  # nwc 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-1.63082808e-19, 1.63082808e-19, 1.97730000e-02],
                    [8.15372799e-20, 1.02630000e-02, -9.88600000e-03],
                    [8.15372799e-20, -1.02630000e-02, -9.88600000e-03],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -55.872832273661,  # nwc 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-1.65334444e-19, 1.65334444e-19, 2.00460000e-02],
                    [8.26672220e-20, 1.04580000e-02, -1.00230000e-02],
                    [8.26672220e-20, -1.04580000e-02, -1.00230000e-02],
                ]
            ),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CONV-FC-CONV  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.201627516796,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0534895025483176,
            "MP2 TOTAL GRADIENT": np.array(
                [  # fnocc findif-5 fc pk+conv
                    0.00000000000000,
                    0.00000000000000,
                    0.00317450456474,
                    0.00000000000000,
                    0.00000000000000,
                    -0.00317450456474,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [-0.00183193, 0.0, 0.0, 0.00183193, 0.0, 0.0],
                    [0.0, -0.00183193, 0.0, 0.0, 0.00183193, 0.0],
                    [0.0, 0.0, 0.64212587, 0.0, 0.0, -0.64212587],
                    [0.00183193, 0.0, 0.0, -0.00183193, 0.0, 0.0],
                    [0.0, 0.00183193, 0.0, 0.0, -0.00183193, 0.0],
                    [0.0, 0.0, -0.64212587, 0.0, 0.0, 0.64212587],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.20453611,  # fnocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04918710,  # fnocc
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000579954803, 0.0, 0.0, 0.000579954803]).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.20670909,  # vcc
            "MP4(T) CORRECTION ENERGY": -0.002027930587,  # vcc
            "CISD CORRELATION ENERGY": -0.20039201434963,  # vcc
            "QCISD CORRELATION ENERGY": -0.20699674383631,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00180789,  # vcc
            "FCI CORRELATION ENERGY": -0.2092292951,  # detci
            "REMP2 CORRELATION ENERGY": -0.20642733451785,  # occ, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.04882718129882,  # occ, tight
            "LCCD CORRELATION ENERGY": -0.2079585027,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.047635656759,  # fnocc
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.002535502057, 0.0, 0.0, -0.002535502057]).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2087915976,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.047754723454,  # fnocc
            "CEPA(1) CORRELATION ENERGY": -0.2064021969,  # fnocc
            "CEPA(1) SINGLES ENERGY": 0.0,
            "CEPA(1) SAME-SPIN CORRELATION ENERGY": -0.0473671686,  # fnocc
            "CEPA(3) CORRELATION ENERGY": -0.2047660232,  # fnocc
            "CEPA(3) SINGLES ENERGY": 0.0,
            "CEPA(3) SAME-SPIN CORRELATION ENERGY": -0.0470742732,  # fnocc
            "ACPF CORRELATION ENERGY": -0.2065398048,  # fnocc
            "ACPF SINGLES ENERGY": 0.0,
            "ACPF SAME-SPIN CORRELATION ENERGY": -0.0473955636,  # fnocc
            "AQCC CORRELATION ENERGY": -0.2046960774,  # fnocc
            "AQCC SINGLES ENERGY": 0.0,
            "AQCC SAME-SPIN CORRELATION ENERGY": -0.0470964567,  # fnocc
            "CCD CORRELATION ENERGY": -0.20622552201818,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.04781602,  # vcc
            "CCD TOTAL GRADIENT": np.array([[0.0, 0.0, 0.0011776783], [0.0, 0.0, -0.0011776783]]),  # vcc
            "BCCD CORRELATION ENERGY": -0.20667693,  # ccenergy only
            "CC2 CORRELATION ENERGY": -0.2023633612,  # ccenergy
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.004483211, 0.0, 0.0, -0.004483211]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.2068152041,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.0478712079,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.002335204281,
                    0.0,
                    0.0,
                    -0.002335204281,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.00134759, 0.0, 0.0, 0.00134759, 0.0, 0.0],
                    [0.0, -0.00134759, 0.0, 0.0, 0.00134759, 0.0],
                    [0.0, 0.0, 0.63866268, 0.0, 0.0, -0.63866268],
                    [0.00134759, 0.0, 0.0, -0.00134759, 0.0, 0.0],
                    [0.0, 0.00134759, 0.0, 0.0, -0.00134759, 0.0],
                    [0.0, 0.0, -0.63866268, 0.0, 0.0, 0.63866268],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.002040855770,  # vcc
            "(T) CORRECTION ENERGY": -0.0019205007159748158,
            "CCSD(T) TOTAL GRADIENT": np.array([[0.0, 0.0, 0.00342858327943], [0.0, 0.0, -0.00342858327943]]),  # ncc fd
            "CCSD(T) TOTAL HESSIAN": np.array(  # ncc fd
                [
                    [-0.001978535996, 0.0, 0.0, 0.001978535996, 0.0, 0.0],
                    [0.0, -0.001978535996, 0.0, 0.0, 0.001978535996, 0.0],
                    [0.0, 0.0, 0.637319868539, 0.0, 0.0, -0.637319868539],
                    [0.001978535996, 0.0, 0.0, -0.001978535996, 0.0, 0.0],
                    [0.0, 0.001978535996, 0.0, -0.0, -0.001978535996, 0.0],
                    [0.0, 0.0, -0.637319868539, 0.0, 0.0, 0.637319868539],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.001944208736203,  # ncc
            "A-CCSD(T) TOTAL GRADIENT": np.array(  # ccenergy fd
                [[0.00000000000, 0.00000000000, 0.00341625320310], [0.00000000000, 0.00000000000, -0.00341625320310]]
            ),
            "B(T) CORRECTION ENERGY": -0.00206676,  # ccenergy only
            "CC3 CORRELATION ENERGY": -0.208823404506030,  # ccenergy
            "CCSDT-1A CORRELATION ENERGY": -0.208822533905,  # ecc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[0.0, 0.0, 0.00350581687383], [0.0, 0.0, -0.00350581687383]]
            ),
            "CCSDT-1A TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-2.02328225e-03, 1.66833702e-08, 2.17609176e-08, 2.02328225e-03, -1.66833702e-08, -2.17609176e-08],
                    [1.66833702e-08, -2.02328515e-03, 1.74087341e-08, -1.66833702e-08, 2.02328515e-03, -1.74087341e-08],
                    [2.17609176e-08, 1.74087341e-08, 6.37250530e-01, -2.17609176e-08, -1.74087341e-08, -6.37250530e-01],
                    [2.02328225e-03, -1.66833702e-08, -2.17609176e-08, -2.02328225e-03, 1.66833702e-08, 2.17609176e-08],
                    [-1.66833702e-08, 2.02328515e-03, -1.74087341e-08, 1.66833702e-08, -2.02328515e-03, 1.74087341e-08],
                    [-2.17609176e-08, -1.74087341e-08, -6.37250530e-01, 2.17609176e-08, 1.74087341e-08, 6.37250530e-01],
                ]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.208825939324,  # ecc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[0.0, 0.0, 0.00350694748535], [0.0, 0.0, -0.00350694748535]]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.208693156430,  # ecc
            "CCSDT-2 TOTAL GRADIENT": np.array(  # ecc by psi fc STAR
                [[0.0, 0.0, 0.00345167960987], [0.0, 0.0, -0.00345167960987]]
            ),
            "CCSDT-3 CORRELATION ENERGY": -0.20869193974177,  # vcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[0.0, 0.0, 0.00344911055208], [0.0, 0.0, -0.00344911055208]]
            ),
            "CCSDT-3 TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.001990549473, 0.0, 0.0, 0.001990549473, -0.0, -0.0],
                    [0.0, -0.001990539318, 0.0, -0.0, 0.001990539318, -0.0],
                    [0.0, 0.0, 0.637308538355, -0.0, -0.0, -0.637308538355],
                    [0.001990549473, -0.0, -0.0, -0.001990549473, 0.0, 0.0],
                    [-0.0, 0.001990539318, -0.0, 0.0, -0.001990539318, 0.0],
                    [-0.0, -0.0, -0.637308538355, 0.0, 0.0, 0.637308538355],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.208825198347707,  # ncc
            "CCSDT TOTAL GRADIENT": np.array(
                [  # ecc
                    0.000000000000000,
                    0.000000000000000,
                    0.003519024898117,
                    0.000000000000000,
                    0.000000000000000,
                    -0.003519024898117,
                ]
            ).reshape((-1, 3)),
            "CCSDT TOTAL HESSIAN": np.array(  # ecc
                [
                    [-0.00203074, 0.0, 0.0, 0.00203074, 0.0, 0.0],
                    [0.0, -0.00203074, 0.0, 0.0, 0.00203074, 0.0],
                    [0.0, 0.0, 0.63717213, 0.0, 0.0, -0.63717213],
                    [0.00203074, 0.0, 0.0, -0.00203074, 0.0, 0.0],
                    [0.0, 0.00203074, 0.0, 0.0, -0.00203074, 0.0],
                    [0.0, 0.0, -0.63717213, 0.0, 0.0, 0.63717213],
                ]
            ),
            "[Q] CORRECTION ENERGY": -0.000379506649993,  # ncc
            "(Q) CORRECTION ENERGY": -0.000413051703749,  # ncc
            "CCSDTQ CORRELATION ENERGY": -0.209218171097884,  # ncc
            "OMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2 CORRELATION ENERGY": _knownmissing,
            "OMP2 TOTAL GRADIENT": _knownmissing,
            "OMP2.5 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2.5 CORRELATION ENERGY": _knownmissing,
            "OMP2.5 TOTAL GRADIENT": _knownmissing,
            "OMP3 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP3 CORRELATION ENERGY": _knownmissing,
            "OMP3 TOTAL GRADIENT": _knownmissing,
            "OREMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OREMP2 CORRELATION ENERGY": _knownmissing,
            "OREMP2 TOTAL GRADIENT": _knownmissing,
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0005535161,  # p4n (core-occ rotations neglected)
            "OLCCD CORRELATION ENERGY": -0.2086206237,  # p4n (core-occ rotations neglected)
            "OLCCD TOTAL GRADIENT": _knownmissing,
            "OCCD REFERENCE CORRECTION ENERGY": 0.000510311,  # qchem
            "OCCD CORRELATION ENERGY": -0.206722125,  # qchem
            "OCCD TOTAL GRADIENT": np.array([0, 0, 0.00227059, 0, 0, -0.00227059]).reshape(  # qchem, rearranged
                (-1, 3)
            ),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_pk_rhf,
            "B2PLYP TOTAL ENERGY": -100.363980302665,  # nwc 99,590
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.2194081478,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.055833980855745646,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.010245839621,
                    0.0,
                    0.005893268945,
                    -0.00512291981,
                    0.0,
                    -0.005893268945,
                    -0.00512291981,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [
                        -9.24437570e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.62218780e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.62218780e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.20231256e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.60115628e-01,
                        2.82410191e-01,
                        0.00000000e00,
                        -3.60115628e-01,
                        -2.82410191e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.67582875e-01,
                        0.00000000e00,
                        2.16920929e-01,
                        -2.33791438e-01,
                        0.00000000e00,
                        -2.16920929e-01,
                        -2.33791438e-01,
                    ],
                    [
                        4.62218780e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.36961760e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.52570200e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.60115628e-01,
                        2.16920929e-01,
                        0.00000000e00,
                        3.88438352e-01,
                        -2.49665560e-01,
                        0.00000000e00,
                        -2.83227242e-02,
                        3.27446311e-02,
                    ],
                    [
                        0.00000000e00,
                        2.82410191e-01,
                        -2.33791438e-01,
                        0.00000000e00,
                        -2.49665560e-01,
                        2.24169862e-01,
                        0.00000000e00,
                        -3.27446311e-02,
                        9.62157610e-03,
                    ],
                    [
                        4.62218780e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.52570200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -4.36961760e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.60115628e-01,
                        -2.16920929e-01,
                        0.00000000e00,
                        -2.83227242e-02,
                        -3.27446311e-02,
                        0.00000000e00,
                        3.88438352e-01,
                        2.49665560e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.82410191e-01,
                        -2.33791438e-01,
                        0.00000000e00,
                        3.27446311e-02,
                        9.62157610e-03,
                        0.00000000e00,
                        2.49665560e-01,
                        2.24169862e-01,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.224155670726,  # fnocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.051069118695,  # fnocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.004593346123, 0.0, 0.002925016535, -0.002296673061, 0.0, -0.002925016535, -0.002296673061]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.22737281,  # vcc
            "MP4(T) CORRECTION ENERGY": -0.005605447265,  # vcc
            "CISD CORRELATION ENERGY": -0.21764746560900,  # vcc
            "QCISD CORRELATION ENERGY": -0.22775040212176,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00485806,  # vcc
            "REMP2 CORRELATION ENERGY": -0.227138458433,  # occ, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.050586597452,  # occ, tight
            "LCCD CORRELATION ENERGY": -0.2296135965,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.049154543318,  # fnocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.009143629163, 0.0, 0.005604924417, -0.004571814581, 0.0, -0.005604924417, -0.004571814581]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2318316308,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.049659952324,  # fnocc
            "CEPA(1) CORRELATION ENERGY": -0.2276613849,  # fnocc
            "CEPA(1) SINGLES ENERGY": 0.0,
            "CEPA(1) SAME-SPIN CORRELATION ENERGY": -0.0489936246,  # fnocc
            "CEPA(3) CORRELATION ENERGY": -0.2248646422,  # fnocc
            "CEPA(3) SINGLES ENERGY": 0.0,
            "CEPA(3) SAME-SPIN CORRELATION ENERGY": -0.0485143833,  # fnocc
            "ACPF CORRELATION ENERGY": -0.2278506904,  # fnocc
            "ACPF SINGLES ENERGY": 0.0,
            "ACPF SAME-SPIN CORRELATION ENERGY": -0.0490397538,  # fnocc
            "AQCC CORRELATION ENERGY": -0.2246992302,  # fnocc
            "AQCC SINGLES ENERGY": 0.0,
            "AQCC SAME-SPIN CORRELATION ENERGY": -0.0485408970,  # fnocc
            "CCD CORRELATION ENERGY": -0.22578168712640,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.04913759,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc findif in psi
                [[0.0, 0.0, 0.00677294], [0.0, 0.00429381, -0.00338647], [0.0, -0.00429381, -0.00338647]]
            ),
            "BCCD CORRELATION ENERGY": -0.226749124382525,  # ccenergy only
            "CC2 CORRELATION ENERGY": -0.2216835625,  # ccenergy
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.012528297777, 0.0, 0.007118519319, -0.006264148888, 0.0, -0.007118519319, -0.006264148888]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.22717334532791256,  # mrcc
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.049398348010672,
            "CCSD TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.00811816], [0.0, 0.00498838, -0.00405908], [0.0, -0.00498838, -0.00405908]]
            ),
            "CCSD TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.00732388, 0.0, 0.0, 0.00366194, 0.0, 0.0, 0.00366194, 0.0, 0.0],
                    [0.0, 0.71380971, 0.0, 0.0, -0.35690486, 0.27918049, 0.0, -0.35690486, -0.27918049],
                    [0.0, 0.0, 0.47324568, 0.0, 0.21526997, -0.23662284, 0.0, -0.21526997, -0.23662284],
                    [0.00366194, 0.0, 0.0, -0.00357321, 0.0, 0.0, -0.00008873, 0.0, 0.0],
                    [0.0, -0.35690486, 0.21526997, 0.0, 0.3871942, -0.24722523, 0.0, -0.03028935, 0.03195526],
                    [0.0, 0.27918049, -0.23662284, 0.0, -0.24722523, 0.22465158, 0.0, -0.03195526, 0.01197127],
                    [0.00366194, 0.0, 0.0, -0.00008873, 0.0, 0.0, -0.00357321, 0.0, 0.0],
                    [0.0, -0.35690486, -0.21526997, 0.0, -0.03028935, -0.03195526, 0.0, 0.3871942, 0.24722523],
                    [0.0, -0.27918049, -0.23662284, 0.0, 0.03195526, 0.01197127, 0.0, 0.24722523, 0.22465158],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.005613685607,  # vcc
            "(T) CORRECTION ENERGY": -0.00521238,
            "CCSD(T) TOTAL GRADIENT": np.array(  # ncc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.01079337838744],
                    [0.00000000000000, 0.00642184654717, -0.00539668919372],
                    [0.00000000000000, -0.00642184654717, -0.00539668919372],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # ncc fd
                [
                    [
                        -0.009738397928,
                        0.0,
                        0.0,
                        0.004869198964,
                        0.0,
                        0.0,
                        0.004869198964,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.711718022073,
                        0.0,
                        0.0,
                        -0.355859011036,
                        0.279305472354,
                        0.0,
                        -0.355859011036,
                        -0.279305472354,
                    ],
                    [
                        0.0,
                        0.0,
                        0.469065457749,
                        0.0,
                        0.215200115829,
                        -0.234532728875,
                        0.0,
                        -0.215200115829,
                        -0.234532728875,
                    ],
                    [
                        0.004869198964,
                        0.0,
                        0.0,
                        -0.004677737322,
                        0.0,
                        0.0,
                        -0.000191461642,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.355859011036,
                        0.215200115829,
                        0.0,
                        0.385717829378,
                        -0.247252794091,
                        0.0,
                        -0.029858818342,
                        0.032052678263,
                    ],
                    [
                        0.0,
                        0.279305472354,
                        -0.234532728875,
                        0.0,
                        -0.247252794091,
                        0.223154182027,
                        0.0,
                        -0.032052678263,
                        0.011378546847,
                    ],
                    [
                        0.004869198964,
                        0.0,
                        0.0,
                        -0.000191461642,
                        0.0,
                        0.0,
                        -0.004677737322,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.355859011036,
                        -0.215200115829,
                        0.0,
                        -0.029858818342,
                        -0.032052678263,
                        0.0,
                        0.385717829378,
                        0.247252794091,
                    ],
                    [
                        0.0,
                        -0.279305472354,
                        -0.234532728875,
                        0.0,
                        0.032052678263,
                        0.011378546847,
                        0.0,
                        0.247252794091,
                        0.223154182027,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.005209699406797,  # ncc
            "A-CCSD(T) TOTAL GRADIENT": np.array(  # ccenergy fd
                [
                    [0.00000000000000, 0.00000000000000, 0.01072309503834],
                    [0.00000000000000, 0.00639691786956, -0.00536154751917],
                    [0.00000000000000, -0.00639691786956, -0.00536154751917],
                ]
            ),
            "B(T) CORRECTION ENERGY": -0.005653584466044,  # ccenergy only
            "CC3 CORRELATION ENERGY": -0.232764956682754,  # ccenergy
            "CCSDT-1A CORRELATION ENERGY": -0.2327434892835214,  # mrcc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01100789973884],
                    [0.00000000000000, 0.00656715463790, -0.00550394986942],
                    [0.00000000000000, -0.00656715463790, -0.00550394986942],
                ]
            ),
            "CCSDT-1A TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.00993141, 0.0, 0.0, 0.00496571, 0.0, 0.0, 0.00496571, 0.0, 0.0],
                    [0.0, 0.71162074, 0.0, 0.0, -0.35581037, 0.27934255, 0.0, -0.35581037, -0.27934255],
                    [0.0, 0.0, 0.4686399, 0.0, 0.21523949, -0.23431995, 0.0, -0.21523949, -0.23431995],
                    [0.00496571, 0.0, 0.0, -0.00477664, 0.0, 0.0, -0.00018907, 0.0, 0.0],
                    [0.0, -0.35581037, 0.21523949, 0.0, 0.38585314, -0.24729102, 0.0, -0.03004277, 0.03205153],
                    [0.0, 0.27934255, -0.23431995, 0.0, -0.24729102, 0.22301133, 0.0, -0.03205153, 0.01130862],
                    [0.00496571, 0.0, 0.0, -0.00018907, 0.0, 0.0, -0.00477664, 0.0, 0.0],
                    [0.0, -0.35581037, -0.21523949, 0.0, -0.03004277, -0.03205153, 0.0, 0.38585314, 0.24729102],
                    [0.0, -0.27934255, -0.23431995, 0.0, 0.03205153, 0.01130862, 0.0, 0.24729102, 0.22301133],
                ]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.23274736711556443,  # mrcc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01100335606981],
                    [0.00000000000000, 0.00656630481721, -0.00550167803491],
                    [0.00000000000000, -0.00656630481721, -0.00550167803491],
                ]
            ),
            "CCSDT-1B TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.00992728, 0.0, 0.0, 0.00496364, 0.0, 0.0, 0.00496364, 0.0, 0.0],
                    [0.0, 0.71160797, 0.0, 0.0, -0.35580398, 0.27933599, 0.0, -0.35580398, -0.27933599],
                    [0.0, 0.0, 0.46863418, 0.0, 0.21524634, -0.23431709, 0.0, -0.21524634, -0.23431709],
                    [0.00496364, 0.0, 0.0, -0.0047753, 0.0, 0.0, -0.00018834, 0.0, 0.0],
                    [0.0, -0.35580398, 0.21524634, 0.0, 0.38559464, -0.24729116, 0.0, -0.02979065, 0.03204483],
                    [0.0, 0.27933599, -0.23431709, 0.0, -0.24729116, 0.22300767, 0.0, -0.03204483, 0.01130942],
                    [0.00496364, 0.0, 0.0, -0.00018834, 0.0, 0.0, -0.0047753, 0.0, 0.0],
                    [0.0, -0.35580398, -0.21524634, 0.0, -0.02979065, -0.03204483, 0.0, 0.38559464, 0.24729116],
                    [0.0, -0.27933599, -0.23431709, 0.0, 0.03204483, 0.01130942, 0.0, 0.24729116, 0.22300767],
                ]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.232250104269,  # ecc
            "CCSDT-2 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01067153233521],
                    [0.00000000000000, 0.00637295285347, -0.00533576616760],
                    [0.00000000000000, -0.00637295285347, -0.00533576616760],
                ]
            ),
            "CCSDT-2 TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
            "CCSDT-3 CORRELATION ENERGY": -0.23227192730338686,  # mrcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01069509383085],
                    [0.00000000000000, 0.00638565480574, -0.00534754691542],
                    [0.00000000000000, -0.00638565480574, -0.00534754691542],
                ]
            ),
            "CCSDT-3 TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.00964914, 0.0, 0.0, 0.00482457, 0.0, 0.0, 0.00482457, 0.0, 0.0],
                    [0.0, 0.7116635, 0.0, 0.0, -0.35583175, 0.27924979, 0.0, -0.35583175, -0.27924979],
                    [0.0, 0.0, 0.46915589, 0.0, 0.21525082, -0.23457795, 0.0, -0.21525082, -0.23457795],
                    [0.00482457, 0.0, 0.0, -0.00464265, 0.0, 0.0, -0.00018191, 0.0, 0.0],
                    [0.0, -0.35583175, 0.21525082, 0.0, 0.38572295, -0.24725031, 0.0, -0.0298912, 0.03199949],
                    [0.0, 0.27924979, -0.23457795, 0.0, -0.24725031, 0.22316784, 0.0, -0.03199949, 0.01141011],
                    [0.00482457, 0.0, 0.0, -0.00018191, 0.0, 0.0, -0.00464265, 0.0, 0.0],
                    [0.0, -0.35583175, -0.21525082, 0.0, -0.0298912, -0.03199949, 0.0, 0.38572295, 0.24725031],
                    [0.0, -0.27924979, -0.23457795, 0.0, 0.03199949, 0.01141011, 0.0, 0.24725031, 0.22316784],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.232619797476334,  # ncc
            "CCSDT TOTAL GRADIENT": np.array(
                [  # ecc
                    0.000000000000000,
                    0.000000000000000,
                    0.010883725577495,
                    0.000000000000000,
                    0.006489002186983,
                    -0.005441862788749,
                    0.000000000000000,
                    -0.006489002186983,
                    -0.005441862788749,
                ]
            ).reshape((-1, 3)),
            "CCSDT TOTAL HESSIAN": np.array(  # ecc
                [
                    [
                        -9.81991250e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.90995620e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.90995620e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.11412515e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.55706258e-01,
                        2.79218906e-01,
                        0.00000000e00,
                        -3.55706258e-01,
                        -2.79218906e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.68988233e-01,
                        0.00000000e00,
                        2.15192437e-01,
                        -2.34494116e-01,
                        0.00000000e00,
                        -2.15192437e-01,
                        -2.34494116e-01,
                    ],
                    [
                        4.90995620e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.72159190e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.88364300e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.55706258e-01,
                        2.15192437e-01,
                        0.00000000e00,
                        3.85602165e-01,
                        -2.47205672e-01,
                        0.00000000e00,
                        -2.98959074e-02,
                        3.20132343e-02,
                    ],
                    [
                        0.00000000e00,
                        2.79218906e-01,
                        -2.34494116e-01,
                        0.00000000e00,
                        -2.47205672e-01,
                        2.23077632e-01,
                        0.00000000e00,
                        -3.20132343e-02,
                        1.14164840e-02,
                    ],
                    [
                        4.90995620e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.88364300e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -4.72159190e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.55706258e-01,
                        -2.15192437e-01,
                        0.00000000e00,
                        -2.98959074e-02,
                        -3.20132343e-02,
                        0.00000000e00,
                        3.85602165e-01,
                        2.47205672e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.79218906e-01,
                        -2.34494116e-01,
                        0.00000000e00,
                        3.20132343e-02,
                        1.14164840e-02,
                        0.00000000e00,
                        2.47205672e-01,
                        2.23077632e-01,
                    ],
                ]
            ),
            "[Q] CORRECTION ENERGY": -0.000288788062660,  # ncc
            "(Q) CORRECTION ENERGY": -0.000503838444143,  # ncc
            "CCSDTQ CORRELATION ENERGY": -0.233074721244323,  # ncc
            "OMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2 CORRELATION ENERGY": _knownmissing,
            "OMP2 TOTAL GRADIENT": _knownmissing,
            "OMP2.5 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2.5 CORRELATION ENERGY": _knownmissing,
            "OMP2.5 TOTAL GRADIENT": _knownmissing,
            "OMP3 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP3 CORRELATION ENERGY": _knownmissing,
            "OMP3 TOTAL GRADIENT": _knownmissing,
            "OREMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OREMP2 CORRELATION ENERGY": _knownmissing,
            "OREMP2 TOTAL GRADIENT": _knownmissing,
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0011942797,  # p4n (core-occ rotations neglected)
            "OLCCD CORRELATION ENERGY": -0.2309009256,  # p4n (core-occ rotations neglected)
            "OLCCD TOTAL GRADIENT": _knownmissing,
            "OCCD REFERENCE CORRECTION ENERGY": 0.001125755,  # qchem
            "OCCD CORRELATION ENERGY": -0.226880992,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem, rearranged
                [0, 0, 0.00797428, 0, 0.00488548, -0.00398714, 0, -0.00488548, -0.00398714]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_adz_pk_rhf,
            "B2PLYP TOTAL ENERGY": -76.379687335481,  # nwc 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.24515185206,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06126410,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.00033347691,
                    0.0,
                    -0.00056224437,
                    -0.000166738455,
                    0.0,
                    0.00056224437,
                    -0.000166738455,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [
                        -3.00880800e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.50440400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.50440400e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.00166972e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.50083486e-01,
                        2.71180080e-01,
                        0.00000000e00,
                        -3.50083486e-01,
                        -2.71180080e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.59237517e-01,
                        0.00000000e00,
                        2.06356325e-01,
                        -2.29618758e-01,
                        0.00000000e00,
                        -2.06356325e-01,
                        -2.29618758e-01,
                    ],
                    [
                        1.50440400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.21172300e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.71612700e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.50083486e-01,
                        2.06356325e-01,
                        0.00000000e00,
                        3.80842059e-01,
                        -2.38768202e-01,
                        0.00000000e00,
                        -3.07585729e-02,
                        3.24118773e-02,
                    ],
                    [
                        0.00000000e00,
                        2.71180080e-01,
                        -2.29618758e-01,
                        0.00000000e00,
                        -2.38768202e-01,
                        2.19990798e-01,
                        0.00000000e00,
                        -3.24118773e-02,
                        9.62796070e-03,
                    ],
                    [
                        1.50440400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.71612700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.21172300e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.50083486e-01,
                        -2.06356325e-01,
                        0.00000000e00,
                        -3.07585729e-02,
                        -3.24118773e-02,
                        0.00000000e00,
                        3.80842059e-01,
                        2.38768202e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.71180080e-01,
                        -2.29618758e-01,
                        0.00000000e00,
                        3.24118773e-02,
                        9.62796070e-03,
                        0.00000000e00,
                        2.38768202e-01,
                        2.19990798e-01,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.24747734,  # fnocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.05558309,  # fnocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.006163797026, 0.0, -0.004106253706, 0.003081898513, 0.0, 0.004106253706, 0.003081898513]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.25063260,  # vcc
            "MP4(T) CORRECTION ENERGY": -0.007755518242,  # vcc
            "CISD CORRELATION ENERGY": -0.24016600239402,  # vcc
            "QCISD CORRELATION ENERGY": -0.25077731041751,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00687323,  # vcc
            "REMP2 CORRELATION ENERGY": -0.251018009939,  # occ, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.055360085349,  # occ, tight
            "LCCD CORRELATION ENERGY": -0.2531942099,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.053842594884,  # fnocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.001471171285, 0.0, -0.001391395837, 0.000735585643, 0.0, 0.001391395837, 0.000735585643]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2553008820,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.054321637599,  # fnocc
            "CEPA(1) CORRELATION ENERGY": -0.25091653,  # fnocc
            "CEPA(1) SINGLES ENERGY": 0.0,
            "CEPA(1) SAME-SPIN CORRELATION ENERGY": -0.05365022,  # fnocc
            "CEPA(3) CORRELATION ENERGY": -0.24794377,  # fnocc
            "CEPA(3) SINGLES ENERGY": 0.0,
            "CEPA(3) SAME-SPIN CORRELATION ENERGY": -0.05315570,  # fnocc
            "ACPF CORRELATION ENERGY": -0.25107948,  # fnocc
            "ACPF SINGLES ENERGY": 0.0,
            "ACPF SAME-SPIN CORRELATION ENERGY": -0.05369031,  # fnocc
            "AQCC CORRELATION ENERGY": -0.24771962,  # fnocc
            "AQCC SINGLES ENERGY": 0.0,
            "AQCC SAME-SPIN CORRELATION ENERGY": -0.05317473,  # fnocc
            "CCD CORRELATION ENERGY": -0.24885097949989,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.05375108,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc
                [[0.0, 0.0, -0.0038684502], [0.0, -0.0026251298, 0.0019342251], [0.0, 0.0026251298, 0.0019342251]]
            ),
            "BCCD CORRELATION ENERGY": -0.24995280,  # ccenergy only
            "CC2 CORRELATION ENERGY": -0.247440256353710,  # ccenergy
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.002577148521, 0.0, 0.000581020448, -0.001288574261, 0.0, -0.000581020448, -0.001288574261]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.250330548844,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.054051928864870,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    -0.002486174824,
                    0.0,
                    -0.001923330621,
                    0.001243087412,
                    0.0,
                    0.001923330621,
                    0.001243087412,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        2.24316860e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.12158430e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.12158430e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.92694865e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.46347433e-01,
                        2.67302412e-01,
                        0.00000000e00,
                        -3.46347433e-01,
                        -2.67302412e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.64489488e-01,
                        0.00000000e00,
                        2.03993780e-01,
                        -2.32244744e-01,
                        0.00000000e00,
                        -2.03993780e-01,
                        -2.32244744e-01,
                    ],
                    [
                        -1.12158430e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.23261320e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.11028900e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.46347433e-01,
                        2.03993780e-01,
                        0.00000000e00,
                        3.79028763e-01,
                        -2.35648096e-01,
                        0.00000000e00,
                        -3.26813308e-02,
                        3.16543160e-02,
                    ],
                    [
                        0.00000000e00,
                        2.67302412e-01,
                        -2.32244744e-01,
                        0.00000000e00,
                        -2.35648096e-01,
                        2.20278013e-01,
                        0.00000000e00,
                        -3.16543160e-02,
                        1.19667307e-02,
                    ],
                    [
                        -1.12158430e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.11028900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.23261320e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.46347433e-01,
                        -2.03993780e-01,
                        0.00000000e00,
                        -3.26813308e-02,
                        -3.16543160e-02,
                        0.00000000e00,
                        3.79028763e-01,
                        2.35648096e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.67302412e-01,
                        -2.32244744e-01,
                        0.00000000e00,
                        3.16543160e-02,
                        1.19667307e-02,
                        0.00000000e00,
                        2.35648096e-01,
                        2.20278013e-01,
                    ],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.007366570060,  # vcc
            "(T) CORRECTION ENERGY": -0.007096579721,
            "CCSD(T) TOTAL GRADIENT": np.array(  # ncc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.00065323185711],
                    [0.00000000000000, -0.00035236679634, -0.00032661592856],
                    [0.00000000000000, 0.00035236679634, -0.00032661592856],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # ncc fd
                [
                    [
                        -0.000589405884,
                        0.0,
                        0.0,
                        0.000294702942,
                        0.0,
                        0.0,
                        0.000294702942,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.690892000177,
                        0.0,
                        0.0,
                        -0.345446000089,
                        0.267700971726,
                        0.0,
                        -0.345446000089,
                        -0.267700971726,
                    ],
                    [
                        0.0,
                        0.0,
                        0.460262946323,
                        0.0,
                        0.204108329476,
                        -0.230131473162,
                        0.0,
                        -0.204108329476,
                        -0.230131473162,
                    ],
                    [
                        0.000294702942,
                        0.0,
                        0.0,
                        -0.000024281097,
                        0.0,
                        0.0,
                        -0.000270421845,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.345446000089,
                        0.204108329476,
                        0.0,
                        0.377790533687,
                        -0.235904650601,
                        0.0,
                        -0.032344533598,
                        0.031796321125,
                    ],
                    [
                        0.0,
                        0.267700971726,
                        -0.230131473162,
                        0.0,
                        -0.235904650601,
                        0.218826959667,
                        0.0,
                        -0.031796321125,
                        0.011304513494,
                    ],
                    [
                        0.000294702942,
                        0.0,
                        0.0,
                        -0.000270421845,
                        0.0,
                        0.0,
                        -0.000024281097,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.345446000089,
                        -0.204108329476,
                        0.0,
                        -0.032344533598,
                        -0.031796321125,
                        0.0,
                        0.377790533687,
                        0.235904650601,
                    ],
                    [
                        0.0,
                        -0.267700971726,
                        -0.230131473162,
                        0.0,
                        0.031796321125,
                        0.011304513494,
                        0.0,
                        0.235904650601,
                        0.218826959667,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.007017918524544,  # ncc
            "A-CCSD(T) TOTAL GRADIENT": np.array(  # ccenergy fd
                [
                    [0.00000000000000, 0.00000000000000, 0.00057233082116],
                    [0.00000000000000, -0.00038770405767, -0.00028616541058],
                    [0.00000000000000, 0.00038770405767, -0.00028616541058],
                ]
            ),
            "B(T) CORRECTION ENERGY": -0.00749264,  # ccenergy only
            "CC3 CORRELATION ENERGY": -0.257776704633084,  # ccenergy
            "CCSDT-1A CORRELATION ENERGY": -0.257751267313,  # ecc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.00086210387602],
                    [0.00000000000000, -0.00023280605874, -0.00043105193801],
                    [0.00000000000000, 0.00023280605874, -0.00043105193801],
                ]
            ),
            "CCSDT-1A TOTAL HESSIAN": np.array(  # mrcc
                [
                    [
                        -7.78035412e-04,
                        0.00000000e00,
                        0.00000000e00,
                        3.89017706e-04,
                        0.00000000e00,
                        0.00000000e00,
                        3.89017706e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.90855814e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.45427907e-01,
                        2.67759669e-01,
                        0.00000000e00,
                        -3.45427907e-01,
                        -2.67759669e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.59961661e-01,
                        0.00000000e00,
                        2.04153847e-01,
                        -2.29980830e-01,
                        0.00000000e00,
                        -2.04153847e-01,
                        -2.29980830e-01,
                    ],
                    [
                        3.89017706e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.13247422e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.75770284e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45427907e-01,
                        2.04153847e-01,
                        0.00000000e00,
                        3.77721655e-01,
                        -2.35956758e-01,
                        0.00000000e00,
                        -3.22937484e-02,
                        3.18029110e-02,
                    ],
                    [
                        0.00000000e00,
                        2.67759669e-01,
                        -2.29980830e-01,
                        0.00000000e00,
                        -2.35956758e-01,
                        2.18732769e-01,
                        0.00000000e00,
                        -3.18029110e-02,
                        1.12480613e-02,
                    ],
                    [
                        3.89017706e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.75770284e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.13247422e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45427907e-01,
                        -2.04153847e-01,
                        0.00000000e00,
                        -3.22937484e-02,
                        -3.18029110e-02,
                        0.00000000e00,
                        3.77721655e-01,
                        2.35956758e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.67759669e-01,
                        -2.29980830e-01,
                        0.00000000e00,
                        3.18029110e-02,
                        1.12480613e-02,
                        0.00000000e00,
                        2.35956758e-01,
                        2.18732769e-01,
                    ],
                ]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.257755250971,  # ecc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.00086535505446],
                    [0.00000000000000, -0.00023045568589, -0.00043267752723],
                    [0.00000000000000, 0.00023045568589, -0.00043267752723],
                ]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.257108846844,  # vcc
            "CCSDT-2 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.00049220649666],
                    [0.00000000000000, -0.00043758137368, -0.00024610324833],
                    [0.00000000000000, 0.00043758137368, -0.00024610324833],
                ]
            ),
            "CCSDT-3 CORRELATION ENERGY": -0.25713217826377,  # vcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.00051263693668],
                    [0.00000000000000, -0.00042772129404, -0.00025631846834],
                    [0.00000000000000, 0.00042772129404, -0.00025631846834],
                ]
            ),
            "CCSDT-3 TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.000462718541, 0.0, 0.0, 0.00023135927, 0.0, 0.0, 0.00023135927, 0.0, 0.0],
                    [
                        0.0,
                        0.690863579709,
                        0.0,
                        0.0,
                        -0.345431789855,
                        0.26764066777,
                        0.0,
                        -0.345431789855,
                        -0.26764066777,
                    ],
                    [
                        0.0,
                        0.0,
                        0.460459119187,
                        0.0,
                        0.204125984085,
                        -0.230229559594,
                        0.0,
                        -0.204125984085,
                        -0.230229559594,
                    ],
                    [0.00023135927, 0.0, 0.0, 0.000033666172, 0.0, 0.0, -0.000265025443, 0.0, 0.0],
                    [
                        0.0,
                        -0.345431789855,
                        0.204125984085,
                        0.0,
                        0.377813669112,
                        -0.235883325927,
                        0.0,
                        -0.032381879257,
                        0.031757341842,
                    ],
                    [
                        0.0,
                        0.26764066777,
                        -0.230229559594,
                        0.0,
                        -0.235883325927,
                        0.218879113252,
                        0.0,
                        -0.031757341842,
                        0.011350446342,
                    ],
                    [0.00023135927, 0.0, 0.0, -0.000265025443, 0.0, 0.0, 0.000033666172, 0.0, 0.0],
                    [
                        0.0,
                        -0.345431789855,
                        -0.204125984085,
                        0.0,
                        -0.032381879257,
                        -0.031757341842,
                        0.0,
                        0.377813669112,
                        0.235883325927,
                    ],
                    [
                        0.0,
                        -0.26764066777,
                        -0.230229559594,
                        0.0,
                        0.031757341842,
                        0.011350446342,
                        0.0,
                        0.235883325927,
                        0.218879113252,
                    ],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.257494892048411,  # ncc
            "CCSDT TOTAL GRADIENT": np.array(
                [  # ecc
                    0.000000000000000,
                    0.000000000000000,
                    0.000614598836608,
                    0.000000000000000,
                    -0.000362082670346,
                    -0.000307299418304,
                    0.000000000000000,
                    0.000362082670346,
                    -0.000307299418304,
                ]
            ).reshape((-1, 3)),
            "CCSDT TOTAL HESSIAN": np.array(  # ecc
                [
                    [
                        -5.54525900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.77262900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.77262900e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.90642317e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.45321159e-01,
                        2.67590889e-01,
                        0.00000000e00,
                        -3.45321159e-01,
                        -2.67590889e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.60364322e-01,
                        0.00000000e00,
                        2.04091988e-01,
                        -2.30182161e-01,
                        0.00000000e00,
                        -2.04091988e-01,
                        -2.30182161e-01,
                    ],
                    [
                        2.77262900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.21557000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        -2.65107200e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45321159e-01,
                        2.04091988e-01,
                        0.00000000e00,
                        3.77718239e-01,
                        -2.35841439e-01,
                        0.00000000e00,
                        -3.23970809e-02,
                        3.17494501e-02,
                    ],
                    [
                        0.00000000e00,
                        2.67590889e-01,
                        -2.30182161e-01,
                        0.00000000e00,
                        -2.35841439e-01,
                        2.18813058e-01,
                        0.00000000e00,
                        -3.17494501e-02,
                        1.13691031e-02,
                    ],
                    [
                        2.77262900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.65107200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.21557000e-05,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45321159e-01,
                        -2.04091988e-01,
                        0.00000000e00,
                        -3.23970809e-02,
                        -3.17494501e-02,
                        0.00000000e00,
                        3.77718239e-01,
                        2.35841439e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.67590889e-01,
                        -2.30182161e-01,
                        0.00000000e00,
                        3.17494501e-02,
                        1.13691031e-02,
                        0.00000000e00,
                        2.35841439e-01,
                        2.18813058e-01,
                    ],
                ]
            ),
            "[Q] CORRECTION ENERGY": -0.000151213601440,  # ncc
            "(Q) CORRECTION ENERGY": -0.000368016662584,  # ncc
            "CCSDTQ CORRELATION ENERGY": -0.257824320323573,  # ncc
            "OMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2 CORRELATION ENERGY": _knownmissing,
            "OMP2 TOTAL GRADIENT": _knownmissing,
            "OMP2.5 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2.5 CORRELATION ENERGY": _knownmissing,
            "OMP2.5 TOTAL GRADIENT": _knownmissing,
            "OMP3 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP3 CORRELATION ENERGY": _knownmissing,
            "OMP3 TOTAL GRADIENT": _knownmissing,
            "OREMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OREMP2 CORRELATION ENERGY": _knownmissing,
            "OREMP2 TOTAL GRADIENT": _knownmissing,
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0012982937,  # p4n (core-occ rotations neglected)
            "OLCCD CORRELATION ENERGY": -0.2544819317,  # p4n (core-occ rotations neglected)
            "OLCCD TOTAL GRADIENT": _knownmissing,
            "OCCD REFERENCE CORRECTION ENERGY": 0.001225751,  # qchem
            "OCCD CORRELATION ENERGY": -0.250044802,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem, rearranged
                [0, 0, -0.00269153, 0, -0.00204152, 0.00134576, 0, 0.00204152, 0.00134576]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_qz2p_pk_rhf,
            "B2PLYP TOTAL ENERGY": -76.409701597936,  # nwc 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.058423513790,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001767468898,
            "MP2 TOTAL GRADIENT": np.array(
                # switched sign from unkn origin
                [
                    0.000000000000000,
                    0.000000000000000,
                    0.012305278627642,
                    0.000000000000000,
                    0.000000000000000,
                    -0.011851332672482,
                    0.000000000000000,
                    0.010327045553422,
                    -0.000226972977580,
                    0.000000000000000,
                    -0.010327045553422,
                    -0.000226972977580,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [
                        3.34735620e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -3.92156600e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.87104900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.87104900e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        3.25884470e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -3.78117870e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.61167000e-04,
                        4.35841400e-04,
                        0.00000000e00,
                        2.61167000e-04,
                        -4.35841400e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.15996143e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.13938386e-01,
                        0.00000000e00,
                        1.11245210e-03,
                        -1.02887840e-03,
                        0.00000000e00,
                        -1.11245210e-03,
                        -1.02887840e-03,
                    ],
                    [
                        -3.92156600e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.26586180e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.72147900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.72147900e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.78117870e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.18188010e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.00350700e-04,
                        8.01285000e-05,
                        0.00000000e00,
                        -2.00350700e-04,
                        -8.01285000e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.13938386e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.14114521e-01,
                        0.00000000e00,
                        1.83716000e-05,
                        -8.80676000e-05,
                        0.00000000e00,
                        -1.83716000e-05,
                        -8.80676000e-05,
                    ],
                    [
                        2.87104900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.72147900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -7.41277320e-03,
                        0.00000000e00,
                        0.00000000e00,
                        7.29781620e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.61167000e-04,
                        1.11245210e-03,
                        0.00000000e00,
                        -2.00350700e-04,
                        1.83716000e-05,
                        0.00000000e00,
                        4.11361597e-01,
                        -8.23396800e-04,
                        0.00000000e00,
                        -4.11422413e-01,
                        -3.07426900e-04,
                    ],
                    [
                        0.00000000e00,
                        4.35841400e-04,
                        -1.02887840e-03,
                        0.00000000e00,
                        8.01285000e-05,
                        -8.80676000e-05,
                        0.00000000e00,
                        -8.23396800e-04,
                        -4.65210960e-03,
                        0.00000000e00,
                        3.07426900e-04,
                        5.76905560e-03,
                    ],
                    [
                        2.87104900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.72147900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        7.29781620e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -7.41277320e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.61167000e-04,
                        -1.11245210e-03,
                        0.00000000e00,
                        -2.00350700e-04,
                        -1.83716000e-05,
                        0.00000000e00,
                        -4.11422413e-01,
                        3.07426900e-04,
                        0.00000000e00,
                        4.11361597e-01,
                        8.23396800e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.35841400e-04,
                        -1.02887840e-03,
                        0.00000000e00,
                        -8.01285000e-05,
                        -8.80676000e-05,
                        0.00000000e00,
                        -3.07426900e-04,
                        5.76905560e-03,
                        0.00000000e00,
                        8.23396800e-04,
                        -4.65210960e-03,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.07375810,  # occ
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00220392,  # occ
            "MP3 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.009100318483,
                    0.0,
                    0.0,
                    -0.008860556313,
                    0.0,
                    0.012722119397,
                    -0.000119881085,
                    0.0,
                    -0.012722119397,
                    -0.000119881085,
                ]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.07825006,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.000237348957,  # vcc only
            "CISD CORRELATION ENERGY": -0.07917581082828,  # vcc
            "QCISD CORRELATION ENERGY": -0.08117897504733,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00060269,  # vcc only
            "REMP2 CORRELATION ENERGY": -0.07701467620060,  # occ, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.00219044944957,  # occ. tight
            "LCCD CORRELATION ENERGY": -0.0824313452,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0022532590,  # occ
            "LCCD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.004402112418,
                    0.0,
                    0.0,
                    -0.004307902536,
                    0.0,
                    0.015259701356,
                    -0.000047104941,
                    0.0,
                    -0.015259701356,
                    -0.000047104941,
                ]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.0837903430,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.0022897,  # vcc
            "CCD CORRELATION ENERGY": -0.08014654379229,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.0022032,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc
                [
                    [0.0, 0.0, 0.0061586096],
                    [0.0, 0.0, -0.0059576188],
                    [0.0, 0.0144468804, -0.0001004954],
                    [0.0, -0.0144468804, -0.0001004954],
                ]
            ),
            "BCCD CORRELATION ENERGY": -0.08117235,  # ccenergy only
            "CC2 CORRELATION ENERGY": -0.0587557596,  # ccenergy
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [
                    0.0,
                    0.0,
                    0.011903275003,
                    0.0,
                    0.0,
                    -0.011481983661,
                    0.0,
                    0.010549441714,
                    -0.000210645671,
                    0.0,
                    -0.010549441714,
                    -0.000210645671,
                ]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.08117105566,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.002231965267,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.00496423512,
                    0.0,
                    0.0,
                    -0.004814203262,
                    0.0,
                    0.014877060204,
                    -0.000075015929,
                    0.0,
                    -0.014877060204,
                    -0.000075015929,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        6.58519500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.09716090e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.19320700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.19320700e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.18973300e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -9.89916500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.85471600e-04,
                        4.42369000e-04,
                        0.00000000e00,
                        1.85471600e-04,
                        -4.42369000e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.06191545e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.04510095e-01,
                        0.00000000e00,
                        1.15490300e-03,
                        -8.40725200e-04,
                        0.00000000e00,
                        -1.15490300e-03,
                        -8.40725200e-04,
                    ],
                    [
                        -1.09716090e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.38536200e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.44100500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.44100500e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -9.89916500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.31932370e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.64703600e-04,
                        5.07376000e-05,
                        0.00000000e00,
                        -1.64703600e-04,
                        -5.07376000e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.04510095e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.04625444e-01,
                        0.00000000e00,
                        1.61035900e-04,
                        -5.76744000e-05,
                        0.00000000e00,
                        -1.61035900e-04,
                        -5.76744000e-05,
                    ],
                    [
                        2.19320700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.44100500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.06335912e-02,
                        0.00000000e00,
                        0.00000000e00,
                        1.05583709e-02,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.85471600e-04,
                        1.15490300e-03,
                        0.00000000e00,
                        -1.64703600e-04,
                        1.61035900e-04,
                        0.00000000e00,
                        4.07395303e-01,
                        -9.04522700e-04,
                        0.00000000e00,
                        -4.07416071e-01,
                        -4.11416200e-04,
                    ],
                    [
                        0.00000000e00,
                        4.42369000e-04,
                        -8.40725200e-04,
                        0.00000000e00,
                        5.07376000e-05,
                        -5.76744000e-05,
                        0.00000000e00,
                        -9.04522700e-04,
                        -8.14010510e-03,
                        0.00000000e00,
                        4.11416200e-04,
                        9.03850480e-03,
                    ],
                    [
                        2.19320700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.44100500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.05583709e-02,
                        0.00000000e00,
                        0.00000000e00,
                        -1.06335912e-02,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.85471600e-04,
                        -1.15490300e-03,
                        0.00000000e00,
                        -1.64703600e-04,
                        -1.61035900e-04,
                        0.00000000e00,
                        -4.07416071e-01,
                        4.11416200e-04,
                        0.00000000e00,
                        4.07395303e-01,
                        9.04522700e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.42369000e-04,
                        -8.40725200e-04,
                        0.00000000e00,
                        -5.07376000e-05,
                        -5.76744000e-05,
                        0.00000000e00,
                        -4.11416200e-04,
                        9.03850480e-03,
                        0.00000000e00,
                        9.04522700e-04,
                        -8.14010510e-03,
                    ],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.000606865286,  # vcc
            "(T) CORRECTION ENERGY": -0.00060401,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.00446268840416],
                    [0.00000000000000, 0.00000000000000, -0.00439493294278],
                    [0.00000000000000, 0.01495635291373, -0.00003387773069],
                    [0.00000000000000, -0.01495635291373, -0.00003387773069],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # vcc fd
                [
                    [
                        0.000493717382,
                        0.0,
                        0.0,
                        -0.00091720077,
                        0.0,
                        0.0,
                        0.000211741694,
                        0.0,
                        0.0,
                        0.000211741694,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.000457061127,
                        0.0,
                        0.0,
                        -0.000812917213,
                        0.0,
                        0.0,
                        0.000177928043,
                        0.000438798747,
                        0.0,
                        0.000177928043,
                        -0.000438798747,
                    ],
                    [
                        0.0,
                        0.0,
                        0.206046933798,
                        0.0,
                        0.0,
                        -0.204504540896,
                        0.0,
                        0.001278605784,
                        -0.000771196451,
                        0.0,
                        -0.001278605784,
                        -0.000771196451,
                    ],
                    [
                        -0.00091720077,
                        0.0,
                        0.0,
                        0.001205452004,
                        0.0,
                        0.0,
                        -0.000144125617,
                        0.0,
                        0.0,
                        -0.000144125617,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.000812917213,
                        0.0,
                        0.0,
                        0.001141063782,
                        0.0,
                        0.0,
                        -0.000164073285,
                        0.000047332929,
                        0.0,
                        -0.000164073285,
                        -0.000047332929,
                    ],
                    [
                        0.0,
                        0.0,
                        -0.204504540896,
                        0.0,
                        0.0,
                        0.204650897031,
                        0.0,
                        0.000135987404,
                        -0.000073178067,
                        0.0,
                        -0.000135987404,
                        -0.000073178067,
                    ],
                    [
                        0.000211741694,
                        0.0,
                        0.0,
                        -0.000144125617,
                        0.0,
                        0.0,
                        -0.010686239837,
                        0.0,
                        0.0,
                        0.01061862376,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.000177928043,
                        0.001278605784,
                        0.0,
                        -0.000164073285,
                        0.000135987404,
                        0.0,
                        0.407325954394,
                        -0.000950362432,
                        0.0,
                        -0.407339809152,
                        -0.000464230756,
                    ],
                    [
                        0.0,
                        0.000438798747,
                        -0.000771196451,
                        0.0,
                        0.000047332929,
                        -0.000073178067,
                        0.0,
                        -0.000950362432,
                        -0.008256200737,
                        0.0,
                        0.000464230756,
                        0.009100575256,
                    ],
                    [
                        0.000211741694,
                        0.0,
                        0.0,
                        -0.000144125617,
                        0.0,
                        0.0,
                        0.01061862376,
                        0.0,
                        0.0,
                        -0.010686239837,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.000177928043,
                        -0.001278605784,
                        0.0,
                        -0.000164073285,
                        -0.000135987404,
                        0.0,
                        -0.407339809152,
                        0.000464230756,
                        0.0,
                        0.407325954394,
                        0.000950362432,
                    ],
                    [
                        0.0,
                        -0.000438798747,
                        -0.000771196451,
                        0.0,
                        -0.000047332929,
                        -0.000073178067,
                        0.0,
                        -0.000464230756,
                        0.009100575256,
                        0.0,
                        0.000950362432,
                        -0.008256200737,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.000587873,  # mrcc only
            "B(T) CORRECTION ENERGY": -0.00061086,  # ccenergy only
            "CC3 CORRELATION ENERGY": -0.081776957269437,  # ccenergy
            "CCSDT-1A CORRELATION ENERGY": -0.081774257938,  # ecc
            "CCSDT-1B CORRELATION ENERGY": -0.081772270576,  # ecc
            "CCSDT-2 CORRELATION ENERGY": -0.081772292290,  # ecc
            "CCSDT-3 CORRELATION ENERGY": -0.08177701734273,  # vcc
            "CCSDT CORRELATION ENERGY": -0.08208821205578,  # vcc
            "OMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2 CORRELATION ENERGY": _knownmissing,
            "OMP2 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP2 TOTAL GRADIENT": _knownmissing,
            "OMP2.5 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2.5 CORRELATION ENERGY": _knownmissing,
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP2.5 TOTAL GRADIENT": _knownmissing,
            "OMP3 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP3 CORRELATION ENERGY": _knownmissing,
            "OMP3 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP3 TOTAL GRADIENT": _knownmissing,
            "OREMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OREMP2 CORRELATION ENERGY": _knownmissing,
            "OREMP2 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OREMP2 TOTAL GRADIENT": _knownmissing,
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0014923056,  # p4n (core-occ rotations neglected)
            "OLCCD CORRELATION ENERGY": -0.0839163095,  # p4n (core-occ rotations neglected)
            "OLCCD SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OLCCD TOTAL GRADIENT": _knownmissing,
            "OCCD REFERENCE CORRECTION ENERGY": 0.001154566,  # qchem
            "OCCD CORRELATION ENERGY": -0.081188108,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem, rearranged
                [0, 0, 0.00494989, 0, 0, -0.00479521, 0, 0.01487079, -0.00007734, 0, -0.01487079, -0.00007734]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_bh3p_dz_pk_uhf,
            "B2PLYP TOTAL ENERGY": -26.072618722961,  # nwc 99,590
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.15242755400188052,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03445360441348938,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.026279427993,
                    0.0,
                    0.013998590506,
                    -0.013139713997,
                    0.0,
                    -0.013998590506,
                    -0.013139713997,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [-0.02266258, 0.0, 0.0, 0.01133129, 0.0, 0.0, 0.01133129, 0.0, 0.0],
                    [0.0, 0.62640007, 0.0, 0.0, -0.31320003, 0.24902164, 0.0, -0.31320003, -0.24902164],
                    [0.0, 0.0, 0.39927018, 0.0, 0.19168818, -0.19963509, 0.0, -0.19168818, -0.19963509],
                    [0.01133129, 0.0, 0.0, -0.01029722, 0.0, 0.0, -0.00103408, 0.0, 0.0],
                    [0.0, -0.31320003, 0.19168818, 0.0, 0.3403478, -0.22035491, 0.0, -0.02714777, 0.02866673],
                    [0.0, 0.24902164, -0.19963509, 0.0, -0.22035491, 0.19072649, 0.0, -0.02866673, 0.0089086],
                    [0.01133129, 0.0, 0.0, -0.00103408, 0.0, 0.0, -0.01029722, 0.0, 0.0],
                    [0.0, -0.31320003, -0.19168818, 0.0, -0.02714777, -0.02866673, 0.0, 0.3403478, 0.22035491],
                    [0.0, -0.24902164, -0.19963509, 0.0, 0.02866673, 0.0089086, 0.0, 0.22035491, 0.19072649],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.16862935,  # occ
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03463618,  # occ
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.027862081662, 0.0, 0.014906293724, -0.013931040831, 0.0, -0.014906293724, -0.013931040831]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.17088317,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.003237304648,  # vcc only
            "CISD CORRELATION ENERGY": -0.16574719763576,  # vcc
            "QCISD CORRELATION ENERGY": -0.17187325792329,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00368626,  # vcc only
            "REMP2 CORRELATION ENERGY": -0.170214503998,  # occ, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.033849141454,  # occ, tight
            "LCCD CORRELATION ENERGY": -0.1747537294,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0334378864,  # occ
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.031468252099, 0.0, 0.017115261595, -0.015734126049, 0.0, -0.017115261595, -0.015734126049]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.1763496376,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.0337624,  # vcc
            "CCD CORRELATION ENERGY": -0.17052107838019,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.033058,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc by psi findif
                [[0.0, 0.0, 0.02892979], [0.0, 0.01570148, -0.01446489], [0.0, -0.01570148, -0.01446489]]
            ),
            "BCCD CORRELATION ENERGY": -0.171452568164837,  # ccenergy only
            "CC2 CORRELATION ENERGY": -0.1536230809,  # ccenergy
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.027451749814, 0.0, 0.014623148752, -0.013725874907, 0.0, -0.014623148752, -0.013725874907]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.1716495276047496,  # mrcc
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.033248190929062,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.030055915902,
                    0.0,
                    0.016307167756,
                    -0.015027957951,
                    0.0,
                    -0.016307167756,
                    -0.015027957951,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.02591931, 0.0, 0.0, 0.01295966, 0.0, 0.0, 0.01295966, 0.0, 0.0],
                    [0.0, 0.6170188, 0.0, 0.0, -0.3085094, 0.24667188, 0.0, -0.3085094, -0.24667188],
                    [0.0, 0.0, 0.39656797, 0.0, 0.19003804, -0.19828399, 0.0, -0.19003804, -0.19828399],
                    [0.01295966, 0.0, 0.0, -0.01187521, 0.0, 0.0, -0.00108444, 0.0, 0.0],
                    [0.0, -0.3085094, 0.19003804, 0.0, 0.33664846, -0.21835496, 0.0, -0.02813906, 0.02831692],
                    [0.0, 0.24667188, -0.19828399, 0.0, -0.21835496, 0.1883856, 0.0, -0.02831692, 0.00989838],
                    [0.01295966, 0.0, 0.0, -0.00108444, 0.0, 0.0, -0.01187521, 0.0, 0.0],
                    [0.0, -0.3085094, -0.19003804, 0.0, -0.02813906, -0.02831692, 0.0, 0.33664846, 0.21835496],
                    [0.0, -0.24667188, -0.19828399, 0.0, 0.02831692, 0.00989838, 0.0, 0.21835496, 0.1883856],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.003956941783,  # vcc
            "(T) CORRECTION ENERGY": -0.00381116,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.03193430184375],
                    [0.00000000000000, 0.01730212130253, -0.01596715092187],
                    [0.00000000000000, -0.01730212130253, -0.01596715092187],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # vcc fd
                [
                    [
                        -0.027539040923,
                        0.0,
                        0.0,
                        0.013769520462,
                        0.0,
                        0.0,
                        0.013769520462,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.615442549333,
                        0.0,
                        0.0,
                        -0.307721274667,
                        0.246688641005,
                        0.0,
                        -0.307721274667,
                        -0.246688641005,
                    ],
                    [
                        0.0,
                        0.0,
                        0.39351580275,
                        0.0,
                        0.189939555413,
                        -0.196757901375,
                        0.0,
                        -0.189939555413,
                        -0.196757901375,
                    ],
                    [
                        0.013769520462,
                        0.0,
                        0.0,
                        -0.012609341037,
                        0.0,
                        0.0,
                        -0.001160179424,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.307721274667,
                        0.189939555413,
                        0.0,
                        0.335534419601,
                        -0.218314098209,
                        0.0,
                        -0.027813144934,
                        0.028374542796,
                    ],
                    [
                        0.0,
                        0.246688641005,
                        -0.196757901375,
                        0.0,
                        -0.218314098209,
                        0.187299796614,
                        0.0,
                        -0.028374542796,
                        0.009458104761,
                    ],
                    [
                        0.013769520462,
                        0.0,
                        0.0,
                        -0.001160179424,
                        0.0,
                        0.0,
                        -0.012609341037,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.307721274667,
                        -0.189939555413,
                        0.0,
                        -0.027813144934,
                        -0.028374542796,
                        0.0,
                        0.335534419601,
                        0.218314098209,
                    ],
                    [
                        0.0,
                        -0.246688641005,
                        -0.196757901375,
                        0.0,
                        0.028374542796,
                        0.009458104761,
                        0.0,
                        0.218314098209,
                        0.187299796614,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.00373191576,  # mrcc only
            "B(T) CORRECTION ENERGY": -0.003997866911315,  # ccenergy only
            "CC3 CORRELATION ENERGY": -0.175583616794047,  # ccenergy
            "CCSDT-1A CORRELATION ENERGY": -0.1755715323437954,  # mrcc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.03195915], [0.0, 0.01732744, -0.01597957], [0.0, -0.01732744, -0.01597957]]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.17556886143889727,  # mrcc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.03195282], [0.0, 0.01732422, -0.01597641], [0.0, -0.01732422, -0.01597641]]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.175300994962,  # ecc
            "CCSDT-3 CORRELATION ENERGY": -0.1753162719579464,  # mrcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.03180282], [0.0, 0.01723825, -0.01590141], [0.0, -0.01723825, -0.01590141]]
            ),
            "CCSDT CORRELATION ENERGY": -0.17591978591647,  # vcc
            "OMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2 CORRELATION ENERGY": _knownmissing,
            "OMP2 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP2 TOTAL GRADIENT": _knownmissing,
            "OMP2.5 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2.5 CORRELATION ENERGY": _knownmissing,
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP2.5 TOTAL GRADIENT": _knownmissing,
            "OMP3 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP3 CORRELATION ENERGY": _knownmissing,
            "OMP3 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP3 TOTAL GRADIENT": _knownmissing,
            "OREMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OREMP2 CORRELATION ENERGY": _knownmissing,
            "OREMP2 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OREMP2 TOTAL GRADIENT": _knownmissing,
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0011137749,  # p4n (core-occ rotations neglected)
            "OLCCD CORRELATION ENERGY": -0.1759973931,  # p4n (core-occ rotations neglected)
            "OLCCD SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OLCCD TOTAL GRADIENT": _knownmissing,
            "OCCD REFERENCE CORRECTION ENERGY": 0.000986681,  # qchem
            "OCCD CORRELATION ENERGY": -0.171498252,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem, rearranged
                [0, 0, 0.03004030, 0, 0.01627564, -0.01502015, 0, -0.01627564, -0.01502015]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_adz_pk_uhf,
            "B2PLYP TOTAL ENERGY": -55.832917564913,  # nwc 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.171184123093,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03822454,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.014740098324,
                    0.0,
                    0.005852228009,
                    -0.007370049162,
                    0.0,
                    -0.005852228009,
                    -0.007370049162,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [-0.01271141, 0.0, 0.0, 0.00635571, 0.0, 0.0, 0.00635571, 0.0, 0.0],
                    [0.0, 0.59697724, 0.0, 0.0, -0.29848862, 0.23391528, 0.0, -0.29848862, -0.23391528],
                    [0.0, 0.0, 0.3890832, 0.0, 0.17739783, -0.1945416, 0.0, -0.17739783, -0.1945416],
                    [0.00635571, 0.0, 0.0, -0.00511412, 0.0, 0.0, -0.00124159, 0.0, 0.0],
                    [0.0, -0.29848862, 0.17739783, 0.0, 0.32667986, -0.20565656, 0.0, -0.02819123, 0.02825872],
                    [0.0, 0.23391528, -0.1945416, 0.0, -0.20565656, 0.18507929, 0.0, -0.02825872, 0.00946231],
                    [0.00635571, 0.0, 0.0, -0.00124159, 0.0, 0.0, -0.00511412, 0.0, 0.0],
                    [0.0, -0.29848862, -0.17739783, 0.0, -0.02819123, -0.02825872, 0.0, 0.32667986, 0.20565656],
                    [0.0, -0.23391528, -0.1945416, 0.0, 0.02825872, 0.00946231, 0.0, 0.20565656, 0.18507929],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.18575628,  # occ
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03792248,  # occ
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.015431661276, 0.0, 0.006103353029, -0.007715830638, 0.0, -0.006103353029, -0.007715830638]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.18773117,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.004470767049,  # vcc only
            "CISD CORRELATION ENERGY": -0.18202095532782,  # vcc
            "QCISD CORRELATION ENERGY": -0.18852342173162,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00487511,  # vcc only
            "REMP2 CORRELATION ENERGY": -0.187474026696,  # occ, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.037232180981,  # occ, tight
            "LCCD CORRELATION ENERGY": -0.1917024115,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0367596656,  # occ
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.019152642358, 0.0, 0.008310773449, -0.009576321179, 0.0, -0.008310773449, -0.009576321179]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.1933416962,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.0370625,  # vcc
            "CCD CORRELATION ENERGY": -0.18708520686363,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.0363299,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc
                [[0.0, 0.0, 0.0166234857], [0.0, 0.0069994421, -0.0083117428], [0.0, -0.0069994421, -0.0083117428]]
            ),
            "BCCD CORRELATION ENERGY": -0.18812319,  # ccenergy only
            "CC2 CORRELATION ENERGY": -0.1725181374,  # ccenergy
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.016128741574, 0.0, 0.006556602185, -0.008064370787, 0.0, -0.006556602185, -0.008064370787]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.188317222733,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.036526852874970,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.017883390799,
                    0.0,
                    0.00765987541,
                    -0.0089416954,
                    0.0,
                    -0.00765987541,
                    -0.0089416954,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.01542209, 0.0, 0.0, 0.00771105, 0.0, 0.0, 0.00771105, 0.0, 0.0],
                    [0.0, 0.58612421, 0.0, 0.0, -0.2930621, 0.23079136, 0.0, -0.2930621, -0.23079136],
                    [0.0, 0.0, 0.38586066, 0.0, 0.17492242, -0.19293033, 0.0, -0.17492242, -0.19293033],
                    [0.00771105, 0.0, 0.0, -0.00638987, 0.0, 0.0, -0.00132118, 0.0, 0.0],
                    [0.0, -0.2930621, 0.17492242, 0.0, 0.32224469, -0.20285689, 0.0, -0.02918259, 0.02793447],
                    [0.0, 0.23079136, -0.19293033, 0.0, -0.20285689, 0.18247704, 0.0, -0.02793447, 0.01045329],
                    [0.00771105, 0.0, 0.0, -0.00132118, 0.0, 0.0, -0.00638987, 0.0, 0.0],
                    [0.0, -0.2930621, -0.17492242, 0.0, -0.02918259, -0.02793447, 0.0, 0.32224469, 0.20285689],
                    [0.0, -0.23079136, -0.19293033, 0.0, 0.02793447, 0.01045329, 0.0, 0.20285689, 0.18247704],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.005108963391,  # vcc
            "(T) CORRECTION ENERGY": -0.00498265,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.02003159253303],
                    [0.00000000000000, 0.00872844491897, -0.01001579626652],
                    [0.00000000000000, -0.00872844491897, -0.01001579626652],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # vcc fd
                [
                    [-0.01727460991, 0.0, 0.0, 0.008637304955, 0.0, 0.0, 0.008637304955, 0.0, 0.0],
                    [
                        0.0,
                        0.584492978582,
                        0.0,
                        0.0,
                        -0.292246489291,
                        0.230876480089,
                        0.0,
                        -0.292246489291,
                        -0.230876480089,
                    ],
                    [
                        0.0,
                        0.0,
                        0.382520093487,
                        0.0,
                        0.174867793441,
                        -0.191260046743,
                        0.0,
                        -0.174867793441,
                        -0.191260046743,
                    ],
                    [0.008637304955, 0.0, 0.0, -0.007206558163, 0.0, 0.0, -0.001430746792, 0.0, 0.0],
                    [
                        0.0,
                        -0.292246489291,
                        0.174867793441,
                        0.0,
                        0.321124482646,
                        -0.202872136765,
                        0.0,
                        -0.028877993355,
                        0.028004343324,
                    ],
                    [
                        0.0,
                        0.230876480089,
                        -0.191260046743,
                        0.0,
                        -0.202872136765,
                        0.181320766941,
                        0.0,
                        -0.028004343324,
                        0.009939279803,
                    ],
                    [0.008637304955, 0.0, 0.0, -0.001430746792, 0.0, 0.0, -0.007206558163, 0.0, 0.0],
                    [
                        0.0,
                        -0.292246489291,
                        -0.174867793441,
                        0.0,
                        -0.028877993355,
                        -0.028004343324,
                        0.0,
                        0.321124482646,
                        0.202872136765,
                    ],
                    [
                        0.0,
                        -0.230876480089,
                        -0.191260046743,
                        0.0,
                        0.028004343324,
                        0.009939279803,
                        0.0,
                        0.202872136765,
                        0.181320766941,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.004882427,  # mrcc only
            "B(T) CORRECTION ENERGY": -0.00518450,  # ccenergy only
            "CC3 CORRELATION ENERGY": -0.193441818385715,  # ccenergy
            "CCSDT-1A CORRELATION ENERGY": -0.193424330972,  # ecc
            "CCSDT-1B CORRELATION ENERGY": -0.193423371134,  # ecc
            "CCSDT-2 CORRELATION ENERGY": -0.193087540038,  # vcc
            "CCSDT-3 CORRELATION ENERGY": -0.19310599643349,  # vcc
            "CCSDT CORRELATION ENERGY": -0.19368177447948,  # vcc
            "OMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2 CORRELATION ENERGY": _knownmissing,
            "OMP2 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP2 TOTAL GRADIENT": _knownmissing,
            "OMP2.5 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2.5 CORRELATION ENERGY": _knownmissing,
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP2.5 TOTAL GRADIENT": _knownmissing,
            "OMP3 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP3 CORRELATION ENERGY": _knownmissing,
            "OMP3 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP3 TOTAL GRADIENT": _knownmissing,
            "OREMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OREMP2 CORRELATION ENERGY": _knownmissing,
            "OREMP2 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OREMP2 TOTAL GRADIENT": _knownmissing,
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0012186690,  # p4n (core-occ rotations neglected)
            "OLCCD CORRELATION ENERGY": -0.1929863522,  # p4n (core-occ rotations neglected)
            "OLCCD SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OLCCD TOTAL GRADIENT": _knownmissing,
            "OCCD REFERENCE CORRECTION ENERGY": 0.001089195,  # qchem
            "OCCD CORRELATION ENERGY": -0.188164420,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem, rearranged
                [0, 0, 0.01780351, 0, 0.00760260, -0.00890176, 0, -0.00760260, -0.00890176]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_qz2p_pk_uhf,
            "B2PLYP TOTAL ENERGY": -55.85411445881,  # nwc 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "CISD CORRELATION ENERGY": -0.08032199708731,  # vcc, sd
            "LCCSD CORRELATION ENERGY": -0.08517989,  # vcc != p4n
            "LCCSD OPPOSITE-SPIN CORRELATION ENERGY": -0.08151590,  # vcc nyi
            "CCD CORRELATION ENERGY": -0.08016689737891,  # vcc nyi == tce, sd
            "CCSD CORRELATION ENERGY": -0.08256719,
            "CCSD SINGLES ENERGY": -0.00117001688,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.00230304,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.005323074361,
                    0.0,
                    0.0,
                    -0.005174249172,
                    0.0,
                    0.014881203442,
                    -0.000074412594,
                    0.0,
                    -0.014881203442,
                    -0.000074412594,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00068823,  # cfour only
            "CCSDT CORRELATION ENERGY": -0.08361110233142,  # vcc
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.059407254257,
            "MP2 SINGLES ENERGY": -0.000688368657,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001851937488,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    # switched sign from unkn ref
                    0.000000000000000,
                    0.000000000000000,
                    0.013388410166131,
                    0.000000000000000,
                    0.000000000000000,
                    -0.012907368096590,
                    0.000000000000000,
                    0.010303507439169,
                    -0.000240521034770,
                    0.000000000000000,
                    -0.010303507439169,
                    -0.000240521034770,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # vcc by psi fd
                [
                    [
                        3.77039407e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.35686969e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.93237813e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.93237813e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        3.68005099e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.21445800e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.67203501e-04,
                        4.37851629e-04,
                        0.00000000e00,
                        2.67203501e-04,
                        -4.37851629e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.17707108e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.15648507e-01,
                        0.00000000e00,
                        1.13660067e-03,
                        -1.02930031e-03,
                        0.00000000e00,
                        -1.13660067e-03,
                        -1.02930031e-03,
                    ],
                    [
                        -4.35686969e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.70636057e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74745438e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74745438e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -4.21445800e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.62108124e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.03311621e-04,
                        8.06365754e-05,
                        0.00000000e00,
                        -2.03311621e-04,
                        -8.06365754e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.15648507e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.15870333e-01,
                        0.00000000e00,
                        -4.83601569e-05,
                        -1.10913000e-04,
                        0.00000000e00,
                        4.83601569e-05,
                        -1.10913000e-04,
                    ],
                    [
                        2.93237813e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74745438e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -7.39767359e-03,
                        0.00000000e00,
                        0.00000000e00,
                        7.27918122e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.67203501e-04,
                        1.13660067e-03,
                        0.00000000e00,
                        -2.03311621e-04,
                        -4.83601569e-05,
                        0.00000000e00,
                        4.11398494e-01,
                        -8.03364358e-04,
                        0.00000000e00,
                        -4.11462386e-01,
                        -2.84876153e-04,
                    ],
                    [
                        0.00000000e00,
                        4.37851629e-04,
                        -1.02930031e-03,
                        0.00000000e00,
                        8.06365754e-05,
                        -1.10913000e-04,
                        0.00000000e00,
                        -8.03364358e-04,
                        -4.61144770e-03,
                        0.00000000e00,
                        2.84876153e-04,
                        5.75166101e-03,
                    ],
                    [
                        2.93237813e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74745438e-04,
                        0.00000000e00,
                        0.00000000e00,
                        7.27918122e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -7.39767359e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.67203501e-04,
                        -1.13660067e-03,
                        0.00000000e00,
                        -2.03311621e-04,
                        4.83601569e-05,
                        0.00000000e00,
                        -4.11462386e-01,
                        2.84876153e-04,
                        0.00000000e00,
                        4.11398494e-01,
                        8.03364358e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.37851629e-04,
                        -1.02930031e-03,
                        0.00000000e00,
                        -8.06365754e-05,
                        -1.10913000e-04,
                        0.00000000e00,
                        -2.84876153e-04,
                        5.75166101e-03,
                        0.00000000e00,
                        8.03364358e-04,
                        -4.61144770e-03,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.074845418046,  # cfour
            "MP3 SINGLES ENERGY": -0.000911987215,  # cfour
            "MP3 TOTAL GRADIENT": np.array(  # vcc by fd psi
                [
                    [0.0, 0.0, 0.0099090428],
                    [0.0, 0.0, -0.009667824547],
                    [0.0, 0.012725362696, -0.000120609126],
                    [0.0, -0.012725362696, -0.000120609126],
                ]
            ),
            "MP4(SDQ) CORRELATION ENERGY": -0.07930873,  # vcc only, sc
            "MP4(T) CORRECTION ENERGY": -0.000289639941,  # vcc only, sc
            "ZAPT2 CORRELATION ENERGY": -0.058899120423767,  # detci (ZAPT uses special canonicalization but grouping here with sc)
            "CISD CORRELATION ENERGY": -0.08031834747884,  # vcc only, sc
            "LCCD CORRELATION ENERGY": -0.0824056198,  # p4n (likely sd since tce matches but leaving here until detangled)
            # "LCCSD CORRELATION ENERGY": -0.0851177481,  # p4n (was uncommented)
            "LCCSD CORRELATION ENERGY": -0.08517509,  # vcc nyi
            "LCCSD OPPOSITE-SPIN CORRELATION ENERGY": -0.08151150,  # vcc nyi
            "CCD CORRELATION ENERGY": -0.08016374258389,  # vcc, sc
            "BCCD CORRELATION ENERGY": -0.08269058,  # ccenergy only
            "CC2 CORRELATION ENERGY": -0.0602475799,  # ccenergy only, sc
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [
                    0.0,
                    0.0,
                    0.011923571677,
                    0.0,
                    0.0,
                    -0.011504255786,
                    0.0,
                    0.010550478287,
                    -0.000209657945,
                    0.0,
                    -0.010550478287,
                    -0.000209657945,
                ]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.082563103526,  # vcc
            "CCSD SINGLES ENERGY": -0.001170121907,  # vcc & ecc
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.0023025,  # ecc
            "CCSD TOTAL GRADIENT": np.array(
                # ccenergy findif-5 (to 1e-6 this sc == sd)
                [
                    0.0,
                    0.0,
                    0.005324268016,
                    0.0,
                    0.0,
                    -0.005175469821,
                    0.0,
                    0.014881252049,
                    -0.000074399098,
                    0.0,
                    -0.014881252049,
                    -0.000074399098,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.000691982886,  # vcc
            "CCSD(T) TOTAL GRADIENT": np.array(
                # ccenergy findif-5 (to 1e-6 this sc == sd)
                [
                    0.0,
                    0.0,
                    0.004576121181,
                    0.0,
                    0.0,
                    -0.004510736115,
                    0.0,
                    0.01496004535,
                    -0.000032692533,
                    0.0,
                    -0.01496004535,
                    -0.000032692533,
                ]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.000678625130,  # mrcc only, sc
            "B(T) CORRECTION ENERGY": -0.00061088,  # ccenergy only, sc
            "CC3 CORRELATION ENERGY": -0.083247773021380,  # ccenergy only, sc
            "CCSDT CORRELATION ENERGY": -0.083606448205,  # mrcc, sc
            "OMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2 CORRELATION ENERGY": _knownmissing,
            "OMP2 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP2 TOTAL GRADIENT": _knownmissing,
            "OMP2.5 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2.5 CORRELATION ENERGY": _knownmissing,
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP2.5 TOTAL GRADIENT": _knownmissing,
            "OMP3 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP3 CORRELATION ENERGY": _knownmissing,
            "OMP3 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP3 TOTAL GRADIENT": _knownmissing,
            "OREMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OREMP2 CORRELATION ENERGY": _knownmissing,
            "OREMP2 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OREMP2 TOTAL GRADIENT": _knownmissing,
            "OLCCD REFERENCE CORRECTION ENERGY": -0.0000318047,  # p4n (core-occ rotations neglected)
            "OLCCD CORRELATION ENERGY": -0.0854404197,  # p4n (core-occ rotations neglected)
            "OLCCD SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OLCCD TOTAL GRADIENT": _knownmissing,
            "OCCD REFERENCE CORRECTION ENERGY": -0.000362931,  # qchem
            "OCCD CORRELATION ENERGY": -0.082712571,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem 3-pt findif
                [
                    0.0,
                    8.460068018933954e-12,
                    4.950754755080671e-03,
                    0.0,
                    -1.880015115318656e-12,
                    -4.795829218979597e-03,
                    0.0,
                    1.487359483495976e-02,
                    -7.746235397720769e-05,
                    0.0,
                    -1.487359480111949e-02,
                    -7.746237653738908e-05,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "CISD CORRELATION ENERGY": -0.16970933,  # vcc, sd
            "LCCSD CORRELATION ENERGY": -0.18080394,  # vcc != p4n (was commented)
            "LCCSD OPPOSITE-SPIN CORRELATION ENERGY": -0.14354870,  # vcc nyi only
            "CCD CORRELATION ENERGY": -0.17252087862035,  # vcc nyi == tce, sd
            "CCSD CORRELATION ENERGY": -0.175988485854028,
            "CCSD SINGLES ENERGY": -0.003256808469230,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.033291143258924,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.030051791297,
                    0.0,
                    0.016301545337,
                    -0.015025895649,
                    0.0,
                    -0.016301545337,
                    -0.015025895649,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.003863167899,  # vcc only
            "CCSDT CORRELATION ENERGY": -0.18030677104047,  # vcc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.15702660833165538,  #  sd == sc
            "MP2 SINGLES ENERGY": -0.0028059971624814647,  #  sd == sc
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03466304269235235,  # sd == sc
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.026398091851,
                    0.0,
                    0.014012163884,
                    -0.013199045925,
                    0.0,
                    -0.014012163884,
                    -0.013199045925,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # vcc by psi fd
                [
                    [-0.0227647, 0.0, 0.0, 0.01138235, 0.0, 0.0, 0.01138235, 0.0, 0.0],
                    [0.0, 0.62606906, 0.0, 0.0, -0.31303453, 0.24893382, 0.0, -0.31303453, -0.24893382],
                    [0.0, 0.0, 0.39895549, 0.0, 0.1914611, -0.19947774, 0.0, -0.1914611, -0.19947774],
                    [0.01138235, 0.0, 0.0, -0.01032718, 0.0, 0.0, -0.00105517, 0.0, 0.0],
                    [0.0, -0.31303453, 0.1914611, 0.0, 0.34009582, -0.22019746, 0.0, -0.02706129, 0.02873636],
                    [0.0, 0.24893382, -0.19947774, 0.0, -0.22019746, 0.1906097, 0.0, -0.02873636, 0.00886804],
                    [0.01138235, 0.0, 0.0, -0.00105517, 0.0, 0.0, -0.01032718, 0.0, 0.0],
                    [0.0, -0.31303453, -0.1914611, 0.0, -0.02706129, -0.02873636, 0.0, 0.34009582, 0.22019746],
                    [0.0, -0.24893382, -0.19947774, 0.0, 0.02873636, 0.00886804, 0.0, 0.22019746, 0.1906097],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.173406218947,  # cfour
            "MP3 SINGLES ENERGY": -0.003217654547,  # cfour
            "MP3 TOTAL GRADIENT": np.array(  # vcc by fd psi
                [[0.0, 0.0, 0.0280810549], [0.0, 0.0150218165, -0.0140405275], [0.0, -0.0150218165, -0.0140405275]]
            ),
            "MP4(SDQ) CORRELATION ENERGY": -0.17541828,  # vcc only, sc
            "MP4(T) CORRECTION ENERGY": -0.00334215131,  # vcc only, sc
            "ZAPT2 CORRELATION ENERGY": -0.156282101871335,  # detci
            "CISD CORRELATION ENERGY": -0.169713778760,  # vcc (-0.16970933003145,  # vcc longstanding)
            "LCCD CORRELATION ENERGY": -0.1769020687,  # p4n (likely sd since tce matches but leaving here until detangled)
            # "LCCSD CORRELATION ENERGY": -0.1807707740,  # p4n (was uncommented)
            "LCCSD CORRELATION ENERGY": -0.18080894256088,  # vcc != p4n, sc
            "LCCSD OPPOSITE-SPIN CORRELATION ENERGY": -0.14355320,  # vcc nyi only
            "CCD CORRELATION ENERGY": -0.17252609,  # vcc
            "BCCD CORRELATION ENERGY": -0.175845488255032,  # ccenergy only
            "CC2 CORRELATION ENERGY": -0.1581932061,  # ccenergy only, sc
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.027576804444, 0.0, 0.014704262054, -0.013788402222, 0.0, -0.014704262054, -0.013788402222]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.175993216083,  # vcc
            "CCSD SINGLES ENERGY": -0.003256554797,  # vcc & ecc
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.033292,  # ecc
            "CCSD TOTAL GRADIENT": np.array(
                # ccenergy findif-5 (to 1e-6 this sc == sd)
                [0.0, 0.0, 0.030052138921, 0.0, 0.016301704865, -0.015026069461, 0.0, -0.016301704865, -0.015026069461]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.003868160727,  # vcc, sc
            "CCSD(T) TOTAL GRADIENT": np.array(  # ccenergy findif-5
                [0.0, 0.0, 0.031949750405, 0.0, 0.017311094784, -0.015974875203, 0.0, -0.017311094784, -0.015974875203]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.003784992966,  # mrcc only, sc
            "B(T) CORRECTION ENERGY": -0.003998082545023,  # ccenergy only, sc
            "CC3 CORRELATION ENERGY": -0.179979065555105,  # ccenergy only, sc
            "CCSDT CORRELATION ENERGY": -0.18031166502580,  # vcc
            "OMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2 CORRELATION ENERGY": _knownmissing,
            "OMP2 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP2 TOTAL GRADIENT": _knownmissing,
            "OMP2.5 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2.5 CORRELATION ENERGY": _knownmissing,
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP2.5 TOTAL GRADIENT": _knownmissing,
            "OMP3 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP3 CORRELATION ENERGY": _knownmissing,
            "OMP3 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP3 TOTAL GRADIENT": _knownmissing,
            "OREMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OREMP2 CORRELATION ENERGY": _knownmissing,
            "OREMP2 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OREMP2 TOTAL GRADIENT": _knownmissing,
            "OLCCD REFERENCE CORRECTION ENERGY": -0.0032999290,  # p4n (core-occ rotations neglected)
            "OLCCD CORRELATION ENERGY": -0.1804110970,  # p4n (core-occ rotations neglected)
            "OLCCD SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OLCCD TOTAL GRADIENT": _knownmissing,
            "OCCD REFERENCE CORRECTION ENERGY": -0.003402152,  # qchem
            "OCCD CORRELATION ENERGY": -0.175891240,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem 3-pt findif
                [
                    0.0,
                    -2.131628207280301e-11,
                    2.994042840143152e-02,
                    0.0,
                    1.624512323772365e-02,
                    -1.497021096596995e-02,
                    0.0,
                    -1.624512344378104e-02,
                    -1.497021098018081e-02,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "CISD CORRELATION ENERGY": -0.18613466061896,  # vcc
            "LCCSD CORRELATION ENERGY": -0.19795428,  # vcc != p4n
            "LCCSD OPPOSITE-SPIN CORRELATION ENERGY": -0.15734510,  # vcc nyi, sd
            "CCD CORRELATION ENERGY": -0.18921914785643,  # vcc nyi == tce, sd
            "CCSD CORRELATION ENERGY": -0.19282621471297376,
            "CCSD SINGLES ENERGY": -0.003354603508621,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.036502859698546,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.017873897449,
                    0.0,
                    0.007653541045,
                    -0.008936948724,
                    0.0,
                    -0.007653541045,
                    -0.008936948724,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00504351,  # cfour only
            "CCSDT CORRELATION ENERGY": -0.19824510672649,  # vcc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.1761163066,
            "MP2 SINGLES ENERGY": -0.00294339,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03837483,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.014894057335,
                    0.0,
                    0.005886660707,
                    -0.007447028667,
                    0.0,
                    -0.005886660707,
                    -0.007447028667,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # vcc by psi fd
                [
                    [-0.0128439, 0.0, 0.0, 0.00642195, 0.0, 0.0, 0.00642195, 0.0, 0.0],
                    [0.0, 0.59650839, 0.0, 0.0, -0.29825419, 0.23378625, 0.0, -0.29825419, -0.23378625],
                    [0.0, 0.0, 0.38870359, 0.0, 0.17712488, -0.1943518, 0.0, -0.17712488, -0.1943518],
                    [0.00642195, 0.0, 0.0, -0.00515858, 0.0, 0.0, -0.00126337, 0.0, 0.0],
                    [0.0, -0.29825419, 0.17712488, 0.0, 0.32634865, -0.20545557, 0.0, -0.02809446, 0.02833069],
                    [0.0, 0.23378625, -0.1943518, 0.0, -0.20545557, 0.18492354, 0.0, -0.02833069, 0.00942826],
                    [0.00642195, 0.0, 0.0, -0.00126337, 0.0, 0.0, -0.00515858, 0.0, 0.0],
                    [0.0, -0.29825419, -0.17712488, 0.0, -0.02809446, -0.02833069, 0.0, 0.32634865, 0.20545557],
                    [0.0, -0.23378625, -0.1943518, 0.0, 0.02833069, 0.00942826, 0.0, 0.20545557, 0.18492354],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.190735179201,  # cfour
            "MP3 SINGLES ENERGY": -0.003322599961,  # cfour
            "MP3 TOTAL GRADIENT": np.array(  # vcc by fd psi
                [
                    [0.0, 0.0, 0.015664483926],
                    [0.0, 0.006224027206, -0.007832241963],
                    [0.0, -0.006224027206, -0.007832241963],
                ]
            ),
            "MP4(SDQ) CORRELATION ENERGY": -0.19243169,  # vcc only, sc
            "MP4(T) CORRECTION ENERGY": -0.004597066465,  # vcc only, sc
            "ZAPT2 CORRELATION ENERGY": -0.17532659,  # gamess
            "CISD CORRELATION ENERGY": -0.186140372647,  # vcc, sc
            "LCCD CORRELATION ENERGY": -0.1939920915,  # p4n (likely sd since tce matches but leaving here until detangled)
            # "LCCSD CORRELATION ENERGY": -0.1979175937,  # p4n (was uncommented)
            "LCCSD CORRELATION ENERGY": -0.19796066,  # vcc nyi, sc
            "LCCSD OPPOSITE-SPIN CORRELATION ENERGY": -0.15734990,  # vcc nyi, sc
            "CCD CORRELATION ENERGY": -0.18922567778994,  # vcc, sc
            "BCCD CORRELATION ENERGY": -0.19269438,  # ccenergy only
            "CC2 CORRELATION ENERGY": -0.1772785627,  # ccenergy only, sc
            "CC2 TOTAL GRADIENT": np.array(  # ccenergy, findif-5
                [0.0, 0.0, 0.016255578053, 0.0, 0.006637178638, -0.008127789026, 0.0, -0.006637178638, -0.008127789026]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.1928322825369395,  # vcc
            "CCSD SINGLES ENERGY": -0.00335427534294,  # vcc & ecc
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.0365048,  # ecc
            "CCSD TOTAL GRADIENT": np.array(
                # ccenergy findif-5 (to 1e-6 this sc == sd)
                [0.0, 0.0, 0.017874684927, 0.0, 0.007653820061, -0.008937342463, 0.0, -0.007653820061, -0.008937342463]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.005047880396,  # vcc
            "CCSD(T) TOTAL GRADIENT": np.array(  # ccenergy findif-5
                [0.0, 0.0, 0.020046694566, 0.0, 0.008736311342, -0.010023347283, 0.0, -0.008736311342, -0.010023347283]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.004942680340,  # mrcc only, sc
            "B(T) CORRECTION ENERGY": -0.00518478,  # ccenergy only, sc
            "CC3 CORRELATION ENERGY": -0.198016086042130,  # ccenergy only, sc
            "CCSDT CORRELATION ENERGY": -0.19825144107785,  # vcc
            "OMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2 CORRELATION ENERGY": _knownmissing,
            "OMP2 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP2 TOTAL GRADIENT": _knownmissing,
            "OMP2.5 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP2.5 CORRELATION ENERGY": _knownmissing,
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP2.5 TOTAL GRADIENT": _knownmissing,
            "OMP3 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OMP3 CORRELATION ENERGY": _knownmissing,
            "OMP3 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OMP3 TOTAL GRADIENT": _knownmissing,
            "OREMP2 REFERENCE CORRECTION ENERGY": _knownmissing,
            "OREMP2 CORRELATION ENERGY": _knownmissing,
            "OREMP2 SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OREMP2 TOTAL GRADIENT": _knownmissing,
            "OLCCD REFERENCE CORRECTION ENERGY": -0.0033910391,  # p4n (core-occ rotations neglected)
            "OLCCD CORRELATION ENERGY": -0.1975960603,  # p4n (core-occ rotations neglected)
            "OLCCD SAME-SPIN CORRELATION ENERGY": _knownmissing,
            "OLCCD TOTAL GRADIENT": _knownmissing,
            "OCCD REFERENCE CORRECTION ENERGY": -0.003478949,  # qchem
            "OCCD CORRELATION ENERGY": -0.192732814,  # qchem
            "OCCD TOTAL GRADIENT": np.array(  # qchem 3-pt findif
                [
                    0.0,
                    1.705302565824240e-10,
                    1.774538645804569e-02,
                    0.0,
                    7.588575865469238e-03,
                    -8.872690465011601e-03,
                    0.0,
                    -7.588575776651396e-03,
                    -8.872690337113909e-03,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": _knownmissing,
            "A-O(T) CORRECTION ENERGY": _knownmissing,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CONV-AE-CD  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.203778449,
            "MP2 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.20648138,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.20990784,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.20874537,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00193646,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00196115,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_pk_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.22188844,
            "MP2 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.22643352,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.23188996,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.22941330,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00523874,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00523635,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_adz_pk_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.27018509,
            "MP2 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.27294541,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.27869144,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.27570541,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00726403,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00718185,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_qz2p_pk_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.059477703268,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001918940186,
            "MP3 CORRELATION ENERGY": -0.07481148,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00235724,  # dfocc
            "LCCD CORRELATION ENERGY": -0.08343267,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.00240067,  # dfocc
            "CCSD CORRELATION ENERGY": -0.08217068,  # dfocc
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00062618,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00060937,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_bh3p_dz_pk_uhf,
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.15485159,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03520588,
            "MP3 CORRELATION ENERGY": -0.17091968,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03533767,  # dfocc
            "LCCD CORRELATION ENERGY": -0.17701281,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.03413088,  # dfocc
            "CCSD CORRELATION ENERGY": -0.17387591,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00384405,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00376422,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_adz_pk_uhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.19552518,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04162160,
            "MP3 CORRELATION ENERGY": -0.21084703,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04130865,  # dfocc
            "LCCD CORRELATION ENERGY": -0.21678793,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.04013550,  # dfocc
            "CCSD CORRELATION ENERGY": -0.21329809,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00516691,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00506463,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_qz2p_pk_uhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.0604664810,
            "MP2 SINGLES ENERGY": -0.000694049858,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.002005152902,
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.15948893,
            "MP2 SINGLES ENERGY": -0.00282963,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03542136,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.20053428,
            "MP2 SINGLES ENERGY": -0.00298375,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04179065,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CONV-FC-CD  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.20162439774,
            "MP2 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.20453779,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.20796060,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.20681721,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00192057,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00194429,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_pk_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.21939933,
            "MP2 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.22415842,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.22961687,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.22717646,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00521255,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00520986,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_adz_pk_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.24514540,
            "MP2 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.24747829,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.25319438,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.25033052,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00709694,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00701833,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_qz2p_pk_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.05841222894,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0017676971,
            "MP3 CORRELATION ENERGY": -0.07375426,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00220422,  # dfocc
            "LCCD CORRELATION ENERGY": -0.08242955,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.00225358,  # dfocc
            "CCSD CORRELATION ENERGY": -0.08116911,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00060405,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00058791,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_bh3p_dz_pk_uhf,
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.15241971,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03445776,
            "MP3 CORRELATION ENERGY": -0.16863252,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03464025,  # dfocc
            "LCCD CORRELATION ENERGY": -0.17475833,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.03344184,  # dfocc
            "CCSD CORRELATION ENERGY": -0.17165381,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00381142,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00373217,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_adz_pk_uhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.17117906,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03822980,
            "MP3 CORRELATION ENERGY": -0.18575660,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03792752,  # dfocc
            "LCCD CORRELATION ENERGY": -0.19170259,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.03676455,  # dfocc
            "CCSD CORRELATION ENERGY": -0.18831733,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00498297,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00488274,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_qz2p_pk_uhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.059395907176,
            "MP2 SINGLES ENERGY": -0.00068836865,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001852162877,
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.15701860,
            "MP2 SINGLES ENERGY": -0.00280600,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03466721,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.17611121,
            "MP2 SINGLES ENERGY": -0.00294339,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03838011,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CONV-AE-DF  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.20377997248921056,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05431321036920538,
            "MP2 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.0027998, 0.0, 0.0, -0.0027998]  # dfmp2 findif-5 ae pk+df
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20657694,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000926859678, 0.0, 0.0, 0.000926859678]).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2100497124,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.002190589954, 0.0, 0.0, -0.002190589954]).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.20888438,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00193859,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00196333,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_pk_rhf,
            "B2PLYP TOTAL ENERGY": -100.3645971549267983,  # psi, 99,590
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.22188894,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05674808,
            "MP3 CORRELATION ENERGY": -0.22652962,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2320261414,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.22954333,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00524393,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00524145,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_adz_pk_rhf,
            "B2PLYP TOTAL ENERGY": -76.3804039258306062,  # psi, 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.27018057,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06530212,
            "MP3 CORRELATION ENERGY": -0.27294176,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2786878429,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.27570207,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00726375,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00718158,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_qz2p_pk_rhf,
            "B2PLYP TOTAL ENERGY": -76.4168035769188378,  # psi, 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.05945820694747983,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0019203155958724552,
            "MP3 CORRELATION ENERGY": -0.07486724,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00235854,  # dfocc
            "LCCD CORRELATION ENERGY": -0.0835080983,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0024018298,  # dfocc
            "CCSD CORRELATION ENERGY": -0.08224363,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00062669,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00060985,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_bh3p_dz_pk_uhf,
            "B2PLYP TOTAL ENERGY": -26.0729378891483954,  # psi, 99,590
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.15484736,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03523329,
            "MP3 CORRELATION ENERGY": -0.17098385,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03530465,  # dfocc
            "LCCD CORRELATION ENERGY": -0.1771107929,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0340809591,  # dfocc
            "CCSD CORRELATION ENERGY": -0.17396848,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00384851,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00376850,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_adz_pk_uhf,
            "B2PLYP TOTAL ENERGY": -55.8336257642485307,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.19551918,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04161696,
            "MP3 CORRELATION ENERGY": -0.21084349,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04130382,  # dfocc
            "LCCD CORRELATION ENERGY": -0.2167841215,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0401306929,  # dfocc
            "CCSD CORRELATION ENERGY": -0.21329436,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00516666,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00506439,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_qz2p_pk_uhf,
            "B2PLYP TOTAL ENERGY": -55.8610710137206539,  # psi, 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.0604460449537298,
            "MP2 SINGLES ENERGY": -0.0006940498589629459,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0020066877639503184,
            "LCCD CORRELATION ENERGY": -0.0834825821,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.15948289,
            "MP2 SINGLES ENERGY": -0.00282963,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03544835,
            "LCCD CORRELATION ENERGY": -0.1792713801,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.20052829,
            "MP2 SINGLES ENERGY": -0.00298375,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04178599,
            "LCCD CORRELATION ENERGY": -0.2191002183,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CONV-FC-DF  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.20162566806258586,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05352569481658172,
            "MP2 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.00315485, 0.0, 0.0, -0.00315485]  # dfmp2 findif-5 fc pk+df
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20463304,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2081020566,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.20695586,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00192267,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00194643,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_pk_rhf,
            "B2PLYP TOTAL ENERGY": -100.3639796418507899,  # psi 55,590
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.21939942,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05588210,
            "MP3 CORRELATION ENERGY": -0.22425409,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2297524911,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.22730597,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00521769,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00521491,  # dfocc, tight
            "B2PLYP TOTAL ENERGY": -76.3796850854499780,  # psi 55,590
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_adz_pk_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.24514425,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06126481,
            "MP3 CORRELATION ENERGY": -0.24747778,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2531939249,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.25033030,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00709666,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00701806,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_qz2p_pk_rhf,
            "B2PLYP TOTAL ENERGY": -76.4096994249378554,  # psi 55,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.058392397606538686,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0017690135626491292,
            "MP3 CORRELATION ENERGY": -0.07380965,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00220548,  # dfocc
            "LCCD CORRELATION ENERGY": -0.0825046579,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0022547041,  # dfocc
            "CCSD CORRELATION ENERGY": -0.08124172,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00060454,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00058837,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_bh3p_dz_pk_uhf,
            "B2PLYP TOTAL ENERGY": -26.0726092204242192,  # psi 55,590
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.15241501,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03448519,
            "MP3 CORRELATION ENERGY": -0.16869625,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03460729,  # dfocc
            "LCCD CORRELATION ENERGY": -0.1748557523,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0333918420,  # dfocc
            "CCSD CORRELATION ENERGY": -0.17174585,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00381584,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00373642,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_adz_pk_uhf,
            "B2PLYP TOTAL ENERGY": -55.8329139653478279,  # psi 55,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.17117615,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03822512,
            "MP3 CORRELATION ENERGY": -0.18575590,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03792262,  # dfocc
            "LCCD CORRELATION ENERGY": -0.1917015960,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0367596684,  # dfocc
            "CCSD CORRELATION ENERGY": -0.18831642,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00498272,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00488249,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_qz2p_pk_uhf,
            "B2PLYP TOTAL ENERGY": -55.8541121912572009,  # psi 55,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.05937514348825628,
            "MP2 SINGLES ENERGY": -0.0006883686516107368,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0018536363586657242,
            "LCCD CORRELATION ENERGY": -0.0824786458,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.15701209,
            "MP2 SINGLES ENERGY": -0.00280600,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03469422,
            "LCCD CORRELATION ENERGY": -0.1770018748,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.17610830,
            "MP2 SINGLES ENERGY": -0.00294339,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03837541,
            "LCCD CORRELATION ENERGY": -0.1939912613,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CD-AE-CD  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_cd_rhf,
            # "HF TOTAL HESSIAN": _hess_scf_hf_dz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.20377328786815951,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05427570,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 ae cd+cd
                [0.0, 0.0, 0.00281146, 0.0, 0.0, -0.00281146]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20647618,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array(
                # dfocc findif-5
                [0.0, 0.0, -0.000926981449, 0.0, 0.0, 0.000926981449]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.208401248910,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.20990226,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD TOTAL GRADIENT": np.array(
                # dfocc findif-5
                [0.0, 0.0, 0.002193849073, 0.0, 0.0, -0.002193849073]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.2081480005,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(
                # dfocc findif-5 tight
                [0.0, 0.0, 0.000827082456, 0.0, 0.0, -0.000827082456]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.20873986003026,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04857381,
            "CCSD TOTAL GRADIENT": np.array(
                # dfocc findif-5 tight
                [0.0, 0.0, 0.001979500372, 0.0, 0.0, -0.001979500372]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00193631092143,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight, findif-5
                [0.0, 0.0, 0.003075112296, 0.0, 0.0, -0.003075112296]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00196099396220,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.000704847713,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.204457056936,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [0.0, 0.0, 0.004099545733, 0.0, 0.0, -0.004099545733]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.000571702068,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.205680612213,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [0.0, 0.0, 0.002008491447, 0.0, 0.0, -0.002008491447]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000467209128,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.206931341471,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [0.0, 0.0, -0.000016455846, 0.0, 0.0, 0.000016455846]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.000553277131,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.208938401219,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [0.0, 0.0, 0.00282065206, 0.0, 0.0, -0.00282065206]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.000552265186,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.210437984945,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight, sc
                [0.0, 0.0, 0.003382226985, 0.0, 0.0, -0.003382226985]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.000511278480,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.208645666360,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.002062122653,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.002042124365,  # dfocc, tight
            "SVWN TOTAL ENERGY": -99.97974226710002,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.019330333461, 0.0, 0.0, -0.019330333461]
            ).reshape((-1, 3)),
            "PBE TOTAL ENERGY": -100.33526326,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590, findif-5
                [0.0, 0.0, 0.020101373239, 0.0, 0.0, -0.020101373239]
            ).reshape((-1, 3)),
            "B3LYP TOTAL ENERGY": -100.43551459,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590, findif-5
                [0.0, 0.0, 0.01136685656, 0.0, 0.0, -0.01136685656]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_cd_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.22188817,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05670210,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 ae cd+cd
                [
                    0.0,
                    0.0,
                    0.009643414073,
                    0.0,
                    0.005501440694,
                    -0.004821707036,
                    0.0,
                    -0.005501440694,
                    -0.004821707036,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.22643303,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "REMP2 CORRELATION ENERGY": -0.229445833755,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.23188949,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCD CORRELATION ENERGY": -0.2280230001,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.22941289840818,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05017955,
            "(T) CORRECTION ENERGY": -0.00523866932915,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00523628844912,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.002352399946,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.224116136177,  # dfocc, tight
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001535058364,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.225648131745,  # dfocc, tight
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000953721118,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.227374133013,  # dfocc, tight
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.00128180471179,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.23069205056701,  # dfocc, right
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001189623873,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.233047836839,  # dfocc, tight
            "OCCD REFERENCE CORRECTION ENERGY": 0.001124461842,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.229120333540,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.005573969849,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.005476248095,  # dfocc, tight
            "SVWN TOTAL ENERGY": -76.07573625160775,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.017008805687, 0.0, 0.012808822434, -0.008504402844, 0.0, -0.012808822434, -0.008504402844]
            ).reshape((-1, 3)),
            "PBE TOTAL ENERGY": -76.35898375776449,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590, findif-5
                [0.0, 0.0, 0.018083226924, 0.0, 0.011576350536, -0.009041613462, 0.0, -0.011576350536, -0.009041613462]
            ).reshape((-1, 3)),
            "B3LYP TOTAL ENERGY": -76.44455249,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590, findif-5
                [0.0, 0.0, 0.007469575531, 0.0, 0.006128608861, -0.003734787766, 0.0, -0.006128608861, -0.003734787766]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_adz_cd_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.27018399,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06530655,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 ae cd+cd
                [
                    0.0,
                    0.0,
                    -0.000546229785,
                    0.0,
                    -0.000967320028,
                    0.000273114892,
                    0.0,
                    0.000967320028,
                    0.000273114892,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.27294416,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "REMP2 CORRELATION ENERGY": -0.276442808601,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.27869015,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCD CORRELATION ENERGY": -0.2741709643,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.27570421327166,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05801141,
            "(T) CORRECTION ENERGY": -0.00726394807275,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00718177622315,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.002513733031,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.272591834175,  # dfocc, tight
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001691501139,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.273207190194,  # dfocc, tight
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001090092456,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.274014744975,  # dfocc, tight
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001444318022,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.277846110170,  # dfocc, tight
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001352168423,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.280004165823,  # dfocc, tight
            "OCCD REFERENCE CORRECTION ENERGY": 0.001279252698,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.275416999785,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.007572980491,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.007457755255,  # dfocc, tight
            "SVWN TOTAL ENERGY": -76.10234910181312,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.011171500733, 0.0, 0.009514061362, -0.005585750366, 0.0, -0.009514061362, -0.005585750366]
            ).reshape((-1, 3)),
            "PBE TOTAL ENERGY": -76.38214387,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590, findif-5
                [0.0, 0.0, 0.012782045518, 0.0, 0.008610884679, -0.006391022759, 0.0, -0.008610884679, -0.006391022759]
            ).reshape((-1, 3)),
            "B3LYP TOTAL ENERGY": -76.46846507,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590, findif-5
                [0.0, 0.0, 0.00202921515, 0.0, 0.003048184166, -0.001014607575, 0.0, -0.003048184166, -0.001014607575]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_qz2p_cd_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_uhf,
            # "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.059476326350818454,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0019188791023,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 ae cd+cd
                [
                    0.0,
                    0.0,
                    0.0125029,
                    0.0,
                    0.0,
                    -0.01205882,
                    0.0,
                    0.01033888,
                    -0.00022204,
                    0.0,
                    -0.01033888,
                    -0.00022204,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.07480952,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00235716,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.009315056264,
                    0.0,
                    0.0,
                    -0.009084753984,
                    0.0,
                    0.012696375212,
                    -0.00011515114,
                    0.0,
                    -0.012696375212,
                    -0.00011515114,
                ]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.078034068730,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.002339588819,  # dfocc, tight
            "LCCD CORRELATION ENERGY": -0.08343038,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.00240059,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.004643130768,
                    0.0,
                    0.0,
                    -0.004557062257,
                    0.0,
                    0.015239807714,
                    -0.000043034255,
                    0.0,
                    -0.015239807714,
                    -0.000043034255,
                ]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.08113343,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(
                # dfocc findif-5 tight
                [
                    0.0,
                    0.0,
                    0.0064041956,
                    0.0,
                    0.0,
                    -0.006210948996,
                    0.0,
                    0.014427521395,
                    -0.000096623302,
                    0.0,
                    -0.014427521395,
                    -0.000096623302,
                ]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.08216852322069,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD TOTAL GRADIENT": np.array(
                # dfocc findif-5 tight
                [
                    0.0,
                    0.0,
                    0.005208056118,
                    0.0,
                    0.0,
                    -0.005066275308,
                    0.0,
                    0.014858029899,
                    -0.000070890405,
                    0.0,
                    -0.014858029899,
                    -0.000070890405,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00062616540400,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(
                # dfocc findif-5 tight
                [
                    0.0,
                    0.0,
                    0.004696859967,
                    0.0,
                    0.0,
                    -0.004638013918,
                    0.0,
                    0.014937987395,
                    -0.000029423024,
                    0.0,
                    -0.014937987395,
                    -0.000029423024,
                ]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00060935179141,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.000395339866,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.059851604202,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.001971334763,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.012014926118,
                    0.0,
                    0.0,
                    -0.011616714081,
                    0.0,
                    0.010556988797,
                    -0.000199106019,
                    0.0,
                    -0.010556988797,
                    -0.000199106019,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.000564647007,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.067670676114,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.002198889579,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.010258655017,
                    0.0,
                    0.0,
                    -0.009975425021,
                    0.0,
                    0.011807301213,
                    -0.000141614998,
                    0.0,
                    -0.011807301213,
                    -0.000141614998,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000777636403,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.075525183445,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.002426255771,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.008454603092,
                    0.0,
                    0.0,
                    -0.008286658589,
                    0.0,
                    0.013068349488,
                    -0.000083972251,
                    0.0,
                    -0.013068349488,
                    -0.000083972251,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001061022340,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.078990084532,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.005896409683,
                    0.0,
                    0.0,
                    -0.005784924233,
                    0.0,
                    0.014365074414,
                    -0.000055742725,
                    0.0,
                    -0.014365074414,
                    -0.000055742725,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001484257667,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.084737023607,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.002448903277,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight, sc
                [
                    0.0,
                    0.0,
                    0.002950898504,
                    0.0,
                    0.0,
                    -0.002932789842,
                    0.0,
                    0.015723337205,
                    -0.000009054331,
                    0.0,
                    -0.015723337205,
                    -0.000009054331,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.001166023651,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.082184596469,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.000628150337,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.000605639575,  # dfocc, tight
            "SVWN TOTAL ENERGY": -25.88284516371439,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    0.0,
                    0.0,
                    -0.007521150796,
                    0.0,
                    0.0,
                    0.002274281261,
                    0.0,
                    0.036134178605,
                    0.002623434767,
                    0.0,
                    -0.036134178605,
                    0.002623434767,
                ]
            ).reshape((-1, 3)),
            "PBE TOTAL ENERGY": -26.04684191,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590, findif-5
                [
                    0.0,
                    0.0,
                    -0.002919057618,
                    0.0,
                    0.0,
                    -0.000173763979,
                    0.0,
                    0.02614258472,
                    0.001546410798,
                    0.0,
                    -0.02614258472,
                    0.001546410798,
                ]
            ).reshape((-1, 3)),
            "B3LYP TOTAL ENERGY": -26.11402203,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590, findif-5
                [
                    0.0,
                    0.0,
                    0.003320289103,
                    0.0,
                    0.0,
                    -0.005826956801,
                    0.0,
                    0.019182511328,
                    0.001253333849,
                    0.0,
                    -0.019182511328,
                    0.001253333849,
                ]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_bh3p_dz_cd_uhf,
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.15485101,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03520580,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 ae cd+cd
                    0.0,
                    0.0,
                    0.025470063809,
                    0.0,
                    0.013535107677,
                    -0.012735031905,
                    0.0,
                    -0.013535107677,
                    -0.012735031905,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.17091879,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03533750,  # dfocc
            "REMP2 CORRELATION ENERGY": -0.172502759037,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.034552222422,  # dfocc, tight
            "LCCD CORRELATION ENERGY": -0.17701192,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.03413070,  # dfocc
            "CCD CORRELATION ENERGY": -0.17274477,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.17387519037716,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00384402655927,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00376419871586,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.001367410529,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.156184908805,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.035978726948,  # dfocc, tight
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001174113090,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.164041684333,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.035851868032,  # dfocc, tight
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001024364167,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.171935686596,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.035751323781,  # dfocc, tight
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.00111244830766,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.17360075617488,  # dfocc, tight
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001111961826,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.178109192111,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.034472712167,  # dfocc, tight
            "OCCD REFERENCE CORRECTION ENERGY": 0.000987767517,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.173723374782,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.003989459873,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.003889358031,  # dfocc, tight
            "SVWN TOTAL ENERGY": -55.57345144327663,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.035469689186, 0.0, 0.020928985861, -0.017734844593, 0.0, -0.020928985861, -0.017734844593]
            ).reshape((-1, 3)),
            "PBE TOTAL ENERGY": -55.81475332612209,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590, findif-5
                [0.0, 0.0, 0.035840315894, 0.0, 0.019756083894, -0.017920157947, 0.0, -0.019756083894, -0.017920157947]
            ).reshape((-1, 3)),
            "B3LYP TOTAL ENERGY": -55.89009263,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590, findif-5
                [0.0, 0.0, 0.0268141821, 0.0, 0.01494743893, -0.01340709105, 0.0, -0.01494743893, -0.01340709105]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_adz_cd_uhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.19552441,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04162127,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 ae cd+cd
                    0.0,
                    0.0,
                    0.013727424376,
                    0.0,
                    0.005348487843,
                    -0.006863712188,
                    0.0,
                    -0.005348487843,
                    -0.006863712188,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.21084618,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04130830,  # dfocc
            "REMP2 CORRELATION ENERGY": -0.212450972170,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.040612908150,  # dfocc, tight
            "LCCD CORRELATION ENERGY": -0.21678706,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.04013515,  # dfocc
            "CCD CORRELATION ENERGY": -0.21199584,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.21329730737718,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00516682217061,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00506455138393,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.001570622893,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.197065813141,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.042464346261,  # dfocc, tight
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001350861495,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.204519387255,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.042091026132,  # dfocc, tight
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001177850332,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.212014665319,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.041748214104,  # dfocc, tight
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001287828667,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.213722428382,  # dfocc, tight
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001285760689,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.218055380962,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.040516816950,  # dfocc, tight
            "OCCD REFERENCE CORRECTION ENERGY": 0.001155093018,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.213139214735,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.005339052936,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.005215408128,  # dfocc, tight
            "SVWN TOTAL ENERGY": -55.59260769222509,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.028884098124, 0.0, 0.016574869248, -0.014442049062, 0.0, -0.016574869248, -0.014442049062]
            ).reshape((-1, 3)),
            "PBE TOTAL ENERGY": -55.83099556,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590, findif-5
                [0.0, 0.0, 0.02962368704, 0.0, 0.015649420616, -0.01481184352, 0.0, -0.015649420616, -0.01481184352]
            ).reshape((-1, 3)),
            "B3LYP TOTAL ENERGY": -55.90727490,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590, findif-5
                [0.0, 0.0, 0.020000075259, 0.0, 0.010436428357, -0.01000003763, 0.0, -0.010436428357, -0.01000003763]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_qz2p_cd_uhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_rohf,
            # "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.06046475293245379,
            "MP2 SINGLES ENERGY": -0.00069387098844,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.002005071400,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 ae cd+cd
                [0.0, 0.0, 0.01359215, 0.0, 0.0, -0.01312116, 0.0, 0.01031541, -0.0002355, 0.0, -0.01031541, -0.0002355]
            ).reshape((-1, 3)),
            "OMP2 REFERENCE CORRECTION ENERGY": -0.001128324491,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.061375267646,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.001971334332,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.012014926118,
                    0.0,
                    0.0,
                    -0.011616714077,
                    0.0,
                    0.010556988794,
                    -0.00019910602,
                    0.0,
                    -0.010556988794,
                    -0.00019910602,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.000959016912,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.069194339558,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.002198888868,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.010258655018,
                    0.0,
                    0.0,
                    -0.009975425018,
                    0.0,
                    0.011807301211,
                    -0.000141615,
                    0.0,
                    -0.011807301211,
                    -0.000141615,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": -0.000746026622,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.077048846889,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.002426255644,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.008454603088,
                    0.0,
                    0.0,
                    -0.008286658586,
                    0.0,
                    0.013068349485,
                    -0.000083972251,
                    0.0,
                    -0.013068349485,
                    -0.000083972251,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.000462639536,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.080513747976,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.00589640968,
                    0.0,
                    0.0,
                    -0.005784924229,
                    0.0,
                    0.014365074413,
                    -0.000055742725,
                    0.0,
                    -0.014365074413,
                    -0.000055742725,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": -0.000039411002,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.086260687051,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.002448904007,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight, sc
                [
                    0.0,
                    0.0,
                    0.002950898507,
                    0.0,
                    0.0,
                    -0.002932789839,
                    0.0,
                    0.015723337202,
                    -0.000009054334,
                    0.0,
                    -0.015723337202,
                    -0.000009054334,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": -0.000357639780,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.083708259924,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.000628150338,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.000605639576,  # dfocc, tight
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.15948823,
            "MP2 SINGLES ENERGY": -0.00282948,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03542128,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 ae cd+cd
                    0.0,
                    0.0,
                    0.025588961002,
                    0.0,
                    0.013550360249,
                    -0.012794480501,
                    0.0,
                    -0.013550360249,
                    -0.012794480501,
                ]
            ).reshape((-1, 3)),
            "OMP2 REFERENCE CORRECTION ENERGY": -0.003046091958,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.160598409777,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.035978726618,  # dfocc, tight
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.003239390456,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.168455185305,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.035851867343,  # dfocc, tight
            "OMP3 REFERENCE CORRECTION ENERGY": -0.003389139644,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.176349187568,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.035751323194,  # dfocc, tight
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.003301052810,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.178014257147,  # dfocc, tight
            "OLCCD REFERENCE CORRECTION ENERGY": -0.003301541633,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.182522693083,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.034472708888,  # dfocc, tight
            "OCCD REFERENCE CORRECTION ENERGY": -0.003425733446,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.178136875756,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.003989459878,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.003889358035,  # dfocc, tight
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "cd",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.20053352,
            "MP2 SINGLES ENERGY": -0.00298373,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04179032,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 ae cd+cd
                    0.0,
                    0.0,
                    0.013884053665,
                    0.0,
                    0.005385412795,
                    -0.006942026833,
                    0.0,
                    -0.005385412795,
                    -0.006942026833,
                ]
            ).reshape((-1, 3)),
            "OMP2 REFERENCE CORRECTION ENERGY": -0.003039111099,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.201675545798,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.042464346330,  # dfocc, tight
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.003258874587,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.209129119912,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.042091025938,  # dfocc, tight
            "OMP3 REFERENCE CORRECTION ENERGY": -0.003431885897,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.216624397975,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.041748214168,  # dfocc, tight
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.003321905366,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.218332161038,  # dfocc, tight
            "OLCCD REFERENCE CORRECTION ENERGY": -0.003323981369,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.222665113625,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.040516826809,  # dfocc, tight
            "OCCD REFERENCE CORRECTION ENERGY": -0.003454639639,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.217748947404,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.005339052938,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.005215408130,  # dfocc, tight
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CD-FC-CD  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_cd_rhf,
            # "HF TOTAL HESSIAN": _hess_scf_hf_dz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.201619244596,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05348825,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd
                [0.0, 0.0, 0.00316665, 0.0, 0.0, -0.00316665]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20453260,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd
                [0.0, 0.0, -0.000588974421, 0.0, 0.0, 0.000588974421]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.206423120122,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.20795503,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd
                [0.0, 0.0, 0.002525704147, 0.0, 0.0, -0.002525704147]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.2062220430,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd tight
                [0.0, 0.0, 0.001167911218, 0.0, 0.0, -0.001167911218]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.20681170792808,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04787083,
            "CCSD TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd tight
                [0.0, 0.0, 0.002325504721, 0.0, 0.0, -0.002325504721]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00192042037371,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight, findif-5
                [0.0, 0.0, 0.003419339394, 0.0, 0.0, -0.003419339394]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00194413458399,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.000702500011,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.202330761197,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [0.0, 0.0, 0.004482045565, 0.0, 0.0, -0.004482045565]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.000571428855,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.203698037577,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [0.0, 0.0, 0.002413412285, 0.0, 0.0, -0.002413412285]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000468982166,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.205112639670,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [0.0, 0.0, 0.000419528739, 0.0, 0.0, -0.000419528739]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.000554016472,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.207066289816,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [0.0, 0.0, 0.003235950588, 0.0, 0.0, -0.003235950588]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.000553491444,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.208617145287,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [0.0, 0.0, 0.003803435612, 0.0, 0.0, -0.003803435612]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.000511777936,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.206827599221,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.002043157128,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.002023030659,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_cd_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.21939907,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05583617,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd
                [
                    0.0,
                    0.0,
                    0.010264703011,
                    0.0,
                    0.00588885358,
                    -0.005132351506,
                    0.0,
                    -0.00588885358,
                    -0.005132351506,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.22415794,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "REMP2 CORRELATION ENERGY": -0.227139386881,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.22961642,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCD CORRELATION ENERGY": -0.2257842561,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.22717606848165,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04939986,
            "(T) CORRECTION ENERGY": -0.00521247873234,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00520979601465,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.002351861704,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.221658768039,  # dfocc, tight
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001538083685,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.223336430535,  # dfocc, tight
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000960343766,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.225227488953,  # dfocc, tight
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.00128588799768,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.22849477192663,  # dfocc, tight
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001194389384,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.230903883683,  # dfocc, tight
            "OCCD REFERENCE CORRECTION ENERGY": 0.001127721989,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.226990468428,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.005543695574,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.005445988059,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_adz_cd_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.24514436,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06126931,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd
                [
                    0.0,
                    0.0,
                    0.000318778691,
                    0.0,
                    -0.000569356625,
                    -0.000159389346,
                    0.0,
                    0.000569356625,
                    -0.000159389346,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.24747710,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "REMP2 CORRELATION ENERGY": -0.251016068140,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.25319315,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCD CORRELATION ENERGY": -0.2488498077,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.25032938698805,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05405638,
            "(T) CORRECTION ENERGY": -0.00709685911318,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00701825978933,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.002435969540,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.247516285552,  # dfocc, tight
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001628337567,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.247905627376,  # dfocc, tight
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001042085727,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.248525576959,  # dfocc, tight
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001387317415,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.252381884524,  # dfocc, tight
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001298302400,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.254480886451,  # dfocc, tight
            "OCCD REFERENCE CORRECTION ENERGY": 0.001226900919,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.250063908572,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.007402343012,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.007289120982,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_qz2p_cd_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_uhf,
            # "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.058410863785614,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001767642489,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd
                [
                    0.0,
                    0.0,
                    0.01230315,
                    0.0,
                    0.0,
                    -0.01184537,
                    0.0,
                    0.01033427,
                    -0.00022889,
                    0.0,
                    -0.01033427,
                    -0.00022889,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.07375232,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00220415,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.009099123102,
                    0.0,
                    0.0,
                    -0.008855857257,
                    0.0,
                    0.012692779327,
                    -0.000121632922,
                    0.0,
                    -0.012692779327,
                    -0.000121632922,
                ]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.077009033372,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.002190684009,  # dfocc, tight
            "LCCD CORRELATION ENERGY": -0.08242726,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.00225350,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.004401363786,
                    0.0,
                    0.0,
                    -0.004303629299,
                    0.0,
                    0.015236787233,
                    -0.000048867243,
                    0.0,
                    -0.015236787233,
                    -0.000048867243,
                ]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.08014246,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(
                # dfocc findif-5 tight
                [
                    0.0,
                    0.0,
                    0.006157735197,
                    0.0,
                    0.0,
                    -0.005953159624,
                    0.0,
                    0.014424722031,
                    -0.000102287787,
                    0.0,
                    -0.014424722031,
                    -0.000102287787,
                ]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.08116696345172,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD TOTAL GRADIENT": np.array(
                # dfocc findif-5 tight
                [
                    0.0,
                    0.0,
                    0.004963224373,
                    0.0,
                    0.0,
                    -0.004809612915,
                    0.0,
                    0.014854893513,
                    -0.000076805729,
                    0.0,
                    -0.014854893513,
                    -0.000076805729,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00060403476990,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(
                # dfocc findif-5 tight
                [
                    0.0,
                    0.0,
                    0.004461517471,
                    0.0,
                    0.0,
                    -0.004390206103,
                    0.0,
                    0.014934316994,
                    -0.000035655684,
                    0.0,
                    -0.014934316994,
                    -0.000035655684,
                ]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00058789584036,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.000393271089,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.058955062039,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.001822163929,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.011799994885,
                    0.0,
                    0.0,
                    -0.01138752044,
                    0.0,
                    0.010550315239,
                    -0.000206237223,
                    0.0,
                    -0.010550315239,
                    -0.000206237223,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.000564442101,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.066788204166,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.002049209044,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.010034895199,
                    0.0,
                    0.0,
                    -0.009737937774,
                    0.0,
                    0.011801086788,
                    -0.000148478713,
                    0.0,
                    -0.011801086788,
                    -0.000148478713,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000780607023,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.074661038269,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.002276029266,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.00821954564,
                    0.0,
                    0.0,
                    -0.008038362344,
                    0.0,
                    0.013062527819,
                    -0.000090591648,
                    0.0,
                    -0.013062527819,
                    -0.000090591648,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001065407270,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.078143718483,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.005658577881,
                    0.0,
                    0.0,
                    -0.005534792789,
                    0.0,
                    0.014360022699,
                    -0.000061892546,
                    0.0,
                    -0.014360022699,
                    -0.000061892546,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001492355224,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.083912298766,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.002307760137,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.002702310243,
                    0.0,
                    0.0,
                    -0.002672567295,
                    0.0,
                    0.015718809974,
                    -0.000014871474,
                    0.0,
                    -0.015718809974,
                    -0.000014871474,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.001173585912,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.081366382141,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.000608522852,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.000586642668,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_bh3p_dz_cd_uhf,
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.15241915,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03445770,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 fc cd+cd
                    0.0,
                    0.0,
                    0.026258239074,
                    0.0,
                    0.01404196652,
                    -0.013129119537,
                    0.0,
                    -0.01404196652,
                    -0.013129119537,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.16863165,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03464009,  # dfocc
            "REMP2 CORRELATION ENERGY": -0.170216188333,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.033852995535,  # dfocc, tight
            "LCCD CORRELATION ENERGY": -0.17475747,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0334416820,  # dfocc
            "CCD CORRELATION ENERGY": -0.17052456,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.17165311633501,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00381139581323,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00373214885816,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.001360881782,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.153818916860,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.035194310016,  # dfocc, tight
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001172035292,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.161781119255,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.035088275618,  # dfocc, tight
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001026576887,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.169784888805,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.035012011899,  # dfocc, tight
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.00111325834527,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.17144631168723,  # dfocc, tight
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001113861889,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.176001256953,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.033743742943,  # dfocc, tight
            "OCCD REFERENCE CORRECTION ENERGY": 0.000988643093,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.171635485628,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.003954325581,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.003854709089,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_adz_cd_uhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.17117831,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03822948,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 fc cd+cd
                    0.0,
                    0.0,
                    0.014735846129,
                    0.0,
                    0.005848618964,
                    -0.007367923065,
                    0.0,
                    -0.005848618964,
                    -0.007367923065,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.18575577,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03792717,  # dfocc
            "REMP2 CORRELATION ENERGY": -0.187472566719,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.037236809266,  # dfocc, tight
            "LCCD CORRELATION ENERGY": -0.19170174,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.03676422,  # dfocc
            "CCD CORRELATION ENERGY": -0.18708450,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.18831657592466,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00498288386522,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00488265656152,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.001480341549,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.172676727210,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.039024778995,  # dfocc, tight
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001273007276,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.179776978593,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.038629873581,  # dfocc, tight
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001112904383,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.186933561283,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.038275663827,  # dfocc, tight
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001217841569,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.188746049630,  # dfocc, tight
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001218721333,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.192985744725,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.037047752001,  # dfocc, tight
            "OCCD REFERENCE CORRECTION ENERGY": 0.001091119790,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.188236874643,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.005146215122,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.005026054902,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_qz2p_cd_uhf,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_rohf,
            # "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.05939419492939635,
            "MP2 SINGLES ENERGY": -0.0006881934,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0018520881544,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd
                [
                    0.0,
                    0.0,
                    0.01338641,
                    0.0,
                    0.0,
                    -0.01290149,
                    0.0,
                    0.01031066,
                    -0.00024246,
                    0.0,
                    -0.01031066,
                    -0.00024246,
                ]
            ).reshape((-1, 3)),
            "OMP2 REFERENCE CORRECTION ENERGY": -0.001130393431,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.060478725483,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.001822163952,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.011799994879,
                    0.0,
                    0.0,
                    -0.011387520433,
                    0.0,
                    0.01055031524,
                    -0.000206237223,
                    0.0,
                    -0.01055031524,
                    -0.000206237223,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.000959218077,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.068311867609,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.002049209005,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight, sc
                [
                    0.0,
                    0.0,
                    0.010034895198,
                    0.0,
                    0.0,
                    -0.009737937769,
                    0.0,
                    0.011801086785,
                    -0.000148478714,
                    0.0,
                    -0.011801086785,
                    -0.000148478714,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": -0.000743052999,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.076184701712,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.002276029277,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.008219545639,
                    0.0,
                    0.0,
                    -0.008038362341,
                    0.0,
                    0.013062527816,
                    -0.000090591649,
                    0.0,
                    -0.013062527816,
                    -0.000090591649,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.000458253707,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.079667381927,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.005658577879,
                    0.0,
                    0.0,
                    -0.005534792785,
                    0.0,
                    0.014360022698,
                    -0.000061892547,
                    0.0,
                    -0.014360022698,
                    -0.000061892547,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": -0.000031311396,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.085435962210,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.002307761175,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, findif-5, tight
                [
                    0.0,
                    0.0,
                    0.002702310244,
                    0.0,
                    0.0,
                    -0.002672567292,
                    0.0,
                    0.015718809973,
                    -0.000014871476,
                    0.0,
                    -0.015718809973,
                    -0.000014871476,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": -0.000350077551,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.082890045477,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.000608522844,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.000586642661,  # dfocc, tight
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.15701792,
            "MP2 SINGLES ENERGY": -0.00280584,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03466715,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 fc cd+cd
                    0.0,
                    0.0,
                    0.026376923581,
                    0.0,
                    0.014055606253,
                    -0.01318846179,
                    0.0,
                    -0.014055606253,
                    -0.01318846179,
                ]
            ).reshape((-1, 3)),
            "OMP2 REFERENCE CORRECTION ENERGY": -0.003052622179,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.158232417832,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.035194309944,  # dfocc, tight
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.003241471192,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.166194620227,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.035088275359,  # dfocc, tight
            "OMP3 REFERENCE CORRECTION ENERGY": -0.003386926686,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.174198389777,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.035012012647,  # dfocc, tight
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.00330024259293,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.17585981265919,  # dfocc, tight
            "OLCCD REFERENCE CORRECTION ENERGY": -0.003299640039,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.180414757924,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.033743734902,  # dfocc, tight
            "OCCD REFERENCE CORRECTION ENERGY": -0.003424857817,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.176048986550,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.003954325599,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.003854709106,  # dfocc, tight
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "cd",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.17611046,
            "MP2 SINGLES ENERGY": -0.00294336,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03837979,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 fc cd+cd
                    0.0,
                    0.0,
                    0.014889762324,
                    0.0,
                    0.00588299146,
                    -0.007444881162,
                    0.0,
                    -0.00588299146,
                    -0.007444881162,
                ]
            ).reshape((-1, 3)),
            "OMP2 REFERENCE CORRECTION ENERGY": -0.003129397648,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.177286459867,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.039024777173,  # dfocc, tight
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.003336731502,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.184386711249,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.038629874099,  # dfocc, tight
            "OMP3 REFERENCE CORRECTION ENERGY": -0.003496834047,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.191543293940,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.038275664440,  # dfocc, tight
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.003391896462,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.193355782287,  # dfocc, tight
            "OLCCD REFERENCE CORRECTION ENERGY": -0.003391009701,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.197595477381,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.037047749578,  # dfocc, tight
            "OCCD REFERENCE CORRECTION ENERGY": -0.003518612862,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.192846607336,  # dfocc, tight
            "O(T) CORRECTION ENERGY": -0.005146215124,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.005026054904,  # dfocc, tight
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CD-AE-DF  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_cd_rhf,
            # "HF TOTAL HESSIAN": _hess_scf_hf_dz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.2037748110768,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.054311870576,
            "MP2 TOTAL GRADIENT": np.array(
                # dfmp2 findif-5 ae cd+df
                [0.0, 0.0, 0.00279182, 0.0, 0.0, -0.00279182]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20657173,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000934631868, 0.0, 0.0, 0.000934631868]).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2100441271,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.00218256974, 0.0, 0.0, -0.00218256974]).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.20887887,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04845784,
            "(T) CORRECTION ENERGY": -0.00193844,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00196317,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_cd_rhf,
            "B2PLYP TOTAL ENERGY": -100.3646277114083887,  # psi 55,590
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.22188866,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05674801,
            "MP3 CORRELATION ENERGY": -0.22652912,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2320256729,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.22954292,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05010092,
            "(T) CORRECTION ENERGY": -0.00524386,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00524138,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_adz_cd_rhf,
            "B2PLYP TOTAL ENERGY": -76.3803817300909031,  # psi 55,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.27017947,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06530177,
            "MP3 CORRELATION ENERGY": -0.27294050,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2786865554,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.27570087,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05800702,
            "(T) CORRECTION ENERGY": -0.00726367,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00718150,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_qz2p_cd_rhf,
            "B2PLYP TOTAL ENERGY": -76.4168421953897052,  # psi 55,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_uhf,
            # "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.059456828193,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.00192025457659,
            "MP3 CORRELATION ENERGY": -0.07486528,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00235846,  # dfocc
            "LCCD CORRELATION ENERGY": -0.0835057932,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0024017496,  # dfocc
            "CCSD CORRELATION ENERGY": -0.08224146,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00062667,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00060983,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_bh3p_dz_cd_uhf,
            "B2PLYP TOTAL ENERGY": -26.0729387146540788,  # psi 55,590
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.15484678,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03523320,
            "MP3 CORRELATION ENERGY": -0.17098295,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03530448,  # dfocc
            "LCCD CORRELATION ENERGY": -0.1771099018,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0340807883,  # dfocc
            "CCSD CORRELATION ENERGY": -0.17396776,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00384848,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00376848,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_adz_cd_uhf,
            "B2PLYP TOTAL ENERGY": -55.8336002883394400,  # psi 55,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.19551841,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04161663,
            "MP3 CORRELATION ENERGY": -0.21084263,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04130347,  # dfocc
            "LCCD CORRELATION ENERGY": -0.2167832515,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0401303480,  # dfocc
            "CCSD CORRELATION ENERGY": -0.21329358,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00516657,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00506430,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_qz2p_cd_uhf,
            "B2PLYP TOTAL ENERGY": -55.8610829799526698,  # psi 55,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_rohf,
            # "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.06044431529,
            "MP2 SINGLES ENERGY": -0.00069387098844,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0020066063,
            "LCCD CORRELATION ENERGY": -0.0834800819,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.15948219,
            "MP2 SINGLES ENERGY": -0.00282948,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03544828,
            "LCCD CORRELATION ENERGY": -0.1792705171,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.20052752,
            "MP2 SINGLES ENERGY": -0.00298373,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04178566,
            "LCCD CORRELATION ENERGY": -0.2190993784,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CD-FC-DF  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_cd_rhf,
            # "HF TOTAL HESSIAN": _hess_scf_hf_dz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.2016205147678,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0535243575,
            "MP2 TOTAL GRADIENT": np.array(
                # dfmp2 findif-5 fc cd+df
                [0.0, 0.0, 0.00314686, 0.0, 0.0, -0.00314686]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20462784,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2080964757,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.20695035,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04775464,
            "(T) CORRECTION ENERGY": -0.00192252,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00194627,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_cd_rhf,
            "B2PLYP TOTAL ENERGY": -100.3640101961093478,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.21939916,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05588204,
            "MP3 CORRELATION ENERGY": -0.22425360,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2297520405,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.22730551,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04932106,
            "(T) CORRECTION ENERGY": -0.00521762,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00521484,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_adz_cd_rhf,
            "B2PLYP TOTAL ENERGY": -76.3796628907550996,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.24514320,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06126448,
            "MP3 CORRELATION ENERGY": -0.24747659,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2531926943,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.25032917,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05405189,
            "(T) CORRECTION ENERGY": -0.00709658,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00701799,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_qz2p_cd_rhf,
            "B2PLYP TOTAL ENERGY": -76.4097380606132361,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_uhf,
            # "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.05839103061,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.00176895897,
            "MP3 CORRELATION ENERGY": -0.07380770,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00220541,  # dfocc
            "LCCD CORRELATION ENERGY": -0.0825023638,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0022546311,  # dfocc
            "CCSD CORRELATION ENERGY": -0.08123956,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00060452,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00058836,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_bh3p_dz_cd_uhf,
            "B2PLYP TOTAL ENERGY": -26.0726100467676396,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.15241445,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03448512,
            "MP3 CORRELATION ENERGY": -0.16869538,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03460714,  # dfocc
            "LCCD CORRELATION ENERGY": -0.1748548876,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0333916888,  # dfocc
            "CCSD CORRELATION ENERGY": -0.17174515,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00381581,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00373640,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_adz_cd_uhf,
            "B2PLYP TOTAL ENERGY": -55.8328884911895713,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.17117540,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03822480,
            "MP3 CORRELATION ENERGY": -0.18575507,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03792228,  # dfocc
            "LCCD CORRELATION ENERGY": -0.1917007514,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0367593319,  # dfocc
            "CCSD CORRELATION ENERGY": -0.18831566,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00498264,  # dfocc, tight
            "A-(T) CORRECTION ENERGY": -0.00488241,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_qz2p_cd_uhf,
            "B2PLYP TOTAL ENERGY": -55.8541241661683614,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_rohf,
            # "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.05937342969795,
            "MP2 SINGLES ENERGY": -0.0006881934,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001853561678,
            "LCCD CORRELATION ENERGY": -0.0824761581,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.15701141,
            "MP2 SINGLES ENERGY": -0.00280584,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03469416,
            "LCCD CORRELATION ENERGY": -0.1770010376,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.17610756,
            "MP2 SINGLES ENERGY": -0.00294336,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03837509,
            "LCCD CORRELATION ENERGY": -0.1939904460,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  DF-AE-DF  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_df_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_df_rhf,
            "MP2 CORRELATION ENERGY": -0.2037649370559149,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05430875283333263,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.00000000000000,
                    0.00000000000000,
                    0.00279211492833,
                    0.00000000000000,
                    0.00000000000000,
                    -0.00279211492833,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20656153,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000933466293, 0.0, 0.0, 0.000933466293]).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.208504245426,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2100337333,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.002183232102, 0.0, 0.0, -0.002183232102]).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.2082764592,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.0, 0.000818124849, -0.0, -0.0, -0.000818124849]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.20886881949604,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04845491,
            "CCSD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.0,
                    0.001970518943,
                    0.0,
                    0.0,
                    -0.001970518943,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00193801686266,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.0, 0.003068357914, -0.0, -0.0, -0.003068357914]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00196274645521,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.000705416559,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.204449284608,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.000000000000, 0.004079891926, 0.0, -0.000000000000, -0.004079891926]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.000572277516,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.205719687105,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.000000000000, 0.001995568826, 0.0, -0.000000000000, -0.001995568826]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000467850505,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.207017324163,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.000000000000, -0.000022521780, 0.0, -0.000000000000, 0.000022521780]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.000553781427,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.209041900252,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.000000000000, 0.002809444261, 0.0, -0.000000000000, -0.002809444261]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.000552755815,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.210569944656,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.000000000000, 0.000000000000, 0.003371636637, -0.000000000000, -0.000000000000, -0.003371636637]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.00051171478318,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.20877455867384,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.0, 0.001829701353, 0.0, 0.0, -0.001829701353]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.002064008068,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.002043954736,  # dfocc, tight
            "SVWN TOTAL ENERGY": -99.97968868729104,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, -0.0, 0.019359029672, 0.0, 0.0, -0.019358980719]
            ).reshape((-1, 3)),
            "SVWN TOTAL HESSIAN": np.array(  # psi v1.8 99,590
                [
                    -0.011171616682,
                    0.0,
                    -0.0,
                    0.011171597439,
                    0.0,
                    -0.0,
                    0.0,
                    -0.011171616682,
                    -0.0,
                    -0.0,
                    0.01117159744,
                    0.0,
                    -0.0,
                    -0.0,
                    0.642262891185,
                    0.0,
                    0.0,
                    -0.642262893846,
                    0.011171597439,
                    -0.0,
                    0.0,
                    -0.011168975069,
                    0.000000000001,
                    -0.000000000001,
                    0.0,
                    0.01117159744,
                    0.0,
                    0.000000000001,
                    -0.01116897507,
                    -0.0,
                    -0.0,
                    0.0,
                    -0.642262893846,
                    -0.000000000001,
                    -0.0,
                    0.64226571697,
                ]
            ).reshape((6, 6)),
            "PBE TOTAL ENERGY": -100.33521027,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.020130499789, 0.0, -0.0, -0.020130524567]
            ).reshape((-1, 3)),
            "B3LYP TOTAL ENERGY": -100.43547355,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, -0.0, 0.011392119703, 0.0, -0.0, -0.011392010986]
            ).reshape((-1, 3)),
            "WB97X TOTAL ENERGY": -100.41421085713486,  # psi 500,1202
            "WB97X TOTAL GRADIENT": np.array(  # psi 500,1202, findif-5
                [0.0, -0.0, 0.006648755885, 0.0, -0.0, -0.006648573196]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_df_rhf,
            "B2PLYP TOTAL ENERGY": -100.3646058112210824,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_df_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_df_rhf,
            "MP2 CORRELATION ENERGY": -0.22187976,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05674571,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.0,
                    0.0,
                    0.00962182765,
                    0.0,
                    0.005498317937,
                    -0.004810913825,
                    0.0,
                    -0.005498317937,
                    -0.004810913825,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.22651881,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.004009128141, 0.0, 0.002552803039, -0.00200456407, 0.0, -0.002552803039, -0.00200456407]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.22954466014956,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2320149229,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.008561006838, 0.0, 0.005236802973, -0.004280503419, 0.0, -0.005236802973, -0.004280503419]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.2281433856,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, -0.0, 0.006179417919, -0.0, 0.00392052373, -0.00308970896, 0.0, -0.003920523729, -0.003089708959]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.22953288030956,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05009877,
            "CCSD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.0,
                    0.007518713989,
                    0.0,
                    0.004612996204,
                    -0.003759356993,
                    0.0,
                    -0.004612996204,
                    -0.003759356995,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00524345048605,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.0, 0.010200006450, -0.0, 0.00604923838, -0.005100003223, -0.0, -0.00604923838, -0.005100003226]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00524100328791,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.002353213091,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.224108722336,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.012090520150,
                    0.0,
                    0.006792038487,
                    -0.006045260075,
                    0.0,
                    -0.006792038487,
                    -0.006045260075,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001535554696,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.225687426127,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.008284981234,
                    0.0,
                    0.004775627693,
                    -0.004142490617,
                    0.0,
                    -0.004775627693,
                    -0.004142490617,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000954089054,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.227460339651,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.004815233651,
                    0.0,
                    0.002952173310,
                    -0.002407616826,
                    0.0,
                    -0.002952173310,
                    -0.002407616826,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001281722402,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.23079090375495,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.009108870642,
                    0.0,
                    0.005403366342,
                    -0.004554435321,
                    0.0,
                    -0.005403366342,
                    -0.004554435321,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001189314489,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.233173062606,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.000000000000,
                    -0.000000000000,
                    0.009761700492,
                    -0.000000000000,
                    0.005853403252,
                    -0.004880850246,
                    -0.000000000000,
                    -0.005853403251,
                    -0.004880850246,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.0011241002379,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.22924045334096,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.007259063497,
                    -0.0,
                    0.004473960460,
                    -0.003629531749,
                    -0.0,
                    -0.004473960460,
                    -0.003629531749,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.005578769043,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.005480838164,  # dfocc, tight
            "SVWN TOTAL ENERGY": -76.07574048609808,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.017021972532, -0.0, 0.012817893242, -0.008511166112, 0.0, -0.012817893242, -0.008511166112]
            ).reshape((-1, 3)),
            "SVWN TOTAL HESSIAN": np.array(  # psi v1.8 99,590
                [
                    -0.015306108227,
                    -0.0,
                    -0.0,
                    0.007679241821,
                    -0.0,
                    0.0,
                    0.007679241834,
                    -0.0,
                    -0.0,
                    -0.0,
                    0.714095826016,
                    0.000000000011,
                    -0.0,
                    -0.357059323956,
                    0.282410783964,
                    0.0,
                    -0.357059323952,
                    -0.282410784003,
                    -0.0,
                    0.000000000011,
                    0.456485072062,
                    0.0,
                    0.22318907951,
                    -0.228247665448,
                    0.0,
                    -0.22318907952,
                    -0.228247665506,
                    0.007679241821,
                    -0.0,
                    0.0,
                    -0.008316923332,
                    -0.0,
                    -0.0,
                    0.000637673685,
                    0.0,
                    -0.0,
                    -0.0,
                    -0.357059323956,
                    0.22318907951,
                    -0.0,
                    0.3859291338,
                    -0.252799920245,
                    0.0,
                    -0.028869886253,
                    0.029610851501,
                    0.0,
                    0.282410783964,
                    -0.228247665448,
                    -0.0,
                    -0.252799920245,
                    0.218979358918,
                    -0.0,
                    -0.029610851534,
                    0.009268235563,
                    0.007679241834,
                    0.0,
                    0.0,
                    0.000637673685,
                    0.0,
                    -0.0,
                    -0.008316923345,
                    -0.0,
                    -0.0,
                    -0.0,
                    -0.357059323952,
                    -0.22318907952,
                    0.0,
                    -0.028869886253,
                    -0.029610851534,
                    -0.0,
                    0.385929133855,
                    0.252799920322,
                    -0.0,
                    -0.282410784003,
                    -0.228247665506,
                    -0.0,
                    0.029610851501,
                    0.009268235563,
                    -0.0,
                    0.252799920322,
                    0.218979358932,
                ]
            ).reshape((9, 9)),
            "PBE TOTAL ENERGY": -76.35898912990783,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.018097092596, -0.0, 0.011586123511, -0.009048894034, 0.0, -0.011586123511, -0.009048894034]
            ).reshape((-1, 3)),
            "B3LYP TOTAL ENERGY": -76.44456455,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.007475949858, -0.0, 0.00613847913, -0.003738203973, 0.0, -0.00613847913, -0.003738203973]
            ).reshape((-1, 3)),
            "WB97X TOTAL ENERGY": -76.42164084415194,  # psi 500,1202
            "WB97X TOTAL GRADIENT": np.array(  # psi 500,1202, findif-5
                [0.0, 0.0, 0.00284541, -0.0, 0.00296728, -0.00142221, 0.0, -0.00296728, -0.00142221]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_adz_df_rhf,
            "B2PLYP TOTAL ENERGY": -76.3804028015970857,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_df_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_df_rhf,
            "MP2 CORRELATION ENERGY": -0.27016105,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06529808,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.0,
                    0.0,
                    -0.000566657943,
                    0.0,
                    -0.000968877215,
                    0.000283328971,
                    0.0,
                    0.000968877215,
                    0.000283328971,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.27292175,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.007037747646, 0.0, -0.004494031919, 0.003518873823, 0.0, 0.004494031919, 0.003518873823]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.276419945808,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2786671617,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.002344345036, 0.0, -0.001783728285, 0.001172172518, 0.0, 0.001783728285, 0.001172172518]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.2741489819,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.0,
                    -0.004790202029,
                    -0.0,
                    -0.003048964224,
                    0.002395101014,
                    -0.0,
                    0.003048964224,
                    0.002395101014,
                ]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.27568224252782,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05800380,
            "CCSD TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.0, -0.003408923965, 0.0, -0.002343243985, 0.001704461980, 0.0, 0.002343243985, 0.001704461985]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00726212675888,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.0, -0.000217412659, -0.0, -0.000742939442, 0.000108706327, 0.0, 0.000742939441, 0.000108706333]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00718000875661,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.002514012770,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.272569214076,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.001818176662,
                    0.0,
                    0.000241592175,
                    -0.000909088331,
                    0.0,
                    -0.000241592175,
                    -0.000909088331,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001691695489,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.273184743555,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    -0.002325903973,
                    0.0,
                    -0.001998665869,
                    0.001162951987,
                    0.0,
                    0.001998665869,
                    0.001162951987,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001090223846,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.273992490969,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    -0.006182600264,
                    0.0,
                    -0.004075383127,
                    0.003091300132,
                    0.0,
                    0.004075383127,
                    0.003091300132,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001444459285,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.277823416296,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    -0.001651592566,
                    0.0,
                    -0.001517469318,
                    0.000825796283,
                    0.0,
                    0.001517469318,
                    0.000825796283,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001352291360,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.279981325469,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.000000000000,
                    -0.000000000000,
                    -0.001075340549,
                    -0.000000000000,
                    -0.001136934323,
                    0.000537670274,
                    0.000000000000,
                    0.001136934323,
                    0.000537670274,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.00127938819983,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.27539517277894,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    -0.003670656144,
                    -0.0,
                    -0.002477953103,
                    0.001835328072,
                    0.0,
                    0.002477953103,
                    0.001835328072,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.007571167323,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.007455996809,  # dfocc, tight
            "SVWN TOTAL ENERGY": -76.10232964924052,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.011132147324, -0.0, 0.009489578319, -0.005566364846, 0.0, -0.009489578319, -0.005566364846]
            ).reshape((-1, 3)),
            "SVWN TOTAL HESSIAN": np.array(  # psi v1.8 99,590
                [
                    -0.009990362405,
                    -0.0,
                    0.000000000001,
                    0.00502229918,
                    -0.0,
                    0.0,
                    0.005022299179,
                    0.0,
                    0.0,
                    -0.0,
                    0.699077256021,
                    0.0,
                    -0.0,
                    -0.349550325666,
                    0.274539445833,
                    0.0,
                    -0.349550325662,
                    -0.274539445833,
                    0.000000000001,
                    0.0,
                    0.449564210084,
                    0.0,
                    0.216190331109,
                    -0.22478695452,
                    0.0,
                    -0.216190331109,
                    -0.224786954525,
                    0.00502229918,
                    -0.0,
                    0.0,
                    -0.005826835703,
                    0.0,
                    -0.0,
                    0.000803572097,
                    0.0,
                    0.0,
                    -0.0,
                    -0.349550325666,
                    0.216190331109,
                    0.0,
                    0.379622159841,
                    -0.245365200208,
                    0.0,
                    -0.030071862218,
                    0.029174558331,
                    0.0,
                    0.274539445833,
                    -0.22478695452,
                    -0.0,
                    -0.245365200208,
                    0.215364149782,
                    0.0,
                    -0.029174558332,
                    0.009422617805,
                    0.005022299179,
                    0.0,
                    0.0,
                    0.000803572097,
                    0.0,
                    0.0,
                    -0.005826835703,
                    0.0,
                    -0.0,
                    0.0,
                    -0.349550325662,
                    -0.216190331109,
                    0.0,
                    -0.030071862218,
                    -0.029174558332,
                    0.0,
                    0.379622159838,
                    0.245365200207,
                    0.0,
                    -0.274539445833,
                    -0.224786954525,
                    0.0,
                    0.029174558331,
                    0.009422617805,
                    -0.0,
                    0.245365200207,
                    0.215364149785,
                ]
            ).reshape((9, 9)),
            "PBE TOTAL ENERGY": -76.38213149,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.012739554295, -0.0, 0.008585626913, -0.006370169853, 0.0, -0.008585626913, -0.006370169853]
            ).reshape((-1, 3)),
            "B3LYP TOTAL ENERGY": -76.46844827,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.001992158396, -0.0, 0.003027194556, -0.00099636428, 0.0, -0.003027194556, -0.00099636428]
            ).reshape((-1, 3)),
            "WB97X TOTAL ENERGY": -76.44427590906572,  # psi 500,1202
            "WB97X TOTAL GRADIENT": np.array(  # psi 500,1202, findif-5
                [0.0, 0.0, -0.002582039801, 0.0, -0.000046718668, 0.001291589931, -0.0, 0.000046718668, 0.001291589931]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_qz2p_df_rhf,
            "B2PLYP TOTAL ENERGY": -76.4168144583403830,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_df_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_df_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_df_uhf,
            "MP2 CORRELATION ENERGY": -0.0594557966607590,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001920220330437888,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.00000000000000,
                    0.00000000000000,
                    0.01252024755551,
                    0.00000000000000,
                    0.00000000000000,
                    -0.01207773525598,
                    0.00000000000000,
                    0.01032204616770,
                    -0.00022125614977,
                    0.00000000000000,
                    -0.01032204616770,
                    -0.00022125614977,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.07486350,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00235837,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.009311184271,
                    0.0,
                    0.0,
                    -0.009082434632,
                    0.0,
                    0.012724190366,
                    -0.00011437482,
                    0.0,
                    -0.012724190366,
                    -0.00011437482,
                ]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.078090599375,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.002340709678,  # dfocc, tight
            "LCCD CORRELATION ENERGY": -0.0835030877,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0024016379,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.004636606273,
                    0.0,
                    0.0,
                    -0.004551154292,
                    0.0,
                    0.015266304532,
                    -0.00004272599,
                    0.0,
                    -0.015266304532,
                    -0.00004272599,
                ]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.08120235,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.0,
                    0.006399006162,
                    -0.0,
                    -0.0,
                    -0.006206323117,
                    -0.0,
                    0.014450800585,
                    -0.000096341522,
                    0.0,
                    -0.014450800585,
                    -0.000096341522,
                ]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.08223900985527,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    0.0,
                    0.005201169988,
                    0.0,
                    -0.0,
                    -0.005060084065,
                    -0.0,
                    0.014882201275,
                    -0.000070542962,
                    0.0,
                    -0.014882201275,
                    -0.000070542961,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00062655599464,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    0.0,
                    0.004689654727,
                    0.0,
                    -0.0,
                    -0.004631718438,
                    -0.0,
                    0.014962374742,
                    -0.000028968145,
                    0.0,
                    -0.014962374742,
                    -0.000028968144,
                ]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00060971952519,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.000396459592,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.059832149673,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.001972801620,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.012031476427,
                    0.0,
                    -0.000000000000,
                    -0.011634888026,
                    0.0,
                    0.010540730273,
                    -0.000198294201,
                    0.0,
                    -0.010540730274,
                    -0.000198294201,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.000566397552,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.067689046439,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.002200309774,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.010263706524,
                    0.0,
                    -0.000000000000,
                    -0.009982176653,
                    0.0,
                    0.011813660618,
                    -0.000140764936,
                    0.0,
                    -0.011813660618,
                    -0.000140764936,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000780187024,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.075581499723,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.002427625806,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.008448066470,
                    0.0,
                    -0.000000000000,
                    -0.008281866938,
                    0.0,
                    0.013097326629,
                    -0.000083099766,
                    0.0,
                    -0.013097326629,
                    -0.000083099766,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001063189693,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.079048591655,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.005890513253,
                    0.0,
                    -0.000000000000,
                    -0.005780178036,
                    0.0,
                    0.014386023476,
                    -0.000055167609,
                    0.0,
                    -0.014386023476,
                    -0.000055167609,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001486131044,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.084811453110,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.002450103204,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.000000000000,
                    0.000000000000,
                    0.002942322545,
                    0.000000000000,
                    -0.000000000000,
                    -0.002924986204,
                    -0.000000000000,
                    0.015750409537,
                    -0.000008668171,
                    0.000000000000,
                    -0.015750409537,
                    -0.000008668171,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.00116757931849,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.08225497590846,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    0.0,
                    0.005192742269,
                    0.0,
                    -0.0,
                    -0.005046526613,
                    -0.0,
                    0.014877327584,
                    -0.000073107828,
                    0.0,
                    -0.014877327584,
                    -0.000073107828,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.000628655467,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.000606118093,  # dfocc, tight
            "SVWN TOTAL ENERGY": -25.88283110277485,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    -0.0,
                    0.0,
                    -0.007528180742,
                    0.0,
                    0.0,
                    0.002278298181,
                    -0.0,
                    0.036125770495,
                    0.002624938728,
                    0.0,
                    -0.036125770495,
                    0.002624938728,
                ]
            ).reshape((-1, 3)),
            "SVWN TOTAL HESSIAN": np.array(  # psi 99,590, findif-5
                [
                    -0.001868632523,
                    0.0,
                    0.0,
                    0.002273478054,
                    0.0,
                    0.0,
                    -0.000202422765,
                    0.0,
                    0.0,
                    -0.000202422765,
                    0.0,
                    0.0,
                    0.0,
                    -0.002099514447,
                    0.0,
                    0.0,
                    0.002410237082,
                    0.0,
                    0.0,
                    -0.000155361318,
                    -0.000139354921,
                    0.0,
                    -0.000155361318,
                    0.000139354921,
                    0.0,
                    0.0,
                    0.209841529106,
                    0.0,
                    0.0,
                    -0.211634651715,
                    0.0,
                    0.003613009863,
                    0.000896561305,
                    0.0,
                    -0.003613009863,
                    0.000896561305,
                    0.002273478054,
                    0.0,
                    0.0,
                    -0.001884996935,
                    0.0,
                    0.0,
                    -0.000194240559,
                    0.0,
                    0.0,
                    -0.000194240559,
                    0.0,
                    0.0,
                    0.0,
                    0.002410237082,
                    0.0,
                    0.0,
                    -0.001966220902,
                    0.0,
                    0.0,
                    -0.00022200809,
                    0.000081337758,
                    0.0,
                    -0.00022200809,
                    -0.000081337758,
                    0.0,
                    0.0,
                    -0.211634651715,
                    0.0,
                    0.0,
                    0.211515553657,
                    0.0,
                    0.000035537687,
                    0.000059549029,
                    0.0,
                    -0.000035537687,
                    0.000059549029,
                    -0.000202422765,
                    0.0,
                    0.0,
                    -0.000194240559,
                    0.0,
                    0.0,
                    -0.025531746834,
                    0.0,
                    0.0,
                    0.025928410159,
                    0.0,
                    0.0,
                    0.0,
                    -0.000155361318,
                    0.003613009863,
                    0.0,
                    -0.00022200809,
                    0.000035537687,
                    0.0,
                    0.40846028102,
                    -0.001795265194,
                    0.0,
                    -0.408082911612,
                    -0.001853282357,
                    0.0,
                    -0.000139354921,
                    0.000896561305,
                    0.0,
                    0.000081337758,
                    0.000059549029,
                    0.0,
                    -0.001795265194,
                    -0.026298945203,
                    0.0,
                    0.001853282357,
                    0.025342834869,
                    -0.000202422765,
                    0.0,
                    0.0,
                    -0.000194240559,
                    0.0,
                    0.0,
                    0.025928410159,
                    0.0,
                    0.0,
                    -0.025531746834,
                    0.0,
                    0.0,
                    0.0,
                    -0.000155361318,
                    -0.003613009863,
                    0.0,
                    -0.00022200809,
                    -0.000035537687,
                    0.0,
                    -0.408082911612,
                    0.001853282357,
                    0.0,
                    0.40846028102,
                    0.001795265194,
                    0.0,
                    0.000139354921,
                    0.000896561305,
                    0.0,
                    -0.000081337758,
                    0.000059549029,
                    0.0,
                    -0.001853282357,
                    0.025342834869,
                    0.0,
                    0.001795265194,
                    -0.026298945203,
                ]
            ).reshape((12, 12)),
            "PBE TOTAL ENERGY": -26.04682962,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    -0.0,
                    -0.0,
                    -0.00292661285,
                    0.0,
                    0.0,
                    -0.000169218875,
                    -0.0,
                    0.026134688736,
                    0.001547921503,
                    0.0,
                    -0.026134688736,
                    0.001547921503,
                ]
            ).reshape((-1, 3)),
            "B3LYP TOTAL ENERGY": -26.11401525,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    0.0,
                    0.0,
                    0.003314065304,
                    0.0,
                    0.0,
                    -0.005823149531,
                    -0.0,
                    0.019174183281,
                    0.001254732996,
                    0.0,
                    -0.019174183281,
                    0.001254732996,
                ]
            ).reshape((-1, 3)),
            "WB97X TOTAL ENERGY": -26.09949176460252,  # psi 500,1202
            "WB97X TOTAL GRADIENT": np.array(  # psi 500,1202 findif-5
                [
                    -0.0,
                    0.0,
                    0.003379717239,
                    -0.0,
                    0.0,
                    -0.006574418407,
                    -0.0,
                    0.018193361281,
                    0.001597213407,
                    0.0,
                    -0.018193361281,
                    0.001597213407,
                ]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_bh3p_dz_df_uhf,
            "B2PLYP TOTAL ENERGY": -26.0729396999567840,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "df",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_df_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_df_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_df_uhf,
            "MP2 CORRELATION ENERGY": -0.15483909,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03523134,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.0,
                    0.0,
                    0.025476049585,
                    0.0,
                    0.013480567736,
                    -0.012738024793,
                    0.0,
                    -0.013480567736,
                    -0.012738024793,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.17097347,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03530244,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.027082665714, 0.0, 0.014406579724, -0.013541332857, 0.0, -0.014406579724, -0.013541332857]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.17256998862940,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.03451443433327,  # dfocc, tight
            "LCCD CORRELATION ENERGY": -0.1770997033,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0340788149,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.030686886723, 0.0, 0.016619966665, -0.015343443362, 0.0, -0.016619966665, -0.015343443362]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.17282784,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [-0.0, -0.0, 0.028141518582, 0.0, 0.01520232422, -0.014070759291, 0.0, -0.01520232422, -0.014070759291]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.17395818499234,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    0.0,
                    0.029266386707,
                    0.0,
                    0.015807793582,
                    -0.014633193357,
                    -0.0,
                    -0.015807793582,
                    -0.014633193351,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00384810072338,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    0.0,
                    0.031148775451,
                    0.0,
                    0.016805034066,
                    -0.015574387726,
                    -0.0,
                    -0.016805034066,
                    -0.015574387725,
                ]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00376812221828,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.001367832301,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.156173452626,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.036004901480,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.026914388853,
                    0.0,
                    0.014240311996,
                    -0.013457194426,
                    0.0,
                    -0.014240311996,
                    -0.013457194426,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001174470264,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.164063461670,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.035847923992,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.027483211370,
                    0.0,
                    0.014576999156,
                    -0.013741605685,
                    0.0,
                    -0.014576999156,
                    -0.013741605685,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001024687110,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.171990731878,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.035717311998,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.028112148199,
                    0.0,
                    0.014949633980,
                    -0.014056074099,
                    0.0,
                    -0.014949633980,
                    -0.014056074099,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.0011125246764,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.17366812020990,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight)
                [
                    0.0,
                    -0.000000000000,
                    0.030213597936,
                    0.0,
                    0.016231443797,
                    -0.015106798968,
                    0.0,
                    -0.016231443797,
                    -0.015106798968,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001111851483,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.178196933112,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.034422402711,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.000000000000,
                    -0.000000000000,
                    0.031864432517,
                    0.000000000000,
                    0.017258987132,
                    -0.015932216258,
                    -0.000000000000,
                    -0.017258987132,
                    -0.015932216258,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.00098766230788,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.17380639841079,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [-0.0, -0.0, 0.029147626743, 0.0, 0.01574493194, -0.014573813372, -0.0, -0.01574493194, -0.014573813372]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.003993549226,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.003893263740,  # dfocc, tight
            "SVWN TOTAL ENERGY": -55.57344983107319,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [-0.0, 0.0, 0.035469475022, -0.0, 0.020928806811, -0.01773474765, 0.0, -0.020928806811, -0.01773474765]
            ).reshape((-1, 3)),
            "SVWN TOTAL HESSIAN": np.array(  # psi v1.9 99,590
                [
                    -0.03057202,
                    0.0,
                    -0.0,
                    0.01529392,
                    0.0,
                    -0.0,
                    0.01529392,
                    -0.0,
                    -0.0,
                    0.0,
                    0.61360803,
                    -0.0,
                    -0.0,
                    -0.30681053,
                    0.24715943,
                    0.0,
                    -0.30681053,
                    -0.24715943,
                    -0.0,
                    -0.0,
                    0.37847696,
                    0.0,
                    0.19396643,
                    -0.18924016,
                    0.0,
                    -0.19396643,
                    -0.18924016,
                    0.01529392,
                    -0.0,
                    0.0,
                    -0.01457079,
                    -0.0,
                    0.0,
                    -0.00072245,
                    0.0,
                    0.0,
                    0.0,
                    -0.30681053,
                    0.19396643,
                    -0.0,
                    0.33243283,
                    -0.22056279,
                    -0.0,
                    -0.02562228,
                    0.0265965,
                    -0.0,
                    0.24715943,
                    -0.18924016,
                    0.0,
                    -0.22056279,
                    0.18252176,
                    0.0,
                    -0.0265965,
                    0.00671853,
                    0.01529392,
                    0.0,
                    0.0,
                    -0.00072245,
                    -0.0,
                    0.0,
                    -0.01457079,
                    0.0,
                    0.0,
                    -0.0,
                    -0.30681053,
                    -0.19396643,
                    0.0,
                    -0.02562228,
                    -0.0265965,
                    0.0,
                    0.33243283,
                    0.22056279,
                    -0.0,
                    -0.24715943,
                    -0.18924016,
                    0.0,
                    0.0265965,
                    0.00671853,
                    0.0,
                    0.22056279,
                    0.18252176,
                ]
            ).reshape((9, 9)),
            "PBE TOTAL ENERGY": -55.81475290212862,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [-0.0, 0.0, 0.035840402792, -0.0, 0.01975585323, -0.017920287524, 0.0, -0.01975585323, -0.017920287524]
            ).reshape((-1, 3)),
            "B3LYP TOTAL ENERGY": -55.89010330,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [0.0, 0.0, 0.026815461209, -0.0, 0.014944070844, -0.013407746678, 0.0, -0.014944070844, -0.013407746678]
            ).reshape((-1, 3)),
            "WB97X TOTAL ENERGY": -55.86922972100922,  # psi 500,1202
            "WB97X TOTAL GRADIENT": np.array(  # psi 500,1202 findif-5
                [0.0, 0.0, 0.02448296743, -0.0, 0.013468925278, -0.012241027965, 0.0, -0.013468925278, -0.012241027965]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_adz_df_uhf,
            "B2PLYP TOTAL ENERGY": -55.8336273896870736,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "df",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_df_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_df_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_df_uhf,
            "MP2 CORRELATION ENERGY": -0.19550726,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04161470,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.0,
                    0.0,
                    0.013708831104,
                    0.0,
                    0.005340400162,
                    -0.006854415552,
                    0.0,
                    -0.005340400162,
                    -0.006854415552,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.21083057,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04130147,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.014412459072, 0.0, 0.005610368822, -0.007206229536, 0.0, -0.005610368822, -0.007206229536]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.212434650458,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.040606132872,  # dfocc, tight
            "LCCD CORRELATION ENERGY": -0.2167706529,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0401283617,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.018122428755, 0.0, 0.007810952273, -0.009061214377, 0.0, -0.007810952273, -0.009061214377]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.21198023,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [-0.0, -0.0, 0.015551859519, -0.0, 0.006470515012, -0.00777592976, 0.0, -0.006470515012, -0.00777592976]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.21328159274710,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, -0.0, 0.016818391166, -0.0, 0.0071380193, -0.008409195584, 0.0, -0.0071380193, -0.008409195582]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00516563729772,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.0,
                    0.019018581448,
                    -0.0,
                    0.008236968052,
                    -0.009509290726,
                    0.0,
                    -0.008236968052,
                    -0.009509290722,
                ]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00506341197521,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.001570633341,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.197048695120,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.042457672915,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.015291382216,
                    0.0,
                    0.006145115573,
                    -0.007645691108,
                    0.0,
                    -0.006145115573,
                    -0.007645691108,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001350847967,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.204503012944,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.042084276114,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.015388626332,
                    0.0,
                    0.006150433332,
                    -0.007694313166,
                    0.0,
                    -0.006150433332,
                    -0.007694313166,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001177820960,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.211999043568,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.041741379825,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.015546142750,
                    0.0,
                    0.006191167357,
                    -0.007773071375,
                    0.0,
                    -0.006191167357,
                    -0.007773071375,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001287786460,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.213706080780,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.017877890596,
                    0.0,
                    0.007580722221,
                    -0.008938945298,
                    0.0,
                    -0.007580722221,
                    -0.008938945298,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001285702841,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.218038927594,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.040510034508,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.000000000000,
                    -0.000000000000,
                    0.019448090911,
                    -0.000000000000,
                    0.008509858260,
                    -0.009724045456,
                    0.000000000000,
                    -0.008509858260,
                    -0.009724045456,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.00115506325268,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.21312358096054,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.0, 0.016675154934, -0.0, 0.007064234315, -0.008337577467, 0.0, -0.007064234315, -0.008337577467]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.005337854371,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.005214252989,  # dfocc, tight
            "SVWN TOTAL ENERGY": -55.59261213872637,  # psi 99,590
            "SVWN TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    -0.0,
                    0.0,
                    0.028854540696,
                    -0.0,
                    0.016550659197,
                    -0.014427331687,
                    0.0,
                    -0.016550659197,
                    -0.014427331687,
                ]
            ).reshape((-1, 3)),
            "SVWN TOTAL HESSIAN": np.array(  # psi v1.9 99,590
                [
                    -0.02486474,
                    0.0,
                    0.0,
                    0.01244166,
                    0.0,
                    -0.0,
                    0.01244166,
                    -0.0,
                    -0.0,
                    0.0,
                    0.59337807,
                    -0.0,
                    0.0,
                    -0.29669581,
                    0.23720953,
                    0.0,
                    -0.29669581,
                    -0.23720953,
                    0.0,
                    -0.0,
                    0.37090816,
                    -0.0,
                    0.18498275,
                    -0.18545575,
                    0.0,
                    -0.18498275,
                    -0.18545575,
                    0.01244166,
                    0.0,
                    -0.0,
                    -0.01169818,
                    -0.0,
                    -0.0,
                    -0.0007449,
                    0.0,
                    0.0,
                    0.0,
                    -0.29669581,
                    0.18498275,
                    -0.0,
                    0.3227077,
                    -0.21109672,
                    -0.0,
                    -0.02601184,
                    0.02611339,
                    -0.0,
                    0.23720953,
                    -0.18545575,
                    -0.0,
                    -0.21109672,
                    0.17826019,
                    -0.0,
                    -0.02611339,
                    0.00719517,
                    0.01244166,
                    0.0,
                    0.0,
                    -0.0007449,
                    -0.0,
                    -0.0,
                    -0.01169818,
                    -0.0,
                    0.0,
                    -0.0,
                    -0.29669581,
                    -0.18498275,
                    0.0,
                    -0.02601184,
                    -0.02611339,
                    -0.0,
                    0.3227077,
                    0.21109672,
                    -0.0,
                    -0.23720953,
                    -0.18545575,
                    0.0,
                    0.02611339,
                    0.00719517,
                    0.0,
                    0.21109672,
                    0.17826019,
                ]
            ).reshape((9, 9)),
            "PBE TOTAL ENERGY": -55.83100498,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [-0.0, 0.0, 0.029591151947, -0.0, 0.015623483695, -0.01479570471, 0.0, -0.015623483695, -0.01479570471]
            ).reshape((-1, 3)),
            "B3LYP TOTAL ENERGY": -55.90728149,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [-0.0, 0.0, 0.01997080535, -0.0, 0.010412948823, -0.009985460257, 0.0, -0.010412948823, -0.009985460257]
            ).reshape((-1, 3)),
            "WB97X TOTAL ENERGY": -55.88530822384336,  # psi 500,1202
            "WB97X TOTAL GRADIENT": np.array(  # psi 500,1202, findif-5
                [-0.0, 0.0, 0.01726863113, -0.0, 0.008837123397, -0.008633551557, 0.0, -0.008837123397, -0.008633551557]
            ).reshape((-1, 3)),
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_qz2p_df_uhf,
            "B2PLYP TOTAL ENERGY": -55.8610813741572159,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_df_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_df_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_df_rohf,
            "MP2 CORRELATION ENERGY": -0.0604436327328384,
            "MP2 SINGLES ENERGY": -0.0006940750313001934,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0020065676314032863,
            "MP2 TOTAL GRADIENT": np.array(
                [  # occ findif-5 ae df+df
                    0.00000000000000,
                    0.00000000000000,
                    0.01361287313486,
                    0.00000000000000,
                    0.00000000000000,
                    -0.01314329502424,
                    0.00000000000000,
                    0.01029838165151,
                    -0.00023478905531,
                    0.00000000000000,
                    -0.01029838165151,
                    -0.00023478905531,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.0834776542,  # p4n
            "LCCD TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            "OMP2 REFERENCE CORRECTION ENERGY": -0.001127578413,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.061356186762,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.001972801188,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.012031487574,
                    0.0,
                    -0.000000000000,
                    -0.011634881724,
                    0.0,
                    0.010540719393,
                    -0.000198302925,
                    0.0,
                    -0.010540719393,
                    -0.000198302925,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.000957639989,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.069213083529,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.002200309067,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.010263733632,
                    0.0,
                    -0.000000000000,
                    -0.009982179290,
                    0.0,
                    0.011813639779,
                    -0.000140777171,
                    0.0,
                    -0.011813639779,
                    -0.000140777171,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": -0.000743849641,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.077105536812,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.002427625679,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.008448077204,
                    0.0,
                    -0.000000000000,
                    -0.008281875978,
                    0.0,
                    0.013097326563,
                    -0.000083100613,
                    0.0,
                    -0.013097326563,
                    -0.000083100613,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.000460845361,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.080572628744,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.005890487280,
                    0.0,
                    -0.000000000000,
                    -0.005780144920,
                    0.0,
                    0.014386015714,
                    -0.000055171180,
                    0.0,
                    -0.014386015714,
                    -0.000055171180,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": -0.000037911160,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.086335490199,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.002450103939,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.000000000000,
                    0.000000000000,
                    0.002942353924,
                    0.000000000000,
                    -0.000000000000,
                    -0.002925017923,
                    -0.000000000000,
                    0.015750403382,
                    -0.000008668000,
                    0.000000000000,
                    -0.015750403382,
                    -0.000008668000,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": -0.00035645775738,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.08377901300913,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    0.000000000000,
                    0.005192742183,
                    0.0,
                    -0.000000000000,
                    -0.005046526527,
                    -0.0,
                    0.014877327583,
                    -0.000073107828,
                    0.0,
                    -0.014877327583,
                    -0.000073107828,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.000628655467,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.000606118094,  # dfocc, tight
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "df",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_df_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_df_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_df_rohf,
            "MP2 CORRELATION ENERGY": -0.15947485,
            "MP2 SINGLES ENERGY": -0.00282982,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03544639,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.0,
                    0.0,
                    0.025593521597,
                    0.0,
                    0.013495283342,
                    -0.012796760798,
                    0.0,
                    -0.013495283342,
                    -0.012796760798,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.1792603912,  # p4n
            "OMP2 REFERENCE CORRECTION ENERGY": -0.003046129441,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.160587412835,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.036004901152,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.026914381459,
                    0.0,
                    0.014240313354,
                    -0.013457190729,
                    0.0,
                    -0.014240313354,
                    -0.013457190729,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.003239492526,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.168477421879,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.035847923306,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.027483206969,
                    0.0,
                    0.014577000408,
                    -0.013741603485,
                    0.0,
                    -0.014577000408,
                    -0.013741603485,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": -0.003389275946,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.176404692086,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.035717311411,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.028112143061,
                    0.0,
                    0.014949636932,
                    -0.014056071530,
                    0.0,
                    -0.014949636932,
                    -0.014056071530,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.003301435451,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.178082080418,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.030213608364,
                    0.0,
                    0.016231443474,
                    -0.015106804182,
                    0.0,
                    -0.016231443474,
                    -0.015106804182,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": -0.003302111181,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.182610893321,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.034422399491,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.000000000000,
                    -0.000000000000,
                    0.031864462561,
                    0.000000000000,
                    0.017259000991,
                    -0.015932231281,
                    -0.000000000000,
                    -0.017259000990,
                    -0.015932231281,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": -0.00342629789193,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.17822035862112,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    -0.000000000000,
                    0.029147627070,
                    0.0,
                    0.015744932006,
                    -0.014573813535,
                    -0.0,
                    -0.015744932006,
                    -0.014573813535,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.003993549230,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.003893263743,  # dfocc, tight
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "df",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_df_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_df_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_df_rohf,
            "MP2 CORRELATION ENERGY": -0.20051655,
            "MP2 SINGLES ENERGY": -0.00298400,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04178365,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.0,
                    0.0,
                    0.013865245912,
                    0.0,
                    0.005377216253,
                    -0.006932622956,
                    0.0,
                    -0.005377216253,
                    -0.006932622956,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2190866990,  # p4n
            "OMP2 REFERENCE CORRECTION ENERGY": -0.003039409981,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.201658737065,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.042457672983,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.015291380391,
                    0.0,
                    0.006145126895,
                    -0.007645690196,
                    0.0,
                    -0.006145126895,
                    -0.007645690196,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.003259197444,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.209113054889,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.042084275918,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.015388624199,
                    0.0,
                    0.006150440601,
                    -0.007694312099,
                    0.0,
                    -0.006150440601,
                    -0.007694312099,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": -0.003432224601,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.216609085513,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.041741379888,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.015546139127,
                    0.0,
                    0.006191172928,
                    -0.007773069564,
                    0.0,
                    -0.006191172928,
                    -0.007773069564,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.003322256873,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.218316122726,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.017877876813,
                    0.0,
                    0.007580722265,
                    -0.008938938407,
                    0.0,
                    -0.007580722265,
                    -0.008938938407,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": -0.003324348494,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.222648969545,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.040510044397,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.000000000000,
                    -0.000000000000,
                    0.019447884570,
                    -0.000000000000,
                    0.008509802850,
                    -0.009723942285,
                    0.000000000000,
                    -0.008509802849,
                    -0.009723942285,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": -0.00345497869314,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.21773362291808,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.016675155033,
                    -0.0,
                    0.007064234369,
                    -0.008337577516,
                    0.0,
                    -0.007064234369,
                    -0.008337577516,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.005337854373,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.005214252991,  # dfocc, tight
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  DF-FC-DF  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_df_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_df_rhf,
            "MP2 CORRELATION ENERGY": -0.201610660387,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0535212487451535,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.00000000000000,
                    0.00000000000000,
                    0.00314716362539,
                    0.00000000000000,
                    0.00000000000000,
                    -0.00314716362539,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20461763,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000595617648, 0.0, 0.0, 0.000595617648]).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.206525739571,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2080860831,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.002514968877, 0.0, 0.0, -0.002514968877]).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.2063501312,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.0, 0.001158833226, -0.0, -0.0, -0.001158833226]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.20694030442285,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04775171,
            "CCSD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.0,
                    0.002316407407,
                    0.0,
                    0.0,
                    -0.002316407407,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00192209183545,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.0, 0.003412477814, -0.0, -0.0, -0.003412477814]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00194585245550,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.000703072086,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.202322737562,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.000000000000, 0.004462211239, 0.0, -0.000000000000, -0.004462211239]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.000572012280,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.203736894925,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.000000000000, 0.002400349139, 0.0, -0.000000000000, -0.002400349139]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000469635507,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.205198445453,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.000000000000, 0.000413292087, 0.0, -0.000000000000, -0.000413292087]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.000554531047,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.207169559461,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.000000000000, 0.003224769580, 0.0, -0.000000000000, -0.003224769580]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.000553992712,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.208748872411,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.000000000000, 0.000000000000, 0.003792898685, -0.000000000000, -0.000000000000, -0.003792898685]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.00051222526754,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.20695630522582,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.0, 0.002245645626, -0.0, -0.0, -0.002245645626]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.002045023614,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.002024842553,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_df_rhf,
            "B2PLYP TOTAL ENERGY": -100.3639883016977876,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_df_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_df_rhf,
            "MP2 CORRELATION ENERGY": -0.21939028,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05587974,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.010243193827,
                    0.0,
                    0.005885789424,
                    -0.005121596913,
                    0.0,
                    -0.005885789424,
                    -0.005121596913,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.22424330,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.004605219936, 0.0, 0.002923029606, -0.002302609968, 0.0, -0.002923029606, -0.002302609968]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.22723769214209,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2297412879,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.009150916979, 0.0, 0.005603501036, -0.00457545849, 0.0, -0.005603501036, -0.00457545849]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.2259041334,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.0,
                    0.006780363587,
                    -0.0,
                    0.004293361962,
                    -0.003390181794,
                    0.0,
                    -0.004293361962,
                    -0.003390181793,
                ]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.22729553766998,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04931891,
            "CCSD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.0,
                    0.008124306221,
                    0.0,
                    0.004987568157,
                    -0.004062153110,
                    0.0,
                    -0.004987568157,
                    -0.004062153111,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00521721307613,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.0,
                    0.010801208563,
                    -0.0,
                    0.006421774051,
                    -0.005400604280,
                    -0.0,
                    -0.006421774051,
                    -0.005400604283,
                ]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00521446434812,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.002352666762,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.221650942141,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.012771605048,
                    0.0,
                    0.007190306206,
                    -0.006385802524,
                    0.0,
                    -0.007190306205,
                    -0.006385802524,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001538577646,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.223375325491,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.008996184043,
                    0.0,
                    0.005183828520,
                    -0.004498092021,
                    0.0,
                    -0.005183828519,
                    -0.004498092021,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000960712694,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.225313300040,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.005564092786,
                    0.0,
                    0.003375822885,
                    -0.002782046393,
                    0.0,
                    -0.003375822885,
                    -0.002782046393,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001285806012,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.228593144538,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.009830485254,
                    0.0,
                    0.005814222254,
                    -0.004915242627,
                    0.0,
                    -0.005814222254,
                    -0.004915242627,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001194079264,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.231028592985,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.000000000000,
                    0.000000000000,
                    0.010488609074,
                    -0.000000000000,
                    0.006266448803,
                    -0.005244304537,
                    -0.000000000000,
                    -0.006266448803,
                    -0.005244304537,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.00112736114914,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.22711013815896,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.007977310114,
                    -0.0,
                    0.004884098446,
                    -0.003988655057,
                    -0.0,
                    -0.004884098446,
                    -0.003988655057,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.005548457534,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.005450541943,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_adz_df_rhf,
            "B2PLYP TOTAL ENERGY": -76.3796839677439721,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_df_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_df_rhf,
            "MP2 CORRELATION ENERGY": -0.24512893,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06126089,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.000298272081,
                    0.0,
                    -0.000570968013,
                    -0.00014913604,
                    0.0,
                    0.000570968013,
                    -0.00014913604,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.24746227,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.006198368687, 0.0, -0.004115138427, 0.003099184344, 0.0, 0.004115138427, 0.003099184344]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.251000783874,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2531777549,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.001506644114, 0.0, -0.001400658245, 0.000753322057, 0.0, 0.001400658245, 0.000753322057]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.2488353928,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.0,
                    -0.003903171069,
                    -0.0,
                    -0.002634101054,
                    0.001951585534,
                    -0.0,
                    0.002634101055,
                    0.001951585534,
                ]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.25031496729288,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05404876,
            "CCSD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.0,
                    -0.002520998723,
                    0.0,
                    -0.001932209248,
                    0.001260499359,
                    0.0,
                    0.001932209248,
                    0.001260499363,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00709505459495,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight
                [0.0, 0.0, 0.000617506530, -0.0, -0.000361314051, -0.000308753267, 0.0, 0.000361314051, -0.000308753262]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00701650721029,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.002436225037,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.247501148905,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.002625578291,
                    0.0,
                    0.000602919468,
                    -0.001312789146,
                    0.0,
                    -0.000602919468,
                    -0.001312789146,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001628506731,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.247890696918,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    -0.001471303306,
                    0.0,
                    -0.001622363348,
                    0.000735651653,
                    0.0,
                    0.001622363348,
                    0.000735651653,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001042188755,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.248510869843,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    -0.005274771244,
                    0.0,
                    -0.003679381392,
                    0.002637385622,
                    0.0,
                    0.003679381392,
                    0.002637385622,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001387432536,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.252366737687,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    -0.000767533552,
                    0.0,
                    -0.001128632785,
                    0.000383766776,
                    0.0,
                    0.001128632785,
                    0.000383766776,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001298398813,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.254465606264,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.000000000000,
                    -0.000000000000,
                    -0.000178270779,
                    -0.000000000000,
                    -0.000741462346,
                    0.000089135390,
                    0.000000000000,
                    0.000741462346,
                    0.000089135390,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.00122701106154,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.25004962087804,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    -0.002738846535,
                    -0.0,
                    -0.002056141369,
                    0.001369423268,
                    0.0,
                    0.002056141369,
                    0.001369423268,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.007400543965,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.007287376322,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_h2o_qz2p_df_rhf,
            "B2PLYP TOTAL ENERGY": -76.4097108446835733,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_df_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_df_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_df_uhf,
            "MP2 CORRELATION ENERGY": -0.058390006825,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001768919072594215,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.00000000000000,
                    0.00000000000000,
                    0.01231996225662,
                    0.00000000000000,
                    0.00000000000000,
                    -0.01186374280678,
                    0.00000000000000,
                    0.01031743020277,
                    -0.00022810972492,
                    0.00000000000000,
                    -0.01031743020277,
                    -0.00022810972492,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.07380592,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00220531,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.00909477332,
                    0.0,
                    0.0,
                    -0.008853050692,
                    0.0,
                    0.012720593477,
                    -0.000120861314,
                    0.0,
                    -0.012720593477,
                    -0.000120861314,
                ]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.077065229797,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.002191757201,  # dfocc, tight
            "LCCD CORRELATION ENERGY": -0.0824996438,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0022545103,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.004394350773,
                    0.0,
                    0.0,
                    -0.004297223912,
                    0.0,
                    0.0152632844,
                    -0.000048563431,
                    0.0,
                    -0.0152632844,
                    -0.000048563431,
                ]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.08021106,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.0,
                    0.006152061332,
                    -0.0,
                    -0.0,
                    -0.005948040465,
                    -0.0,
                    0.014448001505,
                    -0.000102010434,
                    0.0,
                    -0.014448001505,
                    -0.000102010434,
                ]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.08123709133288,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    0.0,
                    0.004955877488,
                    0.0,
                    -0.0,
                    -0.004802960939,
                    -0.0,
                    0.014879065297,
                    -0.000076458275,
                    0.0,
                    -0.014879065297,
                    -0.000076458275,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00060440612636,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    0.0,
                    0.004453849339,
                    0.0,
                    -0.0,
                    -0.004383447765,
                    -0.0,
                    0.014958703713,
                    -0.000035200787,
                    0.0,
                    -0.014958703713,
                    -0.000035200787,
                ]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00058824527447,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.000394385396,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.058935159355,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.001823537679,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.011816043382,
                    0.0,
                    -0.000000000000,
                    -0.011405170159,
                    0.0,
                    0.010534044966,
                    -0.000205436611,
                    0.0,
                    -0.010534044966,
                    -0.000205436611,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.000566186305,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.066806131554,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.002050540370,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.010039510728,
                    0.0,
                    -0.000000000000,
                    -0.009744203205,
                    0.0,
                    0.011807411335,
                    -0.000147653762,
                    0.0,
                    -0.011807411335,
                    -0.000147653762,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.000783155015,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.074716931449,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.002277314389,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.008212538414,
                    0.0,
                    -0.000000000000,
                    -0.008033077982,
                    0.0,
                    0.013091494156,
                    -0.000089730216,
                    0.0,
                    -0.013091494156,
                    -0.000089730216,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001067566021,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.078201861999,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.005652301943,
                    0.0,
                    -0.000000000000,
                    -0.005529627660,
                    0.0,
                    0.014380946092,
                    -0.000061337141,
                    0.0,
                    -0.014380946092,
                    -0.000061337141,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001494223667,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.083986395282,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.002308873939,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.000000000000,
                    0.000000000000,
                    0.002693251383,
                    0.000000000000,
                    -0.000000000000,
                    -0.002664270590,
                    -0.000000000000,
                    0.015745905068,
                    -0.000014490396,
                    0.000000000000,
                    -0.015745905068,
                    -0.000014490396,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.00117513618887,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.08143642271488,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    0.000000000000,
                    0.004941329452,
                    0.0,
                    -0.000000000000,
                    -0.004783766728,
                    -0.0,
                    0.014873123435,
                    -0.000078781362,
                    0.0,
                    -0.014873123435,
                    -0.000078781362,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.000609013952,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.000587107823,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_bh3p_dz_df_uhf,
            "B2PLYP TOTAL ENERGY": -26.0726110365712387,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "df",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_df_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_df_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_df_uhf,
            "MP2 CORRELATION ENERGY": -0.15240678,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03448325,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.026264866471,
                    0.0,
                    0.013987430104,
                    -0.013132433236,
                    0.0,
                    -0.013987430104,
                    -0.013132433236,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.16868589,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03460509,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.027853586058, 0.0, 0.014898421791, -0.013926793029, 0.0, -0.014898421791, -0.013926793029]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.17028287893561,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.03381515362353,  # dfocc, tight
            "LCCD CORRELATION ENERGY": -0.1748446809,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0333897039,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.031455324295, 0.0, 0.017108389967, -0.015727662148, 0.0, -0.017108389967, -0.015727662148]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.17060710,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    -0.0,
                    0.028917529759,
                    0.0,
                    0.015695181131,
                    -0.014458764879,
                    0.0,
                    -0.015695181131,
                    -0.014458764879,
                ]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.17173557387375,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    0.0,
                    0.030043575132,
                    0.0,
                    0.016300990267,
                    -0.015021787567,
                    -0.0,
                    -0.016300990267,
                    -0.015021787565,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00381543842966,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    0.0,
                    0.031923074308,
                    0.0,
                    0.017296807836,
                    -0.015961537154,
                    -0.0,
                    -0.017296807836,
                    -0.015961537154,
                ]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00373604126252,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.001361296352,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.153806958505,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.035220509485,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.027773187697,
                    0.0,
                    0.014760469702,
                    -0.013886593849,
                    0.0,
                    -0.014760469702,
                    -0.013886593849,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001172388039,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.161802412521,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.035084385482,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.028357272969,
                    0.0,
                    0.015100442421,
                    -0.014178636485,
                    0.0,
                    -0.015100442421,
                    -0.014178636485,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001026896343,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.169839459643,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.034978089725,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.029002023504,
                    0.0,
                    0.015477270320,
                    -0.014501011752,
                    0.0,
                    -0.015477270319,
                    -0.014501011752,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001113332083,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.171513122196,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.031096217899,
                    0.0,
                    0.016756071833,
                    -0.015548108949,
                    0.0,
                    -0.016756071832,
                    -0.015548108949,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001113749802,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.176088417240,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.033693410677,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.000000000000,
                    -0.000000000000,
                    0.032749728621,
                    0.000000000000,
                    0.017783947932,
                    -0.016374864311,
                    -0.000000000000,
                    -0.017783947932,
                    -0.016374864310,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.00098853594635,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.17171797869288,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    -0.000000000000,
                    0.030027361324,
                    0.0,
                    0.016268352543,
                    -0.015013680662,
                    -0.0,
                    -0.016268352543,
                    -0.015013680662,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.003958389693,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.003858590576,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_adz_df_uhf,
            "B2PLYP TOTAL ENERGY": -55.8329155987089578,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "df",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_df_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_df_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_df_uhf,
            "MP2 CORRELATION ENERGY": -0.17116675,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03822296,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.01471721142,
                    0.0,
                    0.005840479593,
                    -0.00735860571,
                    0.0,
                    -0.005840479593,
                    -0.00735860571,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.18574570,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03792036,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.015407701186, 0.0, 0.006091123422, -0.007703850593, 0.0, -0.006091123422, -0.007703850593]
            ).reshape((-1, 3)),
            "REMP2 CORRELATION ENERGY": -0.187461784316,  # dfocc, tight
            "REMP2 SINGLES ENERGY": 0.0,
            "REMP2 SAME-SPIN CORRELATION ENERGY": -0.037230047131,  # dfocc, tight
            "LCCD CORRELATION ENERGY": -0.1916908596,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0367574293,
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.019127907852, 0.0, 0.00829827557, -0.009563953926, 0.0, -0.00829827557, -0.009563953926]
            ).reshape((-1, 3)),
            "CCD CORRELATION ENERGY": -0.18707439,  # dfocc, tight
            "CCD SINGLES ENERGY": 0.0,
            "CCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    -0.0,
                    0.016599541828,
                    -0.0,
                    0.006987201946,
                    -0.008299770914,
                    0.0,
                    -0.006987201946,
                    -0.008299770914,
                ]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.18830636267506,  # dfocc, tight
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.0,
                    0.017859476803,
                    -0.0,
                    0.007647663282,
                    -0.008929738402,
                    0.0,
                    -0.007647663282,
                    -0.008929738401,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00498171470106,  # dfocc, tight
            "CCSD(T) TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.0,
                    0.020007178737,
                    -0.0,
                    0.008716203188,
                    -0.010003589371,
                    0.0,
                    -0.008716203188,
                    -0.010003589367,
                ]
            ).reshape((-1, 3)),
            "A-(T) CORRECTION ENERGY": -0.00488153123480,  # dfocc, tight
            "OMP2 REFERENCE CORRECTION ENERGY": 0.001480332224,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.172665177005,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.039018138633,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.016299386418,
                    0.0,
                    0.006629699700,
                    -0.008149693209,
                    0.0,
                    -0.006629699700,
                    -0.008149693209,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": 0.001272971815,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.179766145094,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.038623138099,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.016424137432,
                    0.0,
                    0.006638678143,
                    -0.008212068716,
                    0.0,
                    -0.006638678143,
                    -0.008212068716,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": 0.001112851260,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.186923452541,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.038268824767,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.016608938328,
                    0.0,
                    0.006684117862,
                    -0.008304469164,
                    0.0,
                    -0.006684117862,
                    -0.008304469164,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": 0.001217776401,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.188735219689,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.018940660731,
                    0.0,
                    0.008076822926,
                    -0.009470330365,
                    0.0,
                    -0.008076822926,
                    -0.009470330365,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.001218640112,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.192974799277,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.037040961442,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.000000000000,
                    0.000000000000,
                    0.020520978467,
                    -0.000000000000,
                    0.009009472752,
                    -0.010260489234,
                    0.000000000000,
                    -0.009009472752,
                    -0.010260489234,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": 0.0010910673835,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.18822673037030,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.017781686076,
                    -0.0,
                    0.007590501254,
                    -0.008890843038,
                    0.0,
                    -0.007590501254,
                    -0.008890843038,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.005145029771,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.005024912480,  # dfocc, tight
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_nh2_qz2p_df_uhf,
            "B2PLYP TOTAL ENERGY": -55.8541228738473166,  # psi 99,590
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_df_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_df_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_df_rohf,
            "MP2 CORRELATION ENERGY": -0.059372748391,
            "MP2 SINGLES ENERGY": -0.000688391888527046,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0018535174789756292,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.01340658,
                    0.0,
                    0.0,
                    -0.01292306,
                    0.0,
                    0.01029363,
                    -0.00024176,
                    0.0,
                    -0.01029363,
                    -0.00024176,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.0824737155,  # p4n
            "OMP2 REFERENCE CORRECTION ENERGY": -0.001129652787,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.060459196444,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.001823537706,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.011816050823,
                    0.0,
                    -0.000000000000,
                    -0.011405176978,
                    0.0,
                    0.010534044979,
                    -0.000205436923,
                    0.0,
                    -0.010534044979,
                    -0.000205436923,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.000957847537,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.068330168643,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.002050540345,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.010039462838,
                    0.0,
                    -0.000000000000,
                    -0.009744167884,
                    0.0,
                    0.011807418804,
                    -0.000147647477,
                    0.0,
                    -0.011807418804,
                    -0.000147647477,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": -0.000740878651,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.076240968538,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.002277314395,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.008212498711,
                    0.0,
                    -0.000000000000,
                    -0.008033043130,
                    0.0,
                    0.013091495437,
                    -0.000089727790,
                    0.0,
                    -0.013091495437,
                    -0.000089727790,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.000456469124,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.079725899088,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.005652293263,
                    0.0,
                    -0.000000000000,
                    -0.005529605660,
                    0.0,
                    0.014380930007,
                    -0.000061343802,
                    0.0,
                    -0.014380930007,
                    -0.000061343802,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": -0.000029817721,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.085510432371,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.002308874960,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.000000000000,
                    0.000000000000,
                    0.002693270150,
                    0.000000000000,
                    -0.000000000000,
                    -0.002664286871,
                    -0.000000000000,
                    0.015745889183,
                    -0.000014491640,
                    0.000000000000,
                    -0.015745889183,
                    -0.000014491640,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": -0.00034890091894,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.08296045969717,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    0.000000000000,
                    0.004941329268,
                    0.0,
                    -0.000000000000,
                    -0.004783766592,
                    -0.0,
                    0.014873123401,
                    -0.000078781338,
                    0.0,
                    -0.014873123401,
                    -0.000078781338,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.000609013944,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.000587107816,  # dfocc, tight
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "df",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_df_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_df_rohf,
            "HF TOTAL HESSIAN": _grad_scf_nh2_adz_df_rohf,
            "MP2 CORRELATION ENERGY": -0.15700408,
            "MP2 SINGLES ENERGY": -0.00280619,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03469227,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.026382129796,
                    0.0,
                    0.014000533629,
                    -0.013191064898,
                    0.0,
                    -0.014000533629,
                    -0.013191064898,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.1769909051,  # p4n
            "OMP2 REFERENCE CORRECTION ENERGY": -0.003052666882,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.158220918714,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.035220509402,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.027773186774,
                    0.0,
                    0.014760475104,
                    -0.013886593387,
                    0.0,
                    -0.014760475103,
                    -0.013886593387,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.003241577707,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.166216372729,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.035084385218,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.028357346855,
                    0.0,
                    0.015100453507,
                    -0.014178673427,
                    0.0,
                    -0.015100453507,
                    -0.014178673427,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": -0.003387066455,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.174253419851,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.034978090484,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.029002032180,
                    0.0,
                    0.015477263061,
                    -0.014501016090,
                    0.0,
                    -0.015477263061,
                    -0.014501016090,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.003300628093,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.175927082404,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.031096218276,
                    0.0,
                    0.016756071943,
                    -0.015548109138,
                    0.0,
                    -0.016756071943,
                    -0.015548109138,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": -0.003300211378,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.180502377447,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.033693402688,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.000000000000,
                    -0.000000000000,
                    0.032749671199,
                    0.000000000000,
                    0.017783936393,
                    -0.016374835600,
                    -0.000000000000,
                    -0.017783936393,
                    -0.016374835600,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": -0.00342542420012,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.17613193885178,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    -0.0,
                    -0.000000000000,
                    0.030027364470,
                    0.0,
                    0.016268353265,
                    -0.015013682235,
                    -0.0,
                    -0.016268353265,
                    -0.015013682235,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.003958389711,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.003858590593,  # dfocc, tight
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "df",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
            "sdsc": "sc",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_df_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_df_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_df_rohf,
            "MP2 CORRELATION ENERGY": -0.17609909,
            "MP2 SINGLES ENERGY": -0.00294363,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03837317,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.014870916178,
                    0.0,
                    0.00587474124,
                    -0.007435458089,
                    0.0,
                    -0.00587474124,
                    -0.007435458089,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.1939804718,  # p4n
            "OMP2 REFERENCE CORRECTION ENERGY": -0.003129716190,  # dfocc, tight
            "OMP2 CORRELATION ENERGY": -0.177275218950,  # dfocc, tight
            "OMP2 SAME-SPIN CORRELATION ENERGY": -0.039018136779,  # dfocc, tight
            "OMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.016299403030,
                    0.0,
                    0.006629703904,
                    -0.008149701515,
                    0.0,
                    -0.006629703904,
                    -0.008149701515,
                ]
            ).reshape((-1, 3)),
            "OMP2.5 REFERENCE CORRECTION ENERGY": -0.003337076220,  # dfocc, tight
            "OMP2.5 CORRELATION ENERGY": -0.184376187039,  # dfocc, tight
            "OMP2.5 SAME-SPIN CORRELATION ENERGY": -0.038623138676,  # dfocc, tight
            "OMP2.5 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    -0.000000000000,
                    0.016424136331,
                    0.0,
                    0.006638670952,
                    -0.008212068165,
                    0.0,
                    -0.006638670952,
                    -0.008212068165,
                ]
            ).reshape((-1, 3)),
            "OMP3 REFERENCE CORRECTION ENERGY": -0.003497196377,  # dfocc, tight
            "OMP3 CORRELATION ENERGY": -0.191533494486,  # dfocc, tight
            "OMP3 SAME-SPIN CORRELATION ENERGY": -0.038268825438,  # dfocc, tight
            "OMP3 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.016608946514,
                    0.0,
                    0.006684114421,
                    -0.008304473257,
                    0.0,
                    -0.006684114421,
                    -0.008304473257,
                ]
            ).reshape((-1, 3)),
            "OREMP2 REFERENCE CORRECTION ENERGY": -0.003392270931,  # dfocc, tight
            "OREMP2 CORRELATION ENERGY": -0.193345261634,  # dfocc, tight
            "OREMP2 TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.018940651737,
                    0.0,
                    0.008076829692,
                    -0.009470325868,
                    0.0,
                    -0.008076829692,
                    -0.009470325868,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": -0.003391400191,  # dfocc, tight
            "OLCCD CORRELATION ENERGY": -0.197584841222,  # dfocc, tight
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.037040959020,  # dfocc, tight
            "OLCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.000000000000,
                    0.000000000000,
                    0.020521012807,
                    -0.000000000000,
                    0.009009479704,
                    -0.010260506404,
                    0.000000000000,
                    -0.009009479704,
                    -0.010260506404,
                ]
            ).reshape((-1, 3)),
            "OCCD REFERENCE CORRECTION ENERGY": -0.00351897455648,  # dfocc, tight
            "OCCD CORRELATION ENERGY": -0.19283677235185,  # dfocc, tight
            "OCCD TOTAL GRADIENT": np.array(  # dfocc, tight
                [
                    0.0,
                    0.000000000000,
                    0.017781687051,
                    -0.0,
                    0.007590501825,
                    -0.008890843526,
                    0.0,
                    -0.007590501825,
                    -0.008890843526,
                ]
            ).reshape((-1, 3)),
            "O(T) CORRECTION ENERGY": -0.005145029773,  # dfocc, tight
            "A-O(T) CORRECTION ENERGY": -0.005024912482,  # dfocc, tight
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  lopsided SCF/CORL algorithms  >>>
    # <<<  lopsided DF-FC-CONV  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_df_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_df_rhf,
            "MP2 CORRELATION ENERGY": -0.201612517228,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05348507322421174,
            "MP2 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.00316682, 0.0, 0.0, -0.00316682]  # occ findif-5 fc df+conv
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20452073,
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04918306,
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_df_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    # <<<  lopsided CD-FC-CONV  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_cd_rhf,
            # "HF TOTAL HESSIAN": _hess_scf_hf_dz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.20162236483,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.053488165399,
            "MP2 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.00316652, 0.0, 0.0, -0.00316652]  # occ findif-5 fc cd+conv
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20453091,
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04918609,
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_cd_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    # {
    #    "meta": {
    #        "system": "hf",
    #        "basis": "cc-pvdz",
    #        "scf_type": "df",
    #        "reference": "rhf",
    #        "fcae": "fc",
    #        "corl_type": "cd",
    #        "sdsc": "sd",
    #    },
    #    "data": {
    #        "HF TOTAL ENERGY": _scf_hf_dz_df_rhf,
    #        "MP2 CORRELATION ENERGY": -0.201609396752,
    #        #            "MP2 TOTAL ENERGY": -100.221010002381,
    #        "MP2 SINGLES ENERGY": 0.0,
    #        "MP2 SAME-SPIN CORRELATION ENERGY": -0.4,
    #    },
    # },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "uhf",
    #            "fcae": "fc",
    #            "corl_type": "conv",
    #            "sdsc": "sd",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_uhf,
    #            "MP2 CORRELATION ENERGY": -0.058421122206,
    #            #            "MP2 TOTAL ENERGY": -26.003551681354,
    #            "MP2 SINGLES ENERGY": 0.0,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -0.5,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "uhf",
    #            "fcae": "fc",
    #            "corl_type": "cd",
    #            "sdsc": "sd",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_uhf,
    #            "MP2 CORRELATION ENERGY": -0.058409837177,
    #            #            "MP2 TOTAL ENERGY": -26.003540396324,
    #            "MP2 SINGLES ENERGY": 0.0,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -0.7,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "rohf",
    #            "fcae": "fc",
    #            "corl_type": "conv",
    #            "sdsc": "sc",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_rohf,
    #            "MP2 CORRELATION ENERGY": -0.060939211739,
    #            #            "MP2 TOTAL ENERGY": -26.004545733768,
    #            "MP2 SINGLES ENERGY": 1.1,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -1.1,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "rohf",
    #            "fcae": "fc",
    #            "corl_type": "cd",
    #            "sdsc": "sc",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_rohf,
    #            "MP2 CORRELATION ENERGY": -0.059393510962,
    #            #            "MP2 TOTAL ENERGY": -26.003000032991,
    #            "MP2 SINGLES ENERGY": 1.3,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -1.3,
    #        },
    #    },
    # <<<  lopsided DF-AE-CONV  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_df_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_df_rhf,
            "MP2 CORRELATION ENERGY": -0.2037668844651997,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05427252944164894,
            "MP2 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.00281165, 0.0, 0.0, -0.00281165]  # occ findif-5 ae df+conv
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20646457,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04988633,  # dfocc
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000924553428, 0.0, 0.0, 0.000924553428]).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2098900858,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.048336089041,  # fnocc
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.002196305414, 0.0, 0.0, -0.002196305414]).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2107275173,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.048456320034,  # fnocc
            "CCSD CORRELATION ENERGY": -0.20872812,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04857038,
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_df_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    # <<<  lopsided CD-AE-CONV  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "conv",
            "sdsc": "sd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_cd_rhf,
            # "HF TOTAL HESSIAN": _hess_scf_hf_dz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.2037767503537,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05427563053,
            "MP2 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.00281136, 0.0, 0.0, -0.00281136]  # occ findif-5 ae cd+conv
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20647475,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04988936,  # dfocc
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.00092571521, 0.0, 0.0, 0.00092571521]).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2099004485,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.048339111990,  # fnocc
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.002195646447, 0.0, 0.0, -0.002195646447]).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2107380019,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.048459381537,  # fnocc
            "CCSD CORRELATION ENERGY": -0.20873814,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04857333,
            "B2PLYP FUNCTIONAL TOTAL ENERGY": _b2plyp_hf_dz_cd_rhf,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    #    {
    #        "meta": {
    #            "system": "hf",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "rhf",
    #            "fcae": "ae",
    #            "corl_type": "cd",
    #            "sdsc": "sd",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_hf_dz_df_rhf,
    #            "MP2 CORRELATION ENERGY": -2.3,
    #            "MP2 SINGLES ENERGY": 0.0,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -2.3,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "uhf",
    #            "fcae": "ae",
    #            "corl_type": "conv",
    #            "sdsc": "sd",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_uhf,
    #            "MP2 CORRELATION ENERGY": -2.4,
    #            "MP2 SINGLES ENERGY": 0.0,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -2.4,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "uhf",
    #            "fcae": "ae",
    #            "corl_type": "cd",
    #            "sdsc": "sd",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_uhf,
    #            "MP2 CORRELATION ENERGY": -2.5,
    #            "MP2 SINGLES ENERGY": 0.0,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -2.5,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "rohf",
    #            "fcae": "ae",
    #            "corl_type": "conv",
    #            "sdsc": "sc",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_rohf,
    #            "MP2 CORRELATION ENERGY": -2.7,
    #            "MP2 SINGLES ENERGY": -2.7,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -2.7,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "rohf",
    #            "fcae": "ae",
    #            "corl_type": "cd",
    #            "sdsc": "sc",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_rohf,
    #            "MP2 CORRELATION ENERGY": -2.8,
    #            "MP2 SINGLES ENERGY": -2.8,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -2.8,
    #        },
    #    },
]


def compute_derived_qcvars(std_suite_list):
    for calc in std_suite_list:
        if calc["data"]:
            if "MP2 CORRELATION ENERGY" in calc["data"]:
                calc["data"]["MP2 TOTAL ENERGY"] = (
                    calc["data"]["MP2 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "MP2 SINGLES ENERGY" in calc["data"]:
                    calc["data"]["MP2 DOUBLES ENERGY"] = (
                        calc["data"]["MP2 CORRELATION ENERGY"] - calc["data"]["MP2 SINGLES ENERGY"]
                    )
                    if "MP2 SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        calc["data"]["MP2 OPPOSITE-SPIN CORRELATION ENERGY"] = (
                            calc["data"]["MP2 CORRELATION ENERGY"]
                            - calc["data"]["MP2 SAME-SPIN CORRELATION ENERGY"]
                            - calc["data"]["MP2 SINGLES ENERGY"]
                        )
                        calc["data"]["SCS-MP2 CORRELATION ENERGY"] = (
                            (1 / 3) * calc["data"]["MP2 SAME-SPIN CORRELATION ENERGY"]
                            + (6 / 5) * calc["data"]["MP2 OPPOSITE-SPIN CORRELATION ENERGY"]
                            + calc["data"]["MP2 SINGLES ENERGY"]
                        )
                        calc["data"]["SCS-MP2 TOTAL ENERGY"] = (
                            calc["data"]["SCS-MP2 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                        )

            if "MP3 CORRELATION ENERGY" in calc["data"]:
                calc["data"]["MP3 TOTAL ENERGY"] = (
                    calc["data"]["MP3 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "MP2 CORRELATION ENERGY" in calc["data"]:
                    calc["data"]["MP2.5 CORRELATION ENERGY"] = 0.5 * (
                        calc["data"]["MP3 CORRELATION ENERGY"] + calc["data"]["MP2 CORRELATION ENERGY"]
                    )
                    calc["data"]["MP2.5 TOTAL ENERGY"] = (
                        calc["data"]["MP2.5 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                    )

                if "MP3 SINGLES ENERGY" in calc["data"]:
                    calc["data"]["MP3 DOUBLES ENERGY"] = (
                        calc["data"]["MP3 CORRELATION ENERGY"] - calc["data"]["MP3 SINGLES ENERGY"]
                    )
                    if "MP2 SINGLES ENERGY" in calc["data"]:
                        calc["data"]["MP2.5 SINGLES ENERGY"] = 0.5 * (
                            calc["data"]["MP3 SINGLES ENERGY"] + calc["data"]["MP2 SINGLES ENERGY"]
                        )
                        calc["data"]["MP2.5 DOUBLES ENERGY"] = (
                            calc["data"]["MP2.5 CORRELATION ENERGY"] - calc["data"]["MP2.5 SINGLES ENERGY"]
                        )
                    if "MP3 SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        calc["data"]["MP3 OPPOSITE-SPIN CORRELATION ENERGY"] = (
                            calc["data"]["MP3 CORRELATION ENERGY"]
                            - calc["data"]["MP3 SAME-SPIN CORRELATION ENERGY"]
                            - calc["data"]["MP3 SINGLES ENERGY"]
                        )
                        if "MP2 SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                            calc["data"]["MP2.5 SAME-SPIN CORRELATION ENERGY"] = 0.5 * (
                                calc["data"]["MP3 SAME-SPIN CORRELATION ENERGY"]
                                + calc["data"]["MP2 SAME-SPIN CORRELATION ENERGY"]
                            )
                            calc["data"]["MP2.5 OPPOSITE-SPIN CORRELATION ENERGY"] = (
                                calc["data"]["MP2.5 CORRELATION ENERGY"]
                                - calc["data"]["MP2.5 SAME-SPIN CORRELATION ENERGY"]
                                - calc["data"]["MP2.5 SINGLES ENERGY"]
                            )

            if (
                "MP3 TOTAL GRADIENT" in calc["data"]
                and "MP2 TOTAL GRADIENT" in calc["data"]
                and "HF TOTAL GRADIENT" in calc["data"]
            ):
                calc["data"]["MP2.5 TOTAL GRADIENT"] = 0.5 * (
                    calc["data"]["MP3 TOTAL GRADIENT"] + calc["data"]["MP2 TOTAL GRADIENT"]
                )

            if "MP4(SDQ) CORRELATION ENERGY" in calc["data"]:
                calc["data"]["MP4(SDQ) TOTAL ENERGY"] = (
                    calc["data"]["MP4(SDQ) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "MP4(T) CORRECTION ENERGY" in calc["data"]:
                    calc["data"]["MP4 CORRELATION ENERGY"] = (
                        calc["data"]["MP4(SDQ) CORRELATION ENERGY"] + calc["data"]["MP4(T) CORRECTION ENERGY"]
                    )
                    calc["data"]["MP4 TOTAL ENERGY"] = (
                        calc["data"]["MP4 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                    )
                    if "MP3 CORRELATION ENERGY" in calc["data"]:
                        calc["data"]["MP4 CORRECTION ENERGY"] = (
                            calc["data"]["MP4 CORRELATION ENERGY"] - calc["data"]["MP3 CORRELATION ENERGY"]
                        )

            if "ZAPT2 CORRELATION ENERGY" in calc["data"]:
                calc["data"]["ZAPT2 TOTAL ENERGY"] = (
                    calc["data"]["ZAPT2 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "CISD CORRELATION ENERGY" in calc["data"]:
                calc["data"]["CISD TOTAL ENERGY"] = (
                    calc["data"]["CISD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "QCISD CORRELATION ENERGY" in calc["data"]:
                calc["data"]["QCISD TOTAL ENERGY"] = (
                    calc["data"]["QCISD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "QCISD(T) CORRECTION ENERGY" in calc["data"]:
                    calc["data"]["QCISD(T) CORRELATION ENERGY"] = (
                        calc["data"]["QCISD CORRELATION ENERGY"] + calc["data"]["QCISD(T) CORRECTION ENERGY"]
                    )
                    calc["data"]["QCISD(T) TOTAL ENERGY"] = (
                        calc["data"]["QCISD(T) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                    )

            if "FCI CORRELATION ENERGY" in calc["data"]:
                calc["data"]["FCI TOTAL ENERGY"] = (
                    calc["data"]["FCI CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "REMP2 CORRELATION ENERGY" in calc["data"]:
                calc["data"]["REMP2 TOTAL ENERGY"] = (
                    calc["data"]["REMP2 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "REMP2 SINGLES ENERGY" in calc["data"]:
                    calc["data"]["REMP2 DOUBLES ENERGY"] = (
                        calc["data"]["REMP2 CORRELATION ENERGY"] - calc["data"]["REMP2 SINGLES ENERGY"]
                    )
                    if "REMP2 SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        calc["data"]["REMP2 OPPOSITE-SPIN CORRELATION ENERGY"] = (
                            calc["data"]["REMP2 CORRELATION ENERGY"]
                            - calc["data"]["REMP2 SAME-SPIN CORRELATION ENERGY"]
                            - calc["data"]["REMP2 SINGLES ENERGY"]
                        )

            if "LCCD CORRELATION ENERGY" in calc["data"]:
                calc["data"]["LCCD TOTAL ENERGY"] = (
                    calc["data"]["LCCD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "LCCD SINGLES ENERGY" in calc["data"]:
                    calc["data"]["LCCD DOUBLES ENERGY"] = (
                        calc["data"]["LCCD CORRELATION ENERGY"] - calc["data"]["LCCD SINGLES ENERGY"]
                    )
                    if "LCCD SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        calc["data"]["LCCD OPPOSITE-SPIN CORRELATION ENERGY"] = (
                            calc["data"]["LCCD CORRELATION ENERGY"]
                            - calc["data"]["LCCD SAME-SPIN CORRELATION ENERGY"]
                            - calc["data"]["LCCD SINGLES ENERGY"]
                        )

            if "LCCSD CORRELATION ENERGY" in calc["data"]:
                calc["data"]["LCCSD TOTAL ENERGY"] = (
                    calc["data"]["LCCSD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "LCCSD SINGLES ENERGY" in calc["data"]:
                    calc["data"]["LCCSD DOUBLES ENERGY"] = (
                        calc["data"]["LCCSD CORRELATION ENERGY"] - calc["data"]["LCCSD SINGLES ENERGY"]
                    )
                    if "LCCSD SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        calc["data"]["LCCSD OPPOSITE-SPIN CORRELATION ENERGY"] = (
                            calc["data"]["LCCSD CORRELATION ENERGY"]
                            - calc["data"]["LCCSD SAME-SPIN CORRELATION ENERGY"]
                            - calc["data"]["LCCSD SINGLES ENERGY"]
                        )

            if "CEPA(1) CORRELATION ENERGY" in calc["data"]:
                calc["data"]["CEPA(1) TOTAL ENERGY"] = (
                    calc["data"]["CEPA(1) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "CEPA(1) SINGLES ENERGY" in calc["data"]:
                    calc["data"]["CEPA(1) DOUBLES ENERGY"] = (
                        calc["data"]["CEPA(1) CORRELATION ENERGY"] - calc["data"]["CEPA(1) SINGLES ENERGY"]
                    )
                    if "CEPA(1) SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        calc["data"]["CEPA(1) OPPOSITE-SPIN CORRELATION ENERGY"] = (
                            calc["data"]["CEPA(1) CORRELATION ENERGY"]
                            - calc["data"]["CEPA(1) SAME-SPIN CORRELATION ENERGY"]
                            - calc["data"]["CEPA(1) SINGLES ENERGY"]
                        )

            if "CEPA(3) CORRELATION ENERGY" in calc["data"]:
                calc["data"]["CEPA(3) TOTAL ENERGY"] = (
                    calc["data"]["CEPA(3) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "CEPA(3) SINGLES ENERGY" in calc["data"]:
                    calc["data"]["CEPA(3) DOUBLES ENERGY"] = (
                        calc["data"]["CEPA(3) CORRELATION ENERGY"] - calc["data"]["CEPA(3) SINGLES ENERGY"]
                    )
                    if "CEPA(3) SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        calc["data"]["CEPA(3) OPPOSITE-SPIN CORRELATION ENERGY"] = (
                            calc["data"]["CEPA(3) CORRELATION ENERGY"]
                            - calc["data"]["CEPA(3) SAME-SPIN CORRELATION ENERGY"]
                            - calc["data"]["CEPA(3) SINGLES ENERGY"]
                        )

            if "ACPF CORRELATION ENERGY" in calc["data"]:
                calc["data"]["ACPF TOTAL ENERGY"] = (
                    calc["data"]["ACPF CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "ACPF SINGLES ENERGY" in calc["data"]:
                    calc["data"]["ACPF DOUBLES ENERGY"] = (
                        calc["data"]["ACPF CORRELATION ENERGY"] - calc["data"]["ACPF SINGLES ENERGY"]
                    )
                    if "ACPF SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        calc["data"]["ACPF OPPOSITE-SPIN CORRELATION ENERGY"] = (
                            calc["data"]["ACPF CORRELATION ENERGY"]
                            - calc["data"]["ACPF SAME-SPIN CORRELATION ENERGY"]
                            - calc["data"]["ACPF SINGLES ENERGY"]
                        )

            if "AQCC CORRELATION ENERGY" in calc["data"]:
                calc["data"]["AQCC TOTAL ENERGY"] = (
                    calc["data"]["AQCC CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "AQCC SINGLES ENERGY" in calc["data"]:
                    calc["data"]["AQCC DOUBLES ENERGY"] = (
                        calc["data"]["AQCC CORRELATION ENERGY"] - calc["data"]["AQCC SINGLES ENERGY"]
                    )
                    if "AQCC SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        calc["data"]["AQCC OPPOSITE-SPIN CORRELATION ENERGY"] = (
                            calc["data"]["AQCC CORRELATION ENERGY"]
                            - calc["data"]["AQCC SAME-SPIN CORRELATION ENERGY"]
                            - calc["data"]["AQCC SINGLES ENERGY"]
                        )

            if "CCD CORRELATION ENERGY" in calc["data"]:
                calc["data"]["CCD TOTAL ENERGY"] = (
                    calc["data"]["CCD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "CCD SINGLES ENERGY" in calc["data"]:
                    calc["data"]["CCD DOUBLES ENERGY"] = (
                        calc["data"]["CCD CORRELATION ENERGY"] - calc["data"]["CCD SINGLES ENERGY"]
                    )
                    if "CCD SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        calc["data"]["CCD OPPOSITE-SPIN CORRELATION ENERGY"] = (
                            calc["data"]["CCD CORRELATION ENERGY"]
                            - calc["data"]["CCD SAME-SPIN CORRELATION ENERGY"]
                            - calc["data"]["CCD SINGLES ENERGY"]
                        )

            if "BCCD CORRELATION ENERGY" in calc["data"]:
                calc["data"]["BCCD TOTAL ENERGY"] = (
                    calc["data"]["BCCD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "CC2 CORRELATION ENERGY" in calc["data"]:
                calc["data"]["CC2 TOTAL ENERGY"] = (
                    calc["data"]["CC2 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "CCSD CORRELATION ENERGY" in calc["data"]:
                calc["data"]["CCSD TOTAL ENERGY"] = (
                    calc["data"]["CCSD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "CCSD SINGLES ENERGY" in calc["data"]:
                    calc["data"]["CCSD DOUBLES ENERGY"] = (
                        calc["data"]["CCSD CORRELATION ENERGY"] - calc["data"]["CCSD SINGLES ENERGY"]
                    )
                    if "CCSD SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        calc["data"]["CCSD OPPOSITE-SPIN CORRELATION ENERGY"] = (
                            calc["data"]["CCSD CORRELATION ENERGY"]
                            - calc["data"]["CCSD SAME-SPIN CORRELATION ENERGY"]
                            - calc["data"]["CCSD SINGLES ENERGY"]
                        )

            if "T(CCSD) CORRECTION ENERGY" in calc["data"]:
                calc["data"]["CCSD+T(CCSD) CORRELATION ENERGY"] = (
                    calc["data"]["CCSD CORRELATION ENERGY"] + calc["data"]["T(CCSD) CORRECTION ENERGY"]
                )
                calc["data"]["CCSD+T(CCSD) TOTAL ENERGY"] = (
                    calc["data"]["CCSD+T(CCSD) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "(T) CORRECTION ENERGY" in calc["data"]:
                calc["data"]["CCSD(T) CORRELATION ENERGY"] = (
                    calc["data"]["CCSD CORRELATION ENERGY"] + calc["data"]["(T) CORRECTION ENERGY"]
                )
                calc["data"]["CCSD(T) TOTAL ENERGY"] = (
                    calc["data"]["CCSD(T) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "A-(T) CORRECTION ENERGY" in calc["data"]:
                calc["data"]["A-CCSD(T) CORRELATION ENERGY"] = (
                    calc["data"]["CCSD CORRELATION ENERGY"] + calc["data"]["A-(T) CORRECTION ENERGY"]
                )
                calc["data"]["A-CCSD(T) TOTAL ENERGY"] = (
                    calc["data"]["A-CCSD(T) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "B(T) CORRECTION ENERGY" in calc["data"]:
                calc["data"]["BCCD(T) CORRELATION ENERGY"] = (
                    calc["data"]["BCCD CORRELATION ENERGY"] + calc["data"]["B(T) CORRECTION ENERGY"]
                )
                calc["data"]["BCCD(T) TOTAL ENERGY"] = (
                    calc["data"]["BCCD(T) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "CC3 CORRELATION ENERGY" in calc["data"]:
                calc["data"]["CC3 TOTAL ENERGY"] = (
                    calc["data"]["CC3 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "CCSDT-1A CORRELATION ENERGY" in calc["data"]:
                calc["data"]["CCSDT-1A TOTAL ENERGY"] = (
                    calc["data"]["CCSDT-1A CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "CCSDT-1B CORRELATION ENERGY" in calc["data"]:
                calc["data"]["CCSDT-1B TOTAL ENERGY"] = (
                    calc["data"]["CCSDT-1B CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "CCSDT-2 CORRELATION ENERGY" in calc["data"]:
                calc["data"]["CCSDT-2 TOTAL ENERGY"] = (
                    calc["data"]["CCSDT-2 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "CCSDT-3 CORRELATION ENERGY" in calc["data"]:
                calc["data"]["CCSDT-3 TOTAL ENERGY"] = (
                    calc["data"]["CCSDT-3 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "CCSDT CORRELATION ENERGY" in calc["data"]:
                calc["data"]["CCSDT TOTAL ENERGY"] = (
                    calc["data"]["CCSDT CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "(Q) CORRECTION ENERGY" in calc["data"]:
                calc["data"]["CCSDT(Q) CORRELATION ENERGY"] = (
                    calc["data"]["CCSDT CORRELATION ENERGY"] + calc["data"]["(Q) CORRECTION ENERGY"]
                )
                calc["data"]["CCSDT(Q) TOTAL ENERGY"] = (
                    calc["data"]["CCSDT(Q) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "CCSDTQ CORRELATION ENERGY" in calc["data"]:
                calc["data"]["CCSDTQ TOTAL ENERGY"] = (
                    calc["data"]["CCSDTQ CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "OMP2 CORRELATION ENERGY" in calc["data"]:
                if calc["data"]["OMP2 CORRELATION ENERGY"] == _knownmissing:
                    calc["data"]["OMP2 TOTAL ENERGY"] = _knownmissing
                    calc["data"]["OMP2 OPPOSITE-SPIN CORRELATION ENERGY"] = _knownmissing
                else:
                    calc["data"]["OMP2 TOTAL ENERGY"] = (
                        calc["data"]["OMP2 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                    )
                    if "OMP2 SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        if calc["data"]["OMP2 SAME-SPIN CORRELATION ENERGY"] == _knownmissing:
                            calc["data"]["OMP2 OPPOSITE-SPIN CORRELATION ENERGY"] = _knownmissing
                        else:
                            calc["data"]["OMP2 OPPOSITE-SPIN CORRELATION ENERGY"] = (
                                calc["data"]["OMP2 CORRELATION ENERGY"]
                                - calc["data"]["OMP2 REFERENCE CORRECTION ENERGY"]
                                - calc["data"]["OMP2 SAME-SPIN CORRELATION ENERGY"]
                            )

            if "OMP2.5 CORRELATION ENERGY" in calc["data"]:
                if calc["data"]["OMP2.5 CORRELATION ENERGY"] == _knownmissing:
                    calc["data"]["OMP2.5 TOTAL ENERGY"] = _knownmissing
                    calc["data"]["OMP2.5 OPPOSITE-SPIN CORRELATION ENERGY"] = _knownmissing
                else:
                    calc["data"]["OMP2.5 TOTAL ENERGY"] = (
                        calc["data"]["OMP2.5 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                    )
                    if "OMP2.5 SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        if calc["data"]["OMP2.5 SAME-SPIN CORRELATION ENERGY"] == _knownmissing:
                            calc["data"]["OMP2.5 OPPOSITE-SPIN CORRELATION ENERGY"] = _knownmissing
                        else:
                            calc["data"]["OMP2.5 OPPOSITE-SPIN CORRELATION ENERGY"] = (
                                calc["data"]["OMP2.5 CORRELATION ENERGY"]
                                - calc["data"]["OMP2.5 REFERENCE CORRECTION ENERGY"]
                                - calc["data"]["OMP2.5 SAME-SPIN CORRELATION ENERGY"]
                            )

            if "OMP3 CORRELATION ENERGY" in calc["data"]:
                if calc["data"]["OMP3 CORRELATION ENERGY"] == _knownmissing:
                    calc["data"]["OMP3 TOTAL ENERGY"] = _knownmissing
                    calc["data"]["OMP3 OPPOSITE-SPIN CORRELATION ENERGY"] = _knownmissing
                else:
                    calc["data"]["OMP3 TOTAL ENERGY"] = (
                        calc["data"]["OMP3 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                    )
                    if "OMP3 SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        if calc["data"]["OMP3 SAME-SPIN CORRELATION ENERGY"] == _knownmissing:
                            calc["data"]["OMP3 OPPOSITE-SPIN CORRELATION ENERGY"] = _knownmissing
                        else:
                            calc["data"]["OMP3 OPPOSITE-SPIN CORRELATION ENERGY"] = (
                                calc["data"]["OMP3 CORRELATION ENERGY"]
                                - calc["data"]["OMP3 REFERENCE CORRECTION ENERGY"]
                                - calc["data"]["OMP3 SAME-SPIN CORRELATION ENERGY"]
                            )

            if "OREMP2 CORRELATION ENERGY" in calc["data"]:
                if calc["data"]["OREMP2 CORRELATION ENERGY"] == _knownmissing:
                    calc["data"]["OREMP2 TOTAL ENERGY"] = _knownmissing
                    calc["data"]["OREMP2 OPPOSITE-SPIN CORRELATION ENERGY"] = _knownmissing
                else:
                    calc["data"]["OREMP2 TOTAL ENERGY"] = (
                        calc["data"]["OREMP2 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                    )
                    if "OREMP2 SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        if calc["data"]["OREMP2 SAME-SPIN CORRELATION ENERGY"] == _knownmissing:
                            calc["data"]["OREMP2 OPPOSITE-SPIN CORRELATION ENERGY"] = _knownmissing
                        else:
                            calc["data"]["OREMP2 OPPOSITE-SPIN CORRELATION ENERGY"] = (
                                calc["data"]["OREMP2 CORRELATION ENERGY"]
                                - calc["data"]["OREMP2 REFERENCE CORRECTION ENERGY"]
                                - calc["data"]["OREMP2 SAME-SPIN CORRELATION ENERGY"]
                            )

            if "OLCCD CORRELATION ENERGY" in calc["data"]:
                calc["data"]["OLCCD TOTAL ENERGY"] = (
                    calc["data"]["OLCCD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "OLCCD SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                    if calc["data"]["OLCCD SAME-SPIN CORRELATION ENERGY"] == _knownmissing:
                        calc["data"]["OLCCD OPPOSITE-SPIN CORRELATION ENERGY"] = _knownmissing
                    else:
                        calc["data"]["OLCCD OPPOSITE-SPIN CORRELATION ENERGY"] = (
                            calc["data"]["OLCCD CORRELATION ENERGY"]
                            - calc["data"]["OLCCD REFERENCE CORRECTION ENERGY"]
                            - calc["data"]["OLCCD SAME-SPIN CORRELATION ENERGY"]
                        )

            if "OCCD CORRELATION ENERGY" in calc["data"]:
                calc["data"]["OCCD TOTAL ENERGY"] = (
                    calc["data"]["OCCD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "OCCD SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                    calc["data"]["OCCD OPPOSITE-SPIN CORRELATION ENERGY"] = (
                        calc["data"]["OCCD CORRELATION ENERGY"]
                        - calc["data"]["OCCD REFERENCE CORRECTION ENERGY"]
                        - calc["data"]["OCCD SAME-SPIN CORRELATION ENERGY"]
                    )

            if "O(T) CORRECTION ENERGY" in calc["data"]:
                if calc["data"]["O(T) CORRECTION ENERGY"] == _knownmissing:
                    calc["data"]["OCCD(T) CORRELATION ENERGY"] = _knownmissing
                    calc["data"]["OCCD(T) TOTAL ENERGY"] = _knownmissing
                else:
                    calc["data"]["OCCD(T) CORRELATION ENERGY"] = (
                        calc["data"]["OCCD CORRELATION ENERGY"] + calc["data"]["O(T) CORRECTION ENERGY"]
                    )
                    calc["data"]["OCCD(T) TOTAL ENERGY"] = (
                        calc["data"]["OCCD(T) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                    )

            if "A-O(T) CORRECTION ENERGY" in calc["data"]:
                if calc["data"]["A-O(T) CORRECTION ENERGY"] == _knownmissing:
                    calc["data"]["A-OCCD(T) CORRELATION ENERGY"] = _knownmissing
                    calc["data"]["A-OCCD(T) TOTAL ENERGY"] = _knownmissing
                else:
                    calc["data"]["A-OCCD(T) CORRELATION ENERGY"] = (
                        calc["data"]["OCCD CORRELATION ENERGY"] + calc["data"]["A-O(T) CORRECTION ENERGY"]
                    )
                    calc["data"]["A-OCCD(T) TOTAL ENERGY"] = (
                        calc["data"]["A-OCCD(T) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                    )

        calc["data"].update(_std_generics[f"{calc['meta']['system']}_{calc['meta']['basis']}_{calc['meta']['fcae']}"])


def answer_hash(**kwargs):
    system = kwargs.pop("system")
    basis = kwargs.pop("basis")
    scf_type = kwargs.pop("scf_type")
    reference = kwargs.pop("reference")
    fcae = kwargs.pop("fcae")
    corl_type = kwargs.pop("corl_type")
    sdsc = kwargs.pop("sdsc")

    return "_".join([system, basis, scf_type, reference, fcae, corl_type, sdsc])


compute_derived_qcvars(_std_suite)
std_suite = {answer_hash(**calc["meta"]): calc["data"] for calc in _std_suite}
