// SPDX-License-Identifier: MIT

#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

#include <oqs/oqsconfig.h>
#include <oqs/sha3.h>
#include <oqs/sha3x4.h>

#include "system_info.c"

#ifdef __GNUC__
#define UNUSED __attribute__((unused))
#else
#define UNUSED
#endif

/**
* \file sha3_test.h
* \brief <b>SHA3 Known Answer Tests</b> \n
* Uses Known Answer Tests from official sources to verify the
* correct operation of SHA3 digest and SHAKE implementations.
* \author John Underhill
* \date December 28, 2017
*/

/**
* \brief Compares two arrays of 8 bit integers for equivalence
*
* \param a The first byte array for comparison
* \param b The second byte array for comparison
* \param length The number of bytes to compare
*
* \return Returns zero for failure, one if the arrays match
*/
static int are_equal8(const uint8_t *a, const uint8_t *b, size_t length) {
	size_t i;
	int status = EXIT_SUCCESS;

	for (i = 0; i < length; ++i) {
		if (a[i] != b[i]) {
			status = EXIT_FAILURE;
			break;
		}
	}

	return status;
}

/**
* \brief Set a number of a uint8 array elements to zero
*
* \param a The array to reset
* \param count The number of uint8 integers to clear
*/
static void clear8(uint8_t *a, size_t count) {
	for (size_t i = 0; i < count; ++i) {
		a[i] = 0;
	}
}

/**
* \brief Tests the 256 bit version of the keccak message digest for correct operation,
* using selected vectors from the NIST Fips202 and alternative references.
*
* \return Returns 1 for success, 0 for failure
*
* \remarks <b>Test References:</b> \n
* Fips202: <a href="http://nvlpubs.nist.gov/nistpubs/FIPS/NIST.FIPS.202.pdf">SHA3-Standard</a> \n
* KAT: <a href="https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA3-256_Msg0.pdf">SHA256(0)</a> \n
* KAT: <a href="https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA3-256_1600.pdf">SHA256(1600)</a> \n
* KAT: <a href="https://www.di-mgt.com.au/sha_testvectors.html">SHA256(24, 448)</a>
*/
int sha3_256_kat_test(void) {
	int status = EXIT_SUCCESS;

	uint8_t output[32];

	uint8_t msg0[1] = {0x0};
	uint8_t msg24[3] = {0x61, 0x62, 0x63};
	uint8_t msg448[56] = {
		0x61, 0x62, 0x63, 0x64, 0x62, 0x63, 0x64, 0x65, 0x63, 0x64, 0x65, 0x66, 0x64, 0x65, 0x66, 0x67,
		0x65, 0x66, 0x67, 0x68, 0x66, 0x67, 0x68, 0x69, 0x67, 0x68, 0x69, 0x6A, 0x68, 0x69, 0x6A, 0x6B,
		0x69, 0x6A, 0x6B, 0x6C, 0x6A, 0x6B, 0x6C, 0x6D, 0x6B, 0x6C, 0x6D, 0x6E, 0x6C, 0x6D, 0x6E, 0x6F,
		0x6D, 0x6E, 0x6F, 0x70, 0x6E, 0x6F, 0x70, 0x71
	};
	uint8_t msg1600[200] = {
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3
	};

	uint8_t exp0[32] = {
		0xA7, 0xFF, 0xC6, 0xF8, 0xBF, 0x1E, 0xD7, 0x66, 0x51, 0xC1, 0x47, 0x56, 0xA0, 0x61, 0xD6, 0x62,
		0xF5, 0x80, 0xFF, 0x4D, 0xE4, 0x3B, 0x49, 0xFA, 0x82, 0xD8, 0x0A, 0x4B, 0x80, 0xF8, 0x43, 0x4A
	};
	uint8_t exp24[32] = {
		0x3A, 0x98, 0x5D, 0xA7, 0x4F, 0xE2, 0x25, 0xB2, 0x04, 0x5C, 0x17, 0x2D, 0x6B, 0xD3, 0x90, 0xBD,
		0x85, 0x5F, 0x08, 0x6E, 0x3E, 0x9D, 0x52, 0x5B, 0x46, 0xBF, 0xE2, 0x45, 0x11, 0x43, 0x15, 0x32
	};
	uint8_t exp448[32] = {
		0x41, 0xC0, 0xDB, 0xA2, 0xA9, 0xD6, 0x24, 0x08, 0x49, 0x10, 0x03, 0x76, 0xA8, 0x23, 0x5E, 0x2C,
		0x82, 0xE1, 0xB9, 0x99, 0x8A, 0x99, 0x9E, 0x21, 0xDB, 0x32, 0xDD, 0x97, 0x49, 0x6D, 0x33, 0x76
	};
	uint8_t exp1600[32] = {
		0x79, 0xF3, 0x8A, 0xDE, 0xC5, 0xC2, 0x03, 0x07, 0xA9, 0x8E, 0xF7, 0x6E, 0x83, 0x24, 0xAF, 0xBF,
		0xD4, 0x6C, 0xFD, 0x81, 0xB2, 0x2E, 0x39, 0x73, 0xC6, 0x5F, 0xA1, 0xBD, 0x9D, 0xE3, 0x17, 0x87
	};

	/* test compact api */

	clear8(output, 32);
	OQS_SHA3_sha3_256(output, msg0, 0);

	if (are_equal8(output, exp0, 32) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(output, 32);
	OQS_SHA3_sha3_256(output, msg24, 3);

	if (are_equal8(output, exp24, 32) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(output, 32);
	OQS_SHA3_sha3_256(output, msg448, 56);

	if (are_equal8(output, exp448, 32) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(output, 32);
	OQS_SHA3_sha3_256(output, msg1600, 200);

	if (are_equal8(output, exp1600, 32) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* test long-form api */

	OQS_SHA3_sha3_256_inc_ctx state;
	uint8_t hash[200];

	clear8(hash, 200);
	OQS_SHA3_sha3_256_inc_init(&state);
	OQS_SHA3_sha3_256_inc_absorb(&state, msg0, 0);
	OQS_SHA3_sha3_256_inc_finalize(hash, &state);

	if (are_equal8(hash, exp0, 32) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(hash, 200);
	OQS_SHA3_sha3_256_inc_ctx_reset(&state);
	OQS_SHA3_sha3_256_inc_absorb(&state, msg24, 3);
	OQS_SHA3_sha3_256_inc_finalize(hash, &state);

	if (are_equal8(hash, exp24, 32) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(hash, 200);
	OQS_SHA3_sha3_256_inc_ctx_reset(&state);
	OQS_SHA3_sha3_256_inc_absorb(&state, msg448, 56);
	OQS_SHA3_sha3_256_inc_finalize(hash, &state);

	if (are_equal8(hash, exp448, 32) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(hash, 200);
	OQS_SHA3_sha3_256_inc_ctx_reset(&state);
	OQS_SHA3_sha3_256_inc_absorb(&state, msg1600, 200);
	OQS_SHA3_sha3_256_inc_finalize(hash, &state);
	OQS_SHA3_sha3_256_inc_ctx_release(&state);

	if (are_equal8(hash, exp1600, 32) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	return status;
}

/**
* \brief Tests the 512 bit version of the keccak message digest for correct operation,
* using selected vectors from NIST Fips202 and alternative references.
*
* \return Returns 1 for success, 0 for failure
*
* \remarks <b>Test References:</b> \n
* Fips202: <a href="http://nvlpubs.nist.gov/nistpubs/FIPS/NIST.FIPS.202.pdf">SHA3-Standard</a> \n
* KAT: <a href="https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA3-512_Msg0.pdf">SHA512(0)</a> \n
* KAT: <a href="https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA3-512_1600.pdf">SHA512(1600)</a> \n
* KAT: <a href="https://www.di-mgt.com.au/sha_testvectors.html">SHA512(24, 448)</a>
*/
int sha3_512_kat_test(void) {
	int status = EXIT_SUCCESS;

	uint8_t output[64];

	uint8_t msg0[1] = {0x0};
	uint8_t msg24[3] = {0x61, 0x62, 0x63};
	uint8_t msg448[56] = {
		0x61, 0x62, 0x63, 0x64, 0x62, 0x63, 0x64, 0x65, 0x63, 0x64, 0x65, 0x66, 0x64, 0x65, 0x66, 0x67,
		0x65, 0x66, 0x67, 0x68, 0x66, 0x67, 0x68, 0x69, 0x67, 0x68, 0x69, 0x6A, 0x68, 0x69, 0x6A, 0x6B,
		0x69, 0x6A, 0x6B, 0x6C, 0x6A, 0x6B, 0x6C, 0x6D, 0x6B, 0x6C, 0x6D, 0x6E, 0x6C, 0x6D, 0x6E, 0x6F,
		0x6D, 0x6E, 0x6F, 0x70, 0x6E, 0x6F, 0x70, 0x71
	};
	uint8_t msg1600[200] = {
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3
	};

	uint8_t exp0[64] = {
		0xA6, 0x9F, 0x73, 0xCC, 0xA2, 0x3A, 0x9A, 0xC5, 0xC8, 0xB5, 0x67, 0xDC, 0x18, 0x5A, 0x75, 0x6E,
		0x97, 0xC9, 0x82, 0x16, 0x4F, 0xE2, 0x58, 0x59, 0xE0, 0xD1, 0xDC, 0xC1, 0x47, 0x5C, 0x80, 0xA6,
		0x15, 0xB2, 0x12, 0x3A, 0xF1, 0xF5, 0xF9, 0x4C, 0x11, 0xE3, 0xE9, 0x40, 0x2C, 0x3A, 0xC5, 0x58,
		0xF5, 0x00, 0x19, 0x9D, 0x95, 0xB6, 0xD3, 0xE3, 0x01, 0x75, 0x85, 0x86, 0x28, 0x1D, 0xCD, 0x26
	};
	uint8_t exp24[64] = {
		0xB7, 0x51, 0x85, 0x0B, 0x1A, 0x57, 0x16, 0x8A, 0x56, 0x93, 0xCD, 0x92, 0x4B, 0x6B, 0x09, 0x6E,
		0x08, 0xF6, 0x21, 0x82, 0x74, 0x44, 0xF7, 0x0D, 0x88, 0x4F, 0x5D, 0x02, 0x40, 0xD2, 0x71, 0x2E,
		0x10, 0xE1, 0x16, 0xE9, 0x19, 0x2A, 0xF3, 0xC9, 0x1A, 0x7E, 0xC5, 0x76, 0x47, 0xE3, 0x93, 0x40,
		0x57, 0x34, 0x0B, 0x4C, 0xF4, 0x08, 0xD5, 0xA5, 0x65, 0x92, 0xF8, 0x27, 0x4E, 0xEC, 0x53, 0xF0
	};
	uint8_t exp448[64] = {
		0x04, 0xA3, 0x71, 0xE8, 0x4E, 0xCF, 0xB5, 0xB8, 0xB7, 0x7C, 0xB4, 0x86, 0x10, 0xFC, 0xA8, 0x18,
		0x2D, 0xD4, 0x57, 0xCE, 0x6F, 0x32, 0x6A, 0x0F, 0xD3, 0xD7, 0xEC, 0x2F, 0x1E, 0x91, 0x63, 0x6D,
		0xEE, 0x69, 0x1F, 0xBE, 0x0C, 0x98, 0x53, 0x02, 0xBA, 0x1B, 0x0D, 0x8D, 0xC7, 0x8C, 0x08, 0x63,
		0x46, 0xB5, 0x33, 0xB4, 0x9C, 0x03, 0x0D, 0x99, 0xA2, 0x7D, 0xAF, 0x11, 0x39, 0xD6, 0xE7, 0x5E
	};
	uint8_t exp1600[64] = {
		0xE7, 0x6D, 0xFA, 0xD2, 0x20, 0x84, 0xA8, 0xB1, 0x46, 0x7F, 0xCF, 0x2F, 0xFA, 0x58, 0x36, 0x1B,
		0xEC, 0x76, 0x28, 0xED, 0xF5, 0xF3, 0xFD, 0xC0, 0xE4, 0x80, 0x5D, 0xC4, 0x8C, 0xAE, 0xEC, 0xA8,
		0x1B, 0x7C, 0x13, 0xC3, 0x0A, 0xDF, 0x52, 0xA3, 0x65, 0x95, 0x84, 0x73, 0x9A, 0x2D, 0xF4, 0x6B,
		0xE5, 0x89, 0xC5, 0x1C, 0xA1, 0xA4, 0xA8, 0x41, 0x6D, 0xF6, 0x54, 0x5A, 0x1C, 0xE8, 0xBA, 0x00
	};

	/* test compact api */

	clear8(output, 64);
	OQS_SHA3_sha3_512(output, msg0, 0);

	if (are_equal8(output, exp0, 64) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(output, 64);
	OQS_SHA3_sha3_512(output, msg24, 3);

	if (are_equal8(output, exp24, 64) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(output, 64);
	OQS_SHA3_sha3_512(output, msg448, 56);

	if (are_equal8(output, exp448, 64) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(output, 64);
	OQS_SHA3_sha3_512(output, msg1600, 200);

	if (are_equal8(output, exp1600, 64) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* test long-form api */

	OQS_SHA3_sha3_512_inc_ctx state;
	uint8_t hash[200];

	clear8(hash, 200);
	OQS_SHA3_sha3_512_inc_init(&state);
	OQS_SHA3_sha3_512_inc_absorb(&state, msg0, 0);
	OQS_SHA3_sha3_512_inc_finalize(hash, &state);

	if (are_equal8(hash, exp0, 64) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(hash, 200);
	OQS_SHA3_sha3_512_inc_ctx_reset(&state);
	OQS_SHA3_sha3_512_inc_absorb(&state, msg24, 3);
	OQS_SHA3_sha3_512_inc_finalize(hash, &state);

	if (are_equal8(hash, exp24, 64) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(hash, 200);
	OQS_SHA3_sha3_512_inc_ctx_reset(&state);
	OQS_SHA3_sha3_512_inc_absorb(&state, msg448, 56);
	OQS_SHA3_sha3_512_inc_finalize(hash, &state);

	if (are_equal8(hash, exp448, 64) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(hash, 200);
	OQS_SHA3_sha3_512_inc_ctx_reset(&state);
	OQS_SHA3_sha3_512_inc_absorb(&state, msg1600, 200);
	OQS_SHA3_sha3_512_inc_finalize(hash, &state);
	OQS_SHA3_sha3_512_inc_ctx_release(&state);

	if (are_equal8(hash, exp1600, 64) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	return status;
}

/**
* \brief Tests the 128 bit version of the SHAKE XOF function for correct operation,
* using selected vectors from the NIST reference package.
*
* \return Returns 1 for success, 0 for failure
*
* \remarks <b>Test References:</b> \n
* SP800-185: <a href="http://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-185.pdf">SHA-3 Derived Functions</a> \n
* KAT: <a href="https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHAKE128_Msg0.pdf">SHAKE128(0)</a> \n
* KAT: <a href="https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHAKE128_Msg1600.pdf">SHAKE128(1600)</a>
*/
int shake_128_kat_test(void) {
	int status = EXIT_SUCCESS;

	uint8_t output[512];

	uint8_t msg0[1] = {0x0};
	uint8_t msg1600[200] = {
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3
	};

	uint8_t exp0[512] = {
		0x7F, 0x9C, 0x2B, 0xA4, 0xE8, 0x8F, 0x82, 0x7D, 0x61, 0x60, 0x45, 0x50, 0x76, 0x05, 0x85, 0x3E,
		0xD7, 0x3B, 0x80, 0x93, 0xF6, 0xEF, 0xBC, 0x88, 0xEB, 0x1A, 0x6E, 0xAC, 0xFA, 0x66, 0xEF, 0x26,
		0x3C, 0xB1, 0xEE, 0xA9, 0x88, 0x00, 0x4B, 0x93, 0x10, 0x3C, 0xFB, 0x0A, 0xEE, 0xFD, 0x2A, 0x68,
		0x6E, 0x01, 0xFA, 0x4A, 0x58, 0xE8, 0xA3, 0x63, 0x9C, 0xA8, 0xA1, 0xE3, 0xF9, 0xAE, 0x57, 0xE2,
		0x35, 0xB8, 0xCC, 0x87, 0x3C, 0x23, 0xDC, 0x62, 0xB8, 0xD2, 0x60, 0x16, 0x9A, 0xFA, 0x2F, 0x75,
		0xAB, 0x91, 0x6A, 0x58, 0xD9, 0x74, 0x91, 0x88, 0x35, 0xD2, 0x5E, 0x6A, 0x43, 0x50, 0x85, 0xB2,
		0xBA, 0xDF, 0xD6, 0xDF, 0xAA, 0xC3, 0x59, 0xA5, 0xEF, 0xBB, 0x7B, 0xCC, 0x4B, 0x59, 0xD5, 0x38,
		0xDF, 0x9A, 0x04, 0x30, 0x2E, 0x10, 0xC8, 0xBC, 0x1C, 0xBF, 0x1A, 0x0B, 0x3A, 0x51, 0x20, 0xEA,
		0x17, 0xCD, 0xA7, 0xCF, 0xAD, 0x76, 0x5F, 0x56, 0x23, 0x47, 0x4D, 0x36, 0x8C, 0xCC, 0xA8, 0xAF,
		0x00, 0x07, 0xCD, 0x9F, 0x5E, 0x4C, 0x84, 0x9F, 0x16, 0x7A, 0x58, 0x0B, 0x14, 0xAA, 0xBD, 0xEF,
		0xAE, 0xE7, 0xEE, 0xF4, 0x7C, 0xB0, 0xFC, 0xA9, 0x76, 0x7B, 0xE1, 0xFD, 0xA6, 0x94, 0x19, 0xDF,
		0xB9, 0x27, 0xE9, 0xDF, 0x07, 0x34, 0x8B, 0x19, 0x66, 0x91, 0xAB, 0xAE, 0xB5, 0x80, 0xB3, 0x2D,
		0xEF, 0x58, 0x53, 0x8B, 0x8D, 0x23, 0xF8, 0x77, 0x32, 0xEA, 0x63, 0xB0, 0x2B, 0x4F, 0xA0, 0xF4,
		0x87, 0x33, 0x60, 0xE2, 0x84, 0x19, 0x28, 0xCD, 0x60, 0xDD, 0x4C, 0xEE, 0x8C, 0xC0, 0xD4, 0xC9,
		0x22, 0xA9, 0x61, 0x88, 0xD0, 0x32, 0x67, 0x5C, 0x8A, 0xC8, 0x50, 0x93, 0x3C, 0x7A, 0xFF, 0x15,
		0x33, 0xB9, 0x4C, 0x83, 0x4A, 0xDB, 0xB6, 0x9C, 0x61, 0x15, 0xBA, 0xD4, 0x69, 0x2D, 0x86, 0x19,
		0xF9, 0x0B, 0x0C, 0xDF, 0x8A, 0x7B, 0x9C, 0x26, 0x40, 0x29, 0xAC, 0x18, 0x5B, 0x70, 0xB8, 0x3F,
		0x28, 0x01, 0xF2, 0xF4, 0xB3, 0xF7, 0x0C, 0x59, 0x3E, 0xA3, 0xAE, 0xEB, 0x61, 0x3A, 0x7F, 0x1B,
		0x1D, 0xE3, 0x3F, 0xD7, 0x50, 0x81, 0xF5, 0x92, 0x30, 0x5F, 0x2E, 0x45, 0x26, 0xED, 0xC0, 0x96,
		0x31, 0xB1, 0x09, 0x58, 0xF4, 0x64, 0xD8, 0x89, 0xF3, 0x1B, 0xA0, 0x10, 0x25, 0x0F, 0xDA, 0x7F,
		0x13, 0x68, 0xEC, 0x29, 0x67, 0xFC, 0x84, 0xEF, 0x2A, 0xE9, 0xAF, 0xF2, 0x68, 0xE0, 0xB1, 0x70,
		0x0A, 0xFF, 0xC6, 0x82, 0x0B, 0x52, 0x3A, 0x3D, 0x91, 0x71, 0x35, 0xF2, 0xDF, 0xF2, 0xEE, 0x06,
		0xBF, 0xE7, 0x2B, 0x31, 0x24, 0x72, 0x1D, 0x4A, 0x26, 0xC0, 0x4E, 0x53, 0xA7, 0x5E, 0x30, 0xE7,
		0x3A, 0x7A, 0x9C, 0x4A, 0x95, 0xD9, 0x1C, 0x55, 0xD4, 0x95, 0xE9, 0xF5, 0x1D, 0xD0, 0xB5, 0xE9,
		0xD8, 0x3C, 0x6D, 0x5E, 0x8C, 0xE8, 0x03, 0xAA, 0x62, 0xB8, 0xD6, 0x54, 0xDB, 0x53, 0xD0, 0x9B,
		0x8D, 0xCF, 0xF2, 0x73, 0xCD, 0xFE, 0xB5, 0x73, 0xFA, 0xD8, 0xBC, 0xD4, 0x55, 0x78, 0xBE, 0xC2,
		0xE7, 0x70, 0xD0, 0x1E, 0xFD, 0xE8, 0x6E, 0x72, 0x1A, 0x3F, 0x7C, 0x6C, 0xCE, 0x27, 0x5D, 0xAB,
		0xE6, 0xE2, 0x14, 0x3F, 0x1A, 0xF1, 0x8D, 0xA7, 0xEF, 0xDD, 0xC4, 0xC7, 0xB7, 0x0B, 0x5E, 0x34,
		0x5D, 0xB9, 0x3C, 0xC9, 0x36, 0xBE, 0xA3, 0x23, 0x49, 0x1C, 0xCB, 0x38, 0xA3, 0x88, 0xF5, 0x46,
		0xA9, 0xFF, 0x00, 0xDD, 0x4E, 0x13, 0x00, 0xB9, 0xB2, 0x15, 0x3D, 0x20, 0x41, 0xD2, 0x05, 0xB4,
		0x43, 0xE4, 0x1B, 0x45, 0xA6, 0x53, 0xF2, 0xA5, 0xC4, 0x49, 0x2C, 0x1A, 0xDD, 0x54, 0x45, 0x12,
		0xDD, 0xA2, 0x52, 0x98, 0x33, 0x46, 0x2B, 0x71, 0xA4, 0x1A, 0x45, 0xBE, 0x97, 0x29, 0x0B, 0x6F
	};
	uint8_t exp1600[512] = {
		0x13, 0x1A, 0xB8, 0xD2, 0xB5, 0x94, 0x94, 0x6B, 0x9C, 0x81, 0x33, 0x3F, 0x9B, 0xB6, 0xE0, 0xCE,
		0x75, 0xC3, 0xB9, 0x31, 0x04, 0xFA, 0x34, 0x69, 0xD3, 0x91, 0x74, 0x57, 0x38, 0x5D, 0xA0, 0x37,
		0xCF, 0x23, 0x2E, 0xF7, 0x16, 0x4A, 0x6D, 0x1E, 0xB4, 0x48, 0xC8, 0x90, 0x81, 0x86, 0xAD, 0x85,
		0x2D, 0x3F, 0x85, 0xA5, 0xCF, 0x28, 0xDA, 0x1A, 0xB6, 0xFE, 0x34, 0x38, 0x17, 0x19, 0x78, 0x46,
		0x7F, 0x1C, 0x05, 0xD5, 0x8C, 0x7E, 0xF3, 0x8C, 0x28, 0x4C, 0x41, 0xF6, 0xC2, 0x22, 0x1A, 0x76,
		0xF1, 0x2A, 0xB1, 0xC0, 0x40, 0x82, 0x66, 0x02, 0x50, 0x80, 0x22, 0x94, 0xFB, 0x87, 0x18, 0x02,
		0x13, 0xFD, 0xEF, 0x5B, 0x0E, 0xCB, 0x7D, 0xF5, 0x0C, 0xA1, 0xF8, 0x55, 0x5B, 0xE1, 0x4D, 0x32,
		0xE1, 0x0F, 0x6E, 0xDC, 0xDE, 0x89, 0x2C, 0x09, 0x42, 0x4B, 0x29, 0xF5, 0x97, 0xAF, 0xC2, 0x70,
		0xC9, 0x04, 0x55, 0x6B, 0xFC, 0xB4, 0x7A, 0x7D, 0x40, 0x77, 0x8D, 0x39, 0x09, 0x23, 0x64, 0x2B,
		0x3C, 0xBD, 0x05, 0x79, 0xE6, 0x09, 0x08, 0xD5, 0xA0, 0x00, 0xC1, 0xD0, 0x8B, 0x98, 0xEF, 0x93,
		0x3F, 0x80, 0x64, 0x45, 0xBF, 0x87, 0xF8, 0xB0, 0x09, 0xBA, 0x9E, 0x94, 0xF7, 0x26, 0x61, 0x22,
		0xED, 0x7A, 0xC2, 0x4E, 0x5E, 0x26, 0x6C, 0x42, 0xA8, 0x2F, 0xA1, 0xBB, 0xEF, 0xB7, 0xB8, 0xDB,
		0x00, 0x66, 0xE1, 0x6A, 0x85, 0xE0, 0x49, 0x3F, 0x07, 0xDF, 0x48, 0x09, 0xAE, 0xC0, 0x84, 0xA5,
		0x93, 0x74, 0x8A, 0xC3, 0xDD, 0xE5, 0xA6, 0xD7, 0xAA, 0xE1, 0xE8, 0xB6, 0xE5, 0x35, 0x2B, 0x2D,
		0x71, 0xEF, 0xBB, 0x47, 0xD4, 0xCA, 0xEE, 0xD5, 0xE6, 0xD6, 0x33, 0x80, 0x5D, 0x2D, 0x32, 0x3E,
		0x6F, 0xD8, 0x1B, 0x46, 0x84, 0xB9, 0x3A, 0x26, 0x77, 0xD4, 0x5E, 0x74, 0x21, 0xC2, 0xC6, 0xAE,
		0xA2, 0x59, 0xB8, 0x55, 0xA6, 0x98, 0xFD, 0x7D, 0x13, 0x47, 0x7A, 0x1F, 0xE5, 0x3E, 0x5A, 0x4A,
		0x61, 0x97, 0xDB, 0xEC, 0x5C, 0xE9, 0x5F, 0x50, 0x5B, 0x52, 0x0B, 0xCD, 0x95, 0x70, 0xC4, 0xA8,
		0x26, 0x5A, 0x7E, 0x01, 0xF8, 0x9C, 0x0C, 0x00, 0x2C, 0x59, 0xBF, 0xEC, 0x6C, 0xD4, 0xA5, 0xC1,
		0x09, 0x25, 0x89, 0x53, 0xEE, 0x5E, 0xE7, 0x0C, 0xD5, 0x77, 0xEE, 0x21, 0x7A, 0xF2, 0x1F, 0xA7,
		0x01, 0x78, 0xF0, 0x94, 0x6C, 0x9B, 0xF6, 0xCA, 0x87, 0x51, 0x79, 0x34, 0x79, 0xF6, 0xB5, 0x37,
		0x73, 0x7E, 0x40, 0xB6, 0xED, 0x28, 0x51, 0x1D, 0x8A, 0x2D, 0x7E, 0x73, 0xEB, 0x75, 0xF8, 0xDA,
		0xAC, 0x91, 0x2F, 0xF9, 0x06, 0xE0, 0xAB, 0x95, 0x5B, 0x08, 0x3B, 0xAC, 0x45, 0xA8, 0xE5, 0xE9,
		0xB7, 0x44, 0xC8, 0x50, 0x6F, 0x37, 0xE9, 0xB4, 0xE7, 0x49, 0xA1, 0x84, 0xB3, 0x0F, 0x43, 0xEB,
		0x18, 0x8D, 0x85, 0x5F, 0x1B, 0x70, 0xD7, 0x1F, 0xF3, 0xE5, 0x0C, 0x53, 0x7A, 0xC1, 0xB0, 0xF8,
		0x97, 0x4F, 0x0F, 0xE1, 0xA6, 0xAD, 0x29, 0x5B, 0xA4, 0x2F, 0x6A, 0xEC, 0x74, 0xD1, 0x23, 0xA7,
		0xAB, 0xED, 0xDE, 0x6E, 0x2C, 0x07, 0x11, 0xCA, 0xB3, 0x6B, 0xE5, 0xAC, 0xB1, 0xA5, 0xA1, 0x1A,
		0x4B, 0x1D, 0xB0, 0x8B, 0xA6, 0x98, 0x2E, 0xFC, 0xCD, 0x71, 0x69, 0x29, 0xA7, 0x74, 0x1C, 0xFC,
		0x63, 0xAA, 0x44, 0x35, 0xE0, 0xB6, 0x9A, 0x90, 0x63, 0xE8, 0x80, 0x79, 0x5C, 0x3D, 0xC5, 0xEF,
		0x32, 0x72, 0xE1, 0x1C, 0x49, 0x7A, 0x91, 0xAC, 0xF6, 0x99, 0xFE, 0xFE, 0xE2, 0x06, 0x22, 0x7A,
		0x44, 0xC9, 0xFB, 0x35, 0x9F, 0xD5, 0x6A, 0xC0, 0xA9, 0xA7, 0x5A, 0x74, 0x3C, 0xFF, 0x68, 0x62,
		0xF1, 0x7D, 0x72, 0x59, 0xAB, 0x07, 0x52, 0x16, 0xC0, 0x69, 0x95, 0x11, 0x64, 0x3B, 0x64, 0x39
	};

	/* test compact api */

	clear8(output, 512);
	OQS_SHA3_shake128(output, 512, msg0, 0);

	if (are_equal8(output, exp0, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(output, 512);
	OQS_SHA3_shake128(output, 512, msg1600, 200);

	if (are_equal8(output, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* test long-form api */

	OQS_SHA3_shake128_inc_ctx state;
	uint8_t hash[OQS_SHA3_SHAKE128_RATE * 4];

	clear8(hash, sizeof(hash));
	OQS_SHA3_shake128_inc_init(&state);
	OQS_SHA3_shake128_inc_absorb(&state, msg1600, sizeof(msg1600));
	OQS_SHA3_shake128_inc_finalize(&state);
	OQS_SHA3_shake128_inc_squeeze(hash, sizeof(hash), &state);
	OQS_SHA3_shake128_inc_ctx_release(&state);

	if (are_equal8(hash, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* Test small absorb calls */
	clear8(hash, sizeof(hash));
	OQS_SHA3_shake128_inc_init(&state);
	for (size_t i = 0; i < sizeof(msg1600); i += 8) {
		OQS_SHA3_shake128_inc_absorb(&state, msg1600 + i, 8);
	}
	OQS_SHA3_shake128_inc_finalize(&state);
	OQS_SHA3_shake128_inc_squeeze(hash, sizeof(hash), &state);
	OQS_SHA3_shake128_inc_ctx_release(&state);

	if (are_equal8(hash, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* Test small small squeeze calls */
	clear8(hash, sizeof(hash));
	OQS_SHA3_shake128_inc_init(&state);
	OQS_SHA3_shake128_inc_absorb(&state, msg1600, sizeof(msg1600));
	OQS_SHA3_shake128_inc_finalize(&state);
	for (size_t i = 0; i < sizeof(hash); i += 8) {
		OQS_SHA3_shake128_inc_squeeze(hash + i, 8, &state);
	}
	OQS_SHA3_shake128_inc_ctx_release(&state);

	if (are_equal8(hash, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* Test clone */
	OQS_SHA3_shake128_inc_ctx state2;

	clear8(hash, sizeof(hash));
	OQS_SHA3_shake128_inc_init(&state);
	OQS_SHA3_shake128_inc_absorb(&state, msg1600, sizeof(msg1600));
	OQS_SHA3_shake128_inc_finalize(&state);

	OQS_SHA3_shake128_inc_init(&state2);
	OQS_SHA3_shake128_inc_ctx_clone(&state2, &state);
	OQS_SHA3_shake128_inc_squeeze(hash, sizeof(hash), &state2);
	OQS_SHA3_shake128_inc_ctx_release(&state2);

	if (are_equal8(hash, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* Test reset */
	clear8(hash, sizeof(hash));
	OQS_SHA3_shake128_inc_ctx_reset(&state);
	OQS_SHA3_shake128_inc_absorb(&state, msg1600, sizeof(msg1600));
	OQS_SHA3_shake128_inc_finalize(&state);
	OQS_SHA3_shake128_inc_squeeze(hash, sizeof(hash), &state);
	OQS_SHA3_shake128_inc_ctx_release(&state);

	if (are_equal8(hash, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}
	return status;
}

/**
* \brief Tests the 256 bit version of the SHAKE XOF function for correct operation,
* using selected vectors from the NIST reference package.
*
* \return Returns 1 for success, 0 for failure
*
* \remarks <b>Test References:</b> \n
* SP800-185: <a href="http://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-185.pdf">SHA-3 Derived Functions</a> \n
* KAT: <a href="https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHAKE256_Msg0.pdf">SHAKE256(0)</a> \n
* KAT: <a href="https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHAKE256_Msg1600.pdf">SHAKE256(1600)</a>
*/
int shake_256_kat_test(void) {
	int status = EXIT_SUCCESS;

	uint8_t output[512];

	uint8_t msg0[1] = {0x0};
	uint8_t msg1600[200] = {
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3
	};

	uint8_t exp0[512] = {
		0x46, 0xB9, 0xDD, 0x2B, 0x0B, 0xA8, 0x8D, 0x13, 0x23, 0x3B, 0x3F, 0xEB, 0x74, 0x3E, 0xEB, 0x24,
		0x3F, 0xCD, 0x52, 0xEA, 0x62, 0xB8, 0x1B, 0x82, 0xB5, 0x0C, 0x27, 0x64, 0x6E, 0xD5, 0x76, 0x2F,
		0xD7, 0x5D, 0xC4, 0xDD, 0xD8, 0xC0, 0xF2, 0x00, 0xCB, 0x05, 0x01, 0x9D, 0x67, 0xB5, 0x92, 0xF6,
		0xFC, 0x82, 0x1C, 0x49, 0x47, 0x9A, 0xB4, 0x86, 0x40, 0x29, 0x2E, 0xAC, 0xB3, 0xB7, 0xC4, 0xBE,
		0x14, 0x1E, 0x96, 0x61, 0x6F, 0xB1, 0x39, 0x57, 0x69, 0x2C, 0xC7, 0xED, 0xD0, 0xB4, 0x5A, 0xE3,
		0xDC, 0x07, 0x22, 0x3C, 0x8E, 0x92, 0x93, 0x7B, 0xEF, 0x84, 0xBC, 0x0E, 0xAB, 0x86, 0x28, 0x53,
		0x34, 0x9E, 0xC7, 0x55, 0x46, 0xF5, 0x8F, 0xB7, 0xC2, 0x77, 0x5C, 0x38, 0x46, 0x2C, 0x50, 0x10,
		0xD8, 0x46, 0xC1, 0x85, 0xC1, 0x51, 0x11, 0xE5, 0x95, 0x52, 0x2A, 0x6B, 0xCD, 0x16, 0xCF, 0x86,
		0xF3, 0xD1, 0x22, 0x10, 0x9E, 0x3B, 0x1F, 0xDD, 0x94, 0x3B, 0x6A, 0xEC, 0x46, 0x8A, 0x2D, 0x62,
		0x1A, 0x7C, 0x06, 0xC6, 0xA9, 0x57, 0xC6, 0x2B, 0x54, 0xDA, 0xFC, 0x3B, 0xE8, 0x75, 0x67, 0xD6,
		0x77, 0x23, 0x13, 0x95, 0xF6, 0x14, 0x72, 0x93, 0xB6, 0x8C, 0xEA, 0xB7, 0xA9, 0xE0, 0xC5, 0x8D,
		0x86, 0x4E, 0x8E, 0xFD, 0xE4, 0xE1, 0xB9, 0xA4, 0x6C, 0xBE, 0x85, 0x47, 0x13, 0x67, 0x2F, 0x5C,
		0xAA, 0xAE, 0x31, 0x4E, 0xD9, 0x08, 0x3D, 0xAB, 0x4B, 0x09, 0x9F, 0x8E, 0x30, 0x0F, 0x01, 0xB8,
		0x65, 0x0F, 0x1F, 0x4B, 0x1D, 0x8F, 0xCF, 0x3F, 0x3C, 0xB5, 0x3F, 0xB8, 0xE9, 0xEB, 0x2E, 0xA2,
		0x03, 0xBD, 0xC9, 0x70, 0xF5, 0x0A, 0xE5, 0x54, 0x28, 0xA9, 0x1F, 0x7F, 0x53, 0xAC, 0x26, 0x6B,
		0x28, 0x41, 0x9C, 0x37, 0x78, 0xA1, 0x5F, 0xD2, 0x48, 0xD3, 0x39, 0xED, 0xE7, 0x85, 0xFB, 0x7F,
		0x5A, 0x1A, 0xAA, 0x96, 0xD3, 0x13, 0xEA, 0xCC, 0x89, 0x09, 0x36, 0xC1, 0x73, 0xCD, 0xCD, 0x0F,
		0xAB, 0x88, 0x2C, 0x45, 0x75, 0x5F, 0xEB, 0x3A, 0xED, 0x96, 0xD4, 0x77, 0xFF, 0x96, 0x39, 0x0B,
		0xF9, 0xA6, 0x6D, 0x13, 0x68, 0xB2, 0x08, 0xE2, 0x1F, 0x7C, 0x10, 0xD0, 0x4A, 0x3D, 0xBD, 0x4E,
		0x36, 0x06, 0x33, 0xE5, 0xDB, 0x4B, 0x60, 0x26, 0x01, 0xC1, 0x4C, 0xEA, 0x73, 0x7D, 0xB3, 0xDC,
		0xF7, 0x22, 0x63, 0x2C, 0xC7, 0x78, 0x51, 0xCB, 0xDD, 0xE2, 0xAA, 0xF0, 0xA3, 0x3A, 0x07, 0xB3,
		0x73, 0x44, 0x5D, 0xF4, 0x90, 0xCC, 0x8F, 0xC1, 0xE4, 0x16, 0x0F, 0xF1, 0x18, 0x37, 0x8F, 0x11,
		0xF0, 0x47, 0x7D, 0xE0, 0x55, 0xA8, 0x1A, 0x9E, 0xDA, 0x57, 0xA4, 0xA2, 0xCF, 0xB0, 0xC8, 0x39,
		0x29, 0xD3, 0x10, 0x91, 0x2F, 0x72, 0x9E, 0xC6, 0xCF, 0xA3, 0x6C, 0x6A, 0xC6, 0xA7, 0x58, 0x37,
		0x14, 0x30, 0x45, 0xD7, 0x91, 0xCC, 0x85, 0xEF, 0xF5, 0xB2, 0x19, 0x32, 0xF2, 0x38, 0x61, 0xBC,
		0xF2, 0x3A, 0x52, 0xB5, 0xDA, 0x67, 0xEA, 0xF7, 0xBA, 0xAE, 0x0F, 0x5F, 0xB1, 0x36, 0x9D, 0xB7,
		0x8F, 0x3A, 0xC4, 0x5F, 0x8C, 0x4A, 0xC5, 0x67, 0x1D, 0x85, 0x73, 0x5C, 0xDD, 0xDB, 0x09, 0xD2,
		0xB1, 0xE3, 0x4A, 0x1F, 0xC0, 0x66, 0xFF, 0x4A, 0x16, 0x2C, 0xB2, 0x63, 0xD6, 0x54, 0x12, 0x74,
		0xAE, 0x2F, 0xCC, 0x86, 0x5F, 0x61, 0x8A, 0xBE, 0x27, 0xC1, 0x24, 0xCD, 0x8B, 0x07, 0x4C, 0xCD,
		0x51, 0x63, 0x01, 0xB9, 0x18, 0x75, 0x82, 0x4D, 0x09, 0x95, 0x8F, 0x34, 0x1E, 0xF2, 0x74, 0xBD,
		0xAB, 0x0B, 0xAE, 0x31, 0x63, 0x39, 0x89, 0x43, 0x04, 0xE3, 0x58, 0x77, 0xB0, 0xC2, 0x8A, 0x9B,
		0x1F, 0xD1, 0x66, 0xC7, 0x96, 0xB9, 0xCC, 0x25, 0x8A, 0x06, 0x4A, 0x8F, 0x57, 0xE2, 0x7F, 0x2A
	};
	uint8_t exp1600[512] = {
		0xCD, 0x8A, 0x92, 0x0E, 0xD1, 0x41, 0xAA, 0x04, 0x07, 0xA2, 0x2D, 0x59, 0x28, 0x86, 0x52, 0xE9,
		0xD9, 0xF1, 0xA7, 0xEE, 0x0C, 0x1E, 0x7C, 0x1C, 0xA6, 0x99, 0x42, 0x4D, 0xA8, 0x4A, 0x90, 0x4D,
		0x2D, 0x70, 0x0C, 0xAA, 0xE7, 0x39, 0x6E, 0xCE, 0x96, 0x60, 0x44, 0x40, 0x57, 0x7D, 0xA4, 0xF3,
		0xAA, 0x22, 0xAE, 0xB8, 0x85, 0x7F, 0x96, 0x1C, 0x4C, 0xD8, 0xE0, 0x6F, 0x0A, 0xE6, 0x61, 0x0B,
		0x10, 0x48, 0xA7, 0xF6, 0x4E, 0x10, 0x74, 0xCD, 0x62, 0x9E, 0x85, 0xAD, 0x75, 0x66, 0x04, 0x8E,
		0xFC, 0x4F, 0xB5, 0x00, 0xB4, 0x86, 0xA3, 0x30, 0x9A, 0x8F, 0x26, 0x72, 0x4C, 0x0E, 0xD6, 0x28,
		0x00, 0x1A, 0x10, 0x99, 0x42, 0x24, 0x68, 0xDE, 0x72, 0x6F, 0x10, 0x61, 0xD9, 0x9E, 0xB9, 0xE9,
		0x36, 0x04, 0xD5, 0xAA, 0x74, 0x67, 0xD4, 0xB1, 0xBD, 0x64, 0x84, 0x58, 0x2A, 0x38, 0x43, 0x17,
		0xD7, 0xF4, 0x7D, 0x75, 0x0B, 0x8F, 0x54, 0x99, 0x51, 0x2B, 0xB8, 0x5A, 0x22, 0x6C, 0x42, 0x43,
		0x55, 0x6E, 0x69, 0x6F, 0x6B, 0xD0, 0x72, 0xC5, 0xAA, 0x2D, 0x9B, 0x69, 0x73, 0x02, 0x44, 0xB5,
		0x68, 0x53, 0xD1, 0x69, 0x70, 0xAD, 0x81, 0x7E, 0x21, 0x3E, 0x47, 0x06, 0x18, 0x17, 0x80, 0x01,
		0xC9, 0xFB, 0x56, 0xC5, 0x4F, 0xEF, 0xA5, 0xFE, 0xE6, 0x7D, 0x2D, 0xA5, 0x24, 0xBB, 0x3B, 0x0B,
		0x61, 0xEF, 0x0E, 0x91, 0x14, 0xA9, 0x2C, 0xDB, 0xB6, 0xCC, 0xCB, 0x98, 0x61, 0x5C, 0xFE, 0x76,
		0xE3, 0x51, 0x0D, 0xD8, 0x8D, 0x1C, 0xC2, 0x8F, 0xF9, 0x92, 0x87, 0x51, 0x2F, 0x24, 0xBF, 0xAF,
		0xA1, 0xA7, 0x68, 0x77, 0xB6, 0xF3, 0x71, 0x98, 0xE3, 0xA6, 0x41, 0xC6, 0x8A, 0x7C, 0x42, 0xD4,
		0x5F, 0xA7, 0xAC, 0xC1, 0x0D, 0xAE, 0x5F, 0x3C, 0xEF, 0xB7, 0xB7, 0x35, 0xF1, 0x2D, 0x4E, 0x58,
		0x9F, 0x7A, 0x45, 0x6E, 0x78, 0xC0, 0xF5, 0xE4, 0xC4, 0x47, 0x1F, 0xFF, 0xA5, 0xE4, 0xFA, 0x05,
		0x14, 0xAE, 0x97, 0x4D, 0x8C, 0x26, 0x48, 0x51, 0x3B, 0x5D, 0xB4, 0x94, 0xCE, 0xA8, 0x47, 0x15,
		0x6D, 0x27, 0x7A, 0xD0, 0xE1, 0x41, 0xC2, 0x4C, 0x78, 0x39, 0x06, 0x4C, 0xD0, 0x88, 0x51, 0xBC,
		0x2E, 0x7C, 0xA1, 0x09, 0xFD, 0x4E, 0x25, 0x1C, 0x35, 0xBB, 0x0A, 0x04, 0xFB, 0x05, 0xB3, 0x64,
		0xFF, 0x8C, 0x4D, 0x8B, 0x59, 0xBC, 0x30, 0x3E, 0x25, 0x32, 0x8C, 0x09, 0xA8, 0x82, 0xE9, 0x52,
		0x51, 0x8E, 0x1A, 0x8A, 0xE0, 0xFF, 0x26, 0x5D, 0x61, 0xC4, 0x65, 0x89, 0x69, 0x73, 0xD7, 0x49,
		0x04, 0x99, 0xDC, 0x63, 0x9F, 0xB8, 0x50, 0x2B, 0x39, 0x45, 0x67, 0x91, 0xB1, 0xB6, 0xEC, 0x5B,
		0xCC, 0x5D, 0x9A, 0xC3, 0x6A, 0x6D, 0xF6, 0x22, 0xA0, 0x70, 0xD4, 0x3F, 0xED, 0x78, 0x1F, 0x5F,
		0x14, 0x9F, 0x7B, 0x62, 0x67, 0x5E, 0x7D, 0x1A, 0x4D, 0x6D, 0xEC, 0x48, 0xC1, 0xC7, 0x16, 0x45,
		0x86, 0xEA, 0xE0, 0x6A, 0x51, 0x20, 0x8C, 0x0B, 0x79, 0x12, 0x44, 0xD3, 0x07, 0x72, 0x65, 0x05,
		0xC3, 0xAD, 0x4B, 0x26, 0xB6, 0x82, 0x23, 0x77, 0x25, 0x7A, 0xA1, 0x52, 0x03, 0x75, 0x60, 0xA7,
		0x39, 0x71, 0x4A, 0x3C, 0xA7, 0x9B, 0xD6, 0x05, 0x54, 0x7C, 0x9B, 0x78, 0xDD, 0x1F, 0x59, 0x6F,
		0x2D, 0x4F, 0x17, 0x91, 0xBC, 0x68, 0x9A, 0x0E, 0x9B, 0x79, 0x9A, 0x37, 0x33, 0x9C, 0x04, 0x27,
		0x57, 0x33, 0x74, 0x01, 0x43, 0xEF, 0x5D, 0x2B, 0x58, 0xB9, 0x6A, 0x36, 0x3D, 0x4E, 0x08, 0x07,
		0x6A, 0x1A, 0x9D, 0x78, 0x46, 0x43, 0x6E, 0x4D, 0xCA, 0x57, 0x28, 0xB6, 0xF7, 0x60, 0xEE, 0xF0,
		0xCA, 0x92, 0xBF, 0x0B, 0xE5, 0x61, 0x5E, 0x96, 0x95, 0x9D, 0x76, 0x71, 0x97, 0xA0, 0xBE, 0xEB
	};

	/* test compact api */

	clear8(output, 512);
	OQS_SHA3_shake256(output, 512, msg0, 0);

	if (are_equal8(output, exp0, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(output, 512);
	OQS_SHA3_shake256(output, 512, msg1600, 200);

	if (are_equal8(output, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* test long-form api */

	OQS_SHA3_shake256_inc_ctx state;
	uint8_t hash[OQS_SHA3_SHAKE256_RATE * 4];

	clear8(hash, sizeof(hash));
	OQS_SHA3_shake256_inc_init(&state);
	OQS_SHA3_shake256_inc_absorb(&state, msg1600, sizeof(msg1600));
	OQS_SHA3_shake256_inc_finalize(&state);
	OQS_SHA3_shake256_inc_squeeze(hash, sizeof(hash), &state);
	OQS_SHA3_shake256_inc_ctx_release(&state);

	if (are_equal8(hash, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* Test small absorb calls */
	clear8(hash, sizeof(hash));
	OQS_SHA3_shake256_inc_init(&state);
	for (size_t i = 0; i < sizeof(msg1600); i += 8) {
		OQS_SHA3_shake256_inc_absorb(&state, msg1600 + i, 8);
	}
	OQS_SHA3_shake256_inc_finalize(&state);
	OQS_SHA3_shake256_inc_squeeze(hash, sizeof(hash), &state);
	OQS_SHA3_shake256_inc_ctx_release(&state);

	if (are_equal8(hash, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* Test small small squeeze calls */
	clear8(hash, sizeof(hash));
	OQS_SHA3_shake256_inc_init(&state);
	OQS_SHA3_shake256_inc_absorb(&state, msg1600, sizeof(msg1600));
	OQS_SHA3_shake256_inc_finalize(&state);
	for (size_t i = 0; i < sizeof(hash); i += 8) {
		OQS_SHA3_shake256_inc_squeeze(hash + i, 8, &state);
	}
	OQS_SHA3_shake256_inc_ctx_release(&state);

	if (are_equal8(hash, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* Test clone */
	OQS_SHA3_shake256_inc_ctx state2;

	clear8(hash, sizeof(hash));
	OQS_SHA3_shake256_inc_init(&state);
	OQS_SHA3_shake256_inc_absorb(&state, msg1600, sizeof(msg1600));
	OQS_SHA3_shake256_inc_finalize(&state);

	OQS_SHA3_shake256_inc_init(&state2);
	OQS_SHA3_shake256_inc_ctx_clone(&state2, &state);
	OQS_SHA3_shake256_inc_squeeze(hash, sizeof(hash), &state2);
	OQS_SHA3_shake256_inc_ctx_release(&state2);

	if (are_equal8(hash, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* Test reset */
	clear8(hash, sizeof(hash));
	OQS_SHA3_shake256_inc_ctx_reset(&state);
	OQS_SHA3_shake256_inc_absorb(&state, msg1600, sizeof(msg1600));
	OQS_SHA3_shake256_inc_finalize(&state);
	OQS_SHA3_shake256_inc_squeeze(hash, sizeof(hash), &state);
	OQS_SHA3_shake256_inc_ctx_release(&state);

	if (are_equal8(hash, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}
	return status;
}

/**
* \brief Tests the four way parallel version of the SHAKE XOF function for correct operation,
* using selected vectors from the NIST reference package.
*
* \return Returns 1 for success, 0 for failure
*
* \remarks <b>Test References:</b> \n
* KAT: <a href="https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHAKE128_Msg0.pdf">SHAKE128(0)</a> \n
* KAT: <a href="https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHAKE128_Msg1600.pdf">SHAKE128(1600)</a>
*/
int shake_128_x4_kat_test(void) {
	int status = EXIT_SUCCESS;

	uint8_t output0[512];
	uint8_t output1[512];
	uint8_t output2[512];
	uint8_t output3[512];

	uint8_t msg0[1] = {0x0};
	uint8_t msg1600[200] = {
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3
	};

	uint8_t exp0[512] = {
		0x7F, 0x9C, 0x2B, 0xA4, 0xE8, 0x8F, 0x82, 0x7D, 0x61, 0x60, 0x45, 0x50, 0x76, 0x05, 0x85, 0x3E,
		0xD7, 0x3B, 0x80, 0x93, 0xF6, 0xEF, 0xBC, 0x88, 0xEB, 0x1A, 0x6E, 0xAC, 0xFA, 0x66, 0xEF, 0x26,
		0x3C, 0xB1, 0xEE, 0xA9, 0x88, 0x00, 0x4B, 0x93, 0x10, 0x3C, 0xFB, 0x0A, 0xEE, 0xFD, 0x2A, 0x68,
		0x6E, 0x01, 0xFA, 0x4A, 0x58, 0xE8, 0xA3, 0x63, 0x9C, 0xA8, 0xA1, 0xE3, 0xF9, 0xAE, 0x57, 0xE2,
		0x35, 0xB8, 0xCC, 0x87, 0x3C, 0x23, 0xDC, 0x62, 0xB8, 0xD2, 0x60, 0x16, 0x9A, 0xFA, 0x2F, 0x75,
		0xAB, 0x91, 0x6A, 0x58, 0xD9, 0x74, 0x91, 0x88, 0x35, 0xD2, 0x5E, 0x6A, 0x43, 0x50, 0x85, 0xB2,
		0xBA, 0xDF, 0xD6, 0xDF, 0xAA, 0xC3, 0x59, 0xA5, 0xEF, 0xBB, 0x7B, 0xCC, 0x4B, 0x59, 0xD5, 0x38,
		0xDF, 0x9A, 0x04, 0x30, 0x2E, 0x10, 0xC8, 0xBC, 0x1C, 0xBF, 0x1A, 0x0B, 0x3A, 0x51, 0x20, 0xEA,
		0x17, 0xCD, 0xA7, 0xCF, 0xAD, 0x76, 0x5F, 0x56, 0x23, 0x47, 0x4D, 0x36, 0x8C, 0xCC, 0xA8, 0xAF,
		0x00, 0x07, 0xCD, 0x9F, 0x5E, 0x4C, 0x84, 0x9F, 0x16, 0x7A, 0x58, 0x0B, 0x14, 0xAA, 0xBD, 0xEF,
		0xAE, 0xE7, 0xEE, 0xF4, 0x7C, 0xB0, 0xFC, 0xA9, 0x76, 0x7B, 0xE1, 0xFD, 0xA6, 0x94, 0x19, 0xDF,
		0xB9, 0x27, 0xE9, 0xDF, 0x07, 0x34, 0x8B, 0x19, 0x66, 0x91, 0xAB, 0xAE, 0xB5, 0x80, 0xB3, 0x2D,
		0xEF, 0x58, 0x53, 0x8B, 0x8D, 0x23, 0xF8, 0x77, 0x32, 0xEA, 0x63, 0xB0, 0x2B, 0x4F, 0xA0, 0xF4,
		0x87, 0x33, 0x60, 0xE2, 0x84, 0x19, 0x28, 0xCD, 0x60, 0xDD, 0x4C, 0xEE, 0x8C, 0xC0, 0xD4, 0xC9,
		0x22, 0xA9, 0x61, 0x88, 0xD0, 0x32, 0x67, 0x5C, 0x8A, 0xC8, 0x50, 0x93, 0x3C, 0x7A, 0xFF, 0x15,
		0x33, 0xB9, 0x4C, 0x83, 0x4A, 0xDB, 0xB6, 0x9C, 0x61, 0x15, 0xBA, 0xD4, 0x69, 0x2D, 0x86, 0x19,
		0xF9, 0x0B, 0x0C, 0xDF, 0x8A, 0x7B, 0x9C, 0x26, 0x40, 0x29, 0xAC, 0x18, 0x5B, 0x70, 0xB8, 0x3F,
		0x28, 0x01, 0xF2, 0xF4, 0xB3, 0xF7, 0x0C, 0x59, 0x3E, 0xA3, 0xAE, 0xEB, 0x61, 0x3A, 0x7F, 0x1B,
		0x1D, 0xE3, 0x3F, 0xD7, 0x50, 0x81, 0xF5, 0x92, 0x30, 0x5F, 0x2E, 0x45, 0x26, 0xED, 0xC0, 0x96,
		0x31, 0xB1, 0x09, 0x58, 0xF4, 0x64, 0xD8, 0x89, 0xF3, 0x1B, 0xA0, 0x10, 0x25, 0x0F, 0xDA, 0x7F,
		0x13, 0x68, 0xEC, 0x29, 0x67, 0xFC, 0x84, 0xEF, 0x2A, 0xE9, 0xAF, 0xF2, 0x68, 0xE0, 0xB1, 0x70,
		0x0A, 0xFF, 0xC6, 0x82, 0x0B, 0x52, 0x3A, 0x3D, 0x91, 0x71, 0x35, 0xF2, 0xDF, 0xF2, 0xEE, 0x06,
		0xBF, 0xE7, 0x2B, 0x31, 0x24, 0x72, 0x1D, 0x4A, 0x26, 0xC0, 0x4E, 0x53, 0xA7, 0x5E, 0x30, 0xE7,
		0x3A, 0x7A, 0x9C, 0x4A, 0x95, 0xD9, 0x1C, 0x55, 0xD4, 0x95, 0xE9, 0xF5, 0x1D, 0xD0, 0xB5, 0xE9,
		0xD8, 0x3C, 0x6D, 0x5E, 0x8C, 0xE8, 0x03, 0xAA, 0x62, 0xB8, 0xD6, 0x54, 0xDB, 0x53, 0xD0, 0x9B,
		0x8D, 0xCF, 0xF2, 0x73, 0xCD, 0xFE, 0xB5, 0x73, 0xFA, 0xD8, 0xBC, 0xD4, 0x55, 0x78, 0xBE, 0xC2,
		0xE7, 0x70, 0xD0, 0x1E, 0xFD, 0xE8, 0x6E, 0x72, 0x1A, 0x3F, 0x7C, 0x6C, 0xCE, 0x27, 0x5D, 0xAB,
		0xE6, 0xE2, 0x14, 0x3F, 0x1A, 0xF1, 0x8D, 0xA7, 0xEF, 0xDD, 0xC4, 0xC7, 0xB7, 0x0B, 0x5E, 0x34,
		0x5D, 0xB9, 0x3C, 0xC9, 0x36, 0xBE, 0xA3, 0x23, 0x49, 0x1C, 0xCB, 0x38, 0xA3, 0x88, 0xF5, 0x46,
		0xA9, 0xFF, 0x00, 0xDD, 0x4E, 0x13, 0x00, 0xB9, 0xB2, 0x15, 0x3D, 0x20, 0x41, 0xD2, 0x05, 0xB4,
		0x43, 0xE4, 0x1B, 0x45, 0xA6, 0x53, 0xF2, 0xA5, 0xC4, 0x49, 0x2C, 0x1A, 0xDD, 0x54, 0x45, 0x12,
		0xDD, 0xA2, 0x52, 0x98, 0x33, 0x46, 0x2B, 0x71, 0xA4, 0x1A, 0x45, 0xBE, 0x97, 0x29, 0x0B, 0x6F
	};
	uint8_t exp1600[512] = {
		0x13, 0x1A, 0xB8, 0xD2, 0xB5, 0x94, 0x94, 0x6B, 0x9C, 0x81, 0x33, 0x3F, 0x9B, 0xB6, 0xE0, 0xCE,
		0x75, 0xC3, 0xB9, 0x31, 0x04, 0xFA, 0x34, 0x69, 0xD3, 0x91, 0x74, 0x57, 0x38, 0x5D, 0xA0, 0x37,
		0xCF, 0x23, 0x2E, 0xF7, 0x16, 0x4A, 0x6D, 0x1E, 0xB4, 0x48, 0xC8, 0x90, 0x81, 0x86, 0xAD, 0x85,
		0x2D, 0x3F, 0x85, 0xA5, 0xCF, 0x28, 0xDA, 0x1A, 0xB6, 0xFE, 0x34, 0x38, 0x17, 0x19, 0x78, 0x46,
		0x7F, 0x1C, 0x05, 0xD5, 0x8C, 0x7E, 0xF3, 0x8C, 0x28, 0x4C, 0x41, 0xF6, 0xC2, 0x22, 0x1A, 0x76,
		0xF1, 0x2A, 0xB1, 0xC0, 0x40, 0x82, 0x66, 0x02, 0x50, 0x80, 0x22, 0x94, 0xFB, 0x87, 0x18, 0x02,
		0x13, 0xFD, 0xEF, 0x5B, 0x0E, 0xCB, 0x7D, 0xF5, 0x0C, 0xA1, 0xF8, 0x55, 0x5B, 0xE1, 0x4D, 0x32,
		0xE1, 0x0F, 0x6E, 0xDC, 0xDE, 0x89, 0x2C, 0x09, 0x42, 0x4B, 0x29, 0xF5, 0x97, 0xAF, 0xC2, 0x70,
		0xC9, 0x04, 0x55, 0x6B, 0xFC, 0xB4, 0x7A, 0x7D, 0x40, 0x77, 0x8D, 0x39, 0x09, 0x23, 0x64, 0x2B,
		0x3C, 0xBD, 0x05, 0x79, 0xE6, 0x09, 0x08, 0xD5, 0xA0, 0x00, 0xC1, 0xD0, 0x8B, 0x98, 0xEF, 0x93,
		0x3F, 0x80, 0x64, 0x45, 0xBF, 0x87, 0xF8, 0xB0, 0x09, 0xBA, 0x9E, 0x94, 0xF7, 0x26, 0x61, 0x22,
		0xED, 0x7A, 0xC2, 0x4E, 0x5E, 0x26, 0x6C, 0x42, 0xA8, 0x2F, 0xA1, 0xBB, 0xEF, 0xB7, 0xB8, 0xDB,
		0x00, 0x66, 0xE1, 0x6A, 0x85, 0xE0, 0x49, 0x3F, 0x07, 0xDF, 0x48, 0x09, 0xAE, 0xC0, 0x84, 0xA5,
		0x93, 0x74, 0x8A, 0xC3, 0xDD, 0xE5, 0xA6, 0xD7, 0xAA, 0xE1, 0xE8, 0xB6, 0xE5, 0x35, 0x2B, 0x2D,
		0x71, 0xEF, 0xBB, 0x47, 0xD4, 0xCA, 0xEE, 0xD5, 0xE6, 0xD6, 0x33, 0x80, 0x5D, 0x2D, 0x32, 0x3E,
		0x6F, 0xD8, 0x1B, 0x46, 0x84, 0xB9, 0x3A, 0x26, 0x77, 0xD4, 0x5E, 0x74, 0x21, 0xC2, 0xC6, 0xAE,
		0xA2, 0x59, 0xB8, 0x55, 0xA6, 0x98, 0xFD, 0x7D, 0x13, 0x47, 0x7A, 0x1F, 0xE5, 0x3E, 0x5A, 0x4A,
		0x61, 0x97, 0xDB, 0xEC, 0x5C, 0xE9, 0x5F, 0x50, 0x5B, 0x52, 0x0B, 0xCD, 0x95, 0x70, 0xC4, 0xA8,
		0x26, 0x5A, 0x7E, 0x01, 0xF8, 0x9C, 0x0C, 0x00, 0x2C, 0x59, 0xBF, 0xEC, 0x6C, 0xD4, 0xA5, 0xC1,
		0x09, 0x25, 0x89, 0x53, 0xEE, 0x5E, 0xE7, 0x0C, 0xD5, 0x77, 0xEE, 0x21, 0x7A, 0xF2, 0x1F, 0xA7,
		0x01, 0x78, 0xF0, 0x94, 0x6C, 0x9B, 0xF6, 0xCA, 0x87, 0x51, 0x79, 0x34, 0x79, 0xF6, 0xB5, 0x37,
		0x73, 0x7E, 0x40, 0xB6, 0xED, 0x28, 0x51, 0x1D, 0x8A, 0x2D, 0x7E, 0x73, 0xEB, 0x75, 0xF8, 0xDA,
		0xAC, 0x91, 0x2F, 0xF9, 0x06, 0xE0, 0xAB, 0x95, 0x5B, 0x08, 0x3B, 0xAC, 0x45, 0xA8, 0xE5, 0xE9,
		0xB7, 0x44, 0xC8, 0x50, 0x6F, 0x37, 0xE9, 0xB4, 0xE7, 0x49, 0xA1, 0x84, 0xB3, 0x0F, 0x43, 0xEB,
		0x18, 0x8D, 0x85, 0x5F, 0x1B, 0x70, 0xD7, 0x1F, 0xF3, 0xE5, 0x0C, 0x53, 0x7A, 0xC1, 0xB0, 0xF8,
		0x97, 0x4F, 0x0F, 0xE1, 0xA6, 0xAD, 0x29, 0x5B, 0xA4, 0x2F, 0x6A, 0xEC, 0x74, 0xD1, 0x23, 0xA7,
		0xAB, 0xED, 0xDE, 0x6E, 0x2C, 0x07, 0x11, 0xCA, 0xB3, 0x6B, 0xE5, 0xAC, 0xB1, 0xA5, 0xA1, 0x1A,
		0x4B, 0x1D, 0xB0, 0x8B, 0xA6, 0x98, 0x2E, 0xFC, 0xCD, 0x71, 0x69, 0x29, 0xA7, 0x74, 0x1C, 0xFC,
		0x63, 0xAA, 0x44, 0x35, 0xE0, 0xB6, 0x9A, 0x90, 0x63, 0xE8, 0x80, 0x79, 0x5C, 0x3D, 0xC5, 0xEF,
		0x32, 0x72, 0xE1, 0x1C, 0x49, 0x7A, 0x91, 0xAC, 0xF6, 0x99, 0xFE, 0xFE, 0xE2, 0x06, 0x22, 0x7A,
		0x44, 0xC9, 0xFB, 0x35, 0x9F, 0xD5, 0x6A, 0xC0, 0xA9, 0xA7, 0x5A, 0x74, 0x3C, 0xFF, 0x68, 0x62,
		0xF1, 0x7D, 0x72, 0x59, 0xAB, 0x07, 0x52, 0x16, 0xC0, 0x69, 0x95, 0x11, 0x64, 0x3B, 0x64, 0x39
	};

	/* test compact api */

	clear8(output0, 512);
	clear8(output1, 512);
	clear8(output2, 512);
	clear8(output3, 512);
	OQS_SHA3_shake128_x4(output0, output1, output2, output3, 512,
	                     msg0, msg0, msg0, msg0, 0);

	if (are_equal8(output0, exp0, 512) == EXIT_FAILURE
	        || are_equal8(output1, exp0, 512) == EXIT_FAILURE
	        || are_equal8(output2, exp0, 512) == EXIT_FAILURE
	        || are_equal8(output3, exp0, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(output0, 512);
	clear8(output1, 512);
	clear8(output2, 512);
	clear8(output3, 512);
	OQS_SHA3_shake128_x4(output0, output1, output2, output3, 512,
	                     msg1600, msg1600, msg1600, msg1600, 200);

	if (are_equal8(output0, exp1600, 512) == EXIT_FAILURE
	        || are_equal8(output1, exp1600, 512) == EXIT_FAILURE
	        || are_equal8(output2, exp1600, 512) == EXIT_FAILURE
	        || are_equal8(output3, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* test incremental api */
	clear8(output0, 512);
	clear8(output1, 512);
	clear8(output2, 512);
	clear8(output3, 512);
	OQS_SHA3_shake128_x4_inc_ctx state;
	OQS_SHA3_shake128_x4_inc_init(&state);
	OQS_SHA3_shake128_x4_inc_absorb(&state, msg1600, msg1600, msg1600, msg1600, 32);
	OQS_SHA3_shake128_x4_inc_absorb(&state, msg1600 + 32, msg1600 + 32, msg1600 + 32, msg1600 + 32, 168);
	OQS_SHA3_shake128_x4_inc_finalize(&state);
	OQS_SHA3_shake128_x4_inc_squeeze(output0, output1, output2, output3, 32, &state);
	OQS_SHA3_shake128_x4_inc_squeeze(output0 + 32, output1 + 32, output2 + 32, output3 + 32, 480, &state);
	OQS_SHA3_shake128_x4_inc_ctx_release(&state);

	if (are_equal8(output0, exp1600, 512) == EXIT_FAILURE
	        || are_equal8(output1, exp1600, 512) == EXIT_FAILURE
	        || are_equal8(output2, exp1600, 512) == EXIT_FAILURE
	        || are_equal8(output3, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	return status;
}

/**
* \brief Tests the four way parallel version of the SHAKE-256 XOF function for correct operation,
* using selected vectors from the NIST reference package.
*
* \return Returns 1 for success, 0 for failure
*
* \remarks <b>Test References:</b> \n
* KAT: <a href="https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHAKE128_Msg0.pdf">SHAKE128(0)</a> \n
* KAT: <a href="https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHAKE128_Msg1600.pdf">SHAKE128(1600)</a>
*/
int shake_256_x4_kat_test(void) {
	int status = EXIT_SUCCESS;

	uint8_t output0[512];
	uint8_t output1[512];
	uint8_t output2[512];
	uint8_t output3[512];

	uint8_t msg0[1] = {0x0};
	uint8_t msg1600[200] = {
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
		0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3
	};

	uint8_t exp0[512] = {
		0x46, 0xB9, 0xDD, 0x2B, 0x0B, 0xA8, 0x8D, 0x13, 0x23, 0x3B, 0x3F, 0xEB, 0x74, 0x3E, 0xEB, 0x24,
		0x3F, 0xCD, 0x52, 0xEA, 0x62, 0xB8, 0x1B, 0x82, 0xB5, 0x0C, 0x27, 0x64, 0x6E, 0xD5, 0x76, 0x2F,
		0xD7, 0x5D, 0xC4, 0xDD, 0xD8, 0xC0, 0xF2, 0x00, 0xCB, 0x05, 0x01, 0x9D, 0x67, 0xB5, 0x92, 0xF6,
		0xFC, 0x82, 0x1C, 0x49, 0x47, 0x9A, 0xB4, 0x86, 0x40, 0x29, 0x2E, 0xAC, 0xB3, 0xB7, 0xC4, 0xBE,
		0x14, 0x1E, 0x96, 0x61, 0x6F, 0xB1, 0x39, 0x57, 0x69, 0x2C, 0xC7, 0xED, 0xD0, 0xB4, 0x5A, 0xE3,
		0xDC, 0x07, 0x22, 0x3C, 0x8E, 0x92, 0x93, 0x7B, 0xEF, 0x84, 0xBC, 0x0E, 0xAB, 0x86, 0x28, 0x53,
		0x34, 0x9E, 0xC7, 0x55, 0x46, 0xF5, 0x8F, 0xB7, 0xC2, 0x77, 0x5C, 0x38, 0x46, 0x2C, 0x50, 0x10,
		0xD8, 0x46, 0xC1, 0x85, 0xC1, 0x51, 0x11, 0xE5, 0x95, 0x52, 0x2A, 0x6B, 0xCD, 0x16, 0xCF, 0x86,
		0xF3, 0xD1, 0x22, 0x10, 0x9E, 0x3B, 0x1F, 0xDD, 0x94, 0x3B, 0x6A, 0xEC, 0x46, 0x8A, 0x2D, 0x62,
		0x1A, 0x7C, 0x06, 0xC6, 0xA9, 0x57, 0xC6, 0x2B, 0x54, 0xDA, 0xFC, 0x3B, 0xE8, 0x75, 0x67, 0xD6,
		0x77, 0x23, 0x13, 0x95, 0xF6, 0x14, 0x72, 0x93, 0xB6, 0x8C, 0xEA, 0xB7, 0xA9, 0xE0, 0xC5, 0x8D,
		0x86, 0x4E, 0x8E, 0xFD, 0xE4, 0xE1, 0xB9, 0xA4, 0x6C, 0xBE, 0x85, 0x47, 0x13, 0x67, 0x2F, 0x5C,
		0xAA, 0xAE, 0x31, 0x4E, 0xD9, 0x08, 0x3D, 0xAB, 0x4B, 0x09, 0x9F, 0x8E, 0x30, 0x0F, 0x01, 0xB8,
		0x65, 0x0F, 0x1F, 0x4B, 0x1D, 0x8F, 0xCF, 0x3F, 0x3C, 0xB5, 0x3F, 0xB8, 0xE9, 0xEB, 0x2E, 0xA2,
		0x03, 0xBD, 0xC9, 0x70, 0xF5, 0x0A, 0xE5, 0x54, 0x28, 0xA9, 0x1F, 0x7F, 0x53, 0xAC, 0x26, 0x6B,
		0x28, 0x41, 0x9C, 0x37, 0x78, 0xA1, 0x5F, 0xD2, 0x48, 0xD3, 0x39, 0xED, 0xE7, 0x85, 0xFB, 0x7F,
		0x5A, 0x1A, 0xAA, 0x96, 0xD3, 0x13, 0xEA, 0xCC, 0x89, 0x09, 0x36, 0xC1, 0x73, 0xCD, 0xCD, 0x0F,
		0xAB, 0x88, 0x2C, 0x45, 0x75, 0x5F, 0xEB, 0x3A, 0xED, 0x96, 0xD4, 0x77, 0xFF, 0x96, 0x39, 0x0B,
		0xF9, 0xA6, 0x6D, 0x13, 0x68, 0xB2, 0x08, 0xE2, 0x1F, 0x7C, 0x10, 0xD0, 0x4A, 0x3D, 0xBD, 0x4E,
		0x36, 0x06, 0x33, 0xE5, 0xDB, 0x4B, 0x60, 0x26, 0x01, 0xC1, 0x4C, 0xEA, 0x73, 0x7D, 0xB3, 0xDC,
		0xF7, 0x22, 0x63, 0x2C, 0xC7, 0x78, 0x51, 0xCB, 0xDD, 0xE2, 0xAA, 0xF0, 0xA3, 0x3A, 0x07, 0xB3,
		0x73, 0x44, 0x5D, 0xF4, 0x90, 0xCC, 0x8F, 0xC1, 0xE4, 0x16, 0x0F, 0xF1, 0x18, 0x37, 0x8F, 0x11,
		0xF0, 0x47, 0x7D, 0xE0, 0x55, 0xA8, 0x1A, 0x9E, 0xDA, 0x57, 0xA4, 0xA2, 0xCF, 0xB0, 0xC8, 0x39,
		0x29, 0xD3, 0x10, 0x91, 0x2F, 0x72, 0x9E, 0xC6, 0xCF, 0xA3, 0x6C, 0x6A, 0xC6, 0xA7, 0x58, 0x37,
		0x14, 0x30, 0x45, 0xD7, 0x91, 0xCC, 0x85, 0xEF, 0xF5, 0xB2, 0x19, 0x32, 0xF2, 0x38, 0x61, 0xBC,
		0xF2, 0x3A, 0x52, 0xB5, 0xDA, 0x67, 0xEA, 0xF7, 0xBA, 0xAE, 0x0F, 0x5F, 0xB1, 0x36, 0x9D, 0xB7,
		0x8F, 0x3A, 0xC4, 0x5F, 0x8C, 0x4A, 0xC5, 0x67, 0x1D, 0x85, 0x73, 0x5C, 0xDD, 0xDB, 0x09, 0xD2,
		0xB1, 0xE3, 0x4A, 0x1F, 0xC0, 0x66, 0xFF, 0x4A, 0x16, 0x2C, 0xB2, 0x63, 0xD6, 0x54, 0x12, 0x74,
		0xAE, 0x2F, 0xCC, 0x86, 0x5F, 0x61, 0x8A, 0xBE, 0x27, 0xC1, 0x24, 0xCD, 0x8B, 0x07, 0x4C, 0xCD,
		0x51, 0x63, 0x01, 0xB9, 0x18, 0x75, 0x82, 0x4D, 0x09, 0x95, 0x8F, 0x34, 0x1E, 0xF2, 0x74, 0xBD,
		0xAB, 0x0B, 0xAE, 0x31, 0x63, 0x39, 0x89, 0x43, 0x04, 0xE3, 0x58, 0x77, 0xB0, 0xC2, 0x8A, 0x9B,
		0x1F, 0xD1, 0x66, 0xC7, 0x96, 0xB9, 0xCC, 0x25, 0x8A, 0x06, 0x4A, 0x8F, 0x57, 0xE2, 0x7F, 0x2A
	};
	uint8_t exp1600[512] = {
		0xCD, 0x8A, 0x92, 0x0E, 0xD1, 0x41, 0xAA, 0x04, 0x07, 0xA2, 0x2D, 0x59, 0x28, 0x86, 0x52, 0xE9,
		0xD9, 0xF1, 0xA7, 0xEE, 0x0C, 0x1E, 0x7C, 0x1C, 0xA6, 0x99, 0x42, 0x4D, 0xA8, 0x4A, 0x90, 0x4D,
		0x2D, 0x70, 0x0C, 0xAA, 0xE7, 0x39, 0x6E, 0xCE, 0x96, 0x60, 0x44, 0x40, 0x57, 0x7D, 0xA4, 0xF3,
		0xAA, 0x22, 0xAE, 0xB8, 0x85, 0x7F, 0x96, 0x1C, 0x4C, 0xD8, 0xE0, 0x6F, 0x0A, 0xE6, 0x61, 0x0B,
		0x10, 0x48, 0xA7, 0xF6, 0x4E, 0x10, 0x74, 0xCD, 0x62, 0x9E, 0x85, 0xAD, 0x75, 0x66, 0x04, 0x8E,
		0xFC, 0x4F, 0xB5, 0x00, 0xB4, 0x86, 0xA3, 0x30, 0x9A, 0x8F, 0x26, 0x72, 0x4C, 0x0E, 0xD6, 0x28,
		0x00, 0x1A, 0x10, 0x99, 0x42, 0x24, 0x68, 0xDE, 0x72, 0x6F, 0x10, 0x61, 0xD9, 0x9E, 0xB9, 0xE9,
		0x36, 0x04, 0xD5, 0xAA, 0x74, 0x67, 0xD4, 0xB1, 0xBD, 0x64, 0x84, 0x58, 0x2A, 0x38, 0x43, 0x17,
		0xD7, 0xF4, 0x7D, 0x75, 0x0B, 0x8F, 0x54, 0x99, 0x51, 0x2B, 0xB8, 0x5A, 0x22, 0x6C, 0x42, 0x43,
		0x55, 0x6E, 0x69, 0x6F, 0x6B, 0xD0, 0x72, 0xC5, 0xAA, 0x2D, 0x9B, 0x69, 0x73, 0x02, 0x44, 0xB5,
		0x68, 0x53, 0xD1, 0x69, 0x70, 0xAD, 0x81, 0x7E, 0x21, 0x3E, 0x47, 0x06, 0x18, 0x17, 0x80, 0x01,
		0xC9, 0xFB, 0x56, 0xC5, 0x4F, 0xEF, 0xA5, 0xFE, 0xE6, 0x7D, 0x2D, 0xA5, 0x24, 0xBB, 0x3B, 0x0B,
		0x61, 0xEF, 0x0E, 0x91, 0x14, 0xA9, 0x2C, 0xDB, 0xB6, 0xCC, 0xCB, 0x98, 0x61, 0x5C, 0xFE, 0x76,
		0xE3, 0x51, 0x0D, 0xD8, 0x8D, 0x1C, 0xC2, 0x8F, 0xF9, 0x92, 0x87, 0x51, 0x2F, 0x24, 0xBF, 0xAF,
		0xA1, 0xA7, 0x68, 0x77, 0xB6, 0xF3, 0x71, 0x98, 0xE3, 0xA6, 0x41, 0xC6, 0x8A, 0x7C, 0x42, 0xD4,
		0x5F, 0xA7, 0xAC, 0xC1, 0x0D, 0xAE, 0x5F, 0x3C, 0xEF, 0xB7, 0xB7, 0x35, 0xF1, 0x2D, 0x4E, 0x58,
		0x9F, 0x7A, 0x45, 0x6E, 0x78, 0xC0, 0xF5, 0xE4, 0xC4, 0x47, 0x1F, 0xFF, 0xA5, 0xE4, 0xFA, 0x05,
		0x14, 0xAE, 0x97, 0x4D, 0x8C, 0x26, 0x48, 0x51, 0x3B, 0x5D, 0xB4, 0x94, 0xCE, 0xA8, 0x47, 0x15,
		0x6D, 0x27, 0x7A, 0xD0, 0xE1, 0x41, 0xC2, 0x4C, 0x78, 0x39, 0x06, 0x4C, 0xD0, 0x88, 0x51, 0xBC,
		0x2E, 0x7C, 0xA1, 0x09, 0xFD, 0x4E, 0x25, 0x1C, 0x35, 0xBB, 0x0A, 0x04, 0xFB, 0x05, 0xB3, 0x64,
		0xFF, 0x8C, 0x4D, 0x8B, 0x59, 0xBC, 0x30, 0x3E, 0x25, 0x32, 0x8C, 0x09, 0xA8, 0x82, 0xE9, 0x52,
		0x51, 0x8E, 0x1A, 0x8A, 0xE0, 0xFF, 0x26, 0x5D, 0x61, 0xC4, 0x65, 0x89, 0x69, 0x73, 0xD7, 0x49,
		0x04, 0x99, 0xDC, 0x63, 0x9F, 0xB8, 0x50, 0x2B, 0x39, 0x45, 0x67, 0x91, 0xB1, 0xB6, 0xEC, 0x5B,
		0xCC, 0x5D, 0x9A, 0xC3, 0x6A, 0x6D, 0xF6, 0x22, 0xA0, 0x70, 0xD4, 0x3F, 0xED, 0x78, 0x1F, 0x5F,
		0x14, 0x9F, 0x7B, 0x62, 0x67, 0x5E, 0x7D, 0x1A, 0x4D, 0x6D, 0xEC, 0x48, 0xC1, 0xC7, 0x16, 0x45,
		0x86, 0xEA, 0xE0, 0x6A, 0x51, 0x20, 0x8C, 0x0B, 0x79, 0x12, 0x44, 0xD3, 0x07, 0x72, 0x65, 0x05,
		0xC3, 0xAD, 0x4B, 0x26, 0xB6, 0x82, 0x23, 0x77, 0x25, 0x7A, 0xA1, 0x52, 0x03, 0x75, 0x60, 0xA7,
		0x39, 0x71, 0x4A, 0x3C, 0xA7, 0x9B, 0xD6, 0x05, 0x54, 0x7C, 0x9B, 0x78, 0xDD, 0x1F, 0x59, 0x6F,
		0x2D, 0x4F, 0x17, 0x91, 0xBC, 0x68, 0x9A, 0x0E, 0x9B, 0x79, 0x9A, 0x37, 0x33, 0x9C, 0x04, 0x27,
		0x57, 0x33, 0x74, 0x01, 0x43, 0xEF, 0x5D, 0x2B, 0x58, 0xB9, 0x6A, 0x36, 0x3D, 0x4E, 0x08, 0x07,
		0x6A, 0x1A, 0x9D, 0x78, 0x46, 0x43, 0x6E, 0x4D, 0xCA, 0x57, 0x28, 0xB6, 0xF7, 0x60, 0xEE, 0xF0,
		0xCA, 0x92, 0xBF, 0x0B, 0xE5, 0x61, 0x5E, 0x96, 0x95, 0x9D, 0x76, 0x71, 0x97, 0xA0, 0xBE, 0xEB
	};

	/* test compact api */

	clear8(output0, 512);
	clear8(output1, 512);
	clear8(output2, 512);
	clear8(output3, 512);
	OQS_SHA3_shake256_x4(output0, output1, output2, output3, 512,
	                     msg0, msg0, msg0, msg0, 0);

	if (are_equal8(output0, exp0, 512) == EXIT_FAILURE
	        || are_equal8(output1, exp0, 512) == EXIT_FAILURE
	        || are_equal8(output2, exp0, 512) == EXIT_FAILURE
	        || are_equal8(output3, exp0, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	clear8(output0, 512);
	clear8(output1, 512);
	clear8(output2, 512);
	clear8(output3, 512);
	OQS_SHA3_shake256_x4(output0, output1, output2, output3, 512,
	                     msg1600, msg1600, msg1600, msg1600, 200);

	if (are_equal8(output0, exp1600, 512) == EXIT_FAILURE
	        || are_equal8(output1, exp1600, 512) == EXIT_FAILURE
	        || are_equal8(output2, exp1600, 512) == EXIT_FAILURE
	        || are_equal8(output3, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	/* test incremental api */
	clear8(output0, 512);
	clear8(output1, 512);
	clear8(output2, 512);
	clear8(output3, 512);
	OQS_SHA3_shake256_x4_inc_ctx state;
	OQS_SHA3_shake256_x4_inc_init(&state);
	OQS_SHA3_shake256_x4_inc_absorb(&state, msg1600, msg1600, msg1600, msg1600, 64);
	OQS_SHA3_shake256_x4_inc_absorb(&state, msg1600 + 64, msg1600 + 64, msg1600 + 64, msg1600 + 64, 136);
	OQS_SHA3_shake256_x4_inc_finalize(&state);
	OQS_SHA3_shake256_x4_inc_squeeze(output0, output1, output2, output3, 64, &state);
	OQS_SHA3_shake256_x4_inc_squeeze(output0 + 64, output1 + 64, output2 + 64, output3 + 64, 448, &state);
	OQS_SHA3_shake256_x4_inc_ctx_release(&state);

	if (are_equal8(output0, exp1600, 512) == EXIT_FAILURE
	        || are_equal8(output1, exp1600, 512) == EXIT_FAILURE
	        || are_equal8(output2, exp1600, 512) == EXIT_FAILURE
	        || are_equal8(output3, exp1600, 512) == EXIT_FAILURE) {
		status = EXIT_FAILURE;
	}

	return status;
}

/**
* \brief Run the SHA3 and SHAKE KAT tests
*/
int main(UNUSED int argc, UNUSED char **argv) {
	int ret = EXIT_SUCCESS;

	print_system_info();

	if (sha3_256_kat_test() == EXIT_SUCCESS) {
		printf("Success! passed sha3-256 known answer tests \n");
	} else {
		printf("Failure! failed sha3-256 known answer tests \n");
		ret = EXIT_FAILURE;
	}

	if (sha3_512_kat_test() == EXIT_SUCCESS) {
		printf("Success! passed sha3-512 known answer tests \n");
	} else {
		printf("Failure! failed sha3-512 known answer tests \n");
		ret = EXIT_FAILURE;
	}

	if (shake_128_kat_test() == EXIT_SUCCESS) {
		printf("Success! passed shake-128 known answer tests \n");
	} else {
		printf("Failure! failed shake-128 known answer tests \n");
		ret = EXIT_FAILURE;
	}

	if (shake_256_kat_test() == EXIT_SUCCESS) {
		printf("Success! passed shake-256 known answer tests \n");
	} else {
		printf("Failure! failed shake-256 known answer tests \n");
		ret = EXIT_FAILURE;
	}

	if (shake_128_x4_kat_test() == EXIT_SUCCESS) {
		printf("Success! passed four-way parallel shake-128 known answer tests \n");
	} else {
		printf("Failure! failed four-way parallel shake-128 known answer tests \n");
		ret = EXIT_FAILURE;
	}

	if (shake_256_x4_kat_test() == EXIT_SUCCESS) {
		printf("Success! passed four-way parallel shake-256 known answer tests \n");
	} else {
		printf("Failure! failed four-way parallel shake-256 known answer tests \n");
		ret = EXIT_FAILURE;
	}


	return ret;
}
